/*
 * Copyright (C) 2001 Thomas Roessler <roessler@does-not-exist.org>
 * 
 *     This program is free software; you can redistribute it
 *     and/or modify it under the terms of the GNU General Public
 *     License as published by the Free Software Foundation; either
 *     version 2 of the License, or (at your option) any later
 *     version.
 * 
 *     This program is distributed in the hope that it will be
 *     useful, but WITHOUT ANY WARRANTY; without even the implied
 *     warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *     PURPOSE.  See the GNU General Public License for more
 *     details.
 * 
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 59 Temple Place - Suite 330,
 *     Boston, MA  02111, USA.
 */ 

#include "global.h"

#include <sys/types.h>
#include <dirent.h>

static int check_flag (const char *, char);
static char *flags (char *);
static void sync_one_dir (const char *, int *);
static int sync_one_file (const char *, const char *);

void pm_sync_maildir (const char *path)
{
  char dir_new[_POSIX_PATH_MAX];
  char dir_cur[_POSIX_PATH_MAX];
  
  int count = 0;
  
  snprintf (dir_new, sizeof (dir_new), "%s/new", path);
  snprintf (dir_cur, sizeof (dir_cur), "%s/cur", path);

  sync_one_dir (dir_new, &count);
  sync_one_dir (dir_cur, &count);
  
}

static void sync_one_dir (const char *path, int *countp)
{
  DIR *dirp;
  struct dirent *dep;
  char *flp;

  char fn1[_POSIX_PATH_MAX];
  char fn2[_POSIX_PATH_MAX];
  
  if ((dirp = opendir (path)) == NULL)
  {
    pm_perror ("opendir");
    pm_exit (1);
  }
  
  while ((dep = readdir (dirp)))
  {
    if (*(dep->d_name) == '.')
      continue;

    printf ("Uploading message %d... ", ++(*countp)); fflush (stdout);
    flp = flags (dep->d_name);
    
    if (check_flag (flp, 'T'))
    {
      puts ("skipping");
      continue;
    }
    
    if (sync_one_file (path, dep->d_name) == 0)
    {
      puts ("done");
      snprintf (fn1, sizeof (fn1), "%s/%s", path, dep->d_name);
      snprintf (fn2, sizeof (fn2), "%s/%sT", path, dep->d_name);
      safe_rename (fn1, fn2);
    }
    else
    {
      int oe = errno;
      puts ("failed");
      errno = oe;
      perror ("sync_one_file");
      pm_exit (1);
    }
  }
}

static int sync_one_file (const char *path, const char *fname)
{
  char fn[_POSIX_PATH_MAX];
  char line[LONG_STRING];
  char buff[BUFFLEN];
  FILE *fp;
  
  struct Mail mail;
  int len, rv;
  
  char *t;
  
  memset (&mail, 0, sizeof (mail));
  
  snprintf (fn, sizeof (fn), "%s/%s", path, fname);
  if (!(fp = fopen (fn, "r")))
    return -1;
  
  while (fgets (line, sizeof (line), fp) != NULL)
  {
    if (*line == '\n')		/* end of header */
      break;
    if (ISSPACE (*line))	/* ignore continuation lines */
      continue;
    if ((t = strchr (line, '\n')))
      *t = '\0';
    if ((t = strchr (line, ':')))
    {
      t++;
      SKIPWS (t);
    }
    if (!strncasecmp (line, "To:", 3) && t)
      str_replace (&mail.to, t);
    else if (!strncasecmp (line, "From:", 5) && t)
      str_replace (&mail.from, t);
    else if (!strncasecmp (line, "Cc:", 3) && t)
      str_replace (&mail.cc, t);
    else if (!strncasecmp (line, "Subject:", 8) && t)
      str_replace (&mail.subject, t);
    else if (!strncasecmp (line, "Reply-To:", 9) && t)
      str_replace (&mail.replyTo, t);
    else if (!strncasecmp (line, "Date:", 5) && t)
    {
      time_t d;
      struct tm *dtm;
      
      if ((d = pm_parse_date (t)) != (time_t) -1)
      {
	dtm = localtime (&d);
	mail.date = *dtm;
	mail.dated = 1;
      }
    }
  }
  
  mail.body = safe_calloc (1, Context.sync_pref->truncate);
  len = fread (mail.body, 1, Context.sync_pref->truncate - 1, fp);
  
  if (!feof (fp))
  {
    printf ("(truncated to %d) ", len);
    fflush (stdout);
  }
  
  len = pack_Mail (&mail, buff, sizeof (buff));
  rv = dlp_WriteRecord (Context.sd, Context.db, 0, 0, 0, buff, len, 0);
  
  free_Mail (&mail);
  
  return rv > 0 ? 0 : -1;
}

static int check_flag (const char *flags, char c)
{
  if (flags && strchr (flags, c))
    return 1;
  return 0;
}

static char *flags (char *fname)
{
  char *p = strrchr (fname, ':');
  if (p) 
    return ++p;
  return NULL;
}
