/*
 * Copyright (C) 1995 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.ai.mit.edu.
 *
 * etl_util.c
 * 
 * Description:
 *  Simple functions such as strcasecmp do not exist on all systems.
 * These are wholesale replacements with different names.
 *
 * $Log: etl_util.c,v $
 * Revision 1.4  1995/04/10  21:43:01  zappo
 * Replace GNU version of strtok with my own since theirs was too
 * dependant on thier own library.  Both strdup and strtok will be used
 * only if they don't exist in the system.
 *
 * Revision 1.3  1995/04/08  20:03:49  zappo
 * Added strdup and strtok if not provided by default.
 *
 * Revision 1.2  1995/04/05  00:55:03  zappo
 * Made scmp take const input
 *
 * Revision 1.1  1995/03/30  02:33:32  zappo
 * Initial revision
 *
 *
 * Tokens: ::Header:: etalklib.h
 */
#include "etalklib.h"


/*
 * Function: ETL_scmp
 *
 * Compares s1 and s2 for len characters.  Case is ignored.  Success
 * is returned when they are the same.
 *
 * Returns:     int - Success means they are the same
 * Parameters:  s1  - String 1
 *              s2  - String 2
 *              len - length of string
 * History:
 * eml Jul 30, 1994   Created 
 * zappo   3/29/95    Moved to a generalized function, and cleaned it up.
 */
int ETL_scmp(s1, s2, len)
     const char *s1, *s2;
     int   len;
{
  int i;
  for(i = 0; (i<len) && s1[i] && s2[i]; i++)
    {
      char test1, test2;

      if((s1[i] >= 'A') && (s1[i] <= 'Z'))
	test1 = s1[i] - 'A' + 'a';
      else
	test1 = s1[i];

      if((s2[i] >= 'A') && (s2[i] <= 'Z'))
	test2 = s2[i] - 'A' + 'a';
      else
	test2 = s2[i];

      if(test1 != test2)	/* not equal */
	return Fail;
    }
  if(i == len)
    return Success;		/* all match */
  else
    return Fail;		/* to short */
}

#ifndef HAVE_STRDUP

/*
 * Function: strdup
 *
 *   Return a malloced version of the parameter string.  This is
 * supplied if your system doesn't have a strdup of it's own.
 *
 * Returns:     char * - 
 * Parameters:  str - Pointer toCharacter of string
 *
 * History:
 * zappo   4/4/95     Created
 */
char *strdup(__s)
     const char *__s;
{
  char *tmp;

  /* get some space */
  tmp = (char *)malloc(strlen(__s));

  /* check before copy */
  if(tmp) strcpy(tmp, __s);

  /* return the null if that is what happened */
  return tmp;
}
#endif /* HAVE_STRDUP */
#ifndef HAVE_STRTOK

/*
 * Function: strtok
 *
 *   Emulate strtok found in most libraries.  It is not done
 * efficiently, but that's ok.  It's only used during commandline
 * parsing anyways
 *
 * Returns:     char * - 
 * Parameters:  s     - String to tokenize
 *              delim - token seperator set
 * History:
 * zappo   4/8/95     Created
 */
char *strtok(__s, __delim)
     char *__s;
     const char *__delim;
{
  static char *lastmatch = NULL;
  char *end;

  /* initial cases/errors */
  if(!__s) __s = lastmatch;
  if(!__s || !__delim) return NULL;

  /* first scan over first batch of delimiters */
  while(*__s && strchr(__delim, *__s)) __s++;

  /* now scan until we hit the next delimiter */
  end = __s;
  while(*end && !strchr(__delim, *end)) end++;

  /* terminate the token */
  *end = '\0';
  /* prepare for next scan */
  lastmatch = ++end;
  /* return the result */
  if(*__s)
    return __s;
  else
    return NULL;
}
#endif /* HAVE_STRTOK */
