/* Copyright 1990 GROUPE BULL -- See licence conditions in file COPYRIGHT */
/* create.c:
 *
 *  XPM
 *  Create utility for XPM file format
 *  Developed by Arnaud Le Hors
 */

#include "xpmP.h"

char *ColorKeys[] = {
    "s",			/* key #1: symbol */
    "m",			/* key #2: mono visual */
    "g4",			/* key #3: 4 grays visual */
    "g",			/* key #4: gray visual */
    "c",			/* key #5: color visual */
};

#undef RETURN
#define RETURN(status) \
  { freeColorTable(colorTable, ncolors); \
    if (chars) free(chars); \
    if (pixelindex) free(pixelindex); \
    if (gc) XFreeGC(display, gc); \
    if (ximage) XDestroyImage(ximage); \
    if (pixels) free(pixels); \
    if (infos) XFreeXpmInfo(infos); \
    return(status); }

CreatePixmap(display, visual, d, colormap, data, depth,
	     pixmap_return, width_return, height_return, 
	     pixels_return, npixels_return,
	     colorsymbols, numsymbols, infos)
     Display *display;
     Visual *visual;
     Drawable d;
     Colormap colormap;
     MData *data;
     unsigned int depth;
     Pixmap *pixmap_return;
     unsigned int *width_return, *height_return;
     Pixel **pixels_return;
     unsigned int *npixels_return;
     ColorSymbol colorsymbols[];
     unsigned int numsymbols;
     XpmInfo *infos;

{ Pixmap       pixmap;
  unsigned int cpp;			/* chars per pixel */
  unsigned int ncolors = 0;		/* number of colors */
  unsigned int rncolors = 0;		/* read number of colors */
  char       ***colorTable = NULL;	/* color table */
  unsigned int key;			/* color key */
  Pixel        *pixels = NULL;		/* pixels colors */
  unsigned int width, height;
  char         *chars = NULL, buf[BUFSIZ], *colorname;
  unsigned int *iptr, *pixelindex = NULL;
  XColor       xcolor;
  XGCValues    gcv;
  GC           gc = NULL;
  XImage       *ximage = NULL;
  unsigned int a, b, x, y, l;
  Boolean pixel_defined, defaultCase;

  /*
   * parse the data
   */

  /* read hints: width, height, ncolors, chars_per_pixel
   */
  if (! ((width = mnextui(data)) && (height = mnextui(data))
	 && (rncolors = mnextui(data)) && (cpp =  mnextui(data))))
      RETURN(PixmapFileInvalid);

  ncolors = rncolors;

  /* store the hints comment line
   */
  if (infos) mgetcmt(data, &infos->hints_cmt);

  /* read colors
   */
  colorTable = (char ***)calloc(ncolors, sizeof(char **));
  if (!colorTable) RETURN(PixmapNoMemory);
  for(a = 0; a < ncolors; a++) {
      mnextstring(data);		/* skip the line */
      colorTable[a] = (char **)calloc((NKEYS + 1), sizeof(char *));
      if (!colorTable[a]) RETURN(PixmapNoMemory);
      /* read pixel value
       */
      *colorTable[a] = (char *)malloc(cpp);
      if (!*colorTable[a]) RETURN(PixmapNoMemory);
      for (b = 0; b < cpp; b++)
	  colorTable[a][0][b] = mgetc(data);

      /* read color keys and values
       */
      while (l = mnextw(data, buf)) {
	  for (key = 1; key < NKEYS + 1; key++)
	      if (! strncmp(ColorKeys[key - 1], buf, l))
		  break;
	  if (key <= NKEYS && (l = mnextw(data, buf))) {
	      colorTable[a][key] = (char *)malloc(l + 1);
	      if (!colorTable[a][key])  RETURN(PixmapNoMemory);
	      strncpy(colorTable[a][key], buf, l);
	      colorTable[a][key][l] = '\0';
	  } else
	      RETURN(PixmapFileInvalid); /* key without value */
      }      
  }

  /* store the colors comment line
   */
  if (infos) mgetcmt(data, &infos->colors_cmt);

  /* read pixels and index them on color number
   */
  pixelindex = (unsigned int *)malloc(sizeof(unsigned int)*width*height);
  if (!pixelindex) RETURN(PixmapNoMemory);
  iptr = pixelindex;
  chars = (char *)malloc(cpp);
  if (!chars) RETURN(PixmapNoMemory);
  for (y = 0; y < height; y++) {
      mnextstring(data);
      for (x = 0; x < width; x++, iptr++) {
	  for (a = 0; a < cpp; a++)
	      chars[a] = mgetc(data);
	  for (a = 0; a < ncolors; a++)
	      if (!strncmp(colorTable[a][0], chars, cpp))
		  break;
	  if (a == ncolors) RETURN(PixmapFileInvalid); /* no color matches */
	  *iptr = a;
      }
  }

  /* store the pixels comment line
   */
  if (infos) mgetcmt(data, &infos->pixels_cmt);

  /* 
   * build the image data
   */

  key = visualType(visual);
  pixels = (Pixel *)malloc(sizeof(Pixel) * ncolors);
  if (!pixels) RETURN(PixmapNoMemory);

  /* get pixel colors and index them
   */
  for (a = 0; a < ncolors; a++) {
      colorname = NULL;
      pixel_defined = False;

      /* look for a defined symbol
       */
      if (numsymbols && colorTable[a][1]) {
	  for (l = 0; l < numsymbols; l++)
	      if (!strcmp(colorsymbols[l].name, colorTable[a][1]))
		  break;
	  if (l != numsymbols) {
	      if (colorsymbols[l].value)
		  colorname = colorsymbols[l].value;
	      else
		  pixel_defined = True;
	  }
      }
      if (! pixel_defined) {
	  if (! colorname) {
	      if (colorTable[a][key])
		  b = key;
	      else {
		  for (b = key - 1; b > 1; b--)
		      if (colorTable[a][b])
			  break;
		  if (b == 1) {
		      for (b = key + 1; b < NKEYS + 1; b++)
			  if (colorTable[a][b])
			      break;
		      if (b == NKEYS + 1)
			  RETURN(PixmapFileInvalid);
		  }
	      }
	      colorname = colorTable[a][b];
	  }
	  if (! XParseColor(display, colormap, colorname, &xcolor))
	      RETURN(PixmapParseColorFailed);
	  if (XAllocColor(display, colormap, &xcolor)) {
/* too bad XAllocColorCells doesn't work on some visuals like monochrom....
	      if (infos) {
		  for (b = 0; b < a; b++)
		      if (pixels[b] == xcolor.pixel)
			  break;
		  if (b != a) {
		      if (! XAllocColorCells(display, colormap, False, NULL, 0,
					     pixels + a, 1))
			  RETURN(PixmapAllocColorFailed);
		      XFreeColors(display, colormap, &(xcolor.pixel), 1, 0);
		      xcolor.pixel = pixels[a];
		      XStoreColor(display, colormap, &xcolor);
		  }
	      }
*/
	  } else
	      RETURN(PixmapAllocColorFailed);
	  pixels[a] = xcolor.pixel;
      } else
	  pixels[a] = colorsymbols[l].pixel;
  }

  /*
   * send the image to the server
   *
   * some of the algorithms implemented here come from the CreateXImage 
   * function which is part of the xv-pl3 code written by
   * John Bradley, University of Pennsylvania (bradley@cis.upenn.edu)
   *
   * In case depth is 1, 4, 6, 8, 24 or 32 build the XImage data,
   * otherwise build it by slowly but surely setting every pixel 
   * with XPutPixel.
   */

  pixmap = XCreatePixmap(display, d, width, height, depth);
  gcv.function = GXcopy;
  gc = XCreateGC(display, pixmap, GCFunction, &gcv);
  defaultCase = False;
  switch (depth) {
  case 8:
      {
	  byte *imagedata, *ip;
	  imagedata = (byte *) malloc(width*height);
	  if (!imagedata) RETURN(PixmapNoMemory);
	  ip = imagedata; iptr = pixelindex;
	  for (y = 0; y < height; y++)
	      for (x = 0; x < width; x++, iptr++)
		  *ip++ = (byte) pixels[*iptr];
	  ximage = XCreateImage(display, visual, depth, ZPixmap, 0, 
				(char *)imagedata, width, height, 8, 0);
	  break;
      }
    case 1:
      {
	  byte  *imagedata, *destline, *destptr, destmask;
	  int  bperline;
	  ximage = XCreateImage(display, visual, depth, XYPixmap, 0, 
				NULL, width, height, 8, 0);
	  bperline = ximage->bytes_per_line;
	  imagedata = (byte *) malloc(bperline * height);
	  if (!imagedata) RETURN(PixmapNoMemory);
	  ximage->data = (char *) imagedata;
	  iptr = pixelindex; destline = imagedata;
	  if (ximage->bitmap_bit_order == MSBFirst)
	      for (y = 0; y < height; y++, destline += bperline) {
		  destmask = 0x80; destptr = destline;
		  for (x = 0; x < width; x++, iptr++) {
		      if (pixels[*iptr] & 1)
			  *destptr |= destmask;
		      else
			  *destptr &= ~destmask;
		      if (!(destmask >>= 1)) {
			  destmask = 0x80;
			  destptr++;
		      }
		  }
	      }
	  else
	      for (y = 0; y < height; y++, destline += bperline) {
		  destmask = 1; destptr = destline;
		  for (x = 0; x < width; x++, iptr++) {
		      if (pixels[*iptr] & 1)
			  *destptr |= destmask;
		      else
			  *destptr &= ~destmask;
		      if (!(destmask <<= 1)) {
			  destmask = 1;
			  destptr++;
		      }
		  }
	      }
	  break;
      }
    case 4:
      {
	  byte  *imagedata, *ip, *lip;
	  int  bperline, half;
	  ximage = XCreateImage(display, visual, depth, ZPixmap, 0, 
				NULL, width, height, 8, 0);
	  bperline = ximage->bytes_per_line;
	  imagedata = (byte *) malloc(bperline * height);
	  if (!imagedata) RETURN(PixmapNoMemory);
	  ximage->data = (char *) imagedata;
	  if (ximage->bits_per_pixel == 4) {
	      iptr = pixelindex; lip = imagedata;
	      for (y = 0 ; y < height; y++, lip += bperline)
		  for (x = 0, ip = lip, half = 0; 
		       x < width; x++, iptr++, half++)
		      if (half & 1)
			  *ip++ |= (pixels[*iptr] & 0x0f)<<4;
		      else
			  *ip = pixels[*iptr] & 0x0f;
	  } else if (ximage->bits_per_pixel == 8) {
	      iptr = pixelindex; ip = imagedata;
	      for (y = 0 ; y < height; y++)
		  for (x = 0; x < width; x++, iptr++)
		      *ip++ = (byte) pixels[*iptr];
	  } else {
	      free(imagedata);
	      XDestroyImage(ximage);
	      defaultCase = True;
	  }
	  break;
      }
  case 6:
      {
	  byte *imagedata, *ip;
	  ximage = XCreateImage(display, visual, depth, ZPixmap, 0, 
				NULL, width, height, 8, 0);
	  if (ximage->bits_per_pixel == 8) {
	      imagedata = (byte *) malloc(ximage->bytes_per_line*height);
	      if (!imagedata) RETURN(PixmapNoMemory);
	      ximage->data = (char *)imagedata;
	      ip = imagedata; iptr = pixelindex;
	      for (y = 0; y < height; y++)
		  for (x = 0; x < width; x++, iptr++)
		      *ip++ = (byte) pixels[*iptr];
	  } else {
	      XDestroyImage(ximage);
	      defaultCase = True;
	  }
	  break;
      }
  case 24:
  case 32:
      {
	  byte *imagedata, *ip;
	  imagedata = (byte *) malloc(4*width*height);
	  if (!imagedata) RETURN(PixmapNoMemory);
	  ximage = XCreateImage(display, visual, depth, ZPixmap, 0, 
				(char *)imagedata, width, height, 32, 0);
	  ip = imagedata; iptr = pixelindex;
	  if (ximage->byte_order == MSBFirst)
	      for (y = 0; y < height; y++)
		  for (x = 0; x < width; x++, iptr++) {
		      *ip++ = 0;
		      *ip++ = (pixels[*iptr]>>16) & 0xff;
		      *ip++ = (pixels[*iptr]>>8) & 0xff;
		      *ip++ =  pixels[*iptr] & 0xff;
		  }
	  else
	      for (y = 0; y < height; y++)
		  for (x = 0; x < width; x++, iptr++) {
		      *ip++ =  pixels[*iptr] & 0xff;
		      *ip++ = (pixels[*iptr]>>8) & 0xff;
		      *ip++ = (pixels[*iptr]>>16) & 0xff;
		      *ip++ = 0;
		  }
	  break;
      }
  default:
      defaultCase = True;
  }
  if (defaultCase) {
      ximage = XGetImage(display, pixmap, 0, 0,
			 width, height, AllPlanes, ZPixmap);
      if (!ximage) RETURN(PixmapNoMemory);
      iptr = pixelindex;
      for (y = 0; y < height; y++)
	  for (x = 0; x < width; x++, iptr++)
	      XPutPixel(ximage, x, y, pixels[*iptr]);
  }

  XPutImage(display, pixmap, gc, ximage, 0, 0, 0, 0, width, height);

  free(pixelindex);
  XDestroyImage(ximage);
  XFreeGC(display, gc);
  free(chars);

  /* store color table infos
   */
  if (infos) {
      infos->ncolors = ncolors;
      infos->colorTable = colorTable;
      if (pixels_return) {
	  infos->pixels = (Pixel *)malloc(sizeof(Pixel) * ncolors);
	  bcopy(pixels, infos->pixels, sizeof(Pixel) * ncolors);
      } else {
	  infos->pixels = pixels;
	  pixels = NULL;
      }
  } else
      freeColorTable(colorTable, ncolors);

  if (pixmap_return) *pixmap_return = pixmap;
  if (width_return) *width_return = width;
  if (height_return) *height_return = height;
  if (pixels_return) *pixels_return = pixels; else if (pixels) free(pixels);
  if (npixels_return) *npixels_return = ncolors;

  return(PixmapSuccess);
}
