/* 
 * MonSocksClient.c --
 *
 *      Implements the interface from Socks5 server to Monitor.
 *
 * Copyright(C) 1995, 1996 NEC Corporation. All rights reserved.
 * This is a genuine component of Socks5Toolkit(TM).
 *
 */

#include "config.h"
#include "log.h"

#ifdef  MONITOR
#include "monAPI.h"
#include "monSocks.h"
#include "monHand.h"
#include "monMsgIds.h"
#include "monInt.h"

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif /* HAVE_PTHREAD_H */

/*
 * Thread-specific information maintained for every Socks5 server process
 * or thread. Handle returned by S5SocksMonOpen() contains a pointer to this.
 *
 * We combine the 32-bit process ID with the 32-bit thread handle/ID
 * and 16-bit connection number for a unique 80-bit key.
 */

#define MON_UDP_CACHE_SIZE 20

typedef struct {
    int             ipcType;      /* IPC type = Shared memory, Pipe */
    S5MonHandle     ipcHandle;    /* Opaque handle for IPC */
    pid_t           pid;          /* Process ID */
    Uint32          threadInfo;   /* Thread handle/ID */
    Uint16          connNum;      /* Connection number 1..65536 */
    Uint32          inTotBytes;   /* Last known in/out total byte counts */
    Uint32          outTotBytes;
                                  /* Cache data recs per destination address */
                                  /* Used when UDP has multiple destinations */
    int             iCache;       /* Current index into cache */
    S5MonData       data[MON_UDP_CACHE_SIZE];
} MonSocksInfo_t;

/*
 *----------------------------------------------------------------------
 *
 * S5SocksMonOpen --
 *
 *      Opens data transfer channel from Socks5 server to Monitor.
 *
 * Results:
 *      Returns 0 on success; -1 on failure and sets errno.
 *      Returns opaque handle in *handlePtr.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
S5SocksMonOpen(ipcName, ipcType, handlePtr)
    CONST char FAR*     ipcName;    /* IN  -- Name of IPC */
    int                 ipcType;    /* IN  -- Type of IPC */
    S5MonHandle FAR*    handlePtr;  /* OUT -- Pointer to handle returned */
{
    MonHand_t           hand;
    MonSocksInfo_t FAR* infoPtr;
    static Uint32       threadNum = 0; /* Fail-safe way of generating unique keys */
                                       /* even if configured pthread wrong. */

    *handlePtr = NULL;

    /*
     * Allocate space for the struct pointed to by the handle.
     */

    if (!(hand = malloc(sizeof(*hand)))
    ||  !(infoPtr = malloc(sizeof(*infoPtr)))) {
        S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_MON_HANDLE_MALLOC,
                    "malloc() handle failed");
        if(hand) free(hand);
        return -1;
    }
    memset(hand, 0, sizeof(*hand));
    hand->flag = MON_HAND_DEAD;

    memset(infoPtr, 0, sizeof(*infoPtr));
    infoPtr->ipcType = ipcType;
    infoPtr->pid = getpid();

#ifdef USE_THREADS

#ifdef HAVE_PTHREAD_H
    infoPtr->threadInfo = (Uint32) pthread_self();
#else
    infoPtr->threadInfo = threadNum++;
#endif /* HAVE_PTHREAD_H */

#else
    infoPtr->threadInfo = threadNum++;
#endif /* USE_THREADS */

    hand->infoPtr = (VOID FAR*) infoPtr;
    hand->vec.xfer.openProc    = S5SocksMonOpen;
    hand->vec.xfer.closeProc   = S5SocksMonClose;
    hand->vec.xfer.putProc     = NULL;
    hand->vec.xfer.getProc     = NULL;
    hand->vec.xfer.getNextProc = NULL;
    *handlePtr = (S5MonHandle) hand;

    if (ipcType == S5_MON_MEM) {
        if (S5MonMemOpen(MON_MMAP_NAME, S5_MON_SOCKS5, &infoPtr->ipcHandle)
            == -1) {
            S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_MON_MEM_OPEN,
                        "Failed to open shared memory.");
            free(infoPtr);
            free(hand);
            return -1;
        }
    }
    else if (ipcType == S5_MON_PIPE) {
        if (S5MonFifoOpen(MON_FIFO_NAME, S5_MON_SOCKS5, &infoPtr->ipcHandle)
            == -1) {
            S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_MON_FIFO_OPEN, 
                        "failed to open FIFO");
            free(infoPtr);
            free(hand);
            return -1;
        }
    }
    hand->flag = MON_HAND_SOCKS;
    return 0;

} /* S5SocksMonOpen */

/*
 *----------------------------------------------------------------------
 *
 * S5SocksMonClose --
 *
 *      Close data transfer channel from Socks5 to Monitor.
 *
 * Results:
 *      Returns 0 on success; -1 on failure and sets errno.
 *
 * Side effects:
 *      Destroys opaque handle.
 *
 *----------------------------------------------------------------------
 */

int
S5SocksMonClose(handle)
    S5MonHandle         handle;           /* IN -- Opaque handle */
{
    MonHand_t           hand;
    MonSocksInfo_t FAR* infoPtr;

    if (!(hand = (MonHand_t) handle)
    ||  hand->flag != MON_HAND_SOCKS) {
        S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_MON_BAD_HANDLE,
                    "S5SocksMonClose: bad handle=0x%lx specified", handle);
        return -1;
    }
    infoPtr = (MonSocksInfo_t FAR*) hand->infoPtr;

    hand->flag = MON_HAND_DEAD;

    /*
     * Close IPC handle.
     */

    if (S5MonClose(infoPtr->ipcHandle) == -1) {
        S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_MON_CLOSE,
                    "Failed to close IPC");
        free(infoPtr);
        free(hand);
        return -1;
    }

    free(infoPtr);
    free(hand);
    return 0;

} /* S5SocksMonClose */

/*
 *----------------------------------------------------------------------
 *
 * S5SocksMonPut --
 *
 *      Put Socks5 link information into Monitor table.
 *
 * Results:
 *      Returns 0 on success; -1 on failure and sets errno.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
S5SocksMonPut(handle, state, linkPtr, optDataLen, optDataPtr,
              socksErr, socksErrMsg)
    S5MonHandle         handle;      /* IN -- Opaque handle */
    unsigned int        state;       /* IN -- State of Socks5 link */
    S5LinkInfo FAR*     linkPtr;     /* IN -- Pointer to Socks5 link info */
    unsigned int        optDataLen;  /* IN -- Length of optional data */
    CONST VOID FAR*     optDataPtr;  /* IN -- Pointer to optional data */
    int                 socksErr;    /* IN -- Socks5 error */
    CONST char FAR*     socksErrMsg; /* IN -- Socks5 error message */
{
    MonHand_t           hand;
    MonSocksInfo_t FAR* infoPtr;
    int                 i = 0;
    Uint32              inCurBytes = 0;
    Uint32              outCurBytes = 0;
    static CONST S5NetAddr  noNetAddr;  /* Net address with all zeroes */

    if (!(hand = (MonHand_t) handle)
    ||  hand->flag != MON_HAND_SOCKS) {
        S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_MON_BAD_HANDLE, 
            "S5SocksMonPut: bad handle=0x%lx specified", handle);
        return -1;
    }
    infoPtr = (MonSocksInfo_t FAR*) hand->infoPtr;

    /*
     * New Socks5 link started, clear the cache.
     * generate a new key, set the start timestamp.
     */

    if (state & S5_LINK_START) {
        for (i = 0; i <= infoPtr->iCache; i++) {
            memset(&infoPtr->data[i], 0, sizeof(S5MonData));
        } /* for i */

        i = 0;
        infoPtr->iCache = 0;
        infoPtr->inTotBytes = infoPtr->outTotBytes = 0;

        infoPtr->data[0].startTime = infoPtr->data[i].curTime = time(NULL);

        PUT_PID_IN_KEY(infoPtr->pid, infoPtr->data[0].key);
        PUT_THREAD_IN_KEY(infoPtr->threadInfo, infoPtr->data[0].key);
        infoPtr->connNum++;
        PUT_CONN_IN_KEY(infoPtr->connNum, infoPtr->data[0].key);
    }

    /*
     * Determine the current byte transfer counts.
     */

    inCurBytes  = linkPtr->inbc  - infoPtr->inTotBytes;
    outCurBytes = linkPtr->outbc - infoPtr->outTotBytes;

    infoPtr->outTotBytes = linkPtr->outbc;
    infoPtr->inTotBytes  = linkPtr->inbc;

    /* 
     * If Socks command is UDP, check if destination address is old. 
     * If new, create a new [sub]connection entry in the cache.
     * i.e., generate a new key.
     */

    if (state & S5_LINK_MULTIPLE) {

        for (i = infoPtr->iCache; i >= 0; i--) {
            if (memcmp(&linkPtr->dstAddr, &infoPtr->data[i].dstSockAddr,
                sizeof(S5NetAddr)) == 0) {
                break;
            }
        } /* for i */

        if (i < 0) {
            if ((i = ++(infoPtr->iCache)) >= MON_UDP_CACHE_SIZE) {
                S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, 
                    MSGID_MON_TOO_MANY_UDP,
                    "Too many UDP destinations=%d - cache size=%d overflow",
                     i, MON_UDP_CACHE_SIZE);
                return -1;
            }
            state |= S5_LINK_START;
            infoPtr->data[i] = infoPtr->data[0];
            infoPtr->data[i].inTotBytes = infoPtr->data[i].outTotBytes = 0;
            infoPtr->data[i].optData.len = 0;

            infoPtr->data[i].key = infoPtr->data[0].key;
            infoPtr->connNum++;
            PUT_CONN_IN_KEY(infoPtr->connNum, infoPtr->data[i].key);
        }
    }
    infoPtr->data[i].state = state;
    infoPtr->data[i].inTotBytes  += inCurBytes;
    infoPtr->data[i].outTotBytes += outCurBytes;

    infoPtr->data[i].socksCmd = linkPtr->peerCommand;
    infoPtr->data[i].authen   = linkPtr->peerAuth;

    /*
     * Copy non-empty name fields (null terminated strings)
     * from S5LinkInfo to appropriate fields in S5MonData record.
     */

    if (linkPtr->srcName[0]) {
        strncpy(infoPtr->data[i].srcHostName, linkPtr->srcName,
                S5_HOSTNAME_SIZE);
    }
    if (linkPtr->dstName[0]) {
        strncpy(infoPtr->data[i].dstHostName, linkPtr->dstName,
                S5_HOSTNAME_SIZE);
    }
    if (linkPtr->sckName[0]) {
        strncpy(infoPtr->data[i].proxyHostName, linkPtr->sckName,
                S5_HOSTNAME_SIZE);
    }
    if (linkPtr->srcUser[0]) {
        strncpy(infoPtr->data[i].userName, linkPtr->srcUser, S5_USERNAME_SIZE);
    }
    if (linkPtr->dstServ[0]) {
        strncpy(infoPtr->data[i].appName, linkPtr->dstServ, S5_APPNAME_SIZE);
    }

    /*
     * Copy non-null Net address fields in S5LinkInfo
     * to appropriate fields in S5MonData record.
     */

    if (memcmp(&linkPtr->srcAddr, &noNetAddr, sizeof(S5NetAddr)) != 0) {
        memcpy(&infoPtr->data[i].srcSockAddr, &linkPtr->srcAddr,
               sizeof(S5NetAddr));
    }
    if (memcmp(&linkPtr->dstAddr, &noNetAddr, sizeof(S5NetAddr)) != 0) {
        memcpy(&infoPtr->data[i].dstSockAddr, &linkPtr->dstAddr,
               sizeof(S5NetAddr));
    }
    if (memcmp(&linkPtr->sckAddr, &noNetAddr, sizeof(S5NetAddr)) != 0) {
        memcpy(&infoPtr->data[i].proxySockAddr, &linkPtr->sckAddr,
               sizeof(S5NetAddr));
    }
    if (state & S5_LINK_UPDATE
    ||  state & S5_LINK_ERROR) {
        infoPtr->data[i].curTime = time(NULL);
    }

    /*
     * Error on Socks5 link, save the error number & message.
     */

    if (state & S5_LINK_ERROR) {
        infoPtr->data[i].socksErr = socksErr;
        if (socksErrMsg && socksErrMsg[0]) {
            strncpy(infoPtr->data[i].socksErrMsg, socksErrMsg, S5_ERRMSG_SIZE);
        }
    }

    /* 
     * If a non-zero, valid optional data length is specified,
     * copy the optional user data into the Socks5 data record.
     */

    if (optDataPtr && optDataLen > 0) {
        infoPtr->data[i].optData.len = optDataLen;
        memcpy(infoPtr->data[i].optData.data, optDataPtr,
               optDataLen <= S5_OPTDATA_SIZE ? optDataLen : S5_OPTDATA_SIZE);
    }

    /*
     * Socks5 link terminated, set the end timestamp,
     * put all associated UDP entries in cache into Monitor;
     * after linking them into a circular list using the `nextKey' field.
     */

    if (state & S5_LINK_END) {
        infoPtr->data[0].endTime = time(NULL);

        for (i = 0; i <= infoPtr->iCache; i++) {
            infoPtr->data[i].state |= S5_LINK_END;
            infoPtr->data[i].endTime = infoPtr->data[0].endTime;
            if (i < infoPtr->iCache)
                infoPtr->data[i].nextKey = infoPtr->data[i + 1].key;

            if (S5MonSendData(infoPtr->ipcHandle, &infoPtr->data[i]) == -1) {
                break;
            }
        } /* for i */

        if (infoPtr->iCache)
            infoPtr->data[infoPtr->iCache].nextKey = infoPtr->data[0].key;
        infoPtr->iCache = 0;
        return 0;
    }
    else
        return S5MonSendData(infoPtr->ipcHandle, &infoPtr->data[i]);

} /* S5SocksMonPut */
#endif
/* monSocksClient.c */
