/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1993.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XFTP shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XFTP for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <sys/time.h>
#include <sys/param.h>
#include <ctype.h>
#include <Xm/Xm.h>
#include "xftp.h"
#include "xfer.h"

struct xfer_ctrl_block xfer_ctrl;

int init_local_copy();
int do_local_copy();
int abort_local_copy();

int init_put();
int do_put();
int abort_put();

int init_get();
int do_get();
int abort_get();

int init_remote_copy();
int do_remote_copy();
int abort_remote_copy();

int (*init_fn[2][2])() = {
	{ init_local_copy,  init_put         },
	{ init_get,         init_remote_copy }
};

int (*do_fn[2][2])() = {
	{ do_local_copy,    do_put         },
	{ do_get,           do_remote_copy }
};

int (*abort_fn[2][2])() = {
	{ abort_local_copy, abort_put         },
	{ abort_get,        abort_remote_copy }
};

static struct timeval start_time;
static struct timeval stop_time;

extern struct st_host_info hinfo[];
extern int beep_when_ops_done;
extern int diagnostics;
extern int max_ftp_retries;

char *strrchr();


/*
 * start_timer - Starts the microsecond timer.  Needs to be called at
 *               start of interval to be timed.  Call stop_timer() at
 *               end of interval.
 *
 */
start_timer()
{
    gettimeofday(&start_time, NULL);
}


/*
 * stop_timer - Returns the number of microseconds that have elapsed
 *              since start_timer() was last called.  Undefined results
 *              if start_timer() is not called first.
 */
double
stop_timer()
{
    double fstart_time;
    double fstop_time;

    gettimeofday(&stop_time, NULL);
    fstart_time = (double)start_time.tv_sec+(double)start_time.tv_usec/1000000.;
    fstop_time = (double)stop_time.tv_sec+(double)stop_time.tv_usec/1000000.;
    return fstop_time-fstart_time;
}


/*
 * print_xfer_stats - Prints statistics about a file transfer.  "ftime"
 *                    is the number of seconds the transfer took.  "nbytes"
 *                    is the number of bytes transferred.  Set "nbytes"
 *                    to -1 if file length not known.
 */
print_xfer_stats(ftime, nbytes)
double ftime;
long nbytes;
{
	char msg[120];

	write_log("      ");
	if (nbytes == -1)
		write_log("?");
	else {
		sprintf(msg, "%d", nbytes);
		write_log(msg);
	}
	sprintf(msg, " bytes transferred in %3.1lf seconds", ftime);
	write_log(msg);
	if (nbytes != -1) {
		sprintf(msg, " (%3.1lf Kbytes/s)", (double)nbytes/(ftime*1000.));
		write_log(msg);
	}
	write_log("\n");
}


/*
 * clean_up_xfer_ctrl - Cleans up the control structure used by the file
 *                      transfer work proc.
 */
clean_up_xfer_ctrl()
{
	int i;
	struct entry_link *ptr;

	if (xfer_ctrl.rel_path != NULL) {
		XtFree(xfer_ctrl.rel_path);
		xfer_ctrl.rel_path = NULL;
	}

	if (xfer_ctrl.src_path != NULL) {
		XtFree(xfer_ctrl.src_path);
		xfer_ctrl.src_path = NULL;
	}

	if (xfer_ctrl.snk_path != NULL) {
		XtFree(xfer_ctrl.snk_path);
		xfer_ctrl.snk_path = NULL;
	}

	for(i=0; i<=xfer_ctrl.level; i++)
		while ((ptr = xfer_ctrl.head[i]) != NULL) {
			xfer_ctrl.head[i] = xfer_ctrl.head[i]->next;
			XtFree(ptr->entry);
			XtFree((char *)ptr);
		}
}


/*
 * cb_xfer_files - Work proc for transferring files.
 */
cb_xfer_files()
{
	double ftime;
	struct entry_link *ptr;
	int src_host = xfer_ctrl.src_host;
	int snk_host = xfer_ctrl.snk_host;
	int src_host_type = hinfo[xfer_ctrl.src_host].type;
	int snk_host_type = hinfo[xfer_ctrl.snk_host].type;
	struct sl_struct *list;
	int retval;
	int lost_src_host = False;
	int lost_snk_host = False;
	int len;
	int i;
	int is_a_directory;
	char msg[MAXPATHLEN+40];

	switch (xfer_ctrl.state) {
	case 0:   /* State 0: No transfer in progress */
		if (xfer_ctrl.abort_requested)
			goto done;
		xfer_ctrl.src_path = NULL;
		xfer_ctrl.snk_path = NULL;
		xfer_ctrl.rel_path = NULL;
		if (xfer_ctrl.head[xfer_ctrl.level] == NULL) {
			if (xfer_ctrl.level == 0)
				goto done;
			else {
				xfer_ctrl.level--;
				ptr = xfer_ctrl.head[xfer_ctrl.level];
				xfer_ctrl.head[xfer_ctrl.level] = ptr->next;
				XtFree(ptr->entry);
				XtFree((char *)ptr);
				return False;
			}
		}
		/* Build relative path name */
		len = 1;
		for (i=0; i<=xfer_ctrl.level; i++)
			len += strlen(xfer_ctrl.head[i]->entry)+1;
		xfer_ctrl.rel_path = XtMalloc(len);
		strcpy(xfer_ctrl.rel_path, xfer_ctrl.head[0]->entry);
		for (i=1; i<=xfer_ctrl.level; i++) {
			strcat(xfer_ctrl.rel_path, "/");
			strcat(xfer_ctrl.rel_path, xfer_ctrl.head[i]->entry);
		}

		/* Build source path name */
		len = strlen(hinfo[src_host].wd)+strlen(xfer_ctrl.rel_path)+2;
		xfer_ctrl.src_path = XtMalloc(len);
		strcpy(xfer_ctrl.src_path, hinfo[src_host].wd);
		strcat(xfer_ctrl.src_path, "/");
		strcat(xfer_ctrl.src_path, xfer_ctrl.rel_path);

		/* Build sink path name */
		len = strlen(hinfo[snk_host].wd)+strlen(xfer_ctrl.rel_path)+2;
		xfer_ctrl.snk_path = XtMalloc(len);
		strcpy(xfer_ctrl.snk_path, hinfo[snk_host].wd);
		strcat(xfer_ctrl.snk_path, "/");
		strcat(xfer_ctrl.snk_path, xfer_ctrl.rel_path);

		if (xfer_ctrl.recursive) {
			/* Test for directory by trying to cd to source path */
			is_a_directory = False;
			if (src_host_type == LOCAL) {
				if (local_cd(src_host, xfer_ctrl.src_path, False) == 0)
					is_a_directory = True;
			} else
				switch (remote_cd(src_host, xfer_ctrl.src_path, True, False)) {
				case -3:
					lost_src_host = True;
					goto lost;
				case 0:
					is_a_directory = True;
				}
			if (is_a_directory) {
				/* Make sink directory */
				if (snk_host_type == LOCAL) {
					if (local_mkdir(snk_host, xfer_ctrl.snk_path) != 0) {
						warning_error("Unable to make directory");
						goto done;
					}
				} else
					switch (remote_mkdir(snk_host, xfer_ctrl.snk_path)) {
					case -3:
						lost_snk_host = True;
						goto lost;
					case -1:
						sprintf(msg, "Unable to make directory %s",
									 xfer_ctrl.snk_path);
						warning_error(msg);
						goto done;
					}
				if (diagnostics >= NORMAL) {
					sprintf(msg, "*** Successfully created directory: %s\n",
							xfer_ctrl.rel_path);
					write_log(msg);
				}
				/* Get source directory list */
				if (src_host_type == LOCAL) {
					if (local_ls(xfer_ctrl.src_path, &list, False) != 0) {
						warning_error("Unable to get source directory list");
						goto done;
					}
				} else {
					switch (remote_ls(src_host, xfer_ctrl.src_path, &list,
						False)) {
					case -3:
						lost_src_host = True;
						goto lost;
					case -1:
						warning_error("Unable to get source directory list");
						goto done;
					}
				}
				/* Add source directory list to new level */
				xfer_ctrl.level++;
				if (xfer_ctrl.level == MAXLEVELS) {
					warning_error(
						"File transfer recurses too deeply - aborted.");
					goto done;
				}
				xfer_ctrl.head[xfer_ctrl.level] = NULL;
				for (i=list->nentries-1; i>=0; i--) {
					ptr = XtNew(struct entry_link);
					ptr->entry = XtNewString(list->entries[i]);
					strip_off_symbol(ptr->entry);
					ptr->next = xfer_ctrl.head[xfer_ctrl.level];
					xfer_ctrl.head[xfer_ctrl.level] = ptr;
				}
				release_string_list(list);
				return False;
			}
		}

		/* Initialize file transfer */
		initialize_monitor1();
		retval = (*init_fn[src_host_type][snk_host_type])();
		switch (retval) {
		case -5:
			lost_src_host = True;
			lost_snk_host = True;
			goto lost;
		case -4:
			lost_snk_host = True;
			goto lost;
		case -3:
			lost_src_host = True;
			goto lost;
		case -1:
			sprintf(msg, "Unable to initialize transfer of %s",
						 xfer_ctrl.head[xfer_ctrl.level]->entry);
			warning_error(msg);
			xfer_ctrl.state = 0;
			xfer_ctrl.nretries = 0;
			ptr = xfer_ctrl.head[xfer_ctrl.level];
			xfer_ctrl.head[xfer_ctrl.level] = ptr->next;
			XtFree(xfer_ctrl.src_path);
			XtFree(xfer_ctrl.snk_path);
			XtFree(xfer_ctrl.rel_path);
			XtFree(ptr->entry);
			XtFree((char *)ptr);
			return False;
		case 0:
			initialize_monitor2();
			start_timer();
			xfer_ctrl.state = 1;
		}

	case 1:   /* State 1: transfer in progress */
		if (xfer_ctrl.abort_requested)
			goto abort;
		retval = (*do_fn[src_host_type][snk_host_type])();
		switch (retval) {
		case -5:
			lost_src_host = True;
			lost_snk_host = True;
			goto lost;
		case -4:
			lost_snk_host = True;
			goto lost;
		case -3:
			lost_src_host = True;
			goto lost;
		case -6:    /* LLNL Archive retry logic */
			if (xfer_ctrl.nretries++ < max_ftp_retries)  {
				xfer_ctrl.state = 0;
				sprintf(msg, "*** Transfer of %s failed.  Will retry.\n",
					xfer_ctrl.head[xfer_ctrl.level]->entry);
				write_log(msg);
				return False;
			}
		case -1:
			sprintf(msg, "Unable to complete transfer of %s",
						 xfer_ctrl.head[xfer_ctrl.level]->entry);
			warning_error(msg);
			ptr = xfer_ctrl.head[xfer_ctrl.level];
			xfer_ctrl.head[xfer_ctrl.level] = ptr->next;
			XtFree(xfer_ctrl.src_path);
			XtFree(xfer_ctrl.snk_path);
			XtFree(xfer_ctrl.rel_path);
			XtFree(ptr->entry);
			XtFree((char *)ptr);
			xfer_ctrl.state = 0;
			xfer_ctrl.nretries = 0;
			return False;
		case 0:   /* Transfer complete */
			update_monitor();
			ftime = stop_timer();
			if (diagnostics >= NORMAL) {
				sprintf(msg, "*** Successfully transferred file:  %s\n",
						xfer_ctrl.rel_path);
				write_log(msg);
				print_xfer_stats(ftime, xfer_ctrl.file_len);
			}
			ptr = xfer_ctrl.head[xfer_ctrl.level];
			xfer_ctrl.head[xfer_ctrl.level] = ptr->next;
			XtFree(xfer_ctrl.src_path);
			XtFree(xfer_ctrl.snk_path);
			XtFree(xfer_ctrl.rel_path);
			XtFree(ptr->entry);
			XtFree((char *)ptr);
			xfer_ctrl.state = 0;
			xfer_ctrl.nretries = 0;
			return False;
		case 1:   /* More to transfer */
			update_monitor();
			return False;
		}
	}

done:
	
	clean_up_xfer_ctrl();
	hide_monitor();
	enable_controls(True);
	if (update_dir_displays(snk_host, False) < 0) {
		lost_snk_host = True;
		goto lost;
	}
	clear_selected_entries(src_host);
	update_xfer_controls();
	update_host_controls(src_host);
	if (beep_when_ops_done)
		beep();
	restore_prev_cursor();
	if (xfer_ctrl.abort_requested) {
		xfer_ctrl.abort_requested = False;
		hide_abort_dialog();
		warning_error("File transfer(s) aborted");
	}
	return True;

abort:

	retval = (*abort_fn[src_host_type][snk_host_type])();
	switch (retval) {
	case -5:
		lost_src_host = True;
		lost_snk_host = True;
		goto lost;
	case -4:
		lost_snk_host = True;
		goto lost;
	case -3:
		lost_src_host = True;
		goto lost;
	default:
		clean_up_xfer_ctrl();
		enable_controls(True);
		if (update_dir_displays(snk_host, False) < 0) {
			lost_snk_host = True;
			goto lost;
		}
		clear_selected_entries(src_host);
		update_xfer_controls();
		update_host_controls(src_host);
		if (beep_when_ops_done)
			beep();
		restore_prev_cursor();
		xfer_ctrl.abort_requested = False;
		hide_abort_dialog();
		warning_error("File transfer(s) aborted");
		return True;
	}

lost:

	clean_up_xfer_ctrl();
	hide_monitor();
	enable_controls(True);
	clear_selected_entries(src_host);
	if (xfer_ctrl.abort_requested) {
		xfer_ctrl.abort_requested = False;
		hide_abort_dialog();
	}
	warning_error("File transfer(s) aborted");
	if (lost_src_host) {
		restore_prev_cursor();
		lost_connection(src_host);
		use_busy_cursor();
	}
	if (lost_snk_host) {
		restore_prev_cursor();
		lost_connection(snk_host);
		use_busy_cursor();
	} else {
		if (update_dir_displays(snk_host, False) < 0) {
			restore_prev_cursor();
			lost_connection(snk_host);
			use_busy_cursor();
		}
	}
	update_xfer_controls();
	update_host_controls(src_host);
	if (beep_when_ops_done)
		beep();
	restore_prev_cursor();
	return True;
}


/*
 * extract_file_len - Extracts the file length usually returned in a "150"
 *                    ftp reply.  This routine looks for a pattern of the
 *                    form "(xxxxxx bytes)" or "(xxxxxx)" near the end of
 *                    the reply, where the "xxxxxx" represent digits.  If
 *                    this pattern is found, the file length is returned,
 *                    otherwise -1 is returned.  "reply" is the "150"
 *                    ftp reply to be scanned.
 */
long
extract_file_len(reply)
char *reply;
{
    static char *pat1 = " bytes)";
    int lenpat1 = strlen(pat1);
	static char *pat2 = ")";
	int lenpat2 = strlen(pat2);
	char *ptr;
	long len;

    if ((ptr = strrchr(reply, '(')) && isdigit(*++ptr)) {
        len = 0;
        while (*ptr && isdigit(*ptr))
            len = (len*10)+(*ptr++-'0');
        if (strlen(reply)-strlen(ptr) >= lenpat1)
            if (strncmp(ptr, pat1, lenpat1) == 0)
                return len;
        if (strlen(reply)-strlen(ptr) >= lenpat2)
            if (strncmp(ptr, pat2, lenpat2) == 0)
                return len;
    }

	return -1;
}

