/* 
 * $Id: idrp_rt_policy.c,v 1.23 1996/08/29 04:40:37 skh Exp $ 
 * Merit IDRP release 1.1 (gated 3.5.4).  Copyright (c) 1994 by Merit Network, Inc. 
 */

#include "include.h"
#include "iso.h"
#include "idrp.h"


/* idrp_rt_policy.c - 
 * 
 *  routines that interact with policy to  calculate idrp
 *  route preferences, Do Distribution (DIST), and
 *  aggregation of IDRP routes
 * 
 *  1) preference calculation -
 *     PREF(p_idrp_rt,p_idrp_pref,p_gated_pref)
 *	- do import policy on the idrp - not in service now
 *     idrp_proto_pref (proto
 *     idrp_ext_info_rt
 *     gated import function - ??
 * 
 *  2) Distribution policy for routes:
 * 	DIST(p_idrp_rt,peer)
 *
 *  3) DIST_LIST_INCL and DIST_LIST_EXCL
 *	for attribute records:
 * 
 *	DIST_ATTR(p_att,peer) 
 *
 *  4) export policy for gated
 *
 * %% distribution policy routines  	 
 * 
 *  DIST(idrpRoute,peer) - run policy on a NLRI and peer 
 *  DIST_ATTR(p_att,peer) - run policy on Attribute list
 * 
 * 7) Any outbound aggregretion policy
 *    hopefull none of this will be idrp specific
 *
 *  8)  preference match mediation -
 *	idrp_vs_proto_pref(proto,peer)
 * 	given protocols with route with same preference
 *	as idrp route, should protocols route take
 *	preference for peer
 *	- use preference table in policy structures
 *	eventually, now hard coded routine for all peers	 
 */

	 
/* preference on a route */

int
PREF(p_idrp_rt, p_idrp_pref, p_gated_pref,caller)
idrpRoute	*p_idrp_rt;
int		*p_idrp_pref;
pref_t		*p_gated_pref;
int		caller;
{
pref_t		 preference;
idrpRoute_info	idrp_route_info;
adv_entry	*p_import;
int		use;

	/* set up for import call
	 *
	 * set the import list and interface
	 * based on the protocol
	 */
 
	p_import = p_idrp_rt->peer->idrp_import;

	/* set up idrp Route options 
	 * - don't set route option
	 */ 
	
	idrp_route_info.idrp_pref = IDRP_INVALID_ROUTE_PREF; 
	idrp_route_info.p_idrp_rt = p_idrp_rt;
	idrp_route_info.route_options = 0; 

	/*
	 *   	import call has:
	 *		1) destination
	 *			 sockaddr_un from rt_table
	 *
	 *		2) * peer pointer to import list
	 * 		3) int_list - is the  idrp rdi path list policy  
	 * 
	 * 		4) gw_list - gateway list is zero
	 *
	 *
	 * 		5) preference value to be returned
	 * 		
	 *		6)  ifap - if_addr *
	 *
	 *		interface this will be going out??
	 *
	 *		7) ps_data - specific data for idrp
	 *
	 *			idrpRoute_options 
	 *	
	 *
	 *		call:
	 *	 	dst-> from idrp routes pointer to gated rt_entry
	 *		import list -> from peer
	 *		import_paths -> from peer
	 *		gw_list -> no gateway list
	 *		preference - we set this up
	 *		
	 *		ps_data  -  idrpRoute_info - options plus spot
	 *			    to return idrp preference
	 */

	if (p_idrp_rt->peer->type == IDRP_PEER_EXTERNAL) {
		preference = RTPREF_IDRP_EXT;
	} else {
		preference = RTPREF_IDRP_INT;
		}
	use = import(p_idrp_rt->p_rt->rt_dest,
			p_idrp_rt->p_rt->rt_dest_mask,
			p_import,
			idrp_import_list_any_rdi,
			idrp_import_rdpath_list,
			&preference,
			(if_addr *) NULL,
			(void_t) &idrp_route_info);

	if (!use) {
		/* check to see if this is
		 * a phase 3 change from valid to invalid
		 * - you must adjust: 
		 * 	a) routing table by reseting the gated preference in the route (by import)
		 * 	b) resetting the IDRP route : gated_pref and pref_cal
		 *      c) If it is a phase1 call - the active decision is not made
		 *          so just set NOTINSTALL flag 
		 * 
		 *	d) If it is a phase3 call - the active is set or the route would not get here  		
		 * 	      delete the route -- but *** dont*** reset the IDRP bit
		 */


		trace_tf (idrp_trace_options, TR_NORMAL,0,("PREF noinstall %s (%d,%d) phase: %d old (%d,%d)",iso_ptoa(&p_idrp_rt->nlri),preference,idrp_route_info.idrp_pref,caller,p_idrp_rt->pref_cal,p_idrp_rt->gated_pref));	
		preference = IDRP_GATED_INVALID_ROUTE_PREF;

		/* if this is being used by the 
		 * phase 3 to handle import/export changes
		 * please drop the non-imported route
		 */

		if (caller == IDRP_PREF_USE_PH3 && p_idrp_rt->p_attr->rib_id == 0)
			{

			/* this indicates a route going from
			 * useable to un-usable
			 * -- if we are working on something in the kernel 
			 * -- then release the kernel via a delete
			 */
	
			/* delete the route by gated
			 * - but hold on the IDRP bit so that
			 * - we can keep the route here
			 */

			idrp_del_rt_pref(p_idrp_rt,preference,idrp_route_info.idrp_pref);
			
			}
		if (caller == IDRP_PREF_USE_PH1 || p_idrp_rt->p_attr->rib_id != 0)
		 	{
			/* 
			 *
			 */
			trace_tf (idrp_trace_options, TR_NORMAL,0,("PREF set noinstall %s (%d,%d) Phase 1",iso_ptoa(&p_idrp_rt->nlri),preference,idrp_route_info.idrp_pref));	
			}	 
		} 
	else
		{
		/* if IDRP route preference is invalid, but marked
		 * use - set to default preference for type
		 * skhskh -- Why am I getting here
		 * skhskh -  what should be the default preference
		 * skhskh -  from import for IDRP  
		 */

		 if (idrp_route_info.idrp_pref == IDRP_INVALID_ROUTE_PREF) 
			{
			/* need to invert IDRP preference
			 * 255 - gated preference = IDRP preference
			 */ 

			if (parse_limit_check("IDRP_prefence within range", (u_long) preference,LIMIT_IDRP_PREFERENCE))
				{
				/* no preference default is lower 
				 * idrp higher is better
				 */
 
				idrp_route_info.idrp_pref = IDRP_NOPREF_DEF;
				}
			else
				{
				/* if it is within range - set it to 
				 * 255 - gated_preference
				 */

				idrp_route_info.idrp_pref = 255 - preference;
				}
			}
		/* if this is a change from invalid to valid
		 * over a new policy hop
		 * - then change the default route's install bit
		 * qos - does not install anyway
		 */ 
		 
		 if ((p_idrp_rt->gated_pref == IDRP_GATED_INVALID_ROUTE_PREF) &&
			p_idrp_rt->p_attr->rib_id == 0)
			{
		   	trace_tf (idrp_trace_options, TR_NORMAL,0,("PREF reset noinstall %s (%d,%d)",iso_ptoa(&p_idrp_rt->nlri),preference,idrp_route_info.idrp_pref));	
			/* was restricted for zero and
			 * now installed
			 * skh -- please watch the setting of idrp_pref
			 */

			BIT_RESET(p_idrp_rt->p_rt->rt_state,(RTS_NOTINSTALL|RTS_DELETE));
			p_idrp_rt->gated_pref = preference; 
			p_idrp_rt->pref_cal = idrp_route_info.idrp_pref; 
			idrp_del_rt_pref(p_idrp_rt,preference,idrp_route_info.idrp_pref);
			}

		if (BIT_TEST(p_idrp_rt->p_rt->rt_state,(RTS_NOTINSTALL)))
			{
			if (p_idrp_rt->p_attr->rib_id == 0)
				{
				BIT_RESET(p_idrp_rt->p_rt->rt_state,(RTS_NOTINSTALL));
				}
			}	
		}
 
	/* 
	 * even if import() returns FALSE, the set-up of 
	 *
	 *	idrp_route_info.idrp_pref = IDRP_INVALID_ROUTE_PREF; 
	 * 
	 *
	 * and return of
	 *
	 *	preference = IDRP_GATED_INVALID_ROUTE_PREF (255) 
	 *
	 * will make things work out fine.
	 */

	*p_idrp_pref  = (int) idrp_route_info.idrp_pref;
	*p_gated_pref =  preference; 
	return(*p_idrp_pref);
} 



int 
idrp_proto_pref(proto_rcv,peer)
u_int		proto_rcv;
idrpPeer	*peer;
{
	switch (proto_rcv)
		{
		case RTPROTO_IDRP:
		   trace_tf (idrp_trace_options, TR_NORMAL,0,("idrp_vs_proto_pref called with IDRP as protocol - peer %s", peer->name));	
		   return(FALSE);	

		case RTPROTO_BGP:
			return (TRUE);
			break;

		case RTPROTO_ISIS:
			default:
			return (FALSE);
			break;
		}
}


/* idrp_ext_info_pref (rt_entry *p_rt)
 * 		ext_info preference for idrp
 *
 */
 
int
idrp_ext_info_pref(p_rt)
rt_entry	*p_rt;
{
int	pref = 0;

	/* here's wehere we check to see if we can export this protocol 
	 * - We are currently exporting
	 *	Static - and associated the default attributes
	 *		for local node if the exterior flag is not set (IDRP_EXT_INFO_IGP)
	 *		for ext_info - if the exterior flag is set (IDRP_EXT_INFO 
	 * 
	 *	ISIS - and associating the exterior information	 
	 */ 

	switch (p_rt->rt_gwp->gw_proto)
		{
		case RTPROTO_STATIC:
			if (is_martian(p_rt->rt_dest,p_rt->rt_dest_mask))
				{
				/* not valid for this protocol) 
				 */
				return(pref);
				}
			if (p_rt->rt_state & RTS_EXTERIOR)
				{
				return (IDRP_EXT_INFO_EXT_INFO);
				}
			else
				{
				return(IDRP_EXT_INFO_IGP);
				}
			break;
			

		case RTPROTO_ISIS:
			/* IS-IS routes
			 * - need to test RTS_EXTERIOR
			 *   if set EXT_INFO route
			 *   if not set, IGP route
			 */
 
			if (p_rt->rt_state & RTS_EXTERIOR)
				{
				/* check to see that it is not
				 * my bis 
				 */
				if (!my_bis_peer(p_rt->rt_gwp))
					{
					/* not my peer - so announce the route */
					pref = IDRP_EXT_INFO_EXT_INFO;
					}
				}
			else
				{
				pref = IDRP_EXT_INFO_IGP;
				}				
			
			return(pref);
			break;

		case RTPROTO_IDRP:
			trace_tf (idrp_trace_options, TR_NORMAL,0,("Illegal protocol interactions - IDRP should not call ext_info pref"));
			return (pref);
			break;
		}
	return (pref);
}

int
DIST(p_idrp_rt,peer,type)
idrpRoute	*p_idrp_rt;
idrpPeer	*peer;
int		type;
{
  trace_tf(idrp_trace_options, TR_NORMAL,0,("DIST nlri %s (peer rt %s) peer %s type = %d",iso_ptoa(&p_idrp_rt->nlri),p_idrp_rt->peer->name ,peer->name,type)); 

	switch(type)
		{
		case IDRP_DIST_PEER_UP_DUMP:
		case IDRP_DIST_REFRESH:
			if (!DIST_ATTR(p_idrp_rt->p_attr,peer))
				{
				return(FALSE);
				}
		

		case IDRP_DIST_FLASH:
		case IDRP_DIST_NEWPOLICY_FLASH:
		case IDRP_DIST_PH1_BEST_EXT:
			/* here are simple distribution stuff
			 *
			 */

			if (peer->type == IDRP_PEER_INTERNAL)
				{
				/* don't send any internal routes
				 * to another peer
				 * - cut out redistribution here
				 * - until yes/no/maybe clause done
				 *
				 */

				if (p_idrp_rt->peer->type == IDRP_PEER_INTERNAL)
					{
					return(FALSE);
					}
				}

			if (peer == p_idrp_rt->peer)
				{
				/* Do not send routes back to peer we learned
				 * them from.  A bad idea, although BGP does it
					 */ 
				return (FALSE);
				}	
			break;

		case IDRP_DIST_LOCAL_FLASH:
			/* we want to distribute local routes to internal peers
			 * since we won't hear them in phase 1 
			 */
			if ( ! ((p_idrp_rt->peer->type == IDRP_PEER_LOCAL) && (peer->type == IDRP_PEER_INTERNAL)))
				{
				/* we only want to send to internal peers when we're
				 * using IDRP_DIST_LOCAL_FLASH
				 */
				return (FALSE);
				}
			break;
		}
	return(TRUE);
}

int
DIST_WITH(p_idrp_rt,peer,type)
idrpRoute	*p_idrp_rt;
idrpPeer	*peer;
int		type;
{
	switch(type)
		{
		case IDRP_DIST_LOCAL_FLASH:
			/* we only want to distribute w/draws of local routes to internal
			 * peers, since they will have heard all other w/draws in phase 1.
			 * Everyone else gets all w/draws.
			 */
			if ( ! ((p_idrp_rt->peer->type == IDRP_PEER_LOCAL) && (peer->type == IDRP_PEER_INTERNAL)))
			{
			trace_tf(idrp_trace_options, TR_NORMAL,0,("DIST_WITH peer %s nlri %s and not allowing DIST_LOCAL_FLASH",iso_ptoa(&p_idrp_rt->nlri),peer->name)); 
			return(FALSE);
			}
			trace_tf(idrp_trace_options, TR_NORMAL,0,("DIST_WITH nlri %s peer %s and allowing DIST_LOCAL_FLASH",iso_ptoa(&p_idrp_rt->nlri),peer->name)); 
			break;

		case IDRP_DIST_REFRESH:
			{
			trace_tf(idrp_trace_options, TR_NORMAL,0,("DIST_WITH nlri %s peer %s DIST_LOCAL_REFRESH",iso_ptoa(&p_idrp_rt->nlri),peer->name)); 
			return(FALSE);
			}	
			break;
			
				
		default:
			trace_tf(idrp_trace_options, TR_NORMAL,0,("DIST_WITH peer not DIST_LOCAL_FLASH"));
			break;
		}

	return(TRUE);
}


idrp_attribute_record *
DIST_ATTR(p_att,peer)
idrp_attribute_record	*p_att;
idrpPeer		*peer;
{
idrp_attribute_record *p_valid_att = 0;

	/* attribute list will be modified for this peer
	 * check for the RDI in the DIST_LIST_INCL
	 * or DIST_LIST_EXCL parameter
	 */
	
	if (p_att->idrp_mask & IDRP_ATTBT_DIST_LIST_INCL)
		{

			
		trace_tf (idrp_trace_options, TR_POLICY,0,("DIST_ATTR called with DIST_LIST_INCL in pdu peer = %s\n",peer->name));
		if (rdi_in_dist_list(&p_att->attrib[IDRP_ATTR_DIST_LIST_INCL],&peer->rdi))
			{
			p_valid_att = p_att;
			}
		}
	else
		{
		if (p_att->idrp_mask & IDRP_ATTBT_DIST_LIST_EXCL)
			{
			trace_tf (idrp_trace_options, TR_POLICY,0,("DIST_ATTR called with DIST_list_EXCL in pdu, peer = %s\n",peer->name));
				
			if (!rdi_in_dist_list(&p_att->attrib[IDRP_ATTR_DIST_LIST_EXCL],&peer->rdi))
				{
				p_valid_att = p_att;
				}
			}
		else
			{
			p_valid_att = p_att;
			}
		} 

	/* while we are at the attribute level
	 *  work on NON-DISTINGUISHING atts
	 * 1) Multi-exit-disc (MED) 
	 * 2) CAPACITY
	 */

	/* 
	 *	 
	 * 1) change any outstanding multi-exits to the 
	 *     right value for this peer
	 *     this may be further modified by policy
	 *   - but this is the base local_info and peer statement 	
	 * 
	 */

	IDRP_STATUS_BIT_TEST(p_att->p_opts,IDRP_OPTS_MULTI_EXIT)
		{
		if (!send_multi_exit(p_att,peer))
			{
			IDRP_STATUS_BIT_SET(p_att->p_opts,IDRP_OPTS_MED_OFF);
			}	
		}
	idrp_capacity_mod(p_att,peer);
	idrp_hopcount_mod(p_att,peer);

#ifdef	IDRP_QOS
	/* here we need to modify the
	 * the QOS Distinguishing attributes of Type
	 * (TransitDelay , RDLRE, Expense, Priority)    
	 */
	if (p_att->rib_id != RIB_ID_DEFAULT)
		{
		idrp_qos_type_mods(p_att,peer);
		}	
#endif	/* IDRP_QOS */
			
	trace_tf (idrp_trace_options, TR_POLICY,0,("DIST_ATTR (%x)  returns with p_att = %x\n",p_att,p_valid_att));
			
	return (p_valid_att);
}

void
AGGR(peer,p_ann_list)
idrpPeer	*peer;
idrp_ann_list	*p_ann_list;
{
	trace_tf (idrp_trace_options, TR_NORMAL,0,("AGGR called to aggregated for peer %s ann_list %x ",peer->name,p_ann_list));
}


/* check to see if this is a new route
 */

int
idrp_new_route(p_idrp_rt)
idrpRoute       *p_idrp_rt;
{
int cnt = 0;
idrpPeer	*peer;

	IDRP_LIST(peer, idrp_peers)                        
		{
		/* if any of the routes can be found outbound */ 
		if (peer->idrp_flags == IDRPF_CONNECT)
			{
			/* only process peers that our connected 
			 * -less errors more routes
			 */

			if (p_idrp_rt->route_out[peer->id].p_next)
				{
				/* found a route - it isn't a new route
				 */
				return(FALSE);
				}


			/* increment the count of the  active peers
			 * if we hit it, quit.  Dont keep looping.
			 */

			cnt++;
			if (cnt > idrp_n_established)
				{
				return(TRUE);
				}
			}
		} IDRP_LIST_END
	return(TRUE);
}




int
idrp_dist_policy_change(p_idrp_rt)
idrpRoute	*p_idrp_rt;
{
idrpRoute_options *p_opts = (idrpRoute_options *) NULL;
boolean		was_announced = FALSE;
idrpPeer	*peer;
int		res;
adv_results	result;	
 
	/* check the distribution policy change
	 * for the route
	 * in the gated source
	 * - This may be a withdrawl or anything else
	 *
	 */


	IDRP_LIST(peer,idrp_peers)
		{	
		if ((p_idrp_rt->route_out[peer->id].route_id > 0) ||
			((p_idrp_rt->route_out[peer->id].route_id == 0) && (p_idrp_rt->route_out[peer->id].p_att != NULL))) 
			{
			was_announced = TRUE;
			p_opts = p_idrp_rt->p_export_opts;
			}

		res =  export(p_idrp_rt->p_rt,
			(proto_t) 0,
			 peer->idrp_export,
			(adv_entry *) 0,
			(adv_entry *) 0,
			&result);


		/* if the policy from a peer changes from 
		 */

		if (res == FALSE && was_announced == TRUE)
			return(TRUE);

		if (res == TRUE &&  was_announced == FALSE)
			return(TRUE); 

		/* simplistic comparison for now
		 */

		if (p_opts != p_idrp_rt->p_export_opts)
			return(TRUE);
		}

	return(FALSE);
}




int
idrp_route_export(p_idrp_rt,peer,query_type)
idrpRoute *p_idrp_rt;
idrpPeer  *peer;	
int	query_type;
{
idrp_attribute_record	*p_att;
u_int			omask = 0;
int			type = 0;
int			status = 0;
int			status2 = 0;
int			res;
adv_results		result;

	p_att = p_idrp_rt->p_attr;
	bzero(&result,sizeof(adv_results));

	/* clean-up on the previous export information for this peer 
	 */

	status = export(p_idrp_rt->p_rt,
		(proto_t) 0,
		 peer->idrp_export,
		(adv_entry *) 0,
		(adv_entry *) 0,
		&result);

	if (!status)
	    {
	    status2  = export(p_idrp_rt->p_rt,
	        (proto_t) 0,
		idrp_export_list_any_rdi,
		(adv_entry *) 0,
		(adv_entry *) 0,
		&result);
	
	    if (!status2)
		{	
		/* idrp protocol says send everything to
		 * internal peers.   This code will
		 * need to be smarter yet if we combine
		 * IDRP routes into a domain with aggregation
		 *  or other policy considerations.
		 */ 

	        if (peer->type == IDRP_PEER_INTERNAL)
			return(IDRP_EXPORT_VALID);	
		
	        return(IDRP_EXPORT_INVALID);
	 	}
	   }
			
	if (p_idrp_rt->p_export_opts == (idrpRoute_options *) NULL)
		{
		/* check to see what we need to send out the
		 * door for this route for multi_exit
		 */ 
		return (IDRP_EXPORT_VALID);
		}
	
	/* check the qos values */	
	if (p_idrp_rt->p_export_opts->p_qos != (idrp_distinguish_att *) NULL )
		{
		/* QOS has changed -- this is illegal in IDRP
		 */
		task_quit(0);	
		}
	
	/* New idrp_attribute required for export of route
	 * - check to see what is changed
	 * 1) if the next_hop changes - 
	 * 	cannot change next_hop - location
	 *
	 * 2) status 
	 * 	2.1) MULTI_EXIT - on, values set 
	 * 	2.2) ROUTE_SERVER  - values handled by send_next_hop 
	 *	  -- send_next_hop -> attend to peer 	
	 *
	 *	2.2) EXT_INFO - shouldn't be set here 
	 *   	2.3) HEIR_REC - not supported yet (export/import) 
	 *	   -- here we need check  
	 * 3) values	
	 *	Capacity - if changes -> see new attribute 	
	 *      Hopcount - can be offset 
	 * 	QOS - should stay the same

	 * 4) DIST_LIST
	 * 5) RDC 
	 */


	if (p_idrp_rt->p_export_opts->status != p_idrp_rt->p_attr->p_opts->status)
		{
		/* change in status - see what changed
		 */
		  trace_tf (idrp_trace_options, TR_NORMAL,0,("IDRP export policy status different peer %s (old,new)(%x,%x)", 
				peer->name,p_idrp_rt->p_export_opts->status,p_idrp_rt->p_attr->p_opts->status));	
		if (query_type == IDRP_EXPORT_NOPDU)
			return(IDRP_EXPORT_VALID_MOD);
		}		
	else
		{
		return(IDRP_EXPORT_VALID);
		}

	if (query_type == IDRP_EXPORT_NOPDU)
		{		
		if ((p_idrp_rt->p_export_opts->p_DIST_LIST_INCL != NULL) ||
		    (p_idrp_rt->p_export_opts->p_DIST_LIST_EXCL != NULL))
			{
			return(IDRP_EXPORT_VALID_MOD);			 
			}
		}
	
	/* create the attribute record & copy over values
	 * now that we have finished seeing if added attributes
	 * are valid
	 */ 
	p_att = create_idrp_out_att(p_idrp_rt,peer);

	/* Now test modify the outbound attribute
	  * to link to the used outbound attribute records  
	  */

	
	IDRP_STATUS_BIT_TEST(p_idrp_rt->p_export_opts,IDRP_OPTS_EXT_INFO)
		{
		/* Export opts wants to set EXT_INFO 
		 */ 
	    	
	    	if ((p_idrp_rt->p_attr->p_opts->status & IDRP_OPTS_EXT_INFO) == 0) 
			{
			fill_ext_info(p_att);		
			}
		}

	IDRP_STATUS_BIT_TEST(p_idrp_rt->p_export_opts,IDRP_OPTS_MULTI_EXIT)
		{
		if (send_multi_exit(p_att,peer))
		  {
		   /* 
		    * 1) Export opts wants to set multi_exit disc
		    * 2) we can send multi_exit to this peer 
		    * 3)  multi_exit not set for this attribute record already
                    * 4) or doesn't have the correct value
		    */
 
		    if (((p_idrp_rt->p_attr->p_opts->status & IDRP_OPTS_MULTI_EXIT) == 0) || 
			(p_idrp_rt->p_attr->p_opts->multi_exit[peer->id] !=
				p_idrp_rt->p_export_opts->multi_exit[peer->id]))
			{ 
			fill_MULTI_EXIT(p_att);
			p_att->p_opts->multi_exit_xmt = p_idrp_rt->p_export_opts-> multi_exit[peer->id]; 	
			omask = IDRP_OATT_MULTI_EXIT_DISC;
			}
			
		  }
		}

	
	IDRP_STATUS_BIT_TEST(p_idrp_rt->p_export_opts,IDRP_OPTS_ROUTE_SERVER)
		{
	    	if ((p_idrp_rt->p_attr->p_opts->status & IDRP_OPTS_ROUTE_SERVER) == 0) 
			{
			/* create a  new next_hop entry 
			 */
 
			IDRP_STATUS_BIT_SET(p_att->p_opts,IDRP_OPTS_ROUTE_SERVER);
			fill_next_hop(p_att);	
			omask |= IDRP_OATT_NEXT_HOP;
			}
		}

	IDRP_STATUS_BIT_TEST(p_idrp_rt->p_export_opts,IDRP_OPTS_CAPACITY)
		{
		p_att->attrib[IDRP_ATTR_CAPACITY].data = &p_att->p_opts->capacity_xmit;
		omask |= IDRP_OATT_CAPACITY;

		/* here we need to add a test to see if the 
		 * capacity should be check against the rest
		 *
		 * if (peer->strict_iso & IDRP_ATTR_CAPACITY)
		 *	{
		 * 	skhskhskh	
		 *      only allow capacity to be going upward from previous to now
		 *      }
		 */ 		
 
		}
	
	IDRP_STATUS_BIT_TEST(p_idrp_rt->p_export_opts,IDRP_OPTS_HOPCOUNT)
		{
		 /*
		 * if (peer->strict_iso & IDRP_ATTR_CAPACITY)
		 * 	skhskhskh	
		 *      only allow hopcount by strict rules - ignore this modification
		 */

		p_att->attrib[IDRP_ATTR_HOP_COUNT].data = &p_att->p_opts->hopcount_xmit;
		}
	
	/* check to see if the distribution list 
	 * needs to be changed
	 */

	res  = export_dist_list(p_idrp_rt,peer,&type);
	if (res  == IDRP_OATT_DIST_LIST_NEW || res == IDRP_OATT_DIST_LIST_EXIST)
		{	
		/* build a new distribution list in attribute 
		 */

		fill_DIST_LIST(p_att,p_att->p_opts->p_DIST_LIST[peer->id],type);
		omask |= res;
		}
	 
#ifdef IDRP_RDC

	/* Process confederation change only if
	 * the peer is external (7.12.3.3) 
	 * with different rdcs (7.12.3.3 a)
	 */  	
	if ((peer->type == IDRP_PEER_EXTERNAL) && (peer->no_rdc_diff != 0))
		{
		if (confed_chk(p_idrp_rt,peer,&p_rdc_path,p_dat) !=
			IDRP_UPDATE_ERROR_MISCON_RDCS)
			{
			fill_rdc_path(p_idrp_rt,peer,p_rdc_path,p_hier,&p_rdc_path,p_dat);
			}
			else 	 
			{
			/* we have an error that must be passed up the
			 * return code for this
			 * - peer, send an ERROR message and then
			 *   cut the connection - abort any further link list on this
			 *   peer 
			 */
		
			idrpPeer	*route_peer = p_idrp_rt->peer;  
	
			route_peer->last_error_pdu.error_subcode = IDRP_UPDATE_ERROR;
       	 		route_peer->last_error_pdu.error_subcode = IDRP_UPDATE_ERROR_MISCON_RDCS;
			route_peer->last_error_pdu.data = (u_char *) 0;
			route_peer->last_error_pdu.data_len = 0;
			route_peer->last_error_pdu.data2 = (u_char *) 0;
			route_peer->last_error_pdu.data2_len = 0;
			idrp_sm(route_peer,IDRP_UPDATE_ERROR);	
			}
		}

#endif /* IDRP_RDC */
	

	/* set the attribute in the idrpRoute
	 */
 
	p_idrp_rt->route_out[peer->id].p_att = p_att;
	link_out_att(p_att);

	return(IDRP_EXPORT_VALID);	

}



int
export_dist_list(p_idrp_rt,peer,type)
idrpRoute	*p_idrp_rt;
idrpPeer	*peer;
int		*type;
{
idrp_canon_rdpath	*p_incl; 
idrp_canon_rdpath	*p_excl; 
idrp_canon_rdpath	*p_can;
idrp_canon_rdpath	*p_can_find;
idrp_canon_rdpath	*p_rdi_list;
idrp_canon_rdpath	*p_rdi_last = NULL;
idrp_canon_rdpath	*p_can_next;
idrpRoute_options	*p_export_opt;
idrpRoute_options	*p_opt;

	p_opt = p_idrp_rt->p_attr->p_opts;
	/* check the distribution lists
	 */

	p_export_opt = p_idrp_rt->p_export_opts; 
	p_incl = p_export_opt->p_DIST_LIST_INCL;
	p_excl = p_export_opt->p_DIST_LIST_EXCL;

	/* distribution lists have changed
	 * walk through and create a new distribution list
	 * based on the intersection of the two lists
	 * - put the list for now back in the orginal idrpRoute_options
	 * - if two policies - exclude takes precedence over include
	 *  
	 */
				

	if ((p_excl == (idrp_canon_rdpath *) NULL)  && 
	    (p_incl == (idrp_canon_rdpath *) NULL))
		{
		return(IDRP_OATT_DIST_LIST_NOCHG);
		}

	/* the distribution lists are originally null
	 */
 
	/*
	 * Prefer to set the DIST_LIST_INCLude over
	 * the DIST_LIST_EXCLude, since the former
	 * gives you tighter control over the
	 * further dissemination of the routing
	 * information. -- SJR
	 */

	if ((p_opt->p_DIST_LIST_INCL == (idrp_canon_rdpath *) NULL) 
		&& (p_incl != (idrp_canon_rdpath *) NULL))
		{	 
		/* simply copy the 
		 * rdi list to the peer location
		 * - if there are two use the include list
		 * skh -- fix fix fix
		 */
		p_rdi_list  = copy_canon_rdi_list(p_incl);
		} 
	   else if  ((p_opt->p_DIST_LIST_EXCL == (idrp_canon_rdpath *) NULL) 
		&& (p_excl != (idrp_canon_rdpath *) NULL))
		{
		p_rdi_list  = copy_canon_rdi_list(p_excl);
		}

	/* case 1)  DIST_LIST_INCL & p_incl - make sure 
	 *  -> include list has intersection of both
	 *  -> rdi has to be in both lists
	 *
	 * case 2) DIST_LIST_INCL & p_excl 
	 * 	    - take out of the include list the excluded RDIs
	 * 	
	 * case 3) DIST_LIST_EXCL & p_incl - ignore policy
	 * 	  flag ignoring policy
	 * 
	 * case 4) DIST_LIST_EXCL & p_excl 
	 * 	 add excludes to the routes 
	 */	

	if (p_opt->p_DIST_LIST_INCL != (idrp_canon_rdpath *) NULL)
		{
		*type = IDRP_ATTR_DIST_LIST_INCL;
		/* copy the rdi_list from the idrp_attribute 
		*/

		p_rdi_list  = copy_canon_rdi_list(p_opt->p_DIST_LIST_INCL);

		/* walk the list of routes,
		 * pull out any on the excluded list
		 * pull out any not found on the inclusion list 
		 */
	
		if (p_excl != (idrp_canon_rdpath *) NULL)
			{
			RDPATH_LIST(p_excl,p_can)
				{
				p_can_find = find_global_rdi_rdilist(p_can->p_rdi,p_rdi_list);
				exclude_global_rdilist(p_can_find,&p_rdi_list);
				} RDPATH_LIST_END;
			}
		
		if (p_incl != (idrp_canon_rdpath *) NULL)
			{
			/* look for each rdi in the include list
			 */
	
			for(p_can = p_rdi_list;p_can; p_can = p_can_next)
				{
				p_can_find = find_global_rdi_rdilist(p_can->p_rdi,p_incl);
				if (p_can_find == (idrp_canon_rdpath *) NULL)
					{
					/* didn't find RDI in the include list
					 * drop it out of the rdi list
					 */
					p_can_next = p_can->p_next;
					exclude_global_rdilist(p_can,&p_rdi_list);
					}
				else
					{
					p_can_next = p_can->p_next;
					}
				}
			}

		}
						 
	if (p_opt->p_DIST_LIST_EXCL != (idrp_canon_rdpath *) NULL)
		{
		*type = IDRP_ATTR_DIST_LIST_EXCL;
		p_rdi_list  = copy_canon_rdi_list(p_opt->p_DIST_LIST_EXCL);
		if (p_incl)
			{	
			trace_tf (idrp_trace_options, TR_NORMAL,0,("include policy and DIST_LIST_EXCL in attribute, ignore - peer %s", peer->name));	
			}
		if (p_excl)
			{
			/* get end of rdi_list
			 */
			RDPATH_LIST(p_rdi_list,p_can)
				{
				p_rdi_last = p_can;
				} RDPATH_LIST_END; 		
	
			RDPATH_LIST(p_excl,p_can)
				{
				/* find the excluded RDI in the list
				 */
				p_can_find = find_global_rdi_rdilist(p_can->p_rdi,p_rdi_list);

				/* if not there, add it 
				 */

				if (p_can_find == (idrp_canon_rdpath *) NULL)
					{
					p_rdi_last = idrp_add_end_rdilist(p_can->p_rdi,p_rdi_last);
					}
				} RDPATH_LIST_END;
			}	
		/* copy the new rdi list to the idrp route options 
		 * for that peer, attribute record 
		 */ 

		}

	if (p_opt->p_DIST_LIST[peer->id])
		{
		/* something was there last time
		 */

		if (same_dist_list(p_opt->p_DIST_LIST[peer->id],p_rdi_list))
			{
			/* same list - free the list 
			 */	
		
			free_dist_list(p_rdi_list);
			return(IDRP_OATT_DIST_LIST_EXIST);
			}
		else
			{
			free_dist_list(p_opt->p_DIST_LIST[peer->id]);
			p_opt->p_DIST_LIST[peer->id] = p_rdi_list;
			}	
		}
	else
		{			
		p_opt->p_DIST_LIST[peer->id] = p_rdi_list;
		}
	return(IDRP_OATT_DIST_LIST_NEW);	


}


int
confed_chk(p_idrp_rt,peer,p_rdc_opath,p_data)
idrpRoute		*p_idrp_rt;
idrpPeer		*peer;
rdpath_list		**p_rdc_opath;
u_char			*p_data;
{
int status = 0;	
idrp_attribute_record 	*p_att;
rdpath_list		*p_rd_opath;
rdpath_list		*p_rdc_path;
rdpath			*p_rdp;
rdi_array		*p_array;
rdc_check		rdc_diff[IDRP_MAX_RDCS];	
rdc_check		rdc_not_diff[IDRP_MAX_RDCS];	
rdc_path_chk		path[IDRP_MAX_PATH_SEG];

/* variables for the path parsing
 */

u_int			no_rdc_not_diff;
u_int			no_rdc_diff;
u_int			irdc = 0;
u_int			irdi = 0;
u_int			pseg_cnt = 0;
u_int			global_rdi_cnt = 0;
u_int			pathseg_rdc_cnt = 0;
u_int			array_cnt = 0;
u_int			array_rdicnt = 0;
int			rdc_status = 0;
int			rdc_local_status = 0;

	p_att = p_idrp_rt->p_attr;
	p_rd_opath = p_att->p_rd_opath;
	p_rdc_path = *p_rdc_opath;

	no_rdc_not_diff = peer->no_rdc - peer->no_rdc_diff;
	no_rdc_diff = peer->no_rdc_diff;

	/* confederations are different
         *	
	 * NESTing order of RDC
	 * outer most RDC --> inner most nesting RDC
	 * within same level - numerical ordering  
	 *
	 * 
         * processing algorithm (7.12.3.3 b)
	 *
	 * 1) forward processing check for RDC existing in the
	 *    path sequence:
	 * 
	 *   build an array of the location of different
	 * 	RDCs (rdc_proc)
	 *	1) path segment pointer
	 *	2) array pointer
	 *	3) array location within array pointer (cnt)
	 * 	4) count within full path
	 * 	5) global pointer for rdi
	 *
	 */

	for (irdc = 0; irdc < peer->no_rdc_diff; irdc++)
		{
		rdc_diff[irdc].p_rdi = peer->rdc_diff[irdc].p_rdi;
		rdc_diff[irdc].nest_level = peer->rdc_diff[irdc].nest_level;     
		rdc_diff[irdc].status = RDC_CHK_STATUS_NOTFOUND;
		}

	for (irdc = 0; irdc < peer->no_rdc; irdc++) 	
		{
		rdc_not_diff[irdc].p_rdi = peer->rdc_not_diff[irdc].p_rdi;
		rdc_not_diff[irdc].nest_level = peer->rdc_not_diff[irdc].nest_level;     
		rdc_not_diff[irdc].status = RDC_CHK_STATUS_NOTFOUND;
		}

	/* set the sequence count of path segments to zero 
	 * set the global rdi count to zero
	 */

	pseg_cnt = 0;
	global_rdi_cnt = 0;

	IDRP_RDPATH_LIST(p_rdp,p_rd_opath->p_start)
		{
		status = (p_rdp->status & IDRP_RDPATH_TYPE);
		p_array = &p_rdp->rdi_list;
		array_cnt = 0;
		path[pseg_cnt].p_path = p_rdp;
		pathseg_rdc_cnt = 0;
		for (array_rdicnt = 0;  array_rdicnt < p_rdp->cnt; array_rdicnt++)
			{
			global_rdi_cnt++;
			irdi = array_rdicnt%IDRP_RD_PATH_ARRAY_SIZE;

			if ((irdi == 0) & (array_rdicnt != 0))
				{
				array_cnt++ ;
				p_array = p_array->p_next;
				assert(p_array != NULL);
				}

			/* walk through the rdis in this path element
			 * - looking to see if there are any rdcs
			 * - in the rdi array - either in the differences rdcs
			 */

			rdc_status = 0;	
			for (irdc = 0; irdc < no_rdc_diff; irdc ++)
				{
				if (p_array->p_rdi_array[irdi] == rdc_diff[irdc].p_rdi)
					{
					if (rdc_diff[irdc].status)
						{
						/* error in RDC path - return with
						 * - misconfigured RDCs
						 */

						return(IDRP_UPDATE_ERROR_MISCON_RDCS);
						}
					rdc_status = rdc_diff[irdc].status = RDC_CHK_STATUS_FOUND;
					rdc_diff[irdc].array_cnt = array_cnt;
					rdc_diff[irdc].aseq_cnt = irdi; 	
					rdc_diff[irdc].pseq_cnt = pseg_cnt;
					rdc_diff[irdc].rdi_cnt = global_rdi_cnt;	
					path[pseg_cnt].rdi_index[pathseg_rdc_cnt] = irdc;
					pathseg_rdc_cnt++;
					break;
					}
				} 
			
			/* loop through RDCs that are the same between this node and
			 * the remote node
			 */
	
			if (!rdc_status)
				{		
				for (irdc = 0; irdc < no_rdc_not_diff; irdc++)
					{
					if (p_array->p_rdi_array[irdi] == rdc_not_diff[irdc].p_rdi)
						{
						if (rdc_not_diff[irdc].status != RDC_CHK_STATUS_NOTFOUND)
							{
							/* error in RDC path - return with
							 * - misconfigured RDCs
							 */
	
							return(IDRP_UPDATE_ERROR_MISCON_RDCS);
							}
						rdc_local_status = rdc_not_diff[irdc].status = RDC_CHK_STATUS_FOUND;
						rdc_not_diff[irdc].array_cnt = array_cnt;
						rdc_not_diff[irdc].aseq_cnt = irdi; 	
						rdc_not_diff[irdc].pseq_cnt = pseg_cnt;
						rdc_not_diff[irdc].rdi_cnt = global_rdi_cnt;	
						break;
						}
					}
				} 
			
			
	
			/* process the list based on the type of entry
			 * 
			 */

  	
			switch (status)
				{	
				case IDRP_RDPATH_RDC_ENTRY_SEQ:
					/* now we have an RDI marked in RDC that has entry seq
					 */
					if (rdc_status)
						{
						rdc_diff[irdc].status |= RDC_CHK_STATUS_RDC_ENTRY_SEQ; 
						}
					break;
					
				case IDRP_RDPATH_RDC_ENTRY_SET:
					/* now we have an RDI marked in RDC that has entry set
					 */
					if (rdc_status)
						{
						rdc_diff[irdc].status |= RDC_CHK_STATUS_RDC_ENTRY_SET; 
						}
					break;
					

				case IDRP_RDPATH_SET:
				case IDRP_RDPATH_SEQ: 
				default:
					break;	
				}
			
			}
		path[pseg_cnt].cnt = pathseg_rdc_cnt;
		pseg_cnt++;  
		} IDRP_RDPATH_LIST_END;


	/* check the RDCs to see (7.12.3.3 b) paragraph I and II 
	 *     
	 * 1) if all the exited  RDCs are in the rdpath ENTRY_SET/ENTRY_SEQ segments 
	 * 2) nesting order    
	 */	

	for (irdc = 0; irdc < no_rdc_diff; irdc++)
		{
		if ((rdc_diff[irdc].status & RDC_CHK_STATUS_FOUND) == 0)
			{
			return(IDRP_UPDATE_ERROR_MISCON_RDCS);
			}
	
 		if (((rdc_diff[irdc].status & (RDC_CHK_STATUS_RDC_ENTRY_SET | RDC_CHK_STATUS_RDC_ENTRY_SEQ)) == 0))
			{
			return(IDRP_UPDATE_ERROR_MISCON_RDCS);
			}
			
		/* if more than 1 RDC exiting
		 * type of path sequence is SEQUENCE
		 * same path sequence between two rdcs
		 * and RDC B is nested within RDC A 
		 * - check that the order within path is RDC A precedes RDC B
		 */	

		if ((irdc >= 1) && ((rdc_diff[irdc].status & RDC_CHK_STATUS_RDC_ENTRY_SEQ) != 0)
			&& (rdc_diff[irdc].pseq_cnt == rdc_diff[irdc-1].pseq_cnt))
			{
			/* we have more than one RDC in the ENTRY SEQUENCE
			 * RDC B = nest level 2 
			 * RDC A = nest level 1 
			 * order within the pathway needs to be 
			 * RDC A  
			 */

			if (rdc_diff[irdc].nest_level != rdc_diff[irdc-1].nest_level)
				{
				if (rdc_diff[irdc].rdi_cnt <= rdc_diff[irdc-1].rdi_cnt)
					{
					/* if not in the right order - error
					 */
 
					return(IDRP_UPDATE_ERROR_MISCON_RDCS);
					}
				}
			}
		}
		

	/* now walk through the path creating a new path given
	 * the following things:
	 * 
	 * 1) rdc_chk array for those things we are exiting
	 * 2) original pathway count
	 *
	 * create an alternate path for the rdc and insert
	 * in the global rd_path sequences	  	 
	 */
   
	p_rdc_path = create_rdc_rdpath(no_rdc_diff,rdc_diff,pseg_cnt,path,p_rd_opath,p_data,peer);	   	  
	*p_rdc_opath = idrp_insert_global_rdpaths(p_rdc_path);	
	return(0);

}

rdpath_list *
create_rdc_rdpath(no_dif_rdc,rdc_dif_proc,path_segment_cnt,path,p_rd_opath,p_data,peer)
int		no_dif_rdc;
rdc_check	rdc_dif_proc[];
int		path_segment_cnt;	
rdc_path_chk	path[];
rdpath_list	*p_rd_opath;
u_char		*p_data;
idrpPeer	*peer;
{
int		i;
int		type;  
rdpath_list	*p_rdc_path = 0;


		trace_tf (idrp_trace_options, TR_NORMAL,0,("create_rdc_rdpath parameters: no_dif_rdc %d, rdc_dif_proc %x, path_seg_cnt = %d, p_rd_opath = %x,p_data = %x, peer is %s",
		 no_dif_rdc,rdc_dif_proc[0],path_segment_cnt,p_rd_opath,
		 p_data,peer->name)); 

	/*  
	 *  
	 */

  	for (i = 0;  i <  path_segment_cnt; i++)
		{
		type = path[i].p_path->status;	
		if (path[i].cnt == 0)
			{
			/* no rdcs in this segment, just copy
			 * the rdis into a new path segment
			 */   
	
			p_rdc_path =  copy_rdpath_seg(peer,path[i].p_path,&p_rdc_path,type);
			}
		else
			{

			if (path[i].cnt == path[i].p_path->cnt) 
				{
				/* check to see if this ENTRY_SEQ/SET has only the
				 * rdcs exited
				 *
				 * if so, simply create a PATH SEQ/ SET    
				 */

				if (path[i].p_path->status == IDRP_RDPATH_RDC_ENTRY_SEQ)
					{
					type = IDRP_RDPATH_SEQ;
					}
				if (path[i].p_path->status == IDRP_RDPATH_RDC_ENTRY_SET)
					{
					type = IDRP_RDPATH_SET;
					}
				p_rdc_path = copy_rdpath_seg(peer,path[i].p_path,&p_rdc_path,type);
				}
			else	
				{
				/* if only some of RDCs in this  rdpath segment
				 * have been exited, then follow the IDRP specification
				 * section p 39 (i - iv) 
				 */
		
				convert_rdcpath_seg(peer,path[i],&p_rdc_path);						
				}
			}
		}			  
	return(p_rdc_path);
}

rdpath_list *
copy_rdpath_seg(peer,p_path,p_rdc_path,type)
idrpPeer	*peer;
rdpath		*p_path;
rdpath_list	**p_rdc_path;
int		type;
{
rdpath_list	*p_rdc;
rdpath		*p_seg;  
rdi_array	*p_array,*p_array_old;
u_int		i,j;
rdi		*p_rdi;	

	trace_tf(idrp_trace_options, TR_NORMAL,0,("copy_rdc_seg for peer %s", peer->name));

	p_rdc = *p_rdc_path;	
	if (p_rdc == (rdpath_list *) NULL)
		{
		/* no existing rdpath - create a new rdpath list
		 * and point the segment pointer to it 
		 */
	
		p_rdc = idrp_create_rdpath_list(type,p_path->rdi_list.p_rdi_array[0]);
		p_seg = p_rdc->p_end;
		}
	else
		{
		/* 
		 * path exists, create a segment memory chunk and set the
		 * segment pointer to it 
		 */
	
		p_seg = (rdpath *) idrp_local_mem_fit(sizeof(rdpath));
		p_seg->p_last = &p_seg->rdi_list;
		p_seg->p_prev = p_rdc->p_end;
		p_rdc->p_end->p_next = p_seg;
		p_rdc->p_end = p_seg;
		} 	

	/* segment pointer
	 */

	p_seg->status = type;
	p_array = p_seg->p_last; 
	p_array_old = &p_path->rdi_list;
	for (i = 1; i < p_path->cnt; i++)
		{
		j = i%IDRP_RD_PATH_ARRAY_SIZE;
		if (j == 0 && i != 0)
			{
			/* index has wrapped via one array	
			 */

			p_array->p_next = (rdi_array *) idrp_local_mem_fit(sizeof(rdi_array));
			p_array->p_next->p_prev = p_array;
			p_array = p_rdc->p_end->p_last = p_array->p_next; 
			p_array_old = p_array_old->p_next;
  			}	

		/* copy the rdi into the new path sequence
		 * - increment count on the rdi 
		 */

		p_rdi = p_array->p_rdi_array[j] = p_array_old->p_rdi_array[j];
		p_seg->cnt++;
		}
	return(p_rdc);
}	

void
convert_rdcpath_seg(peer,path,p_rdc_path)
idrpPeer	*peer;
rdc_path_chk	path;
rdpath_list	**p_rdc_path;
{
rdpath		temp_path;
rdpath		*p_temp_path;
rdpath		*p_path;
int		new_type = 0;
int		old_type = 0;
u_int		i;
int		last_rdi_index, rdi_index;	

	trace_tf (idrp_trace_options, TR_NORMAL,0,("convert_rdpath_seg for peer %s ",peer->name));	

	p_temp_path = &temp_path;
	p_path = path.p_path;	
	/* this should be an ordered list 
	 * of rdi's array addresses
	 */

	if (p_path->status == IDRP_RDPATH_RDC_ENTRY_SEQ)
		{
		new_type = IDRP_RDPATH_SEQ;
		} 
	if (p_path->status == IDRP_RDPATH_RDC_ENTRY_SET)
		{
		new_type = IDRP_RDPATH_SEQ;
		} 

	last_rdi_index = 0;
	for (i = 0  ; i < path.cnt; i++)
		{
		/* copy the pathway count up to the first rdi
		 * that an RDC, then copy 
		 * -- review this to see that it works
		 *      
		 */

		rdi_index = path.rdi_index[i]-1;
		copy_part_rdpath_seg(peer,path.p_path,p_rdc_path,old_type,last_rdi_index,rdi_index); 
		copy_part_rdpath_seg(peer,path.p_path,p_rdc_path,new_type,path.rdi_index[i],path.rdi_index[i]); 
		last_rdi_index = path.rdi_index[i]+1;
		}

}

void
copy_part_rdpath_seg(peer,p_path,p_rdc_path,type,start_rdi,end_rdi)
idrpPeer	*peer;
rdpath		*p_path;
rdpath_list	**p_rdc_path;
int		type;
int		start_rdi;
int		end_rdi;
{
rdpath_list	*p_rdc;
rdpath		*p_seg;
int		i,j,i1;
rdi_array	*p_array,*p_array_old;
rdi		*p_rdi;
	/* 
	 */

	trace_tf (idrp_trace_options, TR_NORMAL,0,("copy_part_rdpath_seg for peer %s ",peer->name));	

	p_rdc = *p_rdc_path;	

	if (p_rdc == (rdpath_list *) NULL)
		{
		p_rdc = idrp_create_rdpath_list(type,p_path->rdi_list.p_rdi_array[start_rdi]);
		p_seg = p_rdc->p_end;
		}
	else
		{
		/* 
		 * path exists, create a segment memory chunk and set the
		 * segment pointer to it 
		 */
	
		p_seg = (rdpath *) idrp_local_mem_fit(sizeof(rdpath));
		p_seg->p_last = &p_seg->rdi_list;
		p_seg->p_prev = p_rdc->p_end;
		p_rdc->p_end->p_next = p_seg;
		p_rdc->p_end = p_seg;
		} 	

	/* segment pointer
	 */

	p_array = p_seg->p_last; 
	p_array_old = &p_path->rdi_list;
	for (i = 0, i1 = start_rdi+1; i < end_rdi - start_rdi;i1++, i++)
		{
		j = i%IDRP_RD_PATH_ARRAY_SIZE;
		if (j == 0 && i != 0)
			{
			/* index has wrapped via one array	
			 */

			p_array->p_next = (rdi_array * ) idrp_local_mem_fit(sizeof(rdi_array));
			p_array->p_next->p_prev = p_array;
			p_array = p_rdc->p_end->p_last = p_array->p_next; 
			p_array_old = p_array_old->p_next;
  			}	

		/* copy the rdi into the new path sequence
		 * - increment count on the rdi 
		 */

		p_rdi = p_array->p_rdi_array[j] = p_array_old->p_rdi_array[i1];
		}
}	


void
fill_rdc_path(p_idrp_rt,peer,p_rdc_path,p_hier,p_rdpath,p_data)
idrpRoute	*p_idrp_rt;
idrpPeer	*peer;
rdpath_list	*p_rdc_path;
idrp_attr_data_entry	*p_hier;
idrp_attr_data_entry	*p_rdpath;
u_char			*p_data;
{
rdi_array	*p_array;
rdi		*p_rdi;
rdpath	*p_rdp;
u_char	*p_cout;
u_char	*p_len = 0;
u_int	hold;
int	total_length;
int	len_segment;
u_int	i;
u_int	j;

	trace_tf (idrp_trace_options, TR_NORMAL,0,("fill_rdc_rdpath_seg for peer %s route %s ",peer->name,iso_ptoa(&p_idrp_rt->nlri)));	

	p_cout = p_data;

	/*  take the rdc path and put it in the
	 *  the rdpath
	 */

	p_hier->present = TRUE;
	p_hier->flags = IDRP_ATTRFLG_HIERARCH_REC;   	
	p_hier->length = IDRP_HIERARCHICAL_RECORDING_LENGTH;
	p_hier->data = &p_hier->data_in[0];
	*p_hier->data = 1;
	
	p_rdpath->present = TRUE;
	p_rdpath->flags = IDRP_ATTRFLG_RD_PATH;   	
 	p_rdpath->data = p_data;

	total_length = 0;	
		
	IDRP_RDPATH_RDI_LIST(p_rdp,p_rdc_path)
		{
		/* Here's the data 
		 */

		*p_cout = p_rdp->status;
		p_len = p_data;
		p_cout += 3;

		/* set up the legnth,
		 * rdi count and
		 * this rdi array pointer
		 */
	
		len_segment = 2;
		p_array = &p_rdp->rdi_list;
		IDRP_RDI_ARRAY_WALK(p_rdp,i,1)
			{
			j = i%IDRP_RD_PATH_ARRAY_SIZE;
			if (i != 0 && j == 0)
				{
				p_array = p_array->p_next;
				assert(p_array != (rdi_array *) NULL);
				}

			/* set the length to network byte order
			 */

			p_rdi = p_array->p_rdi_array[i]; 
			hold = p_rdi->rdi.isoa_len;
                        bcopy((caddr_t) &hold,p_cout,2);
			p_cout +=2; 

			/* copy out the rdi 
			 * adjust the length of the segment, and output pointers
			*/

			bcopy(&p_rdi->rdi.isoa_genaddr,p_cout,p_rdi->rdi.isoa_len);
			p_cout += p_rdi->rdi.isoa_len;
			len_segment = len_segment + 2 + p_rdi->rdi.isoa_len; 
			} IDRP_RDI_ARRAY_WALK_END;

	        hold = htons(len_segment);
		bcopy((caddr_t) &hold,p_cout,2);
	
		total_length += len_segment;
		} IDRP_RDPATH_RDI_LIST_END; 	

	p_rdpath->length = total_length;
 				
}


int
dist_unmod(p_irt,peer,flash)
idrpRoute  *p_irt;
idrpPeer    *peer;
int              flash;
{
int status = 0;

  /* check to see if this route can be send to this peer
   */

         if  (DIST(p_irt,peer,flash)== FALSE) 
	      return(FALSE);

       status = idrp_route_export(p_irt,peer,IDRP_EXPORT_NOPDU);

       if (status == IDRP_EXPORT_ERROR)
             return(FALSE);

       if (status == IDRP_EXPORT_VALID)
		return(TRUE);	

	if (status == IDRP_EXPORT_VALID_MOD)
	  	return(FALSE);

     return(TRUE);
}

void
idrp_capacity_mod(p_att,peer)
idrp_attribute_record *p_att;
idrpPeer	*peer;
{
	if ((peer->type == IDRP_PEER_EXTERNAL)
		&& ((idrp_my_rdi_rdpath(p_att)) == 0)
 	      	&& (p_att->p_opts != NULL)
		&& (p_att->p_opts->capacity.value > idrp_this_node.Capacity))
		{
                p_att->p_opts->capacity_xmit = p_att->p_opts->capacity.value;
 		}
	else
		{
		/* local, internal, and test peers get the
  		 * same capacity value as this node
                 * - also if the receiving route has a capacity value
 		 * - also if the receiving route has a capacity value
 		 *   less than this nodes
               	 */
	
                p_att->p_opts->capacity_xmit = idrp_this_node.Capacity;
		}
}

void
idrp_hopcount_mod(p_att,peer)
idrp_attribute_record *p_att;
idrpPeer	*peer;
{
                                
	if (peer->type == IDRP_PEER_EXTERNAL)
       		{
                p_att->p_opts->hopcount_xmit = p_att->p_opts->hopcount + 1;
		}
         else
        	{
                p_att->p_opts->hopcount_xmit =  p_att->p_opts->hopcount;
		}
}

#ifdef	IDRP_QOS

void
idrp_qos_type_mods(p_att,peer)
idrp_attribute_record *p_att;
idrpPeer	*peer;
{
int rib_id = p_att->rib_id;	
int idrp_mask = p_att->p_opts->p_qos->idrp_mask;
idrp_dist_att_xmit *p_qos_xmit = NULL;

	/* here we need to obey the changes in the
	 * values of 
	 * 1) transit delay
	 * 2) residual error
	 * 3) expense
	 * 4) priority
	 *
	 * Rules only apply to transmittal
	 * to external peer with routes received outside
	 * my RDI (not with my rdi path) 	
	 */ 

	if ((peer->type == IDRP_PEER_EXTERNAL)
		&& !(idrp_my_rdi_rdpath(p_att)))
		{
		p_qos_xmit =  (idrp_dist_att_xmit *) 
			idrp_local_mem_fit(sizeof(idrp_dist_att_xmit)); 

		p_att->p_opts->p_qos_xmt = p_qos_xmit;

		/* if transit delay 
		 *  	add local transit delay to recieved
	 	 */
 
		if (idrp_mask && IDRP_ATTBT_TRANSIT_DELAY)
		   {
	           p_qos_xmit->delay = p_att->p_opts->p_qos->delay.value +
			 qos_rib[rib_id]->att.delay.value;
		 }

	 	/* if Residual error 
		*
		*/

		if ((idrp_mask & IDRP_ATTBT_RESIDUAL_ERROR))
		    {
		    u_int rre =  p_att->p_opts->p_qos->expense.value;
		    u_int rdlre = qos_rib[rib_id]->att.expense.value;
		
		    p_qos_xmit->error = (u_int) (rre + rdlre - ((rre * rdlre)/IDRP_ERROR_K));
		    }

		if (idrp_mask & IDRP_ATTBT_EXPENSE)
		     {
		     /*
		      * expense = expense pdu + expense local
		      */  	
		     p_qos_xmit->expense = p_att->p_opts->p_qos->expense.value +
		 	qos_rib[rib_id]->att.expense.value;
		     }		
		 	
		if (idrp_mask & IDRP_ATTBT_PRIORITY)
		     {
		     /* 
	 	      *  priority's value higherof(pdu priority,node priority)
	 	      */
  	
		     p_qos_xmit->priority = p_att->p_opts->p_qos->priority.value;

		     if (( p_att->p_opts->p_qos->priority.value) < 
			 qos_rib[rib_id]->att.priority.value)
			{
		         p_qos_xmit->priority = qos_rib[rib_id]->att.priority.value;
			}
		     }
		}
}

#endif /* IDRP_QOS */ 
