#
#
#  gen_config.perl
#
#    Generates gated configuration files.
#
#  Author:  Jeff Jackson
#
#  Revision History:
#
#    01-26-95   JWJ   First version finished.
#    02-23-95   JWJ   Added -m option, default file path
#    03-01-95   JWJ   Added IP_STATIC_OPEN, IP_STATIC_CLOSE
#                       (to work around gated parser bug)
#    03-02-95   JWJ   Added socket specification
#    03-03-95   JWJ   Minor environment variable name changes
#    03-07-95   JWJ   Minor bug fix in find_machine (for < to for <=)
#    03-23-95   JWJ   Fixed bug - rdi of internal neighbors should
#                     match rdi of host machine of config file
#    03-30-95   JWJ   Added name of this file to config file headers.
#    04-27-95   JWJ   Added ROUTE-SERVER, MULTI-EXIT-DISC options.
#    05-04-95   JWJ   Added ip_gw switch.
#    05-08-95   JWJ   Bug fixes to route_server and multi_exit code.
#                     Changed ip_gw code to be machine-specific.
#    05-10-95   JWJ   Added code to overwrite machine's default RDI
#                     with that from another machine.  Removed the
#                     "bug fix" of 3-23-95 (the solution was insufficient
#                     and is not trivial).  New solution is to have the
#                     user specify the RDI in the .cfg file.
#    06-07-95   JWJ   Added "net" keyword in neighbor statement before
#                     osi address (now required).
#    08-04-95   JWJ   Added env var checking.
#                     Added "traceoptions" var processing.
#                     Changed machine map field parsing (\t to \s).
#                     and fixed portability bug on chop (see below).
#                     Added different gateway (gw) for default route.
#    08-18-95   JWJ   Changed to always put in IP "default" route.
#                     Changed some terminology to avoid confusion.
#                     (ex: IP_DEFAULT_ROUTES to IP_COMMON_STATIC_ROUTES)
#    08-21-95   JWJ   Removed IP_STATIC_OPEN and IP_STATIC_CLOSE
#                     (change - both IP and ISO now use "static" area).
#                     Changed "ip_gw" keyword to "gateway" (ISO routes now use it).
#                     Added first pass of "export" statement handling.
#    08-24-95   JWJ   Eliminated duplicate RDI in import/export processing.
#    09-07-95   JWJ   Put default route name and common static route info
#                     in machine map file.  Added keywords for processing.
#                     Added global IP and OSI prefixes to machine map file.
#    09-28-95   JWJ   Added import and export processing using template files
#                     and setvar handling.
#    09-29-95   JWJ   Updated variable processing (unique to a test set).
#    10-05-95   JWJ   Added overview and usage messages (-h option or error).
#                     Added code to avoid results file generation (-E option).
#                     Added code to avoid export loop code (-L option).
#                     Fixed bug in internal/external neighbor processing
#                      (two separators needed).
#    10-11-95   JWJ   Put in general variable substitution.
#                     Added a more general and helpful error handling routine.
#    10-16-95   JWJ   Fixed user variable substitution in import/export code.
#    10-17-95   JWJ   Put in new RIB generation code (1 template per RIB).
#                     Added -W option (suppress warning messages).
#    10-24-95   JWJ   Added specific RDI and RDPATH processing.
#    10-25-95   JWJ   Added &rdi and &net built-in functions.
#    10-27-95   JWJ   Modified expected result file generation to include
#                     only the RIBs in the "ribs-supported" peer clauses.
#    10-30-95   JWJ   Removed the "$RDI" predefined variable (for stepping
#                     through RDI list without looping).  More efficient now
#                     to use user-defined vars + built-in functions.
#    11-01-95   JWJ   Added code to put in "." when generating OSI addresses
#                     if length of last prefix + index is longer than 4.
#
#
#  Implementation notes:
#
#    * The BSD Unix version of perl is different from Sun Unix.
#      To keep the code portable, the following must be done:
#      - Any local variable declarations must be put in ().
#      - End of input line must be checked before chopping newline;
#        BSDI has \n at end of input line, Sun does not.
#
#  Environment variables looked for:
#
#      gated_scripts         Directory containing gated scripts and templates
#      gated_machine_map     Machine map file
#      gated_template_file   File used as gated.conf template
#      gated_default_rte_gw  Gateway of default (IP) route
#      gated_mainlog_dir     Directory of gated log file
#      gated_log_file_name   Name of gated log file
#
#  Important global variables:
#
#      g001_mach_idx         Current machine being processed
#      g002_curr_sequence    Current sequence
#      g003_curr_test_set    Current test set
#      g004_curr_peers       Current peers of current machine
#      g005_process_scope    Current process level (global, sequence, ...)
#

$PROGRAM_NAME = $0;
$VERSION      = "3.0";
$VDATE        = "November 1, 1995";

require "ctime.pl";
$date = &ctime(time);
chop( $date );

#
#
# Options:
#    -m <machine name> - restrict file generation to a particular machine.
#    -e                - generate expected results files (default).
#    -E                - do not generate expected results files.
#    -W                - do not generate warning messages (do print errors).
#
require "getopts.pl";
&Getopts('m:EhLW');   #sets $opt_m as argument to -m

#
#  Now make sure a test configuration file was specified.
#
if ( ($#ARGV == -1) || ($opt_h) )
{
   &print_overview;
   &print_usage;
   exit;
}

#
#  Print header
#
print "\n";
print "              GATED Configuration File Generator\n";
print "               Version $VERSION - $VDATE\n";
print "            Copyright 1994, 1995 Merit Network, Inc.\n";
print "\n";

if ( $opt_m ne "" )
{
   print "\nGenerating files only for machine $opt_m\n\n";
}

$print_warnings = ( $opt_W ) ? OFF : ON;
if ( ! $print_warnings )
{
   print "Suppressing warning messages\n";
}


#
#  Default path and file names - using env vars
#  Exit if not declared.
#
$gated_scripts    = $ENV{'gated_scripts'};
$machine_map_file = $ENV{'gated_machine_map'};
$template_file    = $ENV{'gated_template_file'};
$default_route_gw = $ENV{'gated_default_rt_gw'};
$log_file_dir     = $ENV{'gated_mainlog_dir'};
$log_file_name    = $ENV{'gated_log_file_name'};

$success = 1;
if ( $gated_scripts eq "" )
{
   print "   *** ERROR - \$gated_scripts not set\n";
   $success = 0;
}
if ( $machine_map_file eq "" )
{
   print "   *** ERROR - \$gated_machine_map not set\n";
   $success = 0;
}
if ( $template_file eq "" )
{
   print "   *** ERROR - \$gated_template_file not set\n";
   $success = 0;
}
if ( $default_route_gw eq "" )
{
   print "   *** ERROR - \$gated_default_rt_gw not set\n";
   $success = 0;
}
if ( $log_file_dir eq "" )
{
   print "   *** ERROR - \$gated_mainlog_dir not set\n";
   $success = 0;
}
if ( $log_file_name eq "" )
{
   print "   *** ERROR - \$gated_log_file_name not set\n";
   $success = 0;
}

if ( $success == 0 )
{
   die "   Edit and run the file 'main_setup.csh' to set up the necessary environment variables.\n";
}

print "     gated script directory = \"$gated_scripts\"\n\n";

#
#  Initial option values (defaults)
#
@internal_static_osi_addr = ();
@internal_static_ip_addr  = ();
$default_ip_route         = "**NOT SET **";

%import_template_list = ();
%export_template_list = ();
$import_order = 10000;  # for keeping array sorted in order of stuffing
$export_order = 10000;  # for keeping array sorted in order of stuffing

$ALL = "#all#";
$default_rib = "0";

@predef_var_list   = ();
@global_var_list   = ();
@sequence_var_list = ();
@machine_var_list  = ();
@test_set_var_list = ();
@entire_ribs_list  = ();
@peer_ribs_list    = ();

$num_cfg_files = 0;
$num_res_files = 0;
$num_sequences = 0;
$gl_exp_count  = 0;

$refresh_on_hup = on;
$gen_res_files  = on;
$gen_loop_code  = on;

$predef_var_list{'$PROTO_SOCK'} = idrp;
@builtin_list = ( "\&rdi", "\&net", "\&ip" );

#  Turn off generation of results files if specified
if ( $opt_E )
{
   $gen_res_files = off;
   print "Expected results files will NOT be generated\n";
}

#  Turn off export loop code if specified.
#  This is export policy to export routes to the remote RDI
#  with routes imported from this node's machine RDI.
#  Example: export { proto idrp rdi xxx } where xxx is this node's rdi.
if ( $opt_L )
{
   $gen_loop_code = off;
   print "Export loop code will NOT be generated\n";
}

#
#   Variables processed:
#   ====================
#   HEADER
#   TRACEOPTIONS
#   RDI_NUMBER
#   NET
#   PROTO_SOCK
#   CAPACITY
#   MULTI-EXIT-DISC
#   ROUTE-SERVER
#   INTERNAL_NEIGHBORS
#   EXTERNAL_NEIGHBORS
#   ISO_COMMON_STATIC_ROUTES
#   ISO_MACHINE_ROUTES
#   IP_COMMON_STATIC_ROUTES
#   IP_MACHINE_ROUTES
#   IMPORT_STATEMENTS
#   EXPORT_STATEMENTS
#

#
#  Read in information on machines.
#
&read_machine_map();

#
#  Look through the configuration file, processing lines which
#  define a sequence, ignoring comments and printing error otherwise.
#
open( TEST, $ARGV[0] )
   || die "Cannot open test configuration file \"$ARGV[0]\": $!\n";

$line_number = 0;
while ( <TEST> )
{
   TEST_SWITCH:
   {
      $line_number++;

      #  Name of the test.
      /^\s*test[\s\n]+/ && ( &begin_test, last TEST_SWITCH );

      #  Socket to use (idrp or ip)
      /^\s*socket[\s\n]+/ && ( &set_socket, last TEST_SWITCH );

      #  Gateway to use on static routes, unless overridden locally
      /^\s*gateway_all[\s\n]+/ && ( &set_gateway_all, last TEST_SWITCH );

      #  Gateway to use on static routes
      /^\s*gateway[\s\n]+/ && ( &set_gateway, last TEST_SWITCH );

      #  Override default RDI with one from another machine
      /^\s*rdi[\s\n]+/ && ( &set_rdi, last TEST_SWITCH );

      #  Begin processing a test set.
      /^\s*test_set/ && ( &begin_test_set, last TEST_SWITCH );

      #  Establish the IP routes to be used in this test set.
      /^\s*ip_routes/ && ( &process_ip_routes, last TEST_SWITCH );

      #  Establish the OSI (IDRP) routes to be used in this test set.
      /^\s*osi_routes/ && ( &process_osi_routes, last TEST_SWITCH );

      #  Specify that an RDI loop is used for import and get template to use.
      /^\s*import_rdi_loop/ && ( &set_import_template(RDI_LOOP), last TEST_SWITCH );

      #  Specify that a specific RDI is used for import and get template to use.
      /^\s*import_rdi/ && ( &set_import_template(RDI), last TEST_SWITCH );

      #  Specify that a specific RD path is used for import and get template to use.
      /^\s*import_rdpath/ && ( &set_import_template(RDPATH), last TEST_SWITCH );

      #  Specify that an RDI loop is used for export and get template to use.
      /^\s*export_rdi_loop/ && ( &set_export_template(RDI_LOOP), last TEST_SWITCH );

      #  Specify that a specific RDI is used for export and get template to use.
      /^\s*export_rdi/ && ( &set_export_template(RDI), last TEST_SWITCH );

      #  Specify that a specific RD path is used for export and get template to use.
      /^\s*export_rdpath/ && ( &set_export_template(RDPATH), last TEST_SWITCH );

      #  Processing a config file variable set.
      /^\s*setvar/ && ( &process_setvar, last TEST_SWITCH );

      #  Begin processing a test sequence.
      /^\s*sequence/ && ( &begin_sequence, last TEST_SWITCH );

      #  Begin processing a machine located in the current test sequence.
      /^\s*machine/ && ( &begin_machine, last TEST_SWITCH );

      #  Processing the "ribs" statement of a peer.
      /^\s*ribs-supported/ && ( &process_ribs_supported, last TEST_SWITCH );

      #  Processing a "rib" statement, in global or machine scope.
      /^\s*rib/ && ( &process_rib, last TEST_SWITCH );

      #  Process a list of internal neighbors.
      /^\s*internal/  && ( &process_internal_neighbors, last TEST_SWITCH );

      #  Process a list of external neighbors.
      /^\s*external/  && ( &process_external_neighbors, last TEST_SWITCH );

      #  Process an option.
      /^\s*option/  && ( &process_option, last TEST_SWITCH );

      #  End of sequence.  Create configuration files.
      /^\s*end/ && ( &end_block, last TEST_SWITCH );

      #  Ignore comment lines.
      /^\s*#/ && last TEST_SWITCH;

      #  Ignore lines of whitespace only.
      /^\s*$/ && last TEST_SWITCH;

      #  Flag an error on other lines.
      &error_and_die( "UNKNOWN LINE\n\t$_" );
   }     # end TEST_SWITCH
}     # end while <TEST>

close( TEST );

#
#  Let the user know we are done.
#
print "==========================\n";
print "Test sets processed:   ",$#test_set_list + 1,"\n";
print "Sequences processed:   ",$num_sequences,"\n";
print "Total files generated: ",$num_cfg_files + $num_res_files,"\n";
print "\nProcessing complete.\n";

#
#====================  SUBROUTINES  ==============================
#


#
#  Subroutine:  READ_MACHINE_MAP
#  Read in information on machines.
#
sub read_machine_map
{
   local ( @static_routes, $route_list, $route );

   open( MACHINES, $machine_map_file )
      || die "Cannot open machine map file: $!\n";

   print "Reading machine map file ... ";

   $i = 0;
   while ( <MACHINES> )  # assigns to Perl variable $_
   {
      LINE_SWITCH:
      {
         /^#/ && do              # comment line - ignore
         {
            last LINE_SWITCH;
         };

         /^$/ && do              # blank line - ignore
         {
            last LINE_SWITCH;
         };

         /^default_ip_route/ && do
         {
            $default_ip_route = &get_name( 'default_ip_route', $_ )
               || &error_and_die( "No default route listed" );
            print "Processing default IP route $default_ip_route ...\n";
            last LINE_SWITCH;
         };

         /^static_ip_routes/ && do
         {
            $route_list = &get_name( 'static_ip_routes', $_ )
               || &error_and_die( "No static IP routes listed" );
            @static_routes = split( /[\s,]+/, $route_list );
            foreach $route ( @static_routes )
            {
               push( @internal_static_ip_addr, $route );
               print "Processing static IP route $route ...\n";
            }
            last LINE_SWITCH;
         };

         /^static_osi_routes/ && do
         {
            $route_list = &get_name( 'static_osi_routes', $_ )
               || &error_and_die( "No static OSI routes listed" );
            @static_routes = split( /[\s,]+/, $route_list );
            foreach $route ( @static_routes )
            {
               push( @internal_static_osi_addr, $route );
               print "Processing static OSI route $route ...\n";
            }
            last LINE_SWITCH;
         };

         # If we got here, it is a machine line.
         #
         ($machine[$i], $rdi[$i], $ip_addr_prefix[$i], $osi_addr_prefix[$i],
          $nlri_prefix[$i], $capacity[$i], $ip_address[$i], $osi_address[$i]) =
             split( /\s+/ );

         #  Make a copy of RDI list for resetting after each sequence
         #   (rdi can be overridden in a sequence)
         $save_rdi[$i] = $rdi[$i];

         if ( $osi_address[$i] =~ /\n$/ )
         {
            chop( $osi_address[$i] );   # Remove trailing newline.
         }

         # Assign some default values to switches.
         $route_server_switch[$i] = off;
         $multi_exit_switch[$i] = off;
         $ip_gw[$i] = $default_route_gw;
         $osi_gw[$i] = $osi_address[0];

         $i++;
      }     # end LINE_SWITCH
   }     # end while

   print $#machine + 1," machine(s) listed\n\n";


   close( MACHINES );
}

#
#  Subroutine:  CREATE_CONFIG_FILE
#  Create a gated configuration file for the given test name,
#  sequence, and subsequence.
#

sub create_config_file
{
   local ( $curr_mach_idx, $curr_seq, $curr_test_set ) = @_;
   local ( $config_file_name, $ribs, $template );
   local ( @int_neighbors, @ext_neighbors );
   local ( $gateway, $rib );
   local ( $sequence, $mach_idx );
   local ( @peers, $peer_idx, $peer );
   local ( $source_idx, $source );
   local ( @imp_exp_rdi_list, %rib_list );

   $config_file_name = "gated.conf.".$test_name.".".$machine[$curr_mach_idx]
                       .".".$curr_seq.".".$curr_test_set;
   print "Generated configuration file name = $config_file_name\n";

   #  Need to set for variable substitution (&find_variable).
   $g001_mach_idx      = $curr_mach_idx;
   $g002_curr_sequence = $curr_seq;
   $g003_curr_test_set = $curr_test_set;

   #  Determine the route ranges to produce.
   @osi_route_range = ();
   ( $osi_route_list[$curr_test_set] eq 'some' ) && do {
      @osi_route_range = ( 0 ); };
   ( $osi_route_list[$curr_test_set] eq 'all' ) && do {
      @osi_route_range = ( 0..9,'a'..'f' ); };

   @ip_route_range = ();
   ( $ip_route_list[$curr_test_set] eq 'some' ) && do {
      @ip_route_range = ( 0 ); };
   ( $ip_route_list[$curr_test_set] eq 'all' ) && do {
      @ip_route_range = ( 0..9 ); };

   #
   #  Generate a unique list of RDI from the peer list.
   #
   @imp_exp_rdi_list = &gen_rdi_list( $curr_mach_idx );

   #
   #  Open the gated configuration file template.
   #
   open( CFG_TEMPLATE, $template_file )
      || die "Cannot open template file: $! for input\n";

   #
   #  Open the output configuration file.
   #
   open( CONFIG_FILE, ">".$config_file_name )
      || die "Cannot open configuration file: $! for output\n";

   #
   #  Process the template, filling in variables, just printing
   #  each line otherwise.
   #
   while ( <CFG_TEMPLATE> )
   {
      LINE_SWITCH:
      {
         #  Print comment lines, but do not look for variables.
         /^\s*#/ && do
         {
            print CONFIG_FILE $_;
            last LINE_SWITCH;
         };

         /\$HEADER/ && do
         {
            print CONFIG_FILE "# $machine[$curr_mach_idx] gated configuration file\n";
            print CONFIG_FILE "# Generated by SCRIPT  $date\n";
            print CONFIG_FILE "#    Script file name: $PROGRAM_NAME\n";
            print CONFIG_FILE "#    Sequence $curr_seq, $curr_test_set of Test $test_name\n";
            last LINE_SWITCH;
         };

         /\$TRACEOPTIONS/ && do
         {
            print CONFIG_FILE "traceoptions \"$log_file_dir/$log_file_name\"\n";
            last LINE_SWITCH;
         };

         /\$RDI_NUMBER/ && do
         {
            print CONFIG_FILE "rdi $rdi[$curr_mach_idx];\n";
            last LINE_SWITCH;
         };

         /\$NET/ && do
         {
            print CONFIG_FILE "net $osi_address[$curr_mach_idx];\n";
            last LINE_SWITCH;
         };

         #  Replace var with current machine's capacity (from machine map).
         /\$CAPACITY/ && do
         {
            s/\$CAPACITY/$capacity[$curr_mach_idx]/;
            print CONFIG_FILE $_;
            last LINE_SWITCH;
         };

         /\$MULTI-EXIT-DISC/ && do
         {
            s/\$MULTI-EXIT-DISC/$multi_exit_switch[$curr_mach_idx]/;
            print CONFIG_FILE $_;
            last LINE_SWITCH;
         };

         /\$ROUTE-SERVER/ && do
         {
            s/\$ROUTE-SERVER/$route_server_switch[$curr_mach_idx]/;
            print CONFIG_FILE $_;
            last LINE_SWITCH;
         };

         /\$RIB_LOOP\b/ && do
         {
            #  Print the RIB template for every RIB in the list for this machine.
            &generate_rib_list( $g002_curr_sequence, $g001_mach_idx, *rib_list );
            while ( ($key,$template) = each %rib_list )
            {
               ($sequence, $mach_idx, $rib) = split( "$;", $key );
               &process_rib_template( $template, $rib );
            }
            last LINE_SWITCH;
         };

         /\$INTERNAL_NEIGHBORS/ && do
         {
            @int_neighbors = split( ":", $int_neighbor_list[$curr_mach_idx] );
            foreach( @int_neighbors )
            {
               $neighbor_idx = &find_machine( $_ );
               $peer = $machine[$neighbor_idx];

               print CONFIG_FILE "\n\t\t# (ON) internal neighbor $peer\n";
               print CONFIG_FILE "\t\tneighbor on net $osi_address[$neighbor_idx]";
               print CONFIG_FILE " $ip_address[$neighbor_idx]";
               print CONFIG_FILE " rdi $rdi[$curr_mach_idx]";
               print CONFIG_FILE "\n\t\t";
               print CONFIG_FILE "intf $ip_address[$curr_mach_idx]";
               print CONFIG_FILE " net $osi_address[$curr_mach_idx]";
               print CONFIG_FILE " proto-sock $predef_var_list{'$PROTO_SOCK'}";
               print CONFIG_FILE "\n\t\t";
               print CONFIG_FILE "refresh-on-hup $refresh_on_hup";

               $ribs = $peer_ribs_list{$g002_curr_sequence, $curr_mach_idx, $peer};
               if ( $ribs ne "") {
                  print CONFIG_FILE " ribs-supported $ribs"; }

               print CONFIG_FILE ";\n";
            }
            last LINE_SWITCH;
         };

         /\$EXTERNAL_NEIGHBORS/ && do
         {
            @ext_neighbors = split( ":", $ext_neighbor_list[$curr_mach_idx] );
            foreach( @ext_neighbors )
            {
               $neighbor_idx = &find_machine( $_ );
               $peer = $machine[$neighbor_idx];

               print CONFIG_FILE "\n\t\t# (ON) external neighbor $peer\n";
               print CONFIG_FILE "\t\tneighbor on net $osi_address[$neighbor_idx]";
               print CONFIG_FILE " $ip_address[$neighbor_idx]";
               print CONFIG_FILE " rdi $rdi[$neighbor_idx]";
               print CONFIG_FILE "\n\t\t";
               print CONFIG_FILE "intf $ip_address[$curr_mach_idx]";
               print CONFIG_FILE " net $osi_address[$curr_mach_idx]";
               print CONFIG_FILE " proto-sock $predef_var_list{'$PROTO_SOCK'}";
               print CONFIG_FILE "\n\t\t";
               print CONFIG_FILE "refresh-on-hup $refresh_on_hup";

               $ribs = $peer_ribs_list{$g002_curr_sequence, $curr_mach_idx, $peer};
               if ( $ribs ne "") {
                  print CONFIG_FILE " ribs-supported $ribs"; }

               print CONFIG_FILE ";\n";
            }
            last LINE_SWITCH;
         };

         #
         #  These are routes which every machine has declared static.
         /\$ISO_COMMON_STATIC_ROUTES/ && do
         {
            if ( @osi_route_range != () )
            {
               foreach( @internal_static_osi_addr )
               {
                  $gateway = $osi_gw[$curr_mach_idx];
                  print CONFIG_FILE "\t$_\t\tgw $gateway;\n";
               }
            }
            last LINE_SWITCH;
         };

         #
         #  These are routes which every machine has declared static.
         /\$IP_COMMON_STATIC_ROUTES/ && do
         {
            # Install the "default" route
            print CONFIG_FILE "\t$default_ip_route\t\tgw $default_route_gw;\n";

            # Install the other IP static routes
            if ( @ip_route_range != () )
            {
               foreach( @internal_static_ip_addr )
               {
                  $gateway = $ip_gw[$curr_mach_idx];
                  print CONFIG_FILE "\t$_\t\tgw $gateway;\n";
               }
            }
            last LINE_SWITCH;
         };

         /\$ISO_MACHINE_ROUTES/ && do
         {
            foreach $i ( @osi_route_range )
            {
               $gateway = $osi_gw[$curr_mach_idx];
               $nlri = &generate_osi_address( $curr_mach_idx, $i );
               $nlri = $nlri."\t\tgw $gateway;";
               print CONFIG_FILE "\t$nlri\n";
            }
            last LINE_SWITCH;
         };

         /\$IP_MACHINE_ROUTES/ && do
         {
            foreach $i ( @ip_route_range )
            {
               $nlri = $ip_addr_prefix[$curr_mach_idx].$nlri_prefix[$curr_mach_idx].$i;
               $nlri = $nlri."\t\tgw $ip_gw[$curr_mach_idx];";
               print CONFIG_FILE "\t$nlri\n";
            }
            last LINE_SWITCH;
         };

         /\$IMPORT_STATEMENTS/ && do
         {
#           print "Processing import statements...\n";
            &process_import_list;
            last LINE_SWITCH;
         };

         /\$EXPORT_STATEMENTS/ && do
         {
#           print "Processing export statements...\n";
            &process_export_list;
            last LINE_SWITCH;
         };

         # Substitute for any user-defined variables.
         &process_user_vars;  # will use $_

         print CONFIG_FILE $_;
      }

   }  # end while

   close ( CONFIG_FILE );
   close ( CFG_TEMPLATE );
}


#
#  Subroutine:  GENERATE_OSI_ADDRESS
#   Generate an OSI address from prefixes and an input index.
#   If necessary, break up each 4-character interval with ".".
#
sub generate_osi_address
{
   local ( $mach_idx, $index ) = @_;
   local ( $address, @parts );

   $address = $osi_addr_prefix[$mach_idx].$nlri_prefix[$mach_idx];

   #  Check to see if address should be in NN.NNNN.NNNN format.
   #  Ignore if using 0xNNNNNNNNNNN format.
   if ( $address =~ /\./ )
   {
      @parts = split( "\\.", $address );

      #  Check length of last segment of address.  If equal to 4, put in
      #  a period before appending the index.
      if ( length($parts[$#parts]) == 4 ) {
         $address = $address.".".$index; }
      else {
         $address = $address.$index; }
   }

   return ( $address );
}

#
#  Subroutine:  PROCESS_RIB_TEMPLATE
#
sub process_rib_template
{
   local ( $rib_template, $rib ) = @_;
   local ( $token, $value );

#  print "Current RIB = $rib\n";

   open( RIB_TEMPLATE, $rib_template )
      || die "Cannot open rib template file \"$rib_template\": $!\n";

   while ( <RIB_TEMPLATE> )
   {
      #
      # If a match is made with the following set of variables,
      # make the appropriate substitution.
      #
      if ( s/\$RIB\b/$rib/ )
      {
#        print "*** Substitution: New line = $_";
      }

      # Substitute for any user-defined variables.
      &process_user_vars;  # will use $_

      print CONFIG_FILE $_;
   }
   close ( RIB_TEMPLATE );
}

#
#  Subroutine:  GENERATE_RIB_LIST
#  Generate a list of RIBs for a given sequence/machine pair.
#
sub generate_rib_list
{
    local ( $curr_sequence, $curr_mach_idx, *generated_rib_list ) = @_;
    local ( $key, $template, $sequence, $mach_idx, $rib );
    local ( %remaining_rib_list );
 
    %generated_rib_list = ();
    %remaining_rib_list = ();

    #
    #  Look through the entire set of defined RIBs.  Start with those
    #  which match globally, override if a local definition redefines it.
    #
    while ( ($key,$template) = each %entire_ribs_list )
    {
       #  Split key into its components (assoc. array separator is $;).
       ($sequence, $mach_idx, $rib) = split( "$;", $key );
 
       if ( $sequence eq "$ALL" )
       {
          if ( $mach_idx ne "$ALL" ) {
              &error_and_die( "Unknown RIB configuration" ); }
          $generated_rib_list{$curr_sequence, $curr_mach_idx, $rib} = $template;
#         print "*** Global:  Handling RIB $sequence, $mach_idx, $rib\n";
       }
       else
       {
          $remaining_rib_list{$sequence, $mach_idx, $rib} = $template;
       }
    }
 
    #  Now process sequence scope RIBs.
    while ( ($key,$template) = each %remaining_rib_list )
    {
       #  Split key into its components (assoc. array separator is $;).
       ($sequence, $mach_idx, $rib) = split( "$;", $key );
 
       if ( ($sequence eq $curr_sequence) && ($mach_idx eq "$ALL") )
       {
          $generated_rib_list{$curr_sequence, $curr_mach_idx, $rib} = $template;
#         print "*** Sequence:  Handling RIB $sequence, $mach_idx, $rib\n";
       }
    }
 
    #  Now process machine scope RIBs.
    while ( ($key,$template) = each %remaining_rib_list )
    {
       #  Split key into its components (assoc. array separator is $;).
       ($sequence, $mach_idx, $rib) = split( "$;", $key );
 
       if ( ($sequence eq $curr_sequence) && ($mach_idx eq $curr_mach_idx) )
       {
          $generated_rib_list{$curr_sequence, $curr_mach_idx, $rib} = $template;
#         print "*** Machine:  Handling RIB $sequence, $mach_idx, $rib\n";
       }
    }

#   while ( ($key,$template) = each %generated_rib_list )
#   {
#      ($sequence, $mach_idx, $rib) = split( "$;", $key );
#      print "--- Generated RIB list entry = $sequence, $mach_idx, $rib\n";
#   }

}

#
#  Subroutine:  PROCESS_USER_VARS
#
sub process_user_vars
{
   local ( $token, $value );

   if ( /\$/ )      # compares against $_
   {
      foreach $token ( split(/\s+/) )
      {
#        print "*** process_user_vars:  Token = \"$token\"\n";
         if ( $token =~ /^\$/ )
         {
            $value = &find_variable( $token );
            $token = "\\".$token;
            s/$token/$value/;   # replaces token in $_
#           print "*** Substitution: New line = $_";
         }
      }
   }
}

#
#  Subroutine:  PROCESS_IMPORT_LIST
#
sub process_import_list
{
   local ( $order, $test_set, $type, $index );
   local ( $key, $template );
   local ( $peer );

   #  Sorting the keys insures the array entries are returned in the
   #  order in which they were added (because the first part of the
   #  key is a large incrementing number).
   foreach $key (sort(keys %import_template_list))
   {
      $template = $import_template_list{$key};
      ($order, $test_set, $type, $index) = split( "$;", $key );
#     print "Import test set array{ $order, $test_set, $type, $index } = $template\n";

      if ( $test_set eq $g003_curr_test_set )
      {
         TYPE_SWITCH:
         {
            ( $type eq "RDI_LOOP" ) && do
            {
               foreach $peer ( @imp_exp_rdi_list ) {
                  &process_import_template( $peer, $template ); }
               last TYPE_SWITCH;
            };

            ( $type eq "RDI" ) && do
            {
               &process_import_template( "", $template );
               last TYPE_SWITCH;
            };

            ( $type eq "RDPATH" ) && do
            {
               &process_import_template( "", $template );
               last TYPE_SWITCH;
            };

            &error_and_die( "Unknown import type" );
         }   # end TYPE_SWITCH
      }   # end if
   }   # end foreach
}

#
#  Subroutine:  PROCESS_IMPORT_TEMPLATE
#
sub process_import_template
{
   local ( $rdi_entry, $imp_template ) = @_;
   local ( $token, $value );

   open( IMP_TEMPLATE, $imp_template )
      || die "Cannot open import template file \"$imp_template\": $!\n";

   while ( <IMP_TEMPLATE> )
   {
      #
      # If a match is made with the following set of variables,
      # make the appropriate substitution.
      #
      if ( s/\$PEER\b/$rdi_entry/ )
      {
#        print "*** Substitution: New line = $_";
      }

      # Substitute for any user-defined variables.
      &process_user_vars;  # will use $_

      print CONFIG_FILE $_;
   }
   close ( IMP_TEMPLATE );
}

#
#  Subroutine:  PROCESS_EXPORT_LIST
#
sub process_export_list
{
   local ( $order, $test_set, $type, $index );
   local ( $key, $template );
   local ( $peer );

   #  Sorting the keys insures the array entries are returned in the
   #  order in which they were added (because the first part of the
   #  key is a large incrementing number).
   foreach $key (sort(keys %export_template_list))
   {
      $template = $export_template_list{$key};
      ($order, $test_set, $type, $index) = split( "$;", $key );
#     print "Export test set array{ $order, $test_set, $type, $index } = $template\n";

      if ( $test_set eq $g003_curr_test_set )
      {
         TYPE_SWITCH:
         {
            ( $type eq "RDI_LOOP" ) && do
            {
               foreach $peer ( @imp_exp_rdi_list ) {
                  &process_export_template( $peer, $template ); }
               last TYPE_SWITCH;
            };

            ( $type eq "RDI" ) && do
            {
               &process_export_template( "", $template );
               last TYPE_SWITCH;
            };

            ( $type eq "RDPATH" ) && do
            {
               &process_export_template( "", $template );
               last TYPE_SWITCH;
            };

            &error_and_die( "Unknown export type" );
         }   # end TYPE_SWITCH
      }   # end if
   }   # end foreach
}

#
#  Subroutine:  PROCESS_EXPORT_TEMPLATE
#
sub process_export_template
{
   local ( $peer, $exp_template ) = @_;
   local ( $sub_template, $peer2, $count, $i, $save_line, $new_pattern );
   local ( $token, $value );
   local ( $exp_filehandle );

   # Calculate a new filehandle name each time so we don't step on it
   #  when doing recursive calls to this routine.
   $exp_filehandle = "EXP_TEMPLATE_".$gl_exp_count++;
   open( $exp_filehandle, $exp_template )
      || die "Cannot open export template file \"$exp_template\": $!\n";

   $count = 0;
   while ( <$exp_filehandle> )
   {
      #
      # If a match is made with the following set of variables,
      # make the appropriate substitution.
      #
      if ( s/\$PEER\b/$peer/ )
      {
#        print "*** Substitution: New line = $_";
      }

      if ( /\$RDI_LOOP\b/ )
      {
         $sub_template = &get_name( '\$RDI_LOOP', $_ )
               || die "***** ERROR -- No sub-template listed\n";

         for ( $i = 0; $i <= $#imp_exp_rdi_list; $i++ )
         {
            $peer2 = $imp_exp_rdi_list[$i];
            #
            # Avoid loop code if specified, otherwise process peer
            if ( ($rdi[$g001_mach_idx] != $peer2) || ($gen_loop_code eq "on") )
            {
#              print "Peer 2 = imp_exp_rdi_list[$i] = $imp_exp_rdi_list[$i]\n";
               &process_export_template( $peer2, $sub_template );
            }
         }
      }

      # Substitute for any user-defined variables.
      &process_user_vars;  # will use $_

      print CONFIG_FILE $_;
   }
   close ( $exp_filehandle );
}

#
#  Subroutine:  FIND_VARIABLE
#  The config file allows for setting of pseudo-variables.
#  Look up the value of one of these variables in the setvar tables.
#
sub find_variable
{
   local ( $token ) = @_;
   local ( $value );

   # First, check the predefined variable list.
   if ( $predef_var_list{$token} ne "" ) {
      $value = $predef_var_list{$token}; }

   # See if user has set it globally.  Override predefined.
   if ( $global_var_list{$token} ne "" ) {
      $value = $global_var_list{$token}; }

   # See if user has set it inside the current sequence.  Override global.
   if ( $sequence_var_list{$g002_curr_sequence, $token} ne "" ) {
      $value = $sequence_var_list{$g002_curr_sequence, $token}; }

   # See if user has set it inside the current machine.  Override sequence.
   if ( $machine_var_list{$g002_curr_sequence, $g001_mach_idx, $token} ne "" ) {
      $value = $machine_var_list{$g002_curr_sequence, $g001_mach_idx, $token} };

   # See if user has set it inside the current test set.  Override machine.
   if ( $test_set_var_list{$test_set, $token} ne "" ) {
      $value = $test_set_var_list{$test_set, $token}; }

   if ( $value eq "" )
   {
      if ( $print_warnings eq ON ) {
      print "WARNING:  \"$token\" has no value\n"; }
   }

#  print "find_variable: token = $token\n";
#  print "find_variable: value = $value\n";

   return( $value );
}

#
#  Subroutine:  GEN_RDI_LIST
#  Generate and return a list of the RDI of host machine and peers
#  for import/export processing.  Eliminate duplicates.
#
sub gen_rdi_list
{
   local ( $curr_mach_idx ) = @_[0];
   local ( @peers, $peer_idx, $duplicate );
   local ( @tmp_rdi_list, $tmp_rdi );
   local ( @rdi_list, $new_rdi );
   @tmp_rdi_list = ();
   @rdi_list = ();

   #  Create a complete list of RDI from the peer list.
#  print "gen_rdi_list:  Current peer list = \"$peer_list[$curr_mach_idx]\"\n";
   @peers = split( ":", $peer_list[$curr_mach_idx] );
   foreach $peer ( @peers )
   {
      $peer_idx = &find_machine( $peer );
      $tmp_rdi_list[$#tmp_rdi_list + 1] = $rdi[$peer_idx];
   }

   #  Add the RDI of the host machine.
   $tmp_rdi_list[$#tmp_rdi_list + 1] = $rdi[$curr_mach_idx];

   #  Now eliminate duplicates.
   foreach $tmp_rdi ( @tmp_rdi_list )
   {
      $duplicate = 0;
      foreach $new_rdi ( @rdi_list )
      {
         if ( $tmp_rdi eq $new_rdi ) {
            $duplicate = 1; }
      }
      if ( ! $duplicate ) {
         $rdi_list[$#rdi_list + 1] = $tmp_rdi; }
   }

#  for ( $i = 0; $i <= $#rdi_list; $i++ )
#  {
#     print "gen_rdi_list: rdi_list[$i] = $rdi_list[$i]\n";
#  }

   return ( @rdi_list );
}

#
#  Subroutine:  CREATE_RESULT_FILE
#  Create a file listing the routes expected to be in the
#  routing table for a particular test, sequence, and test set.
#
sub create_result_file
{
   local ( $curr_mach_idx, $curr_seq, $curr_test_set ) = @_;
   local ( $i, $mach, $rib, $dummy );
   local ( $key, $template, $sequence, $mach_idx );
   local ( $result_file_name );
   local ( @int_neighbors, @ext_neighbors, @mach_list, %rib_list );
   local ( @peer_ribs_supported, %supported_list );

   $result_file_name = "expected.".$test_name.".".$machine[$curr_mach_idx]
                       .".".$curr_seq.".".$curr_test_set;
   print "Expected results file name = $result_file_name\n";

   #  Determine the route ranges to produce.
   @osi_route_range = ();
   ( $osi_route_list[$curr_test_set] eq 'some' ) && do {
      @osi_route_range = ( 0 ); };
   ( $osi_route_list[$curr_test_set] eq 'all' ) && do {
      @osi_route_range = ( 0..9,'a'..'f' ); };

   @ip_route_range = ();
   ( $ip_route_list[$curr_test_set] eq 'some' ) && do {
      @ip_route_range = ( 0 ); };
   ( $ip_route_list[$curr_test_set] eq 'all' ) && do {
      @ip_route_range = ( 0..9 ); };

   #
   #  Open the output results file.
   #
   open( RESULT_FILE, ">".$result_file_name )
      || die "Cannot open expected results file: $! for output\n";

   #
   #  The expected routes include the IP and ISO routes of the
   #  currrent machine and all its neighbors.
   #
   @mach_list  = ( $curr_mach_idx );
   @int_neighbors = split( ":", $int_neighbor_list[$curr_mach_idx] );
   @ext_neighbors = split( ":", $ext_neighbor_list[$curr_mach_idx] );
   foreach( @int_neighbors, @ext_neighbors )
   {
      push( @mach_list, &find_machine($_) );
   }

   #  Create a unique list of all "ribs-supported" by this machine's peers.
   #  If any say "ribs-supported all", use all RIBs defined for this machine.
   foreach $peer ( @int_neighbors, @ext_neighbors )
   {
      #  Always use the default RIB.
      @peer_ribs_supported = ( $default_rib ) ;

      #  Add specific RIBs for this peer.
      push( @peer_ribs_supported, split( /\s+/,
           $peer_ribs_list{$curr_seq, $curr_mach_idx, $peer}) );

      foreach $rib ( @peer_ribs_supported )
      {
         if ( $rib eq "all" )
         {
            # Generate a list of all RIBS supported by this machine.
            &generate_rib_list( $curr_seq, $curr_mach_idx, *rib_list );
            while ( ($key,$template) = each %rib_list )
            {
               ($sequence, $mach_idx, $rib) = split( "$;", $key );
               $supported_list{$rib} = ON;
            }
         }
         else
         {
            $supported_list{$rib} = ON;
         }
      }
   }

   #  Put each supported RIB into expected results file.
   while ( ($rib, $dummy) = each %supported_list )
   {
      #  Do the default IP addresses.
      if ( @ip_route_range != () )
      {
         foreach( @internal_static_ip_addr )
         {
            print RESULT_FILE "$rib\t$_\n";
         }
      }

      #  Do the IP addresses for all machines in the sequence/test set.
      foreach $mach ( @mach_list )
      {
         foreach $i ( @ip_route_range )
         {
            $nlri = $ip_addr_prefix[$mach].$nlri_prefix[$mach].$i;
            print RESULT_FILE "$rib\t$nlri\n";
         }
      }

      #  Do the default ISO addresses.
      if ( @osi_route_range != () )
      {
         foreach( @internal_static_osi_addr )
         {
            print RESULT_FILE "$rib\t$_\n";
         }
      }

      #  Do the ISO addresses for all machines in the sequence/test set.
      foreach $mach ( @mach_list )
      {
         foreach $i ( @osi_route_range )
         {
            $nlri = $osi_addr_prefix[$mach].$nlri_prefix[$mach].$i;
            print RESULT_FILE "$rib\t$nlri\n";
         }
      }
   }     # while loop - each rib

   close ( RESULT_FILE );
}

#
#  Subroutine:  BEGIN_TEST
#
sub begin_test
{
   $g005_process_scope = GLOBAL;
   @test_set_list = ();

   $test_name = &get_name( 'test', $_ )
             || &error_and_die( "Test has no name" );
   print "Now beginning test $test_name ...\n";
}

#
#  Subroutine:  SET_SOCKET
#
#    Establishes whether to use IDRP or IP socket to carry routes.
#
sub set_socket
{
   $socket_name = &get_name( 'socket', $_ )
             || &error_and_die( "No socket type specified" );
   print "Now processing socket $socket_name ...\n";

   if ( ($socket_name ne "idrp") && ($socket_name ne "ip") ) {
      &error_and_die( "Illegal socket type \"$socket_name\"" );
   }

   $predef_var_list{'$PROTO_SOCK'} = $socket_name;
}

#
#  Subroutine:  SET_GATEWAY
#
#    Sets the gateway (next hop) for static routes.
#
sub set_gateway
{
   local ( $ip_mach_idx, $gateway_name );

   $gateway_name = &get_name( 'gateway', $_ )
      || &error_and_die( "No gateway machine specified" );

   # Verify that the target machine is in the machine map
   $ip_mach_idx = &find_machine( $gateway_name );

   # Set gateway of current machine to target machine
   $osi_gw[$g001_mach_idx] = $osi_address[$ip_mach_idx];
   $ip_gw[$g001_mach_idx]  = $ip_address[$ip_mach_idx];
}

#
#  Subroutine:  SET_GATEWAY_ALL
#
#    Sets the gateway (next hop) to the specified machine
#    for static addresses on all machines.
#
sub set_gateway_all
{
   local ( $ip_mach_idx, $gateway_name );

   $gateway_name = &get_name( 'gateway_all', $_ )
      || &error_and_die( "No gateway machine specified" );

   # Verify that the target machine is in the machine map
   $ip_mach_idx = &find_machine( $gateway_name );

   for ( $i = 0; $i <= $#machine; $i++ )
   {
      $osi_gw[$i] = $ip_address[$ip_mach_idx];
      $ip_gw[$i]  = $osi_address[$ip_mach_idx];
   }
}

#
#  Subroutine:  SET_RDI
#
#    Override a machine's default RDI with one from another machine.
#
sub set_rdi
{
   local ( $rdi_mach_idx );

   $rdi_name = &get_name( 'rdi', $_ )
      || &error_and_die( "No machine specified for RDI" );

   # Verify that the target machine is in the machine map
   $rdi_mach_idx = &find_machine( $rdi_name );

   # Set gateway of current machine to target machine
   $rdi[$g001_mach_idx] = $rdi[$rdi_mach_idx];

   print "Assigning RDI of machine $rdi_name ...\n";
}

#
#  Subroutine:  SET_IMPORT_TEMPLATE
#
#    Sets the import template file name for a test set.
#    Takes the kind of import as input parameter.
#
sub set_import_template
{
   local ( $type ) = @_;
   local ( $template_name );

   TYPE_SWITCH:
   {
      #  Add an RDI loop to import.  Note that a loop is legal only once,
      #  so it does not stack like the other types.  Further loop statements
      #  will simply overwrite the previous values.
      ( $type eq "RDI_LOOP" ) && do
      {
         $template_name = &get_name( 'import_rdi_loop', $_ )
            || &error_and_die( "No template specified" );
         $import_template_list{$import_order++, $g003_curr_test_set, RDI_LOOP,
                               0 } = $template_name;
         last TYPE_SWITCH;
      };

      #  Add this RDI to the list for this test set.
      ( $type eq "RDI" ) && do
      {
         $template_name = &get_name( 'import_rdi', $_ )
            || &error_and_die( "No template specified" );

         $import_template_list{$import_order++, $g003_curr_test_set, RDI,
                               $import_rdi_count++} = $template_name;
         last TYPE_SWITCH;
      };

      #  Add this RD path to the list for this test set.
      ( $type eq "RDPATH" ) && do
      {
         $template_name = &get_name( 'import_rdpath', $_ )
            || &error_and_die( "No template specified" );

         $import_template_list{$import_order++, $g003_curr_test_set, RDPATH,
                               $import_rdpath_count++} = $template_name;
         last TYPE_SWITCH;
      };

      &error_and_die( "Unknown import type" );
   }
}

#
#  Subroutine:  SET_EXPORT_TEMPLATE
#
#    Sets the export template file name for a test set.
#    Takes the kind of import as input parameter.
#
sub set_export_template
{
   local ( $type ) = @_;
   local ( $template_name );

   TYPE_SWITCH:
   {
      #  Add an RDI loop to export.  Note that a loop is legal only once,
      #  so it does not stack like the other types.  Further loop statements
      #  will simply overwrite the previous values.
      ( $type eq "RDI_LOOP" ) && do
      {
         $template_name = &get_name( 'export_rdi_loop', $_ )
            || &error_and_die( "No template specified" );
         $export_template_list{$export_order++, $g003_curr_test_set, RDI_LOOP,
                               0 } = $template_name;
         last TYPE_SWITCH;
      };

      #  Add this RDI to the list for this test set.
      ( $type eq "RDI" ) && do
      {
         $template_name = &get_name( 'export_rdi', $_ )
            || &error_and_die( "No template specified" );

         $export_template_list{$export_order++, $g003_curr_test_set, RDI,
                               $export_rdi_count++} = $template_name;
         last TYPE_SWITCH;
      };

      #  Add this RD path to the list for this test set.
      ( $type eq "RDPATH" ) && do
      {
         $template_name = &get_name( 'export_rdpath', $_ )
            || &error_and_die( "No template specified" );

         $export_template_list{$export_order++, $g003_curr_test_set, RDPATH,
                               $export_rdpath_count++} = $template_name;
         last TYPE_SWITCH;
      };

      &error_and_die( "Unknown export type" );
   }
}

#
#  Subroutine:  BEGIN_TEST_SET
#
sub begin_test_set
{
   $g005_process_scope = TEST_SET;

   $g003_curr_test_set = &get_name( 'test_set', $_ )
             || &error_and_die( "Test set has no name" );

   push( @test_set_list, $g003_curr_test_set );
   print "Now processing test_set $g003_curr_test_set ...\n";

   #  Clear the import and export RDI list arrays and indexes.
   $import_rdi_count     = 0;
   $import_rdpath_count  = 0;

   $export_rdi_count     = 0;
   $export_rdpath_count  = 0;
}


#
#  Subroutine:  PROCESS_SETVAR
#  Add a variable and its value to the appropriate variable list.
#  Format: "variable=value"
#
sub process_setvar
{
   local ( $var_pair, $var_name, $var_value );
   local ( $builtin );

   $var_pair = &get_name( 'setvar', $_ )
             || &error_and_die( "Empty setvar statement" );

   ( $var_name, $var_value ) = split( "=", $var_pair );

   #  Allow a null value, but not a null variable name.
   if ( $var_name eq "" ) {
      &error_and_die( "No variable name for \"$var_pair\"" ); }
#  if ( $var_value eq "" ) {
#     &error_and_die( "No variable value for \"$var_pair\"" ); }

   #
   #  Check for any built-in functions.  Substitute if found.
   #
   foreach $builtin ( @builtin_list )
   {
      while ( $var_value =~ /$builtin\(.*\)/ ) {
         &subs_builtin($var_value, $builtin); }
   }

   SCOPE_SWITCH:
   {
      ( $g005_process_scope eq "GLOBAL" ) && do
      {
         $global_var_list{$var_name} = $var_value;
#        print "process_setvar: Global variable $var_name set to $var_value\n";
         last SCOPE_SWITCH;
      };

      ( $g005_process_scope eq "SEQUENCE" ) && do
      {
         $sequence_var_list{$g002_curr_sequence, $var_name} = $var_value;
#        print "process_setvar: Variable $var_name set to $var_value";
#        print " for sequence $g002_curr_sequence\n";
         last SCOPE_SWITCH;
      };

      ( ($g005_process_scope eq "MACHINE") ||
        ($g005_process_scope eq "PEER") ) && do
      {
         $machine_var_list{$g002_curr_sequence, $g001_mach_idx, $var_name} = $var_value;
#        print "process_setvar: Variable $var_name set to $var_value";
#        print " for machine $machine[$g001_mach_idx] ";
#        print " in sequence $g002_curr_sequence\n";
         last SCOPE_SWITCH;
      };

      ( $g005_process_scope eq "TEST_SET" ) && do
      {
         $test_set_var_list{$g003_curr_test_set, $var_name} = $var_value;
#        print "process_setvar: Variable $var_name set to $var_value";
#        print " for test set $g003_curr_test_set\n";
         last SCOPE_SWITCH;
      };

      &error_and_die( "Unknown process scope" );
   }
}

#
#  Subroutine:  SUBS_BUILTIN
#     Substitute information about a machine in a variable string.
#
sub subs_builtin
{
   local ( $str, $builtin ) = @_;   # string to substitute in
   local ( $pattern, $mach_idx );

   #  The pattern we are looking for is "&<builtin>(machine)"
   #  without worrying about whitespace in between the ().
   #  Putting () around the \\S expression puts the machine name in $+.
   $pattern = "${builtin}\\(\\s*(\\S*)\\s*\\)";
   if ( $str =~ /$pattern/ )
   {
      #  If no machine is specified - i.e. &ip() - use current machine.
      $mach_idx =  ( $+ eq "" ) ? $g001_mach_idx : &find_machine( $+ );

      TYPE_SWITCH:
      {
         ( $builtin eq "&rdi" ) && do
         {
            $str =~ s/$pattern/$rdi[$mach_idx]/;
            last TYPE_SWITCH;
         };

         ( $builtin eq "&net" ) && do
         {
            $str =~ s/$pattern/$osi_address[$mach_idx]/;
            last TYPE_SWITCH;
         };

         ( $builtin eq "&ip" ) && do
         {
            $str =~ s/$pattern/$ip_address[$mach_idx]/;
            last TYPE_SWITCH;
         };

         &error_and_die( "Unknown builtin - \"$builtin\"" );
      }
      @_[0] = $str;
#     print "New string is \"$str\"\n";
   }
   else {
      &error_and_die( "Incorrect use of $builtin - \"$str\"" ); }
}

#
#  Subroutine:  BEGIN_SEQUENCE
#
sub begin_sequence
{
   $g005_process_scope = SEQUENCE;
   @machine_list = ();
   @peer_list = ();
   @int_neighbor_list = ();
   @ext_neighbor_list = ();

   $g002_curr_sequence = &get_name( 'sequence', $_ )
             || &error_and_die( "Sequence has no name" );
   print "Now beginning sequence $g002_curr_sequence ...\n";
}

#
#  Subroutine:  END_BLOCK
#
sub end_block
{
   SCOPE_SWITCH:
   {
      ( $g005_process_scope eq "SEQUENCE" ) && do
      {
         &end_sequence;
         last SCOPE_SWITCH;
      };

      # For right now, no "end" statements for machine clauses.
      #   An "end" ends a sequence.
      ( ($g005_process_scope eq "MACHINE") ||
        ($g005_process_scope eq "PEER") ) && do
      {
         &end_sequence;
         last SCOPE_SWITCH;
      };

      ( $g005_process_scope eq "TEST_SET" ) && do
      {
         &end_test_set;
         last SCOPE_SWITCH;
      };

      &error_and_die( "Unmatched END statement" );
   }
}

#
#  Subroutine:  END_MACHINE
#
#  Ends a machine clause.  Resets process scope to SEQUENCE for user-defined
#  variable purposes.
#  NOTE:  an "end" statement is not required for a machine clause, as it is
#         for a sequence.
#
sub end_machine
{
   print "End of machine $machine[$g001_mach_idxx] ...\n";
   $g005_process_scope = SEQUENCE;
}

#
#  Subroutine:  END_TEST_SET
#
#  Ends a test set.  Resets process scope to GLOBAL for user-defined
#  variable purposes.
#  NOTE:  an "end" statement is not required for a test set, as it is
#         for a sequence.
#
sub end_test_set
{
   print "End of test set $g003_curr_test_set ...\n";
   $g005_process_scope = GLOBAL;
}

#
#  Subroutine:  END_SEQUENCE
#
sub end_sequence
{
   local ( $test_set, $mach_idx, $machine_name, $i );

   print "Ending sequence $g002_curr_sequence ...\n";
   $num_sequences++;

   #
   #  For each machine in the sequence, for each test set defined,
   #  generate a configuration file.  Restrict to a particular
   #  machine if specified in -m option.
   #
   foreach $machine_name ( @machine_list )
   {
      if ( ($opt_m eq "") || ($opt_m eq $machine_name) )
      {
         foreach $test_set ( @test_set_list )
         {
            $mach_idx = &find_machine( $machine_name );
            &create_config_file( $mach_idx, $g002_curr_sequence, $test_set );
            if ( $gen_res_files eq "on" )
            {
               &create_result_file( $mach_idx, $g002_curr_sequence, $test_set );
               $num_res_files++;
            }
            $num_cfg_files++;
         }
      }     # endif
   }    # end foreach $machine_name

   #  Reset the RDI to their original values (in machine map)
   for ( $i = 0; $i <= $#machine; $i++ )
   {
         $rdi[$i] = $save_rdi[$i];
   }

   #  Reset process scope to global.
   $g005_process_scope = GLOBAL;
}

#
#  Subroutine:  BEGIN_MACHINE
#
sub begin_machine
{
   $g005_process_scope = MACHINE;

   @g004_curr_peers = ();

   $machine_name = &get_name( 'machine', $_ )
             || &error_and_die( "Machine has no name" );

   $g001_mach_idx = &find_machine( $machine_name );
   if ( $g001_mach_idx == -1 )
   {
      &error_and_die( "Machine $machine_name not found in table" );
   }

   print "Now processing machine $machine_name ...\n";

   push( @machine_list, $machine_name );
}


#
#  Subroutine:  PROCESS_RIB
#  Proces a "rib" statement at global or machine scope.
#  Format of rib statement:  rib <rib_id>=<template>
#
sub process_rib
{
    local( $sequence, $mach_idx );
    local( $name_str, $rib_number, $template );
 
    $name_str = &get_name( 'rib', $_ );
 
    ( $rib_number, $template ) = split( "=", $name_str );
 
    if ( $rib_number eq "" ) {
       &error_and_die( "No RIB ID in \"$name_str\"" ); }
    if ( $template eq "" ) {
       &error_and_die( "No RIB template in \"$name_str\"" ); }
 
    SCOPE_SWITCH:
    {
       ( $g005_process_scope eq "GLOBAL" ) && do
       {
          $sequence = $ALL;
          $mach_idx = $ALL;
          last SCOPE_SWITCH;
       };
 
       ( $g005_process_scope eq "SEQUENCE" ) && do
       {
          $sequence = $g002_curr_sequence;
          $mach_idx = $ALL;
          last SCOPE_SWITCH;
       };
 
       ( $g005_process_scope eq "MACHINE" ) && do
       {
          $sequence = $g002_curr_sequence;
          $mach_idx = $g001_mach_idx;
          last SCOPE_SWITCH;
       };
 
       &error_and_die( "Invalid scope for declaring a RIB statement" );
    }
 
    $entire_ribs_list{ $sequence, $mach_idx, $rib_number } = $template;
}

#
#  Subroutine:  PROCESS_RIBS_SUPPORTED
#  Proces the "ribs-supported" clause in a peer statement.
#
sub process_ribs_supported
{
   local ( $peer, $ribs );

   $ribs = &get_name( 'ribs-supported', $_ );

   #  Error if not currently processing a peer.
   if ( $g005_process_scope ne "PEER" ) {
      &error_and_die( "Not currently processing a peer" ); }

   foreach $peer ( @g004_curr_peers )
   {
      $peer_ribs_list{$g002_curr_sequence, $g001_mach_idx, $peer} = $ribs;
#     print "Supporting ribs \"$ribs\" for peer $peer\n";
   }
}

#
#  Subroutine: PROCESS_INTERNAL_NEIGHBORS
#  Process a list of internal neighbors to a machine in a sequence.
#
sub process_internal_neighbors
{
   local ( $separator1, $separator2 );

   $g005_process_scope = PEER;

   $machine_list = &get_name( 'internal', $_ )
             || &error_and_die( "No neighbors listed" );

   # Extract the neighbor names and stuff into local array.
   @g004_curr_peers = split( /[\s,]+/, $machine_list );

   $separator1 = ( $int_neighbor_list[$g001_mach_idx] eq "" ) ? "" : ":";
   $separator2 = ( $peer_list[$g001_mach_idx] eq "" ) ? "" : ":";

   #  Add the neighbor machine name to the current machine neighbor list.
   #  Since perl does not support multi-dimensional arrays, we will put
   #  the neighbors into a string, then split them out at file gen time.
   foreach( @g004_curr_peers )
   {
      print "Processing internal neighbor $_ ...\n";
      $int_neighbor_list[$g001_mach_idx] =
         $int_neighbor_list[$g001_mach_idx].$separator1.$_;
      $peer_list[$g001_mach_idx] =
         $peer_list[$g001_mach_idx].$separator2.$_;
      $separator1 = ":";
      $separator2 = ":";
   }
}

#
#  Subroutine: PROCESS_EXTERNAL_NEIGHBORS
#  Process a list of external neighbors to a machine in a sequence.
#
sub process_external_neighbors
{
   local ( $separator1, $separator2 );

   $g005_process_scope = PEER;

   $machine_list = &get_name( 'external', $_ )
             || &error_and_die( "No neighbors listed" );

   # Extract the neighbor names and stuff into local array.
   @g004_curr_peers = split( /[\s,]+/, $machine_list );

   $separator1 = ( $ext_neighbor_list[$g001_mach_idx] eq "" ) ? "" : ":";
   $separator2 = ( $peer_list[$g001_mach_idx] eq "" ) ? "" : ":";

   #  Add the neighbor machine name to the current machine neighbor list.
   #  Since perl does not support multi-dimensional arrays, we will put
   #  the neighbors into a string, then split them out at file gen time.
   foreach( @g004_curr_peers )
   {
      print "Processing external neighbor $_ ...\n";
      $ext_neighbor_list[$g001_mach_idx] =
         $ext_neighbor_list[$g001_mach_idx].$separator1.$_;
      $peer_list[$g001_mach_idx] =
         $peer_list[$g001_mach_idx].$separator2.$_;
      $separator1 = ":";
      $separator2 = ":";
   }
}

#
#  Subroutine:  PROCESS_OPTION
#
sub process_option
{
   $option_name = &get_name( 'option', $_ )
             || &error_and_die( "Empty option" );
   print "Processing option $option_name ...\n";

   OPTION_SWITCH:
   {
      ( $option_name eq "rib_refresh" ) && do
      {
         $refresh_on_hup = on;
         last OPTION_SWITCH;
      };

      ( $option_name eq "delta_routes" ) && do
      {
         $refresh_on_hup = off;
         last OPTION_SWITCH;
      };

      ( $option_name eq "multi-exit" ) && do
      {
         $multi_exit_switch[$g001_mach_idx] = on;
         last OPTION_SWITCH;
      };

      ( $option_name eq "route-server" ) && do
      {
         $route_server_switch[$g001_mach_idx] = on;
         last OPTION_SWITCH;
      };

      &error_and_die( "Unknown option $!" );

   }     # end OPTION_SWITCH
}

#
#  Subroutine:  PROCESS_IP_ROUTES
#
sub process_ip_routes
{
   $num_routes = &get_name( 'ip_routes', $_ )
              || &error_and_die( "No route quantity specified" );

   ROUTE_SWITCH:
   {
      #  Find out how many routes are to be added.
      ( ( $num_routes eq "none" ) ||
        ( $num_routes eq "some" ) ||
        ( $num_routes eq "all" ) ) && do
      {
         $ip_route_list[$g003_curr_test_set] = $num_routes;
         last ROUTE_SWITCH;
      };

      &error_and_die( "Unknown route quantity \"$!\"" );
   }
}

#
#  Subroutine:  PROCESS_OSI_ROUTES
#
sub process_osi_routes
{
   $num_routes = &get_name( 'osi_routes', $_ )
              || &error_and_die( "No route quantity specified" );

   ROUTE_SWITCH:
   {
      #  Find out how many routes are to be added.
      ( ( $num_routes eq "none" ) ||
        ( $num_routes eq "some" ) ||
        ( $num_routes eq "all" ) ) && do
      {
         $osi_route_list[$g003_curr_test_set] = $num_routes;
         last ROUTE_SWITCH;
      };

      &error_and_die( "Unknown route quantity \"$!\"" );
   }
}

#
#  Subroutine:  FIND_MACHINE
#  Find the machine's index in the table.
#
sub find_machine
{
   local ( $mach_name ) = @_[0];

#  print "Looking for machine '$mach_name'\n";

   for ( $i = 0; $i <= $#machine; $i++ )
   {
      if ( $machine[$i] eq $mach_name )
      {
         return $i;
      }
   }

   #
   #  Since we got here, the search failed.
   #
   &error_and_die( "Machine \"$mach_name\" not in machine map." );
}

#
#  Subroutine:  GET_NAME
#  Get the name of a keyword (sequence, machine, etc.)
#
sub get_name
{
   ( $keyword, $line ) = @_;
   $pattern = "^\\s\*$keyword\\s\*";

#  print "Keyword = '$keyword'\n";
#  print "Line    = '$line'\n";
#  print "Pattern = '$pattern'\n";

   ( $dummy, $name ) = split( /$pattern/, $line );
   if ( $name =~ /\n$/ )
   {
      chop( $name );
   }

   # Remove trailing whitespace.
   while ( $name =~ /\s$/ )
   {
      chop( $name );
   }

#  print "Returned name = '$name'\n\n";
   return ( $name );
}

#
#  Subroutine:  PRINT_OVERVIEW
#     Print the overview message for this program.
#
sub print_overview
{
   print "\n$PROGRAM_NAME:\n";
   print "     This program generates gated configuration files (*.conf)\n";
   print "     using a template built from a simplified language.\n";
   print "     Version: $VERSION ($VDATE)\n";
}

#
#  Subroutine:  PRINT_USAGE
#     Print the usage message for this program.
#
sub print_usage
{
   print "\nUsage: $PROGRAM_NAME [-m machine_name] [-EhLW] <test config file>\n";
   print "Options:\n";
   print "     -m <machine name> - Generate files only for that machine\n";
   print "     -E                - Do not generate expected results files\n";
   print "     -h                - Print this usage and overview message\n";
   print "     -L                - Do not generate export loop code\n";
   print "     -W                - Do not generate warning messages (do print errors)\n";
}

#
#  Subroutine:  ERROR_AND_DIE
#     Print an error message and halt program.
#
sub error_and_die
{
   local ( $error_message ) = @_;

   print "***** ERROR -- ";
   print "$error_message\n";
   die "      At line # $line_number of file $ARGV[0].\n";
}

#
# end of file
