#
#  cmp_results.perl
#
#    Compares a gated expected results file for a test/sequence/test set
#    against the current routing table.  Reports any differences.
#
#  Author:  Jeff Jackson
#
#  Environment variables used:
#    gated_cmp_pat_file   Name of file containing patterns to compare for
#
#  Revision History:
#
#    01-26-95   JWJ   Created.
#    08-24-95   JWJ   Removed periods when comparing routes
#                     (see explanatory note below).
#    10-13-95   JWJ   Revised for different RIBs.
#    10-16-95   JWJ   Added overview and usage messages, -v and -h options.
#    10-25-95   JWJ   Added code to ignore comment and blank lines.
#    10-26-95   JWJ   Added code to remove leading whitespace from input.
#                     Added file for patterns rather than hard-coded patterns.
#    11-02-95   JWJ   Fixed bug which matched "180.2" with "180.20", e.g.
#                     by replacing pattern match with exact match and editing
#                     lines read in from the netstat table.
#    11-03-95   JWJ   Added "-z" option (remove trailing ".0" from net
#                     addresses).  This is to fix bug for the Suns which
#                     occurred from fixing the above bug for the PCs.
#

$PROGRAM_NAME = $0;
$VERSION      = "1.7";
$VDATE        = "November 3, 1995";

#
#  Process any command line switches.
#
require "getopts.pl";
&Getopts('hr:q:vz');   #sets $opt_r as argument to -r

if ( ($#ARGV == -1) || ($opt_h) )
{
   &print_overview;
   &print_usage;
   exit;
}

#
#  Set verbose mode if requested.
#
$verbose_mode = ( $opt_v ? ON : OFF );

#
#  Set "chop trailing zeros" mode if requested.
#
$chop_zeros_mode = ( $opt_z ? ON : OFF );

#
#  The only other command line arg is the result file name.
#
$result_file = $ARGV[0];

@pattern_list = ();

@reslist  = ();
@reslist2 = ();
@netlist  = ();
@qoslist  = ();
$qos = OFF;

@errorlist1 = ();
@errorlist2 = ();
@errorlist3 = ();
@errorlist4 = ();

#
#  Open the patterns file and store all patterns.
#
$patterns_file = $ENV{'gated_cmp_pat_file'};
if ( $patterns_file eq "" ) {
   die "   *** ERROR - \$gated_cmp_pat_file not set\n"; }

open( PATTERNS, $patterns_file )
   || die "*** Cannot open patterns file $patterns_file for input\n";

while ( <PATTERNS> )
{
   next if ( (/^#/) || (/^\s*$/)  );  # ignore comments and blank lines
   chop( $_ );                        # remove trailing newline
   push( @pattern_list, $_ );
#  print "Pattern = \"$_\"\n";
}
close PATTERNS;

#
#  Open the expected results file and save all lines.
#
open( EXPECTED, $result_file )
   || die "*** Cannot open expected results file $result_file for input\n";

while ( <EXPECTED> )
{
   next if ( (/^#/) || (/^\s*$/)  );  # ignore comments and blank lines
   if ( /^\s+/ ) { $_ = $'; }         # remove leading whitespace

   #  Split line into RIB and route ID.
   ( $rib, $route ) = split( /\s+/ );

   #  For network table, use route ID only.
   &add_unique_route( $route, *reslist );
#  push( @reslist, $route );

   #  For QOS table, use both fields, but ignore RIB 0.
   if ( $rib ne "0" )
   {
      $line = "RIB: $rib\tRoute: $route";
      push( @reslist2, $line );
   }
}

#
#  If "default" is an expected route, look for it.
#  Note that the line counter must be reset with
#  an explicit close and re-open.
#
close EXPECTED;
open( EXPECTED, $result_file );

$look_for_default = 0;
while ( <EXPECTED> )
{
   next if ( (/^#/) || (/^\s*$/)  );  # ignore comments and blank lines
   if ( /^\s+/ ) { $_ = $'; }         # remove leading whitespace

   ( /default/ ) && do {
      $look_for_default = 1;
   }
}
close EXPECTED;

#
#  Generate a current network routing table listing and save all lines
#  which are related to gated.
#
if ( $opt_r eq "" )
{
   open ( NETSTAT, "netstat -rn |" )
      || die "*** Cannot generate a routing table\n";
}
else
{
   open ( NETSTAT, $opt_r )
      || die "*** Cannot open route table $route_table for input\n";
}

while ( <NETSTAT> )
{
   next if ( (/^#/) || (/^\s*$/)  );  # ignore comments and blank lines
   if ( /^\s+/ ) { $_ = $'; }         # remove leading whitespace

   ( $address, @dummy ) = split( /\s+/ );  # take first part of line only
#  print "** Net table address = \"$address\"\n";

   if ( $look_for_default && ($address =~ /^default/) )
   {
      push( @netlist, $address );
   }
   else
   {
      foreach $pattern ( @pattern_list )
      {
#        print "Attempting to mach pattern \"$pattern\"\n";
         if ( $address =~ /$pattern/ )
         {
            #  Remove all trailing ".0" if requested.
            if ( $chop_zeros_mode eq "ON" )
            {
               while ( $address =~ /\.0$/ )
               {
                  chop($address); chop($address);
               }
            }
            push( @netlist, $address );
            last;    # exit on first match
         }
      }   # end foreach
   }   # end if stmt
}
close NETSTAT;


#
#  Read the QOS table and store all lines.
#
if ( $opt_q ne "" )
{
   $qos = ON;
   open ( QOS_FILE, $opt_q )
      || die "*** Cannot open QOS file $opt_q for input\n";
}

while ( <QOS_FILE> )
{
#  print "***** QOS line = $_\n";
   next if ( (/^#/) || (/^\s*$/)  );  # ignore comments and blank lines
   if ( /^\s+/ ) { $_ = $'; }         # remove leading whitespace

   ( $rib, $route, $intf ) = split( /\s+/ );
#  print "***** RIB = $rib, ROUTE = $route, INTF = $intf\n";
   $line = "RIB: $rib\tRoute: $route";
   push( @qoslist, $line );
}
close QOS_FILE;

#
#  Diagnostic:  print all the lists.
#
if ( $verbose_mode eq ON )
{
   print "*** Patterns list ( \@pattern_list )\n";
   foreach $elem ( @pattern_list )
   {
      print "\tPattern = $elem\n";
   }
   print "\n";

   print "*** Expected results list ( \@reslist )\n";
   foreach $elem ( @reslist )
   {
      print "\telement = $elem\n";
   }
   print "\n";

   print "*** Netstat list ( \@netlist )\n";
   foreach $elem ( @netlist )
   {
      print "\telement = $elem\n";
   }
   print "\n";

   print "*** Expected results list 2 ( \@reslist2 )\n";
   foreach $elem ( @reslist2 )
   {
      print "\telement = $elem\n";
   }
   print "\n";

   print "*** QOS list ( \@qoslist )\n";
   foreach $elem ( @qoslist )
   {
      print "\telement = $elem\n";
   }
   print "\n";
}


#
#  Now compare the lists.  Flag any lines which are in one list
#  but not in the other.
#
&compare_lists( *reslist, *netlist, *errorlist1, *errorlist2 );
if ( $verbose_mode eq ON )
{
   print "Errorlist 1 = @errorlist1\n";
   print "Errorlist 2 = @errorlist2\n";
}

if ( $qos eq "ON" )
{
   &compare_lists( *reslist2, *qoslist, *errorlist3, *errorlist4 );
   if ( $verbose_mode eq ON )
   {
      print "Errorlist 3 = @errorlist3\n";
      print "Errorlist 4 = @errorlist4\n";
   }
}


#
#  Print all lines which were in error.
#
if ( @errorlist1 > 0 )
{
   print "\n\nTHE FOLLOWING EXPECTED ROUTES WERE NOT IN THE ROUTING TABLE\n";
   print "===========================================================\n";
   foreach( @errorlist1 )
   {
      print "$_\n";
   }
}

if ( @errorlist2 > 0 )
{
   print "\n\nTHE FOLLOWING ROUTES IN THE ROUTING TABLE WERE NOT EXPECTED\n";
   print "===========================================================\n";
   foreach( @errorlist2 )
   {
      print "$_\n";
   }
}

if ( @errorlist3 > 0 )
{
   print "\n\nTHE FOLLOWING EXPECTED ROUTES WERE NOT IN THE QOS TABLE\n";
   print "=======================================================\n";
   foreach( @errorlist3 )
   {
      print "$_\n";
   }
}

if ( @errorlist4 > 0 )
{
   print "\n\nTHE FOLLOWING ROUTES IN THE QOS TABLE WERE NOT EXPECTED\n";
   print "=======================================================\n";
   foreach( @errorlist4 )
   {
      print "$_\n";
   }
}

if ( ( $#errorlist1 == -1 ) && ( $#errorlist2 == -1 ) &&
     ( $#errorlist3 == -1 ) && ( $#errorlist4 == -1 ) )
{
   print "\n    *** Expected results achieved ***\n\n";
}


#
#  Subroutine:  ADD_UNIQUE_ROUTE
#  Add a route to a list, but only if not there already.
#
sub add_unique_route
{
   local ( $route, *list ) = @_;
   local ( $elem, $unique );

   $unique = TRUE;
   foreach $elem ( @list )
   {
      ( $route eq $elem ) && do { $unique = FALSE; last; };
   }

   if ( $unique eq TRUE )
   {
      push ( @list, $route );
   }
}


#
#  Subroutine:  COMPARE_LISTS
#  Check every line in the results file against the net table.
#  Store any routes which are not in the table.
#  The arrays are declared as "*arr" to make the arrays passed
#  by reference, so they aren't merged into one list in @_.
#
sub compare_lists
{
   local ( *list1, *list2, *errlist1, *errlist2 ) = @_;
   local ( $line, $line2, $pattern1, $pattern2, $i );
   local ( $match_found );

   foreach $line ( @list1 )
   {
      $match_found = FALSE;

      for( $i = 0; $i <= $#list2; $i++ )
      {
         #  Remove all periods from both patterns (differences in
         #  display format of "netstat" causes false mismatches).
         #  This follows change from "0x49xxxxyyyy" format of ISO
         #  addresses to "49.xxxx.yyyy", etc.
         $pattern1 = join ( "", split("\\.", $list2[$i]) );
         $pattern2 = join ( "", split("\\.", $line) );

#        print "** Attempting to match \"$pattern2\" with \"$pattern1\"\n";
         if ( $pattern1 eq $pattern2 )
         {
#           print "   ## MATCH FOUND ##\n";
            $match_found = TRUE;
            $list2[$i] = "";   # mark line in list 2 as found
            last;
         }
      }

      if ( $match_found eq FALSE )
      {
         push( @errlist1, $line );
      }
   }     # end foreach( @list1 )

   #
   #  Go through the (marked up) list #2 and store any lines which
   #  were not matched with a line in list 1.
   #
   foreach $line2 ( @list2 )
   {
      if ( $line2 ne "" )
      {
         push( @errlist2, $line2 );
      }
   }
}


#
#  Subroutine:  PRINT_OVERVIEW
#     Print the overview message for this program.
#
sub print_overview
{
   print "\n$PROGRAM_NAME:\n";
   print "     This program compares the expected results for a gated.conf file\n";
   print "     against the current routing table and the QOS file.\n";
   print "     If no route table is specified, a current \"netstat\" is used.\n";
   print "     QOS comparison is optional.\n";
   print "     Version: $VERSION ($VDATE)\n";
}

#
#  Subroutine:  PRINT_USAGE
#     Print the usage message for this program.
#
sub print_usage
{
   print "\nUsage: $PROGRAM_NAME [-h] [-v] [-z] [-r <route table>] [-q <QOS table>] <result file name>\n";
   print "Options:\n";
   print "     -h                - Print this overview and usage message\n";
   print "     -r <route table>  - Read route table info from this file\n";
   print "     -q <QOS table>    - Read QOS info from this file\n";
   print "     -v                - Set verbose mode (print diagnostics)\n";
   print "     -z                - Chop all trailing \".0\" from net table addresses\n";
}


#
#end of file
