/*
 *  Consortium Release 4
 *
 *  $Id: krt_ifread_ioctl.c,v 1.33 2000/04/13 07:26:30 naamato Exp $
 */
/*
 * GateD Releases Unicast, Multicast, IPv6, RSd
 * 
 * Copyright (c) 1996,1997,1998,1999 
 * The Regents of the University of Michigan.
 * All Rights Reserved.
 * 
 * License to use, copy, modify, and distribute this software and its
 * documentation can be obtained from Merit Network, Inc. at the 
 * University of Michigan.
 * 
 * Merit GateD Consortium
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C
 * Ann Arbor, MI 48105
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT DO NOT WARRANT THAT THE FUNCTIONS 
 * CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR THAT 
 * OPERATION WILL BE UNINTERRUPTED OR ERROR FREE. The Regents of the
 * University of Michigan and Merit shall not be liable for any special, 
 * indirect, incidental or consequential damages with respect to any claim 
 * by Licensee or any third party arising from use of the software. 
 * GateD was originated and developed through release 3.0 by Cornell 
 * University and its collaborators.
 * 
 * Please send questions or comments to gated-people@gated.org.
 *
 * Please submit bugs, bug fixes, and enhancements using the send-pr(1) 
 * utility or via the web at 
 * www.gated.org/gated-web/support/html/report_prob.html.
 *
 * __END_OF_COPYRIGHT__
 */

#define INCLUDE_IOCTL
#define INCLUDE_IF
#define INCLUDE_SOCKIO

/* #ifdef KRT_IFREAD_IOCTL  */

#ifdef  PROTO_INET
#include "inet4core/inet.h"
#endif  /* PROTO_INET */

#include "include.h"
#include "krt/krt.h"
#include "krt/krt_var.h"

#ifdef HAVE_RT_IOCTL
#include <sys/sockio.h>
#endif

/* per Stevens vol 1, loopback mtu is 1536 */
#define LOOPBACK_MTU  1536
#define POINTOPOINT_MTU 256
#define ETHER_MTU     1500
#define DEFAULT_MTU    256

/*
 * Prototypes
 */
static int  addr_size(struct ifreq *);
static void get_interface_address_info(task *, struct sockaddr *,
    if_info *, if_link *, char *, int);
static void get_interface_parameter_info(task *, if_info *, char *, int);
static int read_interface_list(task *, struct ifconf *, int);
static void dump_ifrp(task *, struct ifreq *);

u_long siocgifconf, siocgifaddr, siocgifdstaddr, siocgifbrdaddr,
    siocgifnetmask, siocgifflags, siocgifmtu, siocgifmetric;


/*
 * Get the length of the socket
 */
static int
addr_size(struct ifreq *ifrp)
{

#ifdef USE_SOCKLEN
	return (MAX(sizeof(struct sockaddr), ifrp->ifr_addr.sa_len));
#else /* !USE_SOCKLEN */

	switch (ifrp->ifr_addr.sa_family) {
#ifdef IPV6
	case AF_INET6:
		return (sizeof(struct sockaddr_in6));
		break;
#endif  /* IPV6 */
	case AF_INET:
	default:
		return (sizeof(struct sockaddr));
		break;
	}
#endif  /* !USE_SOCKLEN */
}

/*
 * Read the interfaces installed on the system using ioctl
 */
int
read_interface_list(task *tp, struct ifconf *ifcp, int sd)
{
	int mult, llen, err;

	llen = mult = 0;

	/*
	 * let's read the interface info from the kernel using ioctl
	 * with SIOCGIFCONF request. This is a loop because not all
	 * implementations of ioctl will return an error if we don't
	 * allocate a large enough buffer. The trick around this is
	 * do the ioctl and save the length returned, then do the ioctl 
	 * again with a larger buffer. If the lengths are the same we have all
	 * the data else we increase the size of the buffer and try again.
	 */ 
	for (;;) {

		/*
		 * allocate a task buffer and put buffer pointer 
		 * and length in ioctl structure. 
		 */
		task_alloc_send(tp, task_pagesize << mult);
		ifcp->ifc_len = task_send_buffer_len;
		ifcp->ifc_buf = task_send_buffer;

#if defined(HPSTREAMS)
		NON_INTR(err, ioctl(sd, siocgifconf, ifcp));
#else
		err = task_ioctl(sd, (u_long)siocgifconf,
		    (void_t)ifcp, ifcp->ifc_len); 
#endif
		if (err < 0) {
			if (errno != EINVAL || llen !=0)
				/* 
				 * we got problems, cannot do successful
				 * ioctl call 
				 */
				return(0);
		} else {
			if (ifcp->ifc_len == llen)
				/* 
				 * length is same so last time so we got it
				 * all, break out of loop 
				 */
				return(1);
		}
			
		/* 
		 * either first time or we got a different length,
		 * or buffer must not have been big enough
		 * let's try it again with a larger buffer 
		 */
		llen = ifcp->ifc_len;
		mult++;
	}
}


#ifdef PROTO_INET6
/*
 * Read the interfaces installed on the system using kernel ioctl call.
 */
int
read_interface_list6(task *tp, struct lifconf *ifcp, int sd)
{
	int mult, llen, err;

	llen = mult = 0;

	/*
	 * let's read the interface info from the kernel using ioctl
	 * with SIOCGIFCONF request. This is a loop because not all
	 * implementations of ioctl will return an error if we don't
	 * allocate a large enough buffer. The trick around this is
	 * do the ioctl and save the length returned, then do the ioctl 
	 * again with a larger buffer. If the lengths are the same we have all
	 * the data else we increase the size of the buffer and try again.
	 */ 
	for (;;) {

		/*
		 * allocate a task buffer and put buffer pointer
		 * and length in ioctl structure.
		 */
		task_alloc_send(tp, task_pagesize << mult);
		ifcp->lifc_len = task_send_buffer_len;
		ifcp->lifc_buf =  task_send_buffer;

#if defined(HPSTREAMS) || defined(HAVE_XPG4)
		NON_INTR(err, ioctl(sd, siocgifconf, ifcp));
#else
		err = task_ioctl(sd, (u_long)siocgifconf,
		    (void_t)ifcp, ifcp->lifc_len);
#endif /* HPSTREAMS || HAVE_XPG4 */

			if (err < 0) {
				if (errno != EINVAL || llen !=0)
	        			/*
				 	 * cannot do successful ioctl
  	       			 	 */
					return(0);
			} else {
				if (ifcp->lifc_len == llen)
					/*
         	 			 * length is same so last time so we
					 * got it all, break out of loop
         	 			 */
        				return(1);
    			}

		/*
     		 * either first time or we got a different length,
		 * or buffer must not have been big enough
		 * let's try it again with a larger buffer
		 */
		llen = ifcp->lifc_len;
		mult++;
	}
}
#endif /* PROTO_INET6 */

/*
 * do ioctls to get address specific info, such as dest address,
 * broadcast address,and netmask.
 */
static void
get_interface_address_info(task *tp, struct sockaddr *addr,
    struct _if_info *ifip, if_link *iflp, char *name, int sd)
{
	struct ifreq ifr;
	sockaddr_un *sap;

	/* copy default info into structure */
	ifip->ifi_link = iflp;

	/* ignore interfaces from undesired families */
	switch (addr->sa_family) {
#ifdef PROTO_INET
	case AF_INET:
		ifip->ifi_addr_local =
		    sockdup(sock2gated_nl(addr));
		ifip->ifi_addr_remote = 0;
		ifip->ifi_addr_broadcast = 0;
		ifip->ifi_netmask = 0;

		/* copy the interface name into the ioctl buffer */
		strcpy(ifr.ifr_name, name);
#ifdef SIOCGIFDSTADDR 
		/*
		 * if we are p2p, let's set the mtu and get the address
		 * of the other side
		 */
		if (BIT_TEST(ifip->ifi_state, IFS_POINTOPOINT)) {

			if (ifip->ifi_mtu == DEFAULT_MTU)
				ifip->ifi_mtu = POINTOPOINT_MTU;

			if (task_ioctl(sd, siocgifdstaddr,
			    (caddr_t)&ifr, sizeof (ifr)) < 0)

			   	 trace_log_tp(tp, 0, LOG_ERR,
				    ("krt_ifread: %s: ioctl "
				    "SIOCGIFDSTADDR: %m",
				    ifr.ifr_name));
			else {

				sap = sock2gated_nl(&ifr.ifr_dstaddr);

				if (sap) 
					ifip->ifi_addr_remote = sockdup(sap);
				else
					trace_log_tp(tp, 0, LOG_ERR,
					    ("krt_ifread: no destination "
					    "address for %A (%s)",
					    ifip->ifi_addr_local,
					    ifr.ifr_name));
			}	
		}
#endif /* SIOCGIFDSTADDR */

#ifdef SIOCGIFBRDADDR 
		/*
	 	 * if we are a broadcast medium, set the mtu and get
	 	 * the broadcast address
	 	 */
		if (BIT_TEST(ifip->ifi_state, IFS_BROADCAST)) {

			if (ifip->ifi_mtu == DEFAULT_MTU)
				ifip->ifi_mtu = ETHER_MTU;

			if (task_ioctl(sd, siocgifbrdaddr, (caddr_t)&ifr,
			    sizeof (ifr)) < 0)

				trace_log_tp(tp, 0, LOG_ERR,
			    	    ("krt_ifread: %s: ioctl "
				    "SIOCGIFBRDADDR: %m",
				    ifr.ifr_name));

			else {
				sap = sock2gated_nl(&ifr.ifr_broadaddr);

				if (sap)
					ifip->ifi_addr_broadcast =
					    sockdup(sap);
				else
					trace_log_tp(tp, 0, LOG_ERR,
					    ("krt_ifread: no broadcast "
					    "address for %A (%s)",
				    	    ifip->ifi_addr_local,
					    ifr.ifr_name));
			}
		}
#endif /* SIOCGIFBRDADDR */

#ifdef SIOCGIFNETMASK 
		/*
		 * get the netmask address for the interface
		 */
		if (task_ioctl(sd, siocgifnetmask,
		    (caddr_t) &ifr, sizeof (ifr)) < 0) {

			trace_log_tp(tp, 0, LOG_ERR,
			    ("krt_ifread: %s: ioctl SIOCGIFNETMASK: %m",
			    ifr.ifr_name));
		} else {

			/* build a netmask from kernel info */
			sap = sock2gated_nl(&ifr.ifr_addr);

			if (sap)
				ifip->ifi_netmask = mask_locate(sap);
			else
				trace_log_tp(tp, 0, LOG_ERR,
				    ("krt_ifread: no network mask for %A (%s)",
				    ifip->ifi_addr_local, ifr.ifr_name));
		}
#endif /* SIOCGIFNETMASK */

		if (!ifip->ifi_addr_remote)
			ifip->ifi_addr_remote = sockdup(ifip->ifi_addr_local);
		/*
		 * XXX what if we don't have a netmask?
		 * or its wrong e.g., p2p2 bsdi
		 */
		sockmask(ifip->ifi_addr_remote, ifip->ifi_netmask);

#endif /* PROTO_INET */
	}
}

#ifdef PROTO_INET6
/*
 * do ioctls to get address specific info, such as dest address,
 * broadcast address,and netmask.
 */
static void
get_interface_address_info6(task *tp, struct sockaddr_storage *addr, 
    struct _if_info *ifip, if_link *iflp, char *name, int sd)
{
	struct lifreq ifr;
	sockaddr_un *sap;

	/* copy default info into structure */
	ifip->ifi_link = iflp;

	/* ignore interfaces from undesired families */
	switch ( addr->ss_family)  {
#ifdef PROTO_INET
	case AF_INET6:

		ifip->ifi_addr_local = sockdup(sockstore2gated(addr));
		ifip->ifi_addr_remote = 0;
		ifip->ifi_addr_broadcast = 0;
		ifip->ifi_netmask = 0;

		/* copy the interface name into the ioctl buffer */
		strcpy(ifr.lifr_name, name);

#ifdef SIOCGLIFDSTADDR
		/*
		 * if we are p2p, let's set the mtu and get the address
		 * of the other side
		 */
		if (BIT_TEST(ifip->ifi_state, IFS_POINTOPOINT)) {

			if (ifip->ifi_mtu == DEFAULT_MTU )
				ifip->ifi_mtu = POINTOPOINT_MTU;

			if (task_ioctl(sd, siocgifdstaddr,
			    (caddr_t)&ifr, sizeof (ifr)) < 0)

				trace_log_tp(tp, 0, LOG_ERR,
				    ("krt_ifread: %s: ioctl SIOCGIFDSTADDR: %m",
				    ifr.lifr_name));

			else {
				sap = sockstore2gated(&ifr.lifr_dstaddr);

				if (sap) 
					ifip->ifi_addr_remote = sockdup(sap);
				else
					trace_log_tp(tp, 0, LOG_ERR,
					    ("krt_ifread: no destination "
					    "address for %A (%s)",
					    ifip->ifi_addr_local,
					    ifr.lifr_name));
			}
		}
#endif /* SIOCGIFDSTADDR */

#ifdef SIOCGLIFBRDADDR
		/*
		 * if we are a broadcast medium, set the mtu and get
		 * the broadcast address
		 */
		if (BIT_TEST(ifip->ifi_state, IFS_BROADCAST)) {

			if (ifip->ifi_mtu == DEFAULT_MTU)
				ifip->ifi_mtu = ETHER_MTU;

			if (task_ioctl(sd, siocgifbrdaddr,
			    (caddr_t) &ifr, sizeof (ifr)) < 0)

				trace_log_tp(tp, 0, LOG_ERR,
				    ("krt_ifread: %s: ioctl SIOCGIFBRDADDR: %m",
				    ifr.lifr_name));
			else {
				sap = sockstore2gated(&ifr.lifr_broadaddr);

				if (sap)
					ifip->ifi_addr_broadcast = sockdup(sap);
				else
					trace_log_tp(tp, 0, LOG_ERR,
					    ("krt_ifread: no broadcast "
					    "address for %A (%s)",
					    ifip->ifi_addr_local,
					    ifr.lifr_name));
			}
		}
#endif /* SIOCGIFBRDADDR */

#ifdef SIOCGLIFNETMASK
		/*
		 * get the netmask address for the interface
		 */
		if (task_ioctl(sd, siocgifnetmask,
		    (caddr_t)&ifr, sizeof (ifr)) < 0)

		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: %s: ioctl SIOCGIFNETMASK: %m",
		    ifr.lifr_name));
		else {
			/* build a netmask from kernel info */
			sap = sockstore2gated(&ifr.lifr_addr);

			if (sap)
				ifip->ifi_netmask = mask_locate(sap);
			else
				trace_log_tp(tp, 0, LOG_ERR,
				    ("krt_ifread: no network mask for %A (%s)",
				    ifip->ifi_addr_local, ifr.lifr_name));
		}
#endif /* SIOCGIFNETMASK */
		if (!ifip->ifi_addr_remote)
			ifip->ifi_addr_remote = sockdup(ifip->ifi_addr_local);

		/*
		 * XXX what if we don't have a netmask?
		 * or its wrong e.g., p2p2 bsdi
		 */
		sockmask(ifip->ifi_addr_remote, ifip->ifi_netmask);
#endif  /* PROTO_INET */
  	}

}
#endif /* PROTO_INET6  */

/*
 * Let's do some ioctls to get the interface flags, mtu, and metrics.
 */
static void
get_interface_parameter_info(task *tp, struct _if_info *ifip,
    char *name, int sd)
{
	struct ifreq ifr;

	/* copy interface name to ioctl structure */
	strcpy(ifr.ifr_name, name);

	/*
	 * get the interface flags
	 */
#ifdef SIOCGIFFLAGS 
	if (task_ioctl(sd, (u_long) siocgifflags,
	    (char *)&ifr, sizeof (ifr)) < 0)

		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: %s: ioctl SIOCGIFFLAGS: %m",
		    ifr.ifr_name));
	else
		ifip->ifi_state = krt_if_flags(ifr.ifr_flags);
#else /* !SIOCGIFFLAGS */
		ifip->ifi_state = 0;
#endif /* !SIOCGIFFLAGS */

	/*
	 * get the interface MTU
	 */
#ifdef SIOCGIFMTU 
	bzero ((caddr_t) &ifr.ifr_ifru, sizeof (ifr.ifr_ifru));

	if (task_ioctl(sd, (u_long) siocgifmtu,
	    (char *)&ifr, sizeof (ifr)) < 0) { 

		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: %s: ioctl SIOCGIFMTU: %m, "
		    "Gated using default mtu",
		    ifr.ifr_name));

		ifip->ifi_mtu = DEFAULT_MTU;
	} else
	    	ifip->ifi_mtu = ifr.KRT_IFR_MTU;
#else /* !SIOCGIFMTU */
		ifip->ifi_mtu = DEFAULT_MTU;
#endif 

	/*
	 * get the interface metrics
	 */
#ifdef  SIOCGIFMETRIC     	    

	bzero ((caddr_t) &ifr.ifr_ifru, sizeof (ifr.ifr_ifru));

	if (task_ioctl(sd, (u_long) siocgifmetric,
	    (char *) &ifr, sizeof (ifr)) < 0) { 

 		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: %s: ioctl SIOCGIFMETRIC: %m",
		    ifr.ifr_name));

		ifip->ifi_metric = 0;
 	} else 
 		ifip->ifi_metric = ifr.ifr_metric;

#else  /* !SIOCGIFMETRIC */
		ifip->ifi_metric = 0;
#endif  /* !SIOCGIFMETRIC */
}



#ifdef PROTO_INET6
/*
 * Do some ioctls to get the interface flags, mtu, and metrics
 */
static void 
get_interface_parameter_info6(task *tp,
    struct _if_info *ifip, char *name, int sd)
{
	struct lifreq ifr;

	/* copy interface name to ioctl structure */
	strcpy(ifr.lifr_name, name);

  	/*
   	 * get the interface flags
   	 */
#ifdef SIOCGIFFLAGS
	if (task_ioctl(sd, (u_long)siocgifflags,
	    (char *)&ifr, sizeof (ifr)) < 0) {

		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: %s: ioctl SIOCGIFFLAGS: %m",
                    ifr.lifr_name));

		ifip->ifi_state = 0;
	} else
		ifip->ifi_state = krt_if_flags(ifr.lifr_flags);
#else
		ifip->ifi_state = 0;
#endif

	/*
	 * get the interface MTU
	 */
#ifdef SIOCGIFMTU
	bzero ((caddr_t)&ifr.lifr_lifru, sizeof (ifr.lifr_lifru));

	if (task_ioctl(sd, (u_long)siocgifmtu,
	    (char *)&ifr, sizeof (ifr)) < 0) {

		trace_log_tp(tp, 0, LOG_ERR, 
		    ("krt_ifread: %s: ioctl SIOCGIFMTU: "
		    "%m, Gated using default mtu",
		    ifr.lifr_name));

		ifip->ifi_mtu = DEFAULT_MTU;
	} else
		ifip->ifi_mtu = ifr.lifr_lifru.lifru_mtu;
#else
		ifip->ifi_mtu = DEFAULT_MTU;
#endif

	/*
	 * get the interface metrics
	 */
#ifdef  SIOCGIFMETRIC
	/* do an ioctl to get the interface metrics */
	bzero ((caddr_t)&ifr.lifr_lifru, sizeof (ifr.lifr_lifru));
	if (task_ioctl(sd, (u_long)siocgifmetric,
	    (char *)&ifr, sizeof (ifr)) < 0) {

		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: %s: ioctl SIOCGIFMETRIC: %m",
		    ifr.lifr_name));
		ifip->ifi_metric = 0;

	} else
		ifip->ifi_metric = ifr.lifr_metric;
#else /* !SIOCGIFMETRIC */
		ifip->ifi_metric = 0;
#endif /* !SIOCGIFMETRIC */
}
#endif /* PROTO_INET6  */

/*
 * Read the interfaces from the kernel, IPv4 style
 */
int
krt_ifread_v4(flag_t save_task_state)
{
	static int sd = -1;

	struct ifconf ifc;
	struct ifreq  *ifrp;
	if_info ifi, oifi;
	if_link *if_plink = 0, *iflp;
	if_addr *ifap;
	sockaddr_un *lmp;
	task *tp = krt_task;
	char *cp, *ptr, name[IFNAMSIZ];
	int aidx, isalias, slen, test_bit_set, interface_count;

	tp = krt_task; 
	test_bit_set = 0; 
	interface_count = 0;
	isalias = FALSE; 
	if_plink = NULL;
	aidx = 0;

	siocgifconf = SIOCGIFCONF;
	siocgifaddr = SIOCGIFADDR;
	siocgifdstaddr = SIOCGIFDSTADDR;
	siocgifbrdaddr = SIOCGIFBRDADDR;
	siocgifnetmask = SIOCGIFNETMASK;
#ifdef SIOCGIFFLAGS
	siocgifflags = SIOCGIFFLAGS;
#endif /* SIOCGIFFLAGS */
#ifdef SIOCGIFMTU
	siocgifmtu = SIOCGIFMTU;
#endif /* SIOCGIFMTU */
#ifdef SIOCGIFMETRIC
	siocgifmetric = SIOCGIFMETRIC;
#endif /* SIOCGIFMETRIC */

	/*
	 * grab a socket for use with ioctl calls.
	 * Note task_get_socket checks for test mode so have to reset bit.
	 */
	if (sd == -1) {

		if (BIT_TEST(task_state, TASKS_TEST)) {
			test_bit_set = 1;
			BIT_RESET(task_state, TASKS_TEST);	
		}

		sd = task_floating_socket(tp, 
		    task_get_socket(tp, AF_INET, SOCK_DGRAM, 0),
		    "krt_ifread_task");

		if (test_bit_set)
			BIT_SET(task_state, TASKS_TEST);
	}

	if (krt_task->task_socket < 0)
		return EBADF;

	/* read the interfaces from the kernel */
	if (!read_interface_list(tp, &ifc, sd))  {

		/*
		 * cannot do successful ioctl call
		 */
		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: ioctl SIOCGIFCONF: %m"));
		return errno;
	}

	/* write our status to the log */
	trace_tp(tp, TR_KRT_IFLIST, TRC_NL_BEFORE,
	    ("krt_iflist: SIOCGIFCONF returns %u bytes", ifc.ifc_len));

	/* loop through all the data */
	for (ptr = ifc.ifc_buf; ptr < ifc.ifc_buf + ifc.ifc_len;) { 

		/*
		 * zero out temporary data structure for
		 * storing values from kernel
		 */
		bzero(&ifi, sizeof (ifi));

		/* get pointer to next interface */
		ifrp = (struct ifreq *)ptr;

		/* keep track of how many interfaces we have */
		interface_count++;

		/*
		 * We use the name for two distinct reasons. In GateD
		 * we have a one to many relationship
		 * with a physical interface and its one or more
		 * interface addresses. For this we need a
		 * name that has no alias. e.g. iprb0 and iprb0:1 both
		 * fall under the physical interface iprb0.
		 * For the ioctl calls we need an unmolested name.
		 */
		strcpy(name, ifrp->ifr_name);

		/*
		 * Remove the :n extension from the name 
		 * The "kernel primary" is the intf without :n, or :0
		 */
		cp = index(name, ':');
		if (cp) {
			if (*(cp + 1) != '0')
				isalias = TRUE;

			ifi.ifi_alias_idx = atoi(cp + 1);
			*cp = (char)0;
		} else
			ifi.ifi_alias_idx = 0;

		/* xxx - dump the ifrp we just read */        
		if (TRACE_TP(krt_task, TR_KRT_IFLIST)) 
			dump_ifrp(krt_task, ifrp);                

		/* read interface specific info from kernel */
		get_interface_parameter_info(tp, &ifi, ifrp->ifr_name, sd);

		/*
		 * Have to have a physical interface to link to
		 * If no previous or previous name is different
		 *
		 * XXX the third check seems iffy, what if there is a new
		 * XXX different name of the same length as the
		 * XXX previous with a ':' in it
		 *
		 * naamato 5/12/99
		 * By removing the ':' that should be fixed.  In the
		 * above problem, an interface with ifrp->ifr_name
		 * of "lan1:0" (not an alias) might get linked to
		 * the previous piflp.
		 *
		 * Now, lan1, lan1:0, lan1:1, lan1:2, etc. mean the same
		 * if_link.
		 */

		if (!if_plink 
		    || (strncmp(if_plink->ifl_name, ifrp->ifr_name, IFNAMSIZ) 
		    && (isalias == FALSE))) {
			/*
			 * either no physical interface or the name doesn't
			 * match, and it's not an alias
			 *
			 * xxx - krt_lladdr() is broke, we are using a
			 * hacked krt_lladdr() in krt_lladdr_sunos4.c.
			 * krt_lladdr() on sunos doesn't return correct llevel
			 * addr info, causing the ifrp info to be
			 * munged.  not using this code does away with correct
			 * ll info. so let's try to hack around it.
			 */
 
			slen = strlen(name);
			iflp = ifl_locate_name(name, slen);

			if_plink = ifl_addup(tp, iflp,
			    interface_count, ifi.ifi_state, 
			    ifi.ifi_metric, 
			    ifi.ifi_mtu, name,
			    slen, krt_lladdr(ifrp),
			    (sockaddr_un *)0); 
	                        
			if (BIT_TEST(ifi.ifi_state, IFS_LOOPBACK)) {
				/*
				 * Set the loopback flag and mtu for
				 * this physical interface
				 */
				BIT_SET(if_plink->ifl_state, IFS_LOOPBACK);
				ifi.ifi_mtu = LOOPBACK_MTU;
			}
		}

		/* read address information from kernel. */
		get_interface_address_info(tp, &ifrp->ifr_addr, 
		    &ifi, if_plink, ifrp->ifr_name, sd);

		/*
		 * the alias code gets hairy here because we are not guaranteed
		 * ordering from the kernel.  for example, hme0:2 may be
		 * reported before hme:1
		 */
/*
		if (isalias == FALSE)
			BIT_SET(ifi.ifi_state, IFS_ALIAS_PRIMARY);
		else if (is_primary_for_subnet(ifi, &ifap)) {

			if (ifap) {

				ifi_dup(&ifap->ifa_info, &oifi);

				BIT_RESET(oifi.ifi_state, IFS_ALIAS_PRIMARY);

				if_conf_addaddr(tp, &oifi);
			}

			BIT_SET(ifi.ifi_state, IFS_ALIAS_PRIMARY);
		}
*/

		/* Add the logical interface structure to the ifap list */
		if_conf_addaddr(tp, &ifi);	

		/* all done with that address let's do it again */
		ptr += sizeof (ifrp->ifr_name) + addr_size(ifrp);

	}
	return (1);
}



#ifdef PROTO_INET6
/*
 * Read the interfaces from the kernel, IPv6 style
 */
int
krt_ifread_v6(flag_t save_task_state)
{
	static int sd = -1;

	struct lifconf ifc;
	struct lifreq *ifrp;
	struct _if_info ifi;
	if_link *if_plink, *iflp;
	task *tp;
	char *ptr, *cp, name[IFNAMSIZ];
	int isalias, slen, test_bit_set, interface_count;

	tp = krt_task;
	if_plink = NULL;
	test_bit_set = 0;
	isalias = FALSE;
	interface_count = 0;

	siocgifconf = SIOCGLIFCONF;
	siocgifaddr = SIOCGLIFADDR;
	siocgifdstaddr = SIOCGLIFDSTADDR;
	siocgifbrdaddr = SIOCGLIFBRDADDR;
	siocgifnetmask = SIOCGLIFNETMASK;
	siocgifflags = SIOCGLIFFLAGS;
	siocgifmtu = SIOCGLIFMTU;
	siocgifmetric = SIOCGLIFMETRIC;
	ifc.lifc_family = AF_INET6;
	ifc.lifc_flags = 0;

	/*
	 * grab a socket for use with ioctl calls. Note task_get_socket checks
	 * for test mode so have to reset bit.
	 */
	if (sd == -1) {

		if (BIT_TEST(task_state, TASKS_TEST)) {
			test_bit_set = 1;
			BIT_RESET(task_state, TASKS_TEST);
		}

		sd = task_floating_socket(tp,
		    task_get_socket(tp, AF_INET6, SOCK_DGRAM, 0),
		    "krt_ifread_task");

		if (test_bit_set)
			BIT_SET(task_state, TASKS_TEST);
	}

	if (krt_task->task_socket < 0)
		return EBADF;

	/* read the interfaces from the kernel */
	if (!read_interface_list6(tp, &ifc, sd)) {

		/* cannot do successful ioctl call */
		trace_log_tp(tp, 0, LOG_ERR,
		    ("krt_ifread: ioctl SIOCGIFCONF: %m"));
		return errno;
	}

	/* write our status to the log */
	trace_tp(tp, TR_KRT_IFLIST, TRC_NL_BEFORE,
	    ("krt_iflist: SIOCGIFCONF returns %u bytes", ifc.lifc_len));

	/* loop through all the data */
	for (ptr = ifc.lifc_buf; ptr < ifc.lifc_buf + ifc.lifc_len;) {
		/*
		 * zero out temporary data structure for
		 * storing values from kernel
		 */
		bzero(&ifi, sizeof (ifi));

		/* get pointer to next interface */
		ifrp = (struct lifreq *)ptr;

		/* keep track of how many interfaces we have */
		interface_count++;

		/*
		 * We use the name for two distinct reasons. In
		 * GateD we have a one to many relationship
		 * with a physical interface and its one or more
		 * interface addresses. For this we need a
		 * name that has no alias. e.g. iprb0 and iprb0:1
		 * both fall under the physical interface
		 * iprb0. For the ioctl calls we need an unmolested name.
		 */
		strcpy(name, ifrp->lifr_name);

		/* Remove the :n extension from the name */
		cp = index(name, ':');
		if (cp) {
			if (*(cp + 1) != '0')
				isalias = TRUE;

			*cp = (char)0;
		}


		/* xxx - dump the ifrp we just read */
		if (TRACE_TP(krt_task, TR_KRT_IFLIST))
			dump_ifrp(krt_task, ifrp);

		/* read interface specific info from kernel */
		get_interface_parameter_info6(tp, &ifi, ifrp->lifr_name, sd);

		/*
		 * Have to have a physical interface to link to
		 * If no previous or previous name is different
		 *
		 * XXX the third check seems iffy, what if there is a new
		 * XXX different name of the same length as the
		 * XXX previous with a ':' in it
		 *
		 * naamato 5/12/99
		 * By removing the ':' that should be fixed.  In the
		 * above problem, an interface with ifrp->ifr_name
		 * of "lan1:0" (not an alias) might get linked to
		 * the previous piflp.
		 *
		 * Now, lan1, lan1:0, lan1:1, lan1:2, etc. mean the same
		 * if_link.
		 */

		if (!if_plink
		    || (strncmp(if_plink->ifl_name, ifrp->lifr_name, IFNAMSIZ)
		    && (isalias == FALSE))) {
			/*
			 * either no physical interface or the name doesn't
			 * match, and it's not an alias
			 *
			 * xxx - krt_lladdr() is broke, we are using a
			 * hacked krt_lladdr() in krt_lladdr_sunos4.c.
			 * krt_lladdr() on sunos doesn't return correct llevel
			 * addr info, causing the ifrp info to be
			 * munged.  not using this code does away with correct
			 * ll info. so let's try to hack around it.
			 */
			slen = strlen(name);
			iflp = ifl_locate_name(name, slen);

			if_plink = ifl_addup(tp, iflp,
			    interface_count, ifi.ifi_state,
			    ifi.ifi_metric,
			    ifi.ifi_mtu, name,
			    slen, krt_lladdr(ifrp),
			    (sockaddr_un *)0);
 
			if (BIT_TEST(ifi.ifi_state, IFS_LOOPBACK)) {
				/*
				 * Set the loopback flag and mtu
				 * for this physical interface 
				 */
				BIT_SET(if_plink->ifl_state, IFS_LOOPBACK);
				ifi.ifi_mtu = LOOPBACK_MTU;
			}

		} else 
			/*
			 * XXX i don't know what's going on here,
			 * but why are we decrementing this?  This doesn't
			 * appear in krt_ifread_v4
			 *
			 * - naamato
			 */
			interface_count--;

		/* read address information from kernel. */
		get_interface_address_info6(tp, &ifrp->lifr_addr,
		    &ifi, if_plink, ifrp->lifr_name, sd);

		if (isalias == FALSE)
			BIT_SET(ifi.ifi_state, IFS_PRIMARY);

		/* Add the logical interface structure to the ifap list */
		if_conf_addaddr(tp, &ifi);

		/* all done with that address let's do it again */
		ptr += sizeof(struct lifreq) ;

	}
	return(1);
}
#endif /* PROTO_INET6 */

/*
 * Dump interface information 
 */
static void
dump_ifrp(task *tp, struct ifreq *ifrp)
{
	const char *cp =
	    trace_value(task_domain_bits, ifrp->ifr_addr.sa_family);
	int size;

	size = usalen(&ifrp->ifr_addr);

	tracef("krt_ifread: name %.*s  length %u  family %u",
	    IFNAMSIZ, ifrp->ifr_name, size, ifrp->ifr_addr.sa_family);

	if (cp)
		tracef("(%s)", cp);

	switch (ifrp->ifr_addr.sa_family) {
#ifdef PROTO_INET
 	case AF_INET:
	{
		struct sockaddr_in *sinp;

		sinp = (struct sockaddr_in *)((void_t)&ifrp->ifr_addr);
		tracef("  port %u  addr %A", ntohs(sinp->sin_port),
		    sockbuild_in(0, sinp->sin_addr.s_addr));
	}
		break;
#endif /* PROTO_INET */
#ifdef PROTO_ISO
	case AF_ISO:
	{
		struct sockaddr_iso *siso;
		byte *dp;

		siso = (struct sockaddr_iso *)&ifrp->ifr_addr;
		dp = (byte *)siso->siso_pad;

		tracef("  addr %A",
		    siso->siso_addr.isoa_genaddr,
		    siso->siso_addr.isoa_len);

		if (siso->siso_plen)
			tracef("  psel %A",
			    sockbuild_ll(0, dp, siso->siso_plen));
			    dp += siso->siso_plen;

		if (siso->siso_slen)
			tracef("  ssel %A",
			    sockbuild_ll(0, dp, siso->siso_slen));
			    dp += siso->siso_slen;

		if (siso->siso_tlen)
			tracef("  tsel %A",
			    sockbuild_ll(0, dp, siso->siso_tlen));
	}
		break;
#endif /* PROTO_ISO */

#ifdef  SOCKADDR_DL
	case AF_LINK:
	{
		struct sockaddr_dl *sdl;
		byte *dp;

		sdl = (struct sockaddr_dl *)&ifrp->ifr_addr;
		dp = (byte *)sdl->sdl_data;

		tracef("  index %u  type %u",
		    sdl->sdl_index, sdl->sdl_type);

		if (sdl->sdl_nlen)
			tracef("  name %.*s",
			    sdl->sdl_nlen, dp);
			    dp += sdl->sdl_nlen;

		if (sdl->sdl_alen)
			tracef("  addr %A",
			    sockbuild_ll(0, dp, sdl->sdl_alen));
			    dp += sdl->sdl_alen;

		if (sdl->sdl_slen)
			tracef("  selector %A",
			    sockbuild_ll(0, dp, sdl->sdl_slen));
	}
		break;
#endif /* SOCKADDR_DL */

	default:
		tracef("  addr %A",
		    sockbuild_ll(0,
		    (byte *)ifrp->ifr_addr.sa_data,
		    (size_t)(size -
		    ((byte *)ifrp->ifr_addr.sa_data -
		    (byte *)&ifrp->ifr_addr))));
	}
	trace_only_tp(tp, TRC_NL_BEFORE, (NULL));
}

/*
 * Read the interface list from the kernel
 */
int
krt_ifread(flag_t save_task_state)
{
	task *tp;

	tp = krt_task;

	/* set interface lists to known state, IFC_NOCHANGE */
	if_conf_open(tp, TRUE);

	krt_ifread_v4(save_task_state);
#ifdef PROTO_INET6
#ifdef SIOCGLIFCONF
	krt_ifread_v6(save_task_state);
#endif /* PROTO_INET6  */
#endif /* SIOCGLIFCONF  */

	if_conf_close(tp, FALSE);
}
