/*
 * Public Release 3
 * 
 * $Id: rip6query.c,v 1.5 2000/04/11 05:23:01 swright Exp $
 */

/*
 * ------------------------------------------------------------------------
 * 
 * Copyright (c) 1996, 1997 The Regents of the University of Michigan
 * All Rights Reserved
 *  
 * Royalty-free licenses to redistribute GateD Release
 * 3 in whole or in part may be obtained by writing to:
 * 
 * 	Merit GateDaemon Project
 * 	4251 Plymouth Road, Suite C
 * 	Ann Arbor, MI 48105
 *  
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT DO NOT WARRANT THAT THE
 * FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR
 * THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE. The Regents of the
 * University of Michigan and Merit shall not be liable for
 * any special, indirect, incidental or consequential damages with respect
 * to any claim by Licensee or any third party arising from use of the
 * software. GateDaemon was originated and developed through release 3.0
 * by Cornell University and its collaborators.
 * 
 * Please forward bug fixes, enhancements and questions to the
 * gated mailing list: gated-people@gated.merit.edu.
 * 
 * ------------------------------------------------------------------------
 * 
 * Copyright (c) 1990,1991,1992,1993,1994,1995 by Cornell University.
 *     All rights reserved.
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT
 * LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * GateD is based on Kirton's EGP, UC Berkeley's routing
 * daemon	 (routed), and DCN's HELLO routing Protocol.
 * Development of GateD has been supported in part by the
 * National Science Foundation.
 * 
 * ------------------------------------------------------------------------
 * 
 * Portions of this software may fall under the following
 * copyrights:
 * 
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms are
 * permitted provided that the above copyright notice and
 * this paragraph are duplicated in all such forms and that
 * any documentation, advertising materials, and other
 * materials related to such distribution and use
 * acknowledge that the software was developed by the
 * University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote
 * products derived from this software without specific
 * prior written permission.  THIS SOFTWARE IS PROVIDED
 * ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


#define	INCLUDE_TIME
#define	INCLUDE_CTYPE
#define	MALLOC_OK

#include "include.h"
#include "inet4core/inet.h"
#include "inet6core/inet6.h"
#include "gated/targets.h"
#include "ripng/ripng.h"
#include "ripng/ripng_internal.h"

/* macros to select internet address given pointer to a struct sockaddr */

#define WTIME	5			/* Time to wait for responses */
#define STIME	1			/* Time to wait between packets */

static int s;
static char packet[BUFSIZ];
static int dflag;
static int nflag;
const char *version = "1.25.2.6";
static int vers = RIPNG_VERSION_1;

const bits ripng_cmd_bits[] = {
    { 0,			"Invalid" },
    { RIPNG_COMMAND_REQUEST,	"Request" },
    { RIPNG_COMMAND_RESPONSE,	"Response" },
    { 0 }
};

/* List of destinations to query */

static int n_dests = 0;

struct dest {
    struct dest *d_forw;
    struct dest *d_back;

    struct in6_addr d_prefix;
    u_int32 d_prefixlen;
} ;

static struct dest dests = { &dests, &dests };
static struct sockaddr_in6 router;

/*
 *	Trace RIPng packets
 */
static void
ripngq_trace (const char *dir,
							struct sockaddr_in6 *who,
							void_t cp, 
							register size_t size)
{
    register struct ripng *rpmsg = (struct ripng *) cp;
    register struct ripng_netinfo *n;
    register struct ripng_nexthop *np;
    register const char *cmd = "Invalid";
    char dst[BUFSIZ];

    (void) inet_ntop(AF_INET6, (void_t) &who->sin6_addr, dst, sizeof(dst));
    if (rpmsg->ripng_command && rpmsg->ripng_command < RIPNG_COMMAND_MAX) {
	cmd = trace_state(ripng_cmd_bits, rpmsg->ripng_command);
    }
    (void) fprintf(stderr, "RIPng %s %s+%d vers %d, cmd %s, length %d",
		   dir,
		   dst,
		   ntohs(who->sin6_port),
		   rpmsg->ripng_version,
		   cmd,
		   size);

    switch (rpmsg->ripng_command) {
	case RIPNG_COMMAND_REQUEST:
	case RIPNG_COMMAND_RESPONSE:
	    (void) fprintf(stderr, "\n");
	    size -= 4 * sizeof(char);
	    n = (struct ripng_netinfo *) ((void_t) (rpmsg + 1));
	    for (; size > 0; n++, size -= sizeof(struct ripng_netinfo)) {
		metric_t metric = n->ripng_metric;
		
		if (size < sizeof(struct ripng_netinfo)) {
		    break;
		}
		switch (metric) {
		    /* RTE */
		default:
		    if (rpmsg->ripng_version == RIPNG_VERSION_1) {
			(void) fprintf(stderr,
				       "\tnet %15s/",
				       inet_ntop(AF_INET6, &n->ripng_prefix,
						 dst, sizeof(dst)));
			(void) fprintf(stderr,
				       "%-15d  ",
				       n->ripng_prefixlen);
			(void) fprintf(stderr,
				       "metric %2d  tag %#04X\n",
				       metric,
				       ntohs((u_int16) n->ripng_tag));
		    }
		    break;

		    /* RTE next hop */
		case RIPNG_METRIC_NEXTHOP:
		    np = (struct ripng_nexthop *) n;
		    (void) fprintf(stderr,
				   "\trouter %-15s\n",
				   inet_ntop(AF_INET6, &np->ripng_nexthop,
					     dst, sizeof(dst)));
		}
	    }
	    (void) fprintf(stderr,
			   "RIPng %s end of packet", dir);
	    break;

	default:
	    (void) fprintf(stderr, "\n");
	    break;
    }
    (void) fprintf(stderr, "\n");
}


static void
ripngq_query (char *host, int cmd, u_int ifindex)
{
    struct ripng *msg = (struct ripng *) ((void_t) packet);
    struct ripng_netinfo *nets = (struct ripng_netinfo *) ((void_t) (msg + 1));
    int rc;
    struct msghdr mh;
    struct cmsghdr *cmh;
    struct iovec iov[2];
    byte cmsgbuf[256];
    struct in6_pktinfo *pi;

    bzero((char *) &router, sizeof(router));
    if (!inet_pton(AF_INET6, host, (void_t) &router.sin6_addr)) {
		int error_num;
#ifdef HAVE_GETIPNODEBYNAME
		struct hostent *hp = getipnodebyname(host, AF_INET6, AI_DEFAULT, &error_num);
#else
		struct hostent *hp = gethostbyname(host);
#endif
	if (!hp) {
	    (void) printf("%s: unknown\n", host);
	    exit(1);
	} else {
	    (void) bcopy(hp->h_addr, (char *) &router.sin6_addr, (size_t) hp->h_length);
	}
    }
    router.sin6_family = AF_INET6;
    router.sin6_port = task_get_port((trace *) 0,
				     "router6", "udp",
				     htons(RIPNG_PORT));
#ifdef USE_SOCKLEN
    router.sin6_len = sizeof(struct sockaddr_in6);
#endif
    msg->ripng_command = cmd;
    msg->ripng_version = vers;

    /* Fill in the request */
    if (n_dests) {
	register struct dest *dp;
	
	/* Query specific nets */

	for (dp = dests.d_forw; dp != &dests; dp = dp->d_forw) {
	    bzero((caddr_t) nets, sizeof *nets);
	    nets->ripng_prefix = dp->d_prefix;
	    nets->ripng_prefixlen = dp->d_prefixlen;
	    nets++;
	}
    } else {
	bzero((caddr_t) nets, sizeof *nets);
	nets->ripng_metric = RIPNG_METRIC_UNREACHABLE;
	nets++;
    }

    if (dflag) {
	ripngq_trace("SEND", &router, (void_t) msg, (caddr_t) nets - packet);
    }
    iov[0].iov_base = (caddr_t) packet;
    iov[0].iov_len = (caddr_t) nets - packet;
    mh.msg_iov = iov;
    mh.msg_iovlen = 1;
    mh.msg_name = (caddr_t) & router;
    mh.msg_namelen = sizeof(router);
    mh.msg_flags  = 0;
    bzero((caddr_t *) cmsgbuf, sizeof (cmsgbuf));
    cmh = (struct cmsghdr *) cmsgbuf;
    mh.msg_control = (caddr_t) cmh;
    mh.msg_controllen = CMSG_SPACE(sizeof (struct in6_pktinfo));

    cmh->cmsg_len = CMSG_LEN(sizeof (struct in6_pktinfo));
    cmh->cmsg_level = IPPROTO_IPV6;
    cmh->cmsg_type = IPV6_PKTINFO;
    pi = (struct in6_pktinfo *) CMSG_DATA(cmh);
    bzero((caddr_t *)&pi->ipi6_addr, sizeof (pi->ipi6_addr));
    pi->ipi6_ifindex = ifindex;

    NON_INTR(rc,
	     sendmsg(s,
		     &mh,
		    0));
    if (rc < 0) {
	perror(host);
    }
}


/*
 * Handle an incoming routing packet.
 */
static void
ripngq_input (struct sockaddr_in6 *from,
							struct in6_addr *to, 
							size_t size)
{
    register struct ripng *msg = (struct ripng *) ((void_t) packet);
    struct ripng_netinfo *n;
    struct ripng_nexthop *np;
    struct hostent *hp;
    char dst[BUFSIZ];

    if (dflag) {
	ripngq_trace("RECV", from, (void_t) msg, size);
    }
    if (msg->ripng_command != RIPNG_COMMAND_RESPONSE)
	return;
    if (nflag
	|| !(hp = gethostbyaddr((char *) &from->sin6_addr, sizeof(struct in6_addr), AF_INET6))) {
	(void) printf("%d bytes from %s",
		      size,
		      inet_ntop(AF_INET6, &from->sin6_addr, dst, sizeof(dst)));
    } else {
	(void) printf("%d bytes from %s(%s)",
		      size,
		      hp->h_name,
		      inet_ntop(AF_INET6, &from->sin6_addr, dst, sizeof(dst)));
    }
    if (to) {
	(void) printf(" to %s",
		      inet_ntop(AF_INET6, &to, dst, sizeof(dst)));
    }
    printf(" version %u:\n",
	   msg->ripng_version);
    size -= sizeof(int);
    n = (struct ripng_netinfo *) ((void_t) (msg + 1));
    while (size > 0) {
	if (size < sizeof(struct ripng_netinfo))
	    break;
	switch (msg->ripng_version) {
	case RIPNG_VERSION_1:
	    if (n->ripng_metric == RIPNG_METRIC_NEXTHOP) {
		np = (struct ripng_nexthop *) n;
		(void) printf("router %-15s  ",
			      inet_ntop(AF_INET6, (void_t) &np->ripng_nexthop,
					dst, sizeof(dst)));
	    } else {
		(void) printf("\t%15s/",
			      inet_ntop(AF_INET6, (void_t) &n->ripng_prefix, 
					dst, sizeof(dst)));
		(void) printf("%-4d  ", n->ripng_prefixlen);
		(void) printf("metric %2d  tag %#04X\n",
			      n->ripng_metric,
			      n->ripng_tag);
	    }
	    break;

	}
	size -= sizeof(struct ripng_netinfo), n++;
    }
}


int
main (int argc, char **argv)
{
    int c, cc, fdbits, errflg = 0, cmd = RIPNG_COMMAND_REQUEST;
    static struct timeval *wait_time, long_time = {WTIME, 0}, short_time = {STIME, 0};
    static struct sockaddr_in6 from;
    char dst[BUFSIZ];
    char *ifname = NULL;
    u_int ifindex = 0;
#if	defined(SCM_RIGHTS) && !defined(sun)
    int on = 1;
    struct in6_addr *to;
    static byte control[BUFSIZ];
    static struct iovec iovec = { (caddr_t) packet, sizeof (packet) };
    static struct msghdr msghdr = {
	(caddr_t) &from, sizeof (from),
	&iovec, 1,
	(caddr_t) control, sizeof(control),
	0 } ;
#else	/* defined(SCM_RIGHTS) && !defined(sun) */
    int fromlen = sizeof (from);
#endif	/* defined(SCM_RIGHTS) && !defined(sun) */

    s = socket(AF_INET6, SOCK_DGRAM, 0);
    if (s == -1) {
	perror("socket");
	exit(2);
    }
#if	defined(SCM_RIGHTS) && !defined(sun)
    if (setsockopt(s, IPPROTO_IPV6, IPV6_PKTINFO, &on, sizeof (on)) < 0) {
	perror("setsockopt(IP_RECVDSTADDR)");
	exit (2);
    }
#endif	/* defined(SCM_RIGHTS) && !defined(sun) */

    while ((c = getopt(argc, argv, "dnrw:N:I:")) != EOF) {
	switch (c) {
	case 'd':
	    dflag++;
	    break;

	case 'n':
	    nflag++;
	    break;

	case 'r':
	    cmd = RIPNG_COMMAND_REQUEST;
	    break;

	case 'w':
	    long_time.tv_sec = atoi(optarg);
	    (void) fprintf(stderr, "Wait time set to %d\n", long_time.tv_sec);
	    break;

	case 'N':
	    if (n_dests < 23) {
		struct in6_addr dest;
		int masklen;
		struct dest *dp;
		register char *cp = (char *) strchr(optarg, '/');

		if (cp) {
		    *cp = (char ) 0;

		    if (!inet_pton(AF_INET6, optarg, &dest)) {
			cp = optarg;
		    BadDest:
			(void) fprintf(stderr,
				       "unable to parse destination: %s\n",
				       cp);
			exit(1);
		    }
		    masklen = atoi(++cp);
		    if (masklen < 0 || masklen > 128) {
			goto BadDest;
		    }
		    vers = RIPNG_VERSION_1;
		} else {
		    if (!inet_pton(AF_INET6, optarg, &dest)) {
			cp = optarg;
			goto BadDest;
		    }
		    masklen = 0;
		}
		dp = (struct dest *) calloc(1, sizeof *dp);
		dp->d_prefix = dest;		/* struct copy */
		dp->d_prefixlen = masklen;
		INSQUE(dp, dests.d_back);
		n_dests++;
	    } else {
		(void) fprintf(stderr,
			       "too many destinations specified\n");
		exit(1);
	    }
	    break;

	case 'I':
	    ifname = optarg;
	    if ((ifindex = if_nametoindex(ifname)) == 0)
				errflg++;
	    if (ifindex == 0 || ifindex > 65535)
				errflg++;
	    break;
		
	case '?':
	    errflg++;
	    break;
	}
    }

    if (errflg || (optind >= argc)) {
	(void) printf("usage: rip6query [ -d ] [ -n ] [ -r ] [ -v ] [ -w time] [ -N dest[/mask] ] [-I interface name ] hosts...\n");
	exit(1);
    }
    setnetent(1);

    for (; optind < argc; optind++) {
	ripngq_query(argv[optind], cmd, ifindex);
	fdbits = 1 << s;
	wait_time = &long_time;
	for (;;) {
	    cc = select(s + 1, (fd_set *) & fdbits, (fd_set *) 0, (fd_set *) 0, wait_time);
	    if (cc == 0) {
		if (wait_time == &short_time) {
		    break;
		}
		break;
	    } else if (cc < 0) {
		perror("select");
		(void) close(s);
		exit(1);
	    } else {
		wait_time = &short_time;
#if	defined(SCM_RIGHTS) && !defined(sun)
		msghdr.msg_namelen = sizeof (from);
		msghdr.msg_controllen = sizeof (control) ;
		msghdr.msg_flags = 0;
		NON_INTR(cc, recvmsg(s, &msghdr, 0));
#else	/* defined(SCM_RIGHTS) && !defined(sun) */
		NON_INTR(cc, recvfrom(s,
				      packet,
				      sizeof (packet),
				      0,
				      (struct sockaddr *) &from,
				      (void_t) &fromlen));
#endif	/* defined(SCM_RIGHTS) && !defined(sun) */
		if (cc <= 0) {
		    if (cc < 0) {
			perror("recvmsg");
			(void) close(s);
			exit(1);
		    }
		    continue;
		}
#if	defined(SCM_RIGHTS) && !defined(sun)
#define	ENOUGH_CMSG(cmsg, size)	((cmsg)->cmsg_len >= ((size) + sizeof(struct cmsghdr)))

		if (msghdr.msg_flags & MSG_TRUNC) {
		    (void) fprintf(stderr, "message from %s truncated to %d bytes\n",
				   inet_ntop(AF_INET6, (void_t) &from.sin6_addr,
					     dst, sizeof(dst)),
				   cc);
		}

		to = (struct in6_addr *) 0;
		if (msghdr.msg_controllen >= sizeof (struct cmsghdr)
		    && !(msghdr.msg_flags & MSG_CTRUNC)) {
		    struct cmsghdr *cmsg;

		    for (cmsg = CMSG_FIRSTHDR(&msghdr);
			 cmsg && cmsg->cmsg_len >= sizeof (struct cmsghdr);
			 cmsg = CMSG_NXTHDR(&msghdr, cmsg)) {

			if (cmsg->cmsg_level == IPPROTO_IPV6
			    && cmsg->cmsg_type == IPV6_PKTINFO
			    && ENOUGH_CMSG(cmsg, sizeof (struct in6_addr))) {
			    bcopy((byte *) &((struct in6_pktinfo *)CMSG_DATA(cmsg))->ipi6_addr, &to, sizeof(struct in6_addr));
			}
		    }
		}

		ripngq_input(&from, to, cc);
#else	/* defined(SCM_RIGHTS) && !defined(sun) */
		ripngq_input(&from, (struct in6_addr *) 0, cc);
#endif	/* defined(SCM_RIGHTS) && !defined(sun) */
	    }
	}
    }

    endnetent();
    return 0;
}
