/*  $Revision: 1.1.1.1 $
**
**  Expire news articles.
*/
#include <stdio.h>
#include <sys/types.h>
#include "configdata.h"
#include "clibrary.h"
#include <ctype.h>
#include <sys/stat.h>
#if	defined(DO_NEED_TIME)
#include <time.h>
#endif	/* defined(DO_NEED_TIME) */
#include <sys/time.h>
#include <errno.h>
#include "paths.h"
#include "libinn.h"
#include "inndcomm.h"
#include "dbz.h"
#include "qio.h"
#include "macros.h"
#include "mydir.h"
#include "tree.h"
#include "can.h"

/*
**  Stuff that more or less duplicates stuff in innd.
*/
#define NGH_HASH(Name, p, j)    \
	for (p = Name, j = 0; *p; ) j = (j << 5) + j + *p++
#define NGH_SIZE	2048
#define NGH_BUCKET(j)	&NGHtable[j & (NGH_SIZE - 1)]

typedef struct _BUFFER {
    int		Size;
    int		Used;
    int		Left;
    char	*Data;
} BUFFER;

typedef struct _NEWSGROUP {
    char		*Name;
    char		*Rest;
    unsigned long	Last;
	/* These fields are new. */
    time_t		Keep;
    time_t		Default;
    time_t		Purge;
} NEWSGROUP;

typedef struct _NGHASH {
    int		Size;
    int		Used;
    NEWSGROUP	**Groups;
} NGHASH;


/*
**  Expire-specific stuff.
*/
#define MAGIC_TIME	49710.

typedef struct _BADGROUP {
    struct _BADGROUP	*Next;
    char		*Name;
    BOOL		HasDirectory;
} BADGROUP;

STATIC BOOL		EXPlinks;
STATIC BOOL		EXPquiet;
STATIC BOOL		EXPsizing;
STATIC BOOL		EXPtracing;
STATIC BOOL		EXPusepost;
STATIC BOOL		EXPearliest;
STATIC char		SPOOL[] = _PATH_SPOOL;
STATIC int		nGroups=10000;
STATIC FILE		*EXPunlinkfile;
STATIC long		EXPsaved;
STATIC STRING		EXPreason;
STATIC time_t		EXPremember;
STATIC time_t		Now;
STATIC time_t		RealNow;

/* Statistics; for -v flag. */
STATIC char		*EXPgraph;
STATIC int		EXPverbose;
STATIC long		EXPprocessed;
STATIC long		EXPunlinked;
STATIC long		EXPhistdrop;
STATIC long		EXPhistremember;
STATIC long		EXPallgone;
STATIC long		EXPstillhere;

STATIC int		EXPsplit();

#if ! defined (atof)            /* NEXT defines aotf as a macro */
extern double		atof();
#endif


/*
**  Open a file or give up.
*/
STATIC FILE *
EXPfopen(Remove, Name, Mode)
    BOOL	Remove;
    STRING	Name;
    char	*Mode;
{
    FILE	*F;

    if (Remove && unlink(Name) < 0 && errno != ENOENT)
	(void)fprintf(stderr, "Warning, can't remove %s, %s\n",
		Name, strerror(errno));
    if ((F = fopen(Name, Mode)) == NULL) {
	(void)fprintf(stderr, "Can't open %s in %s mode, %s\n",
		Name, Mode, strerror(errno));
	exit(1);
    }
    return F;
}


/*
**  Split a line at a specified field separator into a vector and return
**  the number of fields found, or -1 on error.
*/
STATIC int
EXPsplit(p, sep, argv, count)
    register char	*p;
    register char	sep;
    register char	**argv;
    register int	count;
{
    register int	i;

    for (i = 1, *argv++ = p; *p; )
	if (*p++ == sep) {
	    if (++i == count)
		/* Overflow. */
		return -1;
	    p[-1] = '\0';
	    for (*argv++ = p; *p == sep; p++)
		continue;
	}
    return i;
}


/*
**  Parse a number field converting it into a "when did this start?".
**  This makes the "keep it" tests fast, but inverts the logic of
**  just about everything you expect.  Print a message and return FALSE
**  on error.
*/
STATIC BOOL
EXPgetnum(line, word, v, name)
    int			line;
    char		*word;
    time_t		*v;
    char		*name;
{
    register char	*p;
    register BOOL	SawDot;
    double		d;

    if (caseEQ(word, "never")) {
	*v = (time_t)0;
	return TRUE;
    }

    /* Check the number.  We don't have strtod yet. */
    for (p = word; ISWHITE(*p); p++)
	continue;
    if (*p == '+' || *p == '-')
	p++;
    for (SawDot = FALSE; *p; p++)
	if (*p == '.') {
	    if (SawDot)
		break;
	    SawDot = TRUE;
	}
	else if (!CTYPE(isdigit, *p))
	    break;
    if (*p) {
	(void)fprintf(stderr, "Line %d, bad `%c' character in %s field\n",
		line, *p, name);
	return FALSE;
    }
    d = atof(word);
    if (d > MAGIC_TIME)
	*v = (time_t)0;
    else
	*v = Now - (time_t)(d * 86400.);
    return TRUE;
}


/*
**  Parse the expiration control file.  Return TRUE if okay.
*/
STATIC BOOL
EXPreadfile(F)
    register FILE	*F;
{
    register char	*p;
    register int	i;
    register int	j;
    char		buff[BUFSIZ];
    char		*fields[6];

    /* Scan all lines. */
    EXPremember = -1;
    for (i = 1; fgets(buff, sizeof buff, F) != NULL; i++) {
	if ((p = strchr(buff, '\n')) == NULL) {
	    (void)fprintf(stderr, "Line %d too long\n", i);
	    return FALSE;
	}
	*p = '\0';
        p = strchr(buff, '#');
	if (p)
	    *p = '\0';
	else
	    p = buff + strlen(buff);
	while (--p >= buff) {
	    if (isspace(*p))
                *p = '\0';
            else
                break;
        }
        if (buff[0] == '\0')
	    continue;
	if ((j = EXPsplit(buff, ':', fields, SIZEOF(fields))) == -1) {
	    (void)fprintf(stderr, "Line %d too many fields\n", i);
	    return FALSE;
	}

	/* Expired-article remember line? */
	if (EQ(fields[0], "/remember/")) {
	    if (j != 2) {
		(void)fprintf(stderr, "Line %d bad format\n", i);
		return FALSE;
	    }
	    if (EXPremember != -1) {
		(void)fprintf(stderr, "Line %d duplicate /remember/\n", i);
		return FALSE;
	    }
	    if (!EXPgetnum(i, fields[1], &EXPremember, "remember"))
		return FALSE;
	    continue;
	}
    }

    return TRUE;
}


/*
**  Do the work of expiring one line.
*/
STATIC BOOL
EXPdoline(out, line, length, arts)
    FILE		*out;
    char		*line;
    int			length;
    char		**arts;
{
    static char		IGNORING[] = "Ignoring bad line, \"%.20s...\"\n";
    static long		Offset;
    register int	i;
    char		*fields[4];
    time_t		Arrived;
    time_t		deleteOfCan;
    long		where;
    datum		key;
    datum		value;

    /* Split up the major fields. */
    i = EXPsplit(line, HIS_FIELDSEP, fields, SIZEOF(fields));
    if (i != 2 && i != 3) {
	(void)fprintf(stderr, IGNORING, line);
	return TRUE;
    }
    if ( strlen(fields[0]) >= (DBZMAXKEY-1) ) { 
      (void)fprintf(stderr, IGNORING, line);
      return TRUE;
    }

    Arrived = atol(fields[1]);

    if (i == 2) {
	/* History line for already-expired article. */
	if (Arrived < EXPremember || Arrived > Now + 86400) {
	    if (EXPverbose > 3)
		(void)printf("forget: %s\n", line);
	    EXPhistdrop++;
	    return TRUE;
	}

	/* Not time to forget about this one yet. */
	if (out) {
	    where = Offset;
	    (void)fprintf(out, "%s%c%s\n", fields[0], HIS_FIELDSEP, fields[1]);
	    Offset += strlen(fields[0]) + 1 + strlen(fields[1]) + 1;
	    if (EXPverbose > 3)
		(void)printf("remember: %s\n", line);
	    EXPhistremember++;
	}
    }
    else {
	EXPprocessed++;
	deleteOfCan=CANdelete(fields[2]);
	
	if( deleteOfCan<Now )
	{   if (EXPremember > 0 && out != NULL) {
		where = Offset;
		if (Arrived > RealNow)
		    Arrived = RealNow;
		(void)fprintf(out, "%s%c%s\n",
			fields[0], HIS_FIELDSEP, fields[1]);
		Offset += strlen(fields[0]) + 1 + strlen(fields[1]) + 1;
	        if (EXPverbose > 3)
		    (void)printf("remember history: %s%c%s\n",
			    fields[0], HIS_FIELDSEP, fields[1]);
		EXPallgone++;
	    }
	}
	else if (out) {
	    where = Offset;
	    (void)fprintf(out, "%s%c%s%c%s\n",
		    fields[0], HIS_FIELDSEP, fields[1], HIS_FIELDSEP,
		    fields[2]);
	    Offset += strlen(fields[0]) + 1 + strlen(fields[1]) + 1
		    + strlen(fields[2]) + 1;
	    if (EXPverbose > 3)
		(void)printf("remember article: %s%c%s%c%s\n",
			fields[0], HIS_FIELDSEP, fields[1], HIS_FIELDSEP,
			fields[2]);
	    EXPstillhere++;
	}
    }

    if (out == NULL)
	return TRUE;

    if (ferror(out)) {
	(void)fprintf(stderr, "Can't write new history, %s\n",
		strerror(errno));
	return FALSE;
    }

    /* Set up the DBZ data.  We don't have to sanitize the Message-ID
     * since it had to have been clean to get in there. */
    key.dptr = fields[0];
    key.dsize = strlen(key.dptr) + 1;
    value.dptr = (char *)&where;
    value.dsize = sizeof where;
    if (EXPverbose > 4)
	(void)printf("\tdbz %s@%ld\n", key.dptr, where);
    if (dbzstore(key, value) < 0) {
	(void)fprintf(stderr, "Can't store key, %s\n", strerror(errno));
	return FALSE;
    }
    return TRUE;
}

STATIC BOOL
EXPremoveCan()
{
    static char	CANTCD[] = "Can't cd to %s, %s\n";
    DIR		*dirp, *shelfDirp;
    DIRENTRY	*dp, *shelfDp;
    time_t	deleteOfCan;
    struct stat	Sb;
    char	name[SPOOLNAMEBUFF];
    char	x=0;

    if (chdir(SPOOL) < 0) {
	(void)fprintf(stderr, CANTCD, SPOOL, strerror(errno));
	exit(1);
    }

    dirp=opendir(".");
    for( dp=readdir(dirp); dp!=NULL; dp=readdir(dirp) )
    {	if( !strncmp(dp->d_name, "shelf.", 6))
	{   shelfDirp=opendir(dp->d_name);
	
	    for(shelfDp=readdir(shelfDirp);shelfDp;shelfDp=readdir(shelfDirp) )
	    {	if( strlen(shelfDp->d_name)!=strlen("yyyymmddhhxx") )
		    continue;
			    
		if( (deleteOfCan=CANdelete(shelfDp->d_name))==0 )
		    continue;
		
		if( deleteOfCan<Now )
		{   sprintf(name, "%s/%s", dp->d_name, shelfDp->d_name);
		
		    if( EXPsizing && stat(name, &Sb) >= 0)
		    {	EXPsaved+=(int)(((long)Sb.st_size >> 10) +
					(((long)Sb.st_size >> 9) & 1));
		    }
		    if (EXPverbose > 1)
			(void)printf("\tunlink %s\n", name);
		    EXPunlinked++;

		    if (EXPtracing) 
		    	(void)printf("%s\n", name);
		    else if (unlink(name) < 0 && errno != ENOENT)
		    {	(void)fprintf(stderr, "Can't unlink %s, %s\n", 
							name, strerror(errno));
			x=1;
		    }
		}		
	    }

	    closedir(shelfDirp);
	}
    }
    closedir(dirp);
    
    return x;
}

/*
**  Clean up link with the server and exit.
*/
STATIC NORETURN
CleanupAndExit(Server, Paused, x)
    BOOL	Server;
    BOOL	Paused;
    int		x;
{
    FILE	*F;

    if( EXPremoveCan() )
	x = 1;

    if (Server)
	(void)ICCreserve("");
    if (Paused && ICCgo(EXPreason) != 0) {
	(void)fprintf(stderr, "Can't unpause server, %s\n",
		strerror(errno));
	x = 1;
    }
    if (Server && ICCclose() < 0) {
	(void)fprintf(stderr, "Can't close communication link, %s\n",
		strerror(errno));
	x = 1;
    }
    if (EXPunlinkfile && fclose(EXPunlinkfile) == EOF) {
	(void)fprintf(stderr, "Can't close -z file, %s\n", strerror(errno));
	x = 1;
    }

    /* Report stats. */
    if (EXPsizing)
	(void)printf("%s approximately %ldk\n",
	    EXPtracing ? "Would remove" : "Removed", EXPsaved);
    if (EXPverbose) {
	(void)printf("Article lines processed %8ld\n", EXPprocessed);
	(void)printf("Articles retained       %8ld\n", EXPstillhere);
	(void)printf("Entries expired         %8ld\n", EXPallgone);
	(void)printf("Files unlinked          %8ld\n", EXPunlinked);
	(void)printf("Old entries dropped     %8ld\n", EXPhistdrop);
	(void)printf("Old entries retained    %8ld\n", EXPhistremember);
    }

    /* Append statistics to a summary file */
    if (EXPgraph) {
	F = EXPfopen(FALSE, EXPgraph, "a");
	(void)fprintf(F, "%ld %ld %ld %ld %ld %ld %ld\n",
		(long)Now, EXPprocessed, EXPstillhere, EXPallgone,
		EXPunlinked, EXPhistdrop, EXPhistremember);
	(void)fclose(F);
    }

    exit(x);
}


/*
**  Print a usage message and exit.
*/
STATIC NORETURN
Usage()
{
    (void)fprintf(stderr, "Usage: expire [flags] [expire.ctl]\n");
    exit(1);
}


int
main(ac, av)
    int			ac;
    char		*av[];
{
    static char		CANTCD[] = "Can't cd to %s, %s\n";
    register int	i;
    register int	line;
    register char	*p;
    register QIOSTATE	*qp;
    FILE		*F;
    char		**arts;
    STRING		History;
    STRING		HistoryText;
    STRING		HistoryPath;
    STRING		HistoryDB;
    char		*Historydir;
    char		*Historypag;
    char		*NHistory;
    char		*NHistorydir;
    char		*NHistorypag;
    char		*EXPhistdir;
    char		buff[SMBUF];
    register FILE	*out;
    BOOL		Server;
    BOOL		Paused=FALSE;
    BOOL		Bad=FALSE;
    BOOL		IgnoreOld;
    BOOL		Writing;
    BOOL		UnlinkFile;
    time_t		TimeWarp;

    /* Set defaults. */
    Server = TRUE;
    IgnoreOld = FALSE;
    History = "history";
    HistoryText = _PATH_HISTORY;
    HistoryPath = NULL;
    Writing = TRUE;
    TimeWarp = 0;
    UnlinkFile = FALSE;
    (void)umask(NEWSUMASK);

    /* find the default history file directory */
    EXPhistdir = COPY(_PATH_HISTORY);
    p = strrchr(EXPhistdir, '/');
    if (p != NULL) {
	*p = '\0';
    }

    /* Parse JCL. */
    while ((i = getopt(ac, av, "f:h:d:eg:ilnpqr:stv:w:xz:")) != EOF)
	switch (i) {
	default:
	    Usage();
	    /* NOTREACHED */
	case 'd':
	    HistoryPath = optarg;
	    break;
        case 'e':
	    EXPearliest = TRUE;
	    break;
	case 'f':
	    History = optarg;
	    break;
	case 'g':
	    EXPgraph = optarg;
	    break;
	case 'h':
	    HistoryText = optarg;
	    break;
	case 'i':
	    IgnoreOld = TRUE;
	    break;
	case 'l':
	    EXPlinks = TRUE;
	    break;
	case 'n':
	    Server = FALSE;
	    break;
	case 'p':
	    EXPusepost = TRUE;
	    break;
	case 'q':
	    EXPquiet = TRUE;
	    break;
	case 'r':
	    EXPreason = optarg;
	    break;
	case 's':
	    EXPsizing = TRUE;
	    break;
	case 't':
	    EXPtracing = TRUE;
	    break;
	case 'v':
	    EXPverbose = atoi(optarg);
	    break;
	case 'w':
	    TimeWarp = (time_t)(atof(optarg) * 86400.);
	    break;
	case 'x':
	    Writing = FALSE;
	    break;
	case 'z':
	    EXPunlinkfile = EXPfopen(TRUE, optarg, "a");
	    UnlinkFile = TRUE;
	    break;
	}
    ac -= optind;
    av += optind;
    if (ac != 0 && ac != 1)
	Usage();

    (void)time(&Now);
    RealNow = Now;
    Now += TimeWarp;

    /* Parse the control file. */
    if (av[0])
	F = EQ(av[0], "-") ? stdin : EXPfopen(FALSE, av[0], "r");
    else
	F = EXPfopen(FALSE, _PATH_EXPIRECTL, "r");
    if (!EXPreadfile(F)) {
	(void)fclose(F);
	(void)fprintf(stderr, "Format error in expire.ctl\n");
	exit(1);
    }
    (void)fclose(F);

    /* Set up the link, reserve the lock. */
    if (EXPreason == NULL) {
	(void)sprintf(buff, "Expiring process %ld", (long)getpid());
	EXPreason = COPY(buff);
    }
    if (Server) {
	/* If we fail, leave evidence behind. */
	if (ICCopen() < 0) {
	    (void)fprintf(stderr, "Can't open channel to server, %s\n",
		    strerror(errno));
	    CleanupAndExit(FALSE, FALSE, 1);
	}
	if (ICCreserve(EXPreason) != 0) {
	    (void)fprintf(stderr, "Can't reserve server\n");
	    CleanupAndExit(FALSE, FALSE, 1);
	}
    }

    /* Make the history filenames. */
    HistoryDB = COPY(HistoryText);
    (void)sprintf(buff, "%s.dir", HistoryDB);
    Historydir = COPY(buff);
    (void)sprintf(buff, "%s.pag", HistoryDB);
    Historypag = COPY(buff);
    if (HistoryPath)
	(void)sprintf(buff, "%s/%s.n", HistoryPath, History);
    else
	(void)sprintf(buff, "%s.n", HistoryText);
    NHistory = COPY(buff);
    (void)sprintf(buff, "%s.dir", NHistory);
    NHistorydir = COPY(buff);
    (void)sprintf(buff, "%s.pag", NHistory);
    NHistorypag = COPY(buff);

    if (!Writing)
	out = NULL;
    else {
	/* Open new history files, relative to news lib. */
	if (chdir(EXPhistdir) < 0) {
	    (void)fprintf(stderr, CANTCD, EXPhistdir, strerror(errno));
	    exit(1);
	}
	out = EXPfopen(TRUE, NHistory, "w");
	(void)fclose(EXPfopen(TRUE, NHistorydir, "w"));
	(void)fclose(EXPfopen(TRUE, NHistorypag, "w"));
	if (EXPverbose > 3)
	    (void)printf("created: %s %s %s\n",
		    NHistory, NHistorydir, NHistorypag);
	(void)dbzincore(1);
	if (IgnoreOld) {
	    if (dbzfresh(NHistory, dbzsize(0L), '\t', 'C', 0L) < 0) {
		(void)fprintf(stderr, "Can't create database, %s\n",
			strerror(errno));
		exit(1);
	    }
	}
	else if (dbzagain(NHistory, HistoryDB) < 0) {
	    (void)fprintf(stderr, "Can't dbzagain, %s\n", strerror(errno));
	    exit(1);
	}
    }

    if (chdir(SPOOL) < 0) {
	(void)fprintf(stderr, CANTCD, SPOOL, strerror(errno));
	exit(1);
    }

    if (Writing)
    {   /* Main processing loop. */
	arts = NEW(char*, nGroups);
	if ((qp = QIOopen(HistoryText, QIO_BUFFER)) == NULL) {
	    (void)fprintf(stderr, "Can't open history file, %s\n",
		    strerror(errno));
	    CleanupAndExit(Server, FALSE, 1);
	}
	for (Bad = FALSE, line = 1, Paused = FALSE; ; line++) {
	    if ((p = QIOread(qp)) != NULL) {
		if (!EXPdoline(out, p, QIOlength(qp), arts)) {
		    Bad = TRUE;
		    if (errno == ENOSPC) {
			(void)unlink(NHistory);
			(void)unlink(NHistorydir);
			(void)unlink(NHistorypag);
		    }
		    break;
		}
		continue;
	    }
    
	    /* Read or line-format error? */
	    if (QIOtoolong(qp)) {
		(void)fprintf(stderr, "Line %d too long\n", line);
		QIOclose(qp);
		CleanupAndExit(Server, Paused, 1);
	    }
	    if (QIOerror(qp)) {
		(void)fprintf(stderr, "Can't read line %d, %s\n",
			line, strerror(errno));
		QIOclose(qp);
		CleanupAndExit(Server, Paused, 1);
	    }
    
	    /* We hit EOF. */
	    if (Paused || !Server)
		/* Already paused or we don't want to pause -- we're done. */
		break;
	    if (ICCpause(EXPreason) != 0) {
		(void)fprintf(stderr, "Can't pause server, %s\n",
			strerror(errno));
		QIOclose(qp);
		CleanupAndExit(Server, Paused, 1);
	    }
	    Paused = TRUE;
	}
	QIOclose(qp);
	DISPOSE(arts);

	/* Close the output files. */
	if (ferror(out) || fflush(out) == EOF || fclose(out) == EOF) {
	    (void)fprintf(stderr, "Can't close %s, %s\n",
		NHistory, strerror(errno));
	    Bad = TRUE;
	}
	if (dbmclose() < 0) {
	    (void)fprintf(stderr, "Can't close history, %s\n",
		    strerror(errno));
	    Bad = TRUE;
	}

	if (UnlinkFile && EXPunlinkfile == NULL)
	    /* Got -z but file was closed; oops. */
	    Bad = TRUE;

	/* If we're done okay, and we're not tracing, slip in the new files. */
	if (EXPverbose) {
	    if (Bad)
		(void)printf("Expire errors: history files not updated.\n");
	    if (EXPtracing)
		(void)printf("Expire tracing: history files not updated.\n");
	}
	if (!Bad && !EXPtracing) {
	    if (chdir(EXPhistdir) < 0) {
		(void)fprintf(stderr, CANTCD, EXPhistdir, strerror(errno));
		CleanupAndExit(Server, Paused, 1);
	    }
	    /* If user used the -d flag, mark we're done and exit. */
	    if (HistoryPath != NULL) {
		(void)sprintf(buff, "%s.done", NHistory);
		(void)fclose(EXPfopen(FALSE, buff, "w"));
		CleanupAndExit(Server, FALSE, 0);
	    }

	    if (rename(NHistory, HistoryText) < 0
	     || rename(NHistorydir, Historydir) < 0
	     || rename(NHistorypag, Historypag) < 0) {
		(void)fprintf(stderr, "Can't replace history files, %s\n",
			strerror(errno));
		/* Yes -- leave the server paused. */
		CleanupAndExit(Server, FALSE, 1);
	    }
	}
    }

    CleanupAndExit(Server, Paused, Bad ? 1 : 0);
    /* NOTREACHED */
    return 0;
}
