/* osdep_linux.c: Key engine/PF_KEY socket OS dependencies for Linux

   Copyrights on this file are hairy.

   Many parts were derived from af_unix.c and other parts of the Linux
   kernel source tree. See the GPLv2 or that source file for license details 
   on those parts.

   Other parts are Copyright 1996, Craig Metz, All Rights Reserved. You
   may use those parts under the terms of the BSD license (just put my
   name in place of the Regents's) or the GPLv2.

   Several critical bug fixes as well as technical help were contributed
   by Alan Cox.

   This software comes with NO WARRANTY of ANY KIND, including any
   implied warranties of mechantability or fitness for a particular
   purpose. Use it entirely at your own risk.

   This is free software. If you have a problem with the licenses on it
   (for example, some people get rabid at the thought of mixing GPLv2 and
   BSD style licenses), please just relax and deal with it.
*/

#include "key.h"

#include <linux/module.h>
#include <net/sock.h>
#include <linux/random.h>
#include <linux/netdevice.h>
#include <linux/errno.h>

#define min(a,b)	(((a)<(b))?(a):(b))

static struct sockaddr key_addr = { pseudo_AF_KEY, };
static SOCKET *key_socket_list = NULL;

volatile SOCKET *key_last_sendup;

long random(void)
{
	long rval;
	get_random_bytes(&rval, sizeof(rval));
	return rval;
}

int key_sendup(SOCKET *sk, struct key_msghdr *km, int flags)
{
  struct sk_buff *skb;
  int error = 0, size = km->key_msglen;

  /*
   *	Allocate a buffer
   */
  if (!(skb = alloc_skb(size, GFP_ATOMIC))) {
    DPRINTF(IDL_ERROR, ("key_sendup: alloc_skb failed!\n"));
    return -1;
  }
  
  /*
   *	Not locked, push in the data.
   */
  	 
  skb->free = 1;
  skb->h.raw = skb_put(skb, size);
  memcpy(skb->h.raw, km, size);
  
  /*
   *	Queue to target socket.
   */
  
  if ((error = sock_queue_rcv_skb(sk, skb)) < 0) {
    DPRINTF(IDL_ERROR, ("key_sendup: sock_queue_rcv_skb failed (error = %d)!\n", error));
    skb->sk = NULL;
    kfree_skb(skb, FREE_READ);
    return -1;
  }
  DPRINTF(IDL_MAJOR_EVENT, ("key_sendup: returning successfully\n"));
  return 0;
}

int my_addr(SOCKADDR *sa)
{
  switch (sa->sa_family) {
  case AF_INET:
    return (ip_chk_addr(((struct sockaddr_in *)sa)->sin_addr.s_addr) == IS_MYADDR);
  default:
    return 0;
  }
}

/*
 *	Note: Sockets may not be removed _during_ an interrupt or net_bh
 *	handler using this technique. They can be added although we do not
 *	use this facility.
 */

static void key_remove_socket(SOCKET *sk)
{
  SOCKET **s;
	
  cli();
  s = &key_socket_list;
  
  while(*s) {
    if (*s == sk) {
      *s = sk->next;
      sti();
      return;
    }
    s = &((*s)->next);
  }
  sti();
}

static void key_insert_socket(SOCKET *sk)
{
  cli();
  sk->next = key_socket_list;
  key_socket_list = sk;
  sti();
}

#if 0
static int key_einval(void)
{
	DPRINTF(IDL_MAJOR_EVENT, ("Entering key_einval\n"));
	return -EINVAL;
}
#endif /* 0 */

static void key_statechange(struct sock *sk)
{
	DPRINTF(IDL_MAJOR_EVENT, ("Entering key_statechange\n"));
	if(!sk->dead)
		wake_up_interruptible(sk->sleep);
}

static void key_dataready(struct sock *sk, int len)
{
	DPRINTF(IDL_MAJOR_EVENT, ("Entering key_dataready(sk, len=%d)\n", len));
	if(!sk->dead) 
	{
		wake_up_interruptible(sk->sleep);
		sock_wake_async(sk->socket, 1);
	}
}

static void key_writespace(struct sock *sk)
{
	DPRINTF(IDL_MAJOR_EVENT, ("Entering key_writespace\n"));
	if(!sk->dead) 
	{
		wake_up_interruptible(sk->sleep);
		sock_wake_async(sk->socket, 2);
	}
}

static int key_create(struct socket *sock, int protocol)
{
  struct sock *sk;
  
  DPRINTF(IDL_MAJOR_EVENT, (">key_create(%08x, %x)\n", (unsigned int)sock, (unsigned int)protocol));

  if (!sock) {
    DPRINTF(IDL_ERROR, ("key_create: passed NULL sock?!\n"));
    return -EINVAL;
  }
  
  if (protocol)
    return -EPROTONOSUPPORT;

  if (!(sk = sk_alloc(GFP_KERNEL)))
    return -ENOMEM;

  if (sock->type != SOCK_RAW) {
    sk_free(sk);
    return -ESOCKTNOSUPPORT;
  }

  MOD_INC_USE_COUNT;

  memset(sk, 0, sizeof(*sk));
  
  sk->type = sock->type;
  skb_queue_head_init(&sk->receive_queue);
  skb_queue_head_init(&sk->back_log);

  sk->protocol = protocol;
  sk->rcvbuf=SK_RMEM_MAX;
  sk->sndbuf=SK_WMEM_MAX;
  sk->allocation=GFP_KERNEL;
  sk->state=TCP_CLOSE;
  sk->priority=SOPRI_NORMAL;
  sk->state_change = key_statechange;
  sk->data_ready = key_dataready;
  sk->write_space = key_writespace;
  sk->error_report = key_statechange;
  sk->mtu=4096;
  sk->socket=sock;
  sk->sleep=sock->wait;

  sock->data=(void *)sk;
  
  key_insert_socket(sk);
  return 0;
};

static int key_release(struct socket *sock, struct socket *peer)
{
  struct sock *sk = (struct sock *)sock->data;
  struct sk_buff *skb;

  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_release\n"));
  
  if (!sk)
    return 0;
  
  sk->state_change(sk);
  sk->dead = 1;
  /* If you may queue on an interrupt - cli around this loop */
  while ((skb = skb_dequeue(&sk->receive_queue)))
    kfree_skb(skb, FREE_WRITE);
  key_remove_socket(sk);
  sk_free(sk);
  MOD_DEC_USE_COUNT;
  return 0;
}

static int key_getname(struct socket *sock, struct sockaddr *uaddr, int 
*uaddr_len, int peer)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_getname\n"));
  memcpy(uaddr, &key_addr, *uaddr_len = sizeof(key_addr));
  return 0;
}

static int key_select(struct socket *sock, int sel_type, select_table *wait)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_select\n"));
  return datagram_select((struct sock *)sock->data, sel_type, wait);
}

static int key_shutdown(struct socket *sock, int mode)
{
  struct sock *sk=(struct sock *)sock->data;

  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_shutdown\n"));
  
  if (mode & SEND_SHUTDOWN) {
    sk->shutdown |= SEND_SHUTDOWN;
    sk->state_change(sk);
  }
  if (mode & RCV_SHUTDOWN) {
    sk->shutdown |= RCV_SHUTDOWN;
    sk->state_change(sk);
  }
  return 0;
}

/* We ignore msg->msg_name */
static int key_sendmsg(struct socket *socket, struct msghdr *msg, int len, int nonblock, int flags)
{
	struct sock *sk = socket->data;
	int error = EINVAL;
	struct key_msghdr *km;

	DPRINTF(IDL_MAJOR_EVENT, (">key_sendmsg()\n"));
  
	if (sk->err)
		return sock_error(sk);

	if (flags || msg->msg_control)
		return -EINVAL;

	if (len < sizeof(struct key_msghdr)) {
		DPRINTF(IDL_ERROR, ("key_sendmsg: len < sizeof(struct key_msghdr)\n"));
		goto flush;
        }
  
	KMALLOC(km, struct key_msghdr *, len);
        if (!km)
		return -ENOBUFS;

	memcpy_fromiovec((void *)km, msg->msg_iov, len);

	if (len != km->key_msglen) {
		DPRINTF(IDL_ERROR, ("key_sendmsg: len != km->key_msglen\n"));
		goto flush;
	}

	error = key_parse(&km, sk, NULL);
flush:
	km->key_errno = error;
	key_sendup(sk, km, 1);
  	KFREE(km);
	return len;
}

/* stolen from udp_recvmsg */
int key_recvmsg(struct socket *sock, struct msghdr *msg, int len,
		int noblock, int flags, int *addr_len)
{
  struct sock *sk = (struct sock *)sock->data;
  int copied;
  struct sk_buff *skb;
  int er;
  struct sockaddr *sa = (struct sockaddr *)msg->msg_name;
  
  DPRINTF(IDL_MAJOR_EVENT, (">key_recvmsg(struct sock *sk = %08x, struct msghdr *msg = %08x\n", (unsigned int)sk, (unsigned int)msg));
  DPRINTF(IDL_MAJOR_EVENT, (" int len = %x, int noblock = %x, int flags = %08x, int *addr_len = %08x\n", (unsigned int)len, (unsigned int)noblock, (unsigned int)flags, (unsigned int)addr_len));

  if (sk->err)
    return sock_error(sk);
  
  if (!(skb = skb_recv_datagram(sk, flags, noblock, &er)))
    return er;
  
  skb_copy_datagram_iovec(skb, 0, msg->msg_iov, copied = min(len, skb->len));
  
  sk->stamp = skb->stamp;
  
  if (sa) {
    memset(sa, 0, sizeof(*sa));
    sa->sa_family = pseudo_AF_KEY;
#ifdef SA_LEN
    sa->sa_len = sizeof(struct sockaddr);
#endif /* SA_LEN */
  }

  skb_free_datagram(sk, skb);
  return(copied);
}

static int key_dup(struct socket *newsock, struct socket *oldsock)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_dup\n"));
  return -EINVAL;
}

static int key_bind(struct socket *sock, struct sockaddr *umyaddr, int sockaddr_len)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_bind\n"));
  return -EINVAL;
}

static int key_connect(struct socket *sock, struct sockaddr *uservaddr, int sockaddr_len, int flags)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_connect\n"));
  return -EINVAL;
}

static int key_socketpair(struct socket *sock1, struct socket *sock2)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_socketpair\n"));
  return -EINVAL;
}

static int key_accept(struct socket *sock, struct socket *newsock, int flags)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_accept\n"));
  return -EINVAL;
}

static int key_ioctl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_ioctl\n"));
  return -EINVAL;
}

static int key_listen(struct socket *sock, int len)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_listen\n"));
  return -EINVAL;
}

static int key_getsockopt(struct socket *socket, int level, int optname, char *optval, int *optlen)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_getsockopt\n"));
	
  if (level != SOL_SOCKET)
    return -EOPNOTSUPP;
  
  return sock_getsockopt((struct sock *)socket->data, level,
			 optname, optval, optlen);
}

static int key_setsockopt(struct socket *socket, int level, int optname, char *optval, int optlen)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_setsockopt\n"));
  
  if (level != SOL_SOCKET)
    return -EOPNOTSUPP;
  return sock_setsockopt((struct sock *)socket->data, level,optname, optval, optlen);
}

static int key_fcntl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
  /* Probably ought to put the standard code for this in here */
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_fcntl\n"));
  return -EINVAL;
}

static struct proto_ops key_proto_ops = {
  PF_KEY,
  key_create,
  key_dup,      /* (void *)key_einval, */
  key_release,
  key_bind,     /* (void *)key_einval, */
  key_connect,  /* (void *)key_einval, */
  key_socketpair,/*(void *)key_einval, */
  key_accept,   /* (void *)key_einval, */
  key_getname,
  key_select,
  key_ioctl,    /* (void *)key_einval, */	
  key_listen,   /* (void *)key_einval, */
  key_shutdown,
  key_setsockopt, /* (void *)key_einval, */
  key_getsockopt, /* (void *)key_einval, */
  key_fcntl,    /* (void *)key_einval, */
  key_sendmsg,
  key_recvmsg
};

void key_init(void)
{
  printk("NRL key management engine, Linux version 0.01b\n");
  sock_register(key_proto_ops.family, &key_proto_ops);
  key_inittables();
};

#ifdef MODULE
int init_module(void)
{
  key_init();
  return 0;
};

void cleanup_module(void)
{
  key_cleanup();
  sock_unregister(key_proto_ops.family);
};
#else /* MODULE */
void key_proto_init(struct protocol *p)
{
  key_init();
};
#endif /* MODULE */
