/*
 * addr2ascii.c  --  Implements translation of addresses to human-readable
 *                   form.
 *
 * Copyright 1995 by Randall Atkinson, Bao Phan, and Dan McDonald
 *	All Rights Reserved.  
 *      All Rights under this copyright have been assigned to NRL.
 */

/*----------------------------------------------------------------------
#	@(#)COPYRIGHT	1.1a (NRL) 17 August 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

This software and documentation were developed at NRL by various
people.  Those developers have each copyrighted the portions that they
developed at NRL and have assigned All Rights for those portions to
NRL.  Outside the USA, NRL also has copyright on the software
developed at NRL. The affected files all contain specific copyright
notices and those notices must be retained in any derived work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:

	This product includes software developed at the Information
	Technology Division, US Naval Research Laboratory.

4. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/

#include <sys/types.h>

#include <netinet/in.h>
#ifdef INET6
#include <netinet6/in6.h>
#endif /* INET6 */
#ifndef linux
#include <netns/ns.h>
#include <netiso/iso.h>
#include <net/if_dl.h>
#endif /* linux */

#include <arpa/inet.h>
#include <sys/socket.h>
#include <stdio.h>
#include <string.h>

#include "support.h"

static char addr2ascii_buffer[ADDR2ASCII_BUFSZ_MAX+1];

#ifdef INET6
static char hextab[] = { '0', '1', '2', '3', '4', '5', '6', '7',
                         '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
/*
 * Convert network-format IN6/IPv6 address
 * to canonical hexadecimal user format of h:h:h:h
 ... perfect for unrolling ...
 */
static char *_inet6_ntoa(ap)
     void *ap;
{
  static char buffer[40];
  char *op = buffer;
  int i,doubled = 0;

  /*
   * Treat v4-mapped and v4-compatible differently.
   * (In fact v4-mapped, may want to just return the v4 address.)
   */
  if (IS_IN6_IPV4_MAPPED(*(struct in_addr6 *)ap))
    {
      strcpy(op,"::ffff:");
      op += 7;
      (void)addr2ascii(AF_INET, (ap + 12), 4, op);
      return buffer;
    }
  if (IS_IN6_IPV4_COMPAT(*(struct in_addr6 *)ap))
    {
      strcpy(op,"::");
      op += 2;
      (void)addr2ascii(AF_INET, (ap + 12), 4, op);
      return buffer;
    }

  *op = '\0';

#if 0
  for (i=0;i<8;i++)
    {
      if (doubled <= 0)
	{
	  if (sp[i] == 0)
	    {
	      if ((op > buffer + 1) && op[-2] != ':')
		*op++ = ':';
	      doubled = -1;
	      continue;
	    }
	  if (op == buffer && i > 1)
	    {
	      strcpy(op,"::");
	      op+=2;
	      doubled = -1;
	    }
	}
      if (doubled == -1 && sp[i] != 0)
	doubled = 1;
      {
        int ii, j, k, val = htons(sp[i]);

        for (ii = k = 0; ii < 4; ii++) {
          if (j = ((val >> 12) & 0x0f))
            k++;
	  if (k)
            *op++ = hextab[j]; 
          val = val << 4;
        }
      }
      if (i < 7)
	*op++ = ':';
    }
  if (op == buffer)
    strcpy(buffer,"::0");
  else if (op[-1] == ':')
    strcpy(op,"0");
  else *op = '\0';
#endif /* 0 */

  {
    char unpack[4];
    int i, j, k;
    int start = 0, len = 0;
    unsigned short *sp;

    for (i = j = k = 0, sp = (unsigned short *)ap; i < 8; i++) {
      if (*(sp++)) {
	if (j >= len) {
	  start = k;
	  len = j;
	  j = 0;
	}
      } else {
	if (!(j++))
	  k = i;
      }
    }
    if (j >= len) {
      start = k;
      len = j;
    }

#ifdef DEBUG
    fprintf(stderr, "start = %d, len = %d\n", start, len);
#endif /* DEBUG */

    if (len < 2)
      start = 9;

    if (!start)
      *(op++) = ':';

    i = 0;
    while(i < 8) {
      if (i == start) {
	*(op++) = ':';
	ap += len * 2;
	i += len;
	continue;
      }

      unpack[0] = ((*(char *)ap) & 0xf0) >> 4;
      unpack[1] =  (*(char *)ap) & 0x0f; ap++;
      unpack[2] = ((*(char *)ap) & 0xf0) >> 4;
      unpack[3] =  (*(char *)ap) & 0x0f; ap++;

      for (j = 0; (j < 3) && !unpack[j]; j++);
      while(j < 4)
	*(op++) = hextab[unpack[j++]];
      
      *(op++) = ':';
      i++;
    }
    if ((start + len) == 8) 
      *(op++) = '0';
    else
      --op;
    *op = 0;
  }

  return buffer;
}
#endif /* INET6 */

/* Standard API entry point. In the interest of moving other AFs to use this
   routine, try to include support for as much as possible by calling the
   existing conversion functions. */
char *addr2ascii(af, ap, len, cp)
int af;
/* const */ void *ap;
int len;
char *cp;
{
  char *ptr;
  int maxlen;

  ptr = NULL;

  switch(af) {

  case AF_INET:
    if (len != sizeof(struct in_addr))
      break;
    ptr = inet_ntoa(*((struct in_addr *)ap));
    maxlen = 16;
    break;

#ifdef INET6
  case AF_INET6:
    if (len != sizeof(struct in_addr6))
      break;
    ptr = _inet6_ntoa(ap);
    maxlen = 46;
    break;
#endif /* INET6 */

#ifndef linux
#ifdef AF_ISO
  case AF_ISO:
    ptr = iso_ntoa((struct iso_addr *)ap);
    maxlen = 42; /*?*/
    break;
#endif /* AF_ISO */

#ifdef AF_NS
  case AF_NS:
    ptr = ns_ntoa(*((struct ns_addr *)ap));
    maxlen = 42; /*?*/
    break;
#endif /* AF_NS */

#ifdef AF_LINK
  case AF_LINK:
    ptr = link_ntoa((struct sockaddr_dl *)ap);
    maxlen = 42; /*?*/
    break;
#endif /* AF_LINK */
#endif /* linux */

  }

  if (!cp)
    cp = addr2ascii_buffer;

  if (ptr) {
    strncpy(cp, ptr, maxlen);
    cp[maxlen] = 0;
    return cp;
  }

  return NULL;
}
