/*
 * ifconfig.c  --  Implement ifconfig(8) command.
 *
 * Copyright 1995 by Dan McDonald, Randall Atkinson, and Bao Phan
 *      All Rights Reserved.  
 *      All Rights under this copyright have been assigned to NRL.
 */

/*----------------------------------------------------------------------
#	@(#)COPYRIGHT	1.1 (NRL) 17 January 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

Portions of the software are derived from the Net/2 and 4.4 Berkeley
Software Distributions (BSD) of the University of California at
Berkeley and those portions are copyright by The Regents of the
University of California. All Rights Reserved.  The UC Berkeley
Copyright and License agreement is binding on those portions of the
software.  In all cases, the NRL developers have retained the original
UC Berkeley copyright and license notices in the respective files in
accordance with the UC Berkeley copyrights and license.

Portions of this software and documentation were developed at NRL by
various people.  Those developers have each copyrighted the portions
that they developed at NRL and have assigned All Rights for those
portions to NRL.  Outside the USA, NRL has copyright on some of the
software developed at NRL. The affected files all contain specific
copyright notices and those notices must be retained in any derived
work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. All terms of the UC Berkeley copyright and license must be followed.
2. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
3. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
4. All advertising materials mentioning features or use of this software
   must display the following acknowledgements:

	This product includes software developed by the University of
	California, Berkeley and its contributors.

	This product includes software developed at the Information
	Technology Division, US Naval Research Laboratory.

5. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/
/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1983, 1993\n\
	The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)ifconfig.c	8.2 (Berkeley) 2/16/94";
#endif /* not lint */

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define	NSIP
#include <netns/ns.h>
#include <netns/ns_if.h>
#include <netdb.h>

#define EON
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#include <sys/protosw.h>

#ifdef INET6
#include <netinet/if_ether.h>
#include <netinet6/in6.h>
#include <netinet6/in6_var.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#endif /* INET6 */

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

struct	ifreq		ifr, ridreq;
struct	ifaliasreq	addreq;
struct	iso_ifreq	iso_ridreq;
struct	iso_aliasreq	iso_addreq;
struct	sockaddr_in	netmask;
#ifdef INET6
struct inet6_ifreq inet6_ifreq,inet6_ridreq;
struct inet6_aliasreq inet6_addreq;
#endif

char	name[30];
int	flags;
int	metric;
int	nsellength = 1;
int	setaddr;
int	setipdst;
int	doalias;
int	clearaddr;
int	newaddr = 1;
int	s;
extern	int errno;

int	setifflags(), setifaddr(), setifdstaddr(), setifnetmask();
int	setifmetric(), setifbroadaddr(), setifipdst();
int	notealias(), setsnpaoffset(), setnsellength(), notrailers();
#ifdef INET6
int inet6_auto();
#endif

#define	NEXTARG		0xffffff

struct	cmd {
	char	*c_name;
	int	c_parameter;		/* NEXTARG means next argv */
	int	(*c_func)();
} cmds[] = {
	{ "up",		IFF_UP,		setifflags } ,
	{ "down",	-IFF_UP,	setifflags },
#ifdef INET6
	{ "-auto",	0,	inet6_auto },
	{ "auto",	1,	inet6_auto },
#endif
	{ "trailers",	-1,		notrailers },
	{ "-trailers",	1,		notrailers },
	{ "arp",	-IFF_NOARP,	setifflags },
	{ "-arp",	IFF_NOARP,	setifflags },
	{ "debug",	IFF_DEBUG,	setifflags },
	{ "-debug",	-IFF_DEBUG,	setifflags },
	{ "alias",	IFF_UP,		notealias },
	{ "-alias",	-IFF_UP,	notealias },
	{ "delete",	-IFF_UP,	notealias },
#ifdef notdef
#define	EN_SWABIPS	0x1000
	{ "swabips",	EN_SWABIPS,	setifflags },
	{ "-swabips",	-EN_SWABIPS,	setifflags },
#endif
	{ "netmask",	NEXTARG,	setifnetmask },
	{ "metric",	NEXTARG,	setifmetric },
	{ "broadcast",	NEXTARG,	setifbroadaddr },
	{ "ipdst",	NEXTARG,	setifipdst },
	{ "snpaoffset",	NEXTARG,	setsnpaoffset },
	{ "nsellength",	NEXTARG,	setnsellength },
	{ "link0",	IFF_LINK0,	setifflags } ,
	{ "-link0",	-IFF_LINK0,	setifflags } ,
	{ "link1",	IFF_LINK1,	setifflags } ,
	{ "-link1",	-IFF_LINK1,	setifflags } ,
	{ "link2",	IFF_LINK2,	setifflags } ,
	{ "-link2",	-IFF_LINK2,	setifflags } ,
	{ 0,		0,		setifaddr },
	{ 0,		0,		setifdstaddr },
};

/*
 * XNS support liberally adapted from code written at the University of
 * Maryland principally by James O'Toole and Chris Torek.
 */
int	in_status(), in_getaddr();
int	xns_status(), xns_getaddr();
int	iso_status(), iso_getaddr();
#ifdef INET6
int     inet6_status(), inet6_getaddr();
#endif

/* Known address families */
struct afswtch {
	char *af_name;
	short af_af;
	int (*af_status)();
	int (*af_getaddr)();
	int af_difaddr;
	int af_aifaddr;
	caddr_t af_ridreq;
	caddr_t af_addreq;
} afs[] = {
#define C(x) ((caddr_t) &x)
	{ "inet", AF_INET, in_status, in_getaddr,
	     SIOCDIFADDR, SIOCAIFADDR, C(ridreq), C(addreq) },
	{ "ns", AF_NS, xns_status, xns_getaddr,
	     SIOCDIFADDR, SIOCAIFADDR, C(ridreq), C(addreq) },
	{ "iso", AF_ISO, iso_status, iso_getaddr,
	     SIOCDIFADDR_ISO, SIOCAIFADDR_ISO, C(iso_ridreq), C(iso_addreq) },
#ifdef INET6
	{ "inet6", AF_INET6, inet6_status, inet6_getaddr,
     SIOCDIFADDR_INET6, SIOCAIFADDR_INET6, C(inet6_ridreq), C(inet6_addreq) },
#endif
	{ 0,	0,	    0,		0 }
};

struct afswtch *afp;	/*the address family being set or asked about*/

#ifdef INET6

/*----------------------------------------------------------------------
 * This function uses the new SIOCVIFADDR_INET6 ioctl() to verify a new
 * IPv6 address survived duplicate address detection.
 ----------------------------------------------------------------------*/

inet6_verify(ifreq)
     void *ifreq;    /* Should be inet6_aliasreq, which can pass off as
			and inet6_ifreq for SIOCVIFADDR_INET6. */
{
  int rc;
  struct sockaddr_in6 *sin6 = &(((struct inet6_ifreq *)ifreq)->ifr_addr);
  char buffer[60];

  addr2ascii(AF_INET6, &sin6->sin6_addr, sizeof(struct in_addr6),buffer);

  /*
   * Spin-wait doing calls to kernel to verify that new IPv6 address
   * passed duplicate address detection.
   */
  do
    sleep(1);
  while ((rc = ioctl(s,SIOCVIFADDR_INET6,ifreq)) == -1 && 
	 errno == EWOULDBLOCK);

  printf("Duplicate address detection ");

  /*
   * Depending on SIOCVIFADDR_INET6 results, print appropriate message.
   */
  if (rc == -1)
    if (errno == EADDRNOTAVAIL)
      printf("failed ");
    else perror("results unknown.\nStrange SIOCVIFADDR_INET6 error");
  else printf("succeeded ");

  printf("for %s.\n",buffer);
}
#endif /* INET6 */

main(argc, argv)
	int argc;
	char *argv[];
{
	int af = AF_INET;
	register struct afswtch *rafp;

	if (argc < 2) {
		fprintf(stderr, "usage: ifconfig interface\n%s%s%s%s%s",
		    "\t[ af [ address [ dest_addr ] ] [ up ] [ down ]",
			    "[ netmask mask ] ]\n",
		    "\t[ metric n ]\n",
		    "\t[ arp | -arp ]\n",
		    "\t[ link0 | -link0 ] [ link1 | -link1 ] [ link2 | -link2 ] \n");
		exit(1);
	}
	argc--, argv++;
	strncpy(name, *argv, sizeof(name));
	strncpy(ifr.ifr_name, name, sizeof(ifr.ifr_name));
	argc--, argv++;
	if (argc > 0) {
		for (afp = rafp = afs; rafp->af_name; rafp++)
			if (strcmp(rafp->af_name, *argv) == 0) {
				afp = rafp; argc--; argv++;
				break;
			}
		rafp = afp;
		af = ifr.ifr_addr.sa_family = rafp->af_af;
	}
	s = socket(af, SOCK_DGRAM, 0);
	if (s < 0) {
		perror("ifconfig: socket");
		exit(1);
	}
	if (ioctl(s, SIOCGIFFLAGS, (caddr_t)&ifr) < 0) {
		Perror("ioctl (SIOCGIFFLAGS)");
		exit(1);
	}
	strncpy(ifr.ifr_name, name, sizeof ifr.ifr_name);
	flags = ifr.ifr_flags;
	if (ioctl(s, SIOCGIFMETRIC, (caddr_t)&ifr) < 0)
		perror("ioctl (SIOCGIFMETRIC)");
	else
		metric = ifr.ifr_metric;
	if (argc == 0) {
		status();
		exit(0);
	}
	while (argc > 0) {
		register struct cmd *p;

		for (p = cmds; p->c_name; p++)
			if (strcmp(*argv, p->c_name) == 0)
				break;
		if (p->c_name == 0 && setaddr)
			p++;	/* got src, do dst */
		if (p->c_func) {
			if (p->c_parameter == NEXTARG) {
				if (argv[1] == NULL)
					errx(1, "'%s' requires argument",
					    p->c_name);
				(*p->c_func)(argv[1]);
				argc--, argv++;
			} else
				(*p->c_func)(*argv, p->c_parameter);
		}
		argc--, argv++;
	}

	if (af == AF_ISO)
		adjust_nsellength();
	if (setipdst && af==AF_NS) {
		struct nsip_req rq;
		int size = sizeof(rq);

		rq.rq_ns = addreq.ifra_addr;
		rq.rq_ip = addreq.ifra_dstaddr;

		if (setsockopt(s, 0, SO_NSIP_ROUTE, &rq, size) < 0)
			Perror("Encapsulation Routing");
	}
	if (clearaddr) {
		int ret;
		strncpy(rafp->af_ridreq, name, sizeof ifr.ifr_name);
		if ((ret = ioctl(s, rafp->af_difaddr, rafp->af_ridreq)) < 0) {
			if (errno == EADDRNOTAVAIL && (doalias >= 0)) {
				/* means no previous address for interface */
			} else
				Perror("ioctl (SIOCDIFADDR)");
		}
	}
	if (newaddr) {
		strncpy(rafp->af_addreq, name, sizeof ifr.ifr_name);
		if (ioctl(s, rafp->af_aifaddr, rafp->af_addreq) < 0)
			Perror("ioctl (SIOCAIFADDR)");
#ifdef INET6
		/*
		 * Verify new address is unique on the link.
		 */
		if (rafp->af_af == AF_INET6)
		  inet6_verify(rafp->af_addreq);
#endif /* INET6 */
	}
	exit(0);
}
#define RIDADDR 0
#define ADDR	1
#define MASK	2
#define DSTADDR	3

/*ARGSUSED*/
setifaddr(addr, param)
	char *addr;
	short param;
{
	/*
	 * Delay the ioctl to set the interface addr until flags are all set.
	 * The address interpretation may depend on the flags,
	 * and the flags may change when the address is set.
	 */
	setaddr++;
	if (doalias == 0
#ifdef INET6
	    && afp->af_af != AF_INET6   /* Always add INET6 addresses w/o
					   nuking others... */
#endif
	    )
		clearaddr = 1;

#ifdef INET6
	if (afp->af_af == AF_INET6)
	  {
	    printf("Bringing up IPv6 link-local address as well.\n");
	    inet6_auto(NULL,2);  /* Link-local must be configured for i/f. */
	  }
#endif

	(*afp->af_getaddr)(addr, (doalias >= 0 ? ADDR : RIDADDR));
}

setifnetmask(addr)
	char *addr;
{
	(*afp->af_getaddr)(addr, MASK);
}

setifbroadaddr(addr)
	char *addr;
{
	(*afp->af_getaddr)(addr, DSTADDR);
}

setifipdst(addr)
	char *addr;
{
	in_getaddr(addr, DSTADDR);
	setipdst++;
	clearaddr = 0;
	newaddr = 0;
}
#define rqtosa(x) (&(((struct ifreq *)(afp->x))->ifr_addr))
/*ARGSUSED*/
notealias(addr, param)
	char *addr;
{
	if (setaddr && doalias == 0 && param < 0)
		bcopy((caddr_t)rqtosa(af_addreq),
		      (caddr_t)rqtosa(af_ridreq),
		      rqtosa(af_addreq)->sa_len);
	doalias = param;
	if (param < 0) {
		clearaddr = 1;
		newaddr = 0;
	} else
		clearaddr = 0;
}

/*ARGSUSED*/
notrailers(vname, value)
	char *vname;
	int value;
{
	printf("Note: trailers are no longer sent, but always received\n");
}

/*ARGSUSED*/
setifdstaddr(addr, param)
	char *addr;
	int param;
{
	(*afp->af_getaddr)(addr, DSTADDR);
}

#ifdef INET6
/*----------------------------------------------------------------------
 * Perform IPv6 link-local address configuration, based on sockaddr_dl
 * found for the requested interface.
 *
 * When address collision detection happens, this would be the user-space
 * place to put it, if we do collision detection in user space.
 ----------------------------------------------------------------------*/

inet6_auto(ignoreme,doit)
     char *ignoreme;
     int doit;     /*
		    * 0 -> Remove link-local
		    * 1 -> Add link-local & exit program
		    * 2 -> Add link-local & return for further processing
		    */
{
#define BUFLEN 1024
  char buffer[BUFLEN];  /* Storage for... */
  char *data;
  struct ifconf ifc;    /* stuff this variable points to! */
  struct ifreq *ifr = (struct ifreq *)buffer;
  struct sockaddr *sa = NULL;
  struct sockaddr_in6 *sin6,*local = NULL;
  struct sockaddr_dl *sdl = NULL;
  off_t ifnetaddr;
  struct ifnet ifnet;
  int unit,namelen=0;

  while (!isdigit(name[namelen]) && name[namelen] !='\0')
    namelen++;

  sscanf((char *)((u_long)name+(u_long)namelen),"%d",&unit);
  if (afp->af_af != AF_INET6)
    {     
      printf("Sorry, the [-]auto option is only usable with \"inet6\".\n");
      exit(1);
    } 
    
  if (doit)
    {
      printf("Configuring IPv6 local-use address.\n");
      ifc.ifc_len = BUFLEN;
      ifc.ifc_buf = (caddr_t) buffer;

      if (ioctl(s,SIOCGIFCONF,&ifc) < 0)
        {
          perror("SIOCGIFCONF");
          exit(1);
        }

      data = (char *)ifr;
      while ((data - buffer) < ifc.ifc_len )
	{
	  sa = &(ifr->ifr_addr);
	  if (strncmp(name,ifr->ifr_name,sizeof(name)) == 0)
	    {
	      if (sa->sa_family == AF_LINK)
		if (sdl)
		  {
		    printf("More than one AF_LINK on the same if !\n");
		    exit(1);
		  }
		else sdl = (struct sockaddr_dl *)sa;
	      if (sa->sa_family == AF_INET6)
		{
		  sin6 = (struct sockaddr_in6 *)sa;
		  if (flags & IFF_LOOPBACK)
		    if (IS_IN6_LOOPBACK(sin6->sin6_addr))
		      {
			printf("Loopback already configured.\n");
			if (doit == 1)
			  exit(0);
			else return;
		      }
		    else
		      {
			printf("Non-loopback on a loopback?\n");
			exit(1);
		      }
		  else if (IS_IN6_LINKLOC(sin6->sin6_addr))
		    if (local)
		      {
			printf("Two IPv6 link-local addrs on 1 interface.\n");
			exit(1);
		      }
		    else local = sin6;
		}
	    }

	  data += (sa->sa_len <= sizeof(struct sockaddr))?
	        sizeof(struct ifreq):
	        (sizeof(struct ifreq) - sizeof(struct sockaddr) + sa->sa_len);
	  ifr = (struct ifreq *)data;
	}

      if (flags & IFF_LOOPBACK)
	{
	  /*
	   * Configure ::1 for loopback interface.
	   */
	  IN6_ADDR_ASSIGN(inet6_addreq.ifra_addr.sin6_addr,0,0,0,htonl(1));
	}
      else
	{
	  /*
	   * Verify sdl, and see that we don't have one already.
	   */
	  if (sdl == NULL)
	    {
	      printf("No sdl for interface %s.\n",name);
	      exit(1);
	    }
	  if (local)
	    {
	      data = (char *)&local->sin6_addr;
	      data +=  (sizeof(struct in_addr6) - sdl->sdl_alen); /* Get past
								     prefix. */
	      if (memcmp(data,(void *)LLADDR(sdl),sdl->sdl_alen) == 0)
		{
		  printf("IPv6 link-local address already configured.\n");
		  if (doit == 1)
		    exit(0);  /* Already configured. */
		  else return;
		}
	      printf("Already have link-local that isn't the same.\n");
	      exit(1);
	    }
	  /* Tsk tsk tsk, Kebe says that using HARD WIRED numbers is bad. */
	  IN6_ADDR_ASSIGN(inet6_addreq.ifra_addr.sin6_addr,htonl(0xFE800000),\
			  0,0,0);
	  data = (char *)&inet6_addreq.ifra_addr.sin6_addr;
	  data += (sizeof(struct in_addr6) - sdl->sdl_alen);
	  memcpy(data,(void *)LLADDR(sdl),sdl->sdl_alen);
	}

      inet6_addreq.ifra_addr.sin6_len = sizeof(struct sockaddr_in6);
      inet6_addreq.ifra_addr.sin6_family = AF_INET6;
      strncpy(inet6_addreq.ifra_name,name,IFNAMSIZ);
      inet6_addreq.ifra_mask.sin6_len = sizeof(struct sockaddr_in6);
      inet6_addreq.ifra_mask.sin6_family = AF_INET6;
      IN6_ADDR_ASSIGN(inet6_addreq.ifra_mask.sin6_addr,0xFFFFFFFF,0xFFFFFFFF,\
		      0xFFFFFFFF,0xFFFFFFFF);

      /*
       * Verification of address w/solicits & whatnot, will take place
       * in the kernel.  It's up to me to verify the address.
       */
      if (ioctl(s, SIOCAIFADDR_INET6, &inet6_addreq) < 0)  
	{
	  perror("ioctl (SIOCAIFADDR_INET6)");
	  exit (1);
	}

      /*
       * Do busy-wait loop for duplicate address detection.
       */
      inet6_verify(&inet6_addreq);
    }
  else
    {
      printf("Removal of auto-configured addresses not yet implemented.\n");
      exit(1);
    }

  if (doit == 1)
    exit (0);
}
#endif

setifflags(vname, value)
	char *vname;
	short value;
{
 	if (ioctl(s, SIOCGIFFLAGS, (caddr_t)&ifr) < 0) {
 		Perror("ioctl (SIOCGIFFLAGS)");
 		exit(1);
 	}
	strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
 	flags = ifr.ifr_flags;

	if (value < 0) {
		value = -value;
		flags &= ~value;
	} else
		flags |= value;
	ifr.ifr_flags = flags;
	if (ioctl(s, SIOCSIFFLAGS, (caddr_t)&ifr) < 0)
		Perror(vname);
}

setifmetric(val)
	char *val;
{
	strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
	ifr.ifr_metric = atoi(val);
	if (ioctl(s, SIOCSIFMETRIC, (caddr_t)&ifr) < 0)
		perror("ioctl (set metric)");
}

setsnpaoffset(val)
	char *val;
{
	iso_addreq.ifra_snpaoffset = atoi(val);
}

#define	IFFBITS \
"\020\1UP\2BROADCAST\3DEBUG\4LOOPBACK\5POINTOPOINT\6NOTRAILERS\7RUNNING\10NOARP\
\11PROMISC\12ALLMULTI\13OACTIVE\14SIMPLEX\15LINK0\16LINK1\17LINK2\20MULTICAST"

/*
 * Print the status of the interface.  If an address family was
 * specified, show it and it only; otherwise, show them all.
 */
status()
{
	register struct afswtch *p = afp;
	short af = ifr.ifr_addr.sa_family;

	printf("%s: ", name);
	printb("flags", flags, IFFBITS);
	if (metric)
		printf(" metric %d", metric);
	putchar('\n');
	if ((p = afp) != NULL) {
		(*p->af_status)(1);
	} else for (p = afs; p->af_name; p++) {
		ifr.ifr_addr.sa_family = p->af_af;
		(*p->af_status)(0);
	}
}

in_status(force)
	int force;
{
	struct sockaddr_in *sin;
	char *inet_ntoa();

	strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
	if (ioctl(s, SIOCGIFADDR, (caddr_t)&ifr) < 0) {
		if (errno == EADDRNOTAVAIL || errno == EAFNOSUPPORT) {
			if (!force)
				return;
			bzero((char *)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
		} else
			perror("ioctl (SIOCGIFADDR)");
	}
	sin = (struct sockaddr_in *)&ifr.ifr_addr;
	printf("\tinet %s ", inet_ntoa(sin->sin_addr));
	strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
	if (ioctl(s, SIOCGIFNETMASK, (caddr_t)&ifr) < 0) {
		if (errno != EADDRNOTAVAIL)
			perror("ioctl (SIOCGIFNETMASK)");
		bzero((char *)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
	} else
		netmask.sin_addr =
		    ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr;
	if (flags & IFF_POINTOPOINT) {
		if (ioctl(s, SIOCGIFDSTADDR, (caddr_t)&ifr) < 0) {
			if (errno == EADDRNOTAVAIL)
			    bzero((char *)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
			else
			    perror("ioctl (SIOCGIFDSTADDR)");
		}
		strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
		sin = (struct sockaddr_in *)&ifr.ifr_dstaddr;
		printf("--> %s ", inet_ntoa(sin->sin_addr));
	}
	printf("netmask 0x%x ", ntohl(netmask.sin_addr.s_addr));
	if (flags & IFF_BROADCAST) {
		if (ioctl(s, SIOCGIFBRDADDR, (caddr_t)&ifr) < 0) {
			if (errno == EADDRNOTAVAIL)
			    bzero((char *)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
			else
			    perror("ioctl (SIOCGIFADDR)");
		}
		strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
		sin = (struct sockaddr_in *)&ifr.ifr_addr;
		if (sin->sin_addr.s_addr != 0)
			printf("broadcast %s", inet_ntoa(sin->sin_addr));
	}
	putchar('\n');
}


xns_status(force)
	int force;
{
	struct sockaddr_ns *sns;

	close(s);
	s = socket(AF_NS, SOCK_DGRAM, 0);
	if (s < 0) {
		if (errno == EPROTONOSUPPORT)
			return;
		perror("ifconfig: socket");
		exit(1);
	}
	if (ioctl(s, SIOCGIFADDR, (caddr_t)&ifr) < 0) {
		if (errno == EADDRNOTAVAIL || errno == EAFNOSUPPORT) {
			if (!force)
				return;
			bzero((char *)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
		} else
			perror("ioctl (SIOCGIFADDR)");
	}
	strncpy(ifr.ifr_name, name, sizeof ifr.ifr_name);
	sns = (struct sockaddr_ns *)&ifr.ifr_addr;
	printf("\tns %s ", ns_ntoa(sns->sns_addr));
	if (flags & IFF_POINTOPOINT) { /* by W. Nesheim@Cornell */
		if (ioctl(s, SIOCGIFDSTADDR, (caddr_t)&ifr) < 0) {
			if (errno == EADDRNOTAVAIL)
			    bzero((char *)&ifr.ifr_addr, sizeof(ifr.ifr_addr));
			else
			    Perror("ioctl (SIOCGIFDSTADDR)");
		}
		strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
		sns = (struct sockaddr_ns *)&ifr.ifr_dstaddr;
		printf("--> %s ", ns_ntoa(sns->sns_addr));
	}
	putchar('\n');
}

iso_status(force)
	int force;
{
	struct sockaddr_iso *siso;
	struct iso_ifreq ifr;

	close(s);
	s = socket(AF_ISO, SOCK_DGRAM, 0);
	if (s < 0) {
		if (errno == EPROTONOSUPPORT)
			return;
		perror("ifconfig: socket");
		exit(1);
	}
	bzero((caddr_t)&ifr, sizeof(ifr));
	strncpy(ifr.ifr_name, name, sizeof(ifr.ifr_name));
	if (ioctl(s, SIOCGIFADDR_ISO, (caddr_t)&ifr) < 0) {
		if (errno == EADDRNOTAVAIL || errno == EAFNOSUPPORT) {
			if (!force)
				return;
			bzero((char *)&ifr.ifr_Addr, sizeof(ifr.ifr_Addr));
		} else {
			perror("ioctl (SIOCGIFADDR_ISO)");
			exit(1);
		}
	}
	strncpy(ifr.ifr_name, name, sizeof ifr.ifr_name);
	siso = &ifr.ifr_Addr;
	printf("\tiso %s ", iso_ntoa(&siso->siso_addr));
	if (ioctl(s, SIOCGIFNETMASK_ISO, (caddr_t)&ifr) < 0) {
		if (errno != EADDRNOTAVAIL)
			perror("ioctl (SIOCGIFNETMASK_ISO)");
	} else {
		printf(" netmask %s ", iso_ntoa(&siso->siso_addr));
	}
	if (flags & IFF_POINTOPOINT) {
		if (ioctl(s, SIOCGIFDSTADDR_ISO, (caddr_t)&ifr) < 0) {
			if (errno == EADDRNOTAVAIL)
			    bzero((char *)&ifr.ifr_Addr, sizeof(ifr.ifr_Addr));
			else
			    Perror("ioctl (SIOCGIFDSTADDR_ISO)");
		}
		strncpy(ifr.ifr_name, name, sizeof (ifr.ifr_name));
		siso = &ifr.ifr_Addr;
		printf("--> %s ", iso_ntoa(&siso->siso_addr));
	}
	putchar('\n');
}

Perror(cmd)
	char *cmd;
{
	extern int errno;

	switch (errno) {

	case ENXIO:
		errx(1, "%s: no such interface", cmd);
		break;

	case EPERM:
		errx(1, "%s: permission denied", cmd);
		break;

	default:
		err(1, "%s", cmd);
	}
}

#ifdef INET6

/*
 * Kebe asks:  How do I print ALL of the addresses associated with the
 * interface?
 */

inet6_status(force)
     int force;
{
  char outbuf[50];

  close(s);

	s = socket(AF_INET6, SOCK_DGRAM, 0);
	if (s < 0) {
		if (errno == EPROTONOSUPPORT)
			return;
		perror("ifconfig: socket");
		exit(1);
	}
  bzero(&inet6_ifreq,sizeof(inet6_ifreq));
  strncpy(inet6_ifreq.ifr_name, name,sizeof(inet6_ifreq.ifr_name));
  if (ioctl(s, SIOCGIFADDR_INET6, (caddr_t)&inet6_ifreq) < 0)
    {
      if (errno == EADDRNOTAVAIL || errno == EAFNOSUPPORT)
	{
	  if (!force)
	    return;
	  bzero((char *)&inet6_ifreq.ifr_addr,sizeof(struct sockaddr_in6));
	}
      else perror("ioctl (SIOCGIFADDR_INET6)");
    }
  if (!(addr2ascii(AF_INET6, &inet6_ifreq.ifr_addr.sin6_addr, 
	     sizeof(inet6_ifreq.ifr_addr.sin6_addr),outbuf))) {
    fprintf(stderr, "addr2ascii failed\n");
    exit(1);
  }

  printf("\tinet6 address %s\n",outbuf);

  strncpy(inet6_ifreq.ifr_name, name,sizeof(inet6_ifreq.ifr_name));

  /* do PPP */
  if (flags & IFF_POINTOPOINT)
    {
      if (ioctl(s, SIOCGIFDSTADDR_INET6, (caddr_t)&inet6_ifreq) <0)
	if (errno == EADDRNOTAVAIL)
	  bzero((char *)&inet6_ifreq.ifr_addr,sizeof(struct sockaddr_in6));
	else perror("ioctl (SIOCGIFDSTADDR_INET6)");
      if (!(addr2ascii(AF_INET6, &inet6_ifreq.ifr_addr.sin6_addr, 
		 sizeof(inet6_ifreq.ifr_addr.sin6_addr),outbuf))) {
        fprintf(stderr, "addr2ascii failed\n");
        exit(1);
      }
      printf("\t ---> %s\n", outbuf);
    }

  strncpy(inet6_ifreq.ifr_name, name,sizeof(inet6_ifreq.ifr_name));
  if (ioctl(s, SIOCGIFNETMASK_INET6, (caddr_t)&inet6_ifreq) <0)
    {
      if (errno != EADDRNOTAVAIL)
	perror("ioctl (SIOCGIFNETMASK_INET6)");
      bzero((char *)&inet6_ifreq.ifr_addr,sizeof(struct sockaddr_in6));
    }
  if (!(addr2ascii(AF_INET6, &inet6_ifreq.ifr_addr.sin6_addr, 
	     sizeof(inet6_ifreq.ifr_addr.sin6_addr),outbuf))) {
    fprintf(stderr, "addr2ascii failed\n");
    exit(1);
  }
  printf("\tinet6 netmask %s\n",outbuf);
}
#endif /* INET6 */

struct	in_addr inet_makeaddr();

#define SIN(x) ((struct sockaddr_in *) &(x))
struct sockaddr_in *sintab[] = {
SIN(ridreq.ifr_addr), SIN(addreq.ifra_addr),
SIN(addreq.ifra_mask), SIN(addreq.ifra_broadaddr)};

in_getaddr(s, which)
	char *s;
{
	register struct sockaddr_in *sin = sintab[which];
	struct hostent *hp;
	struct netent *np;
	int val;

	sin->sin_len = sizeof(*sin);
	if (which != MASK)
		sin->sin_family = AF_INET;

	if ((val = inet_addr(s)) != -1)
		sin->sin_addr.s_addr = val;
	else if (hp = gethostbyname(s))
		bcopy(hp->h_addr, (char *)&sin->sin_addr, hp->h_length);
	else if (np = getnetbyname(s))
		sin->sin_addr = inet_makeaddr(np->n_net, INADDR_ANY);
	else
		errx(1, "%s: bad value", s);
}

/*
 * Print a value a la the %b format of the kernel's printf
 */
printb(s, v, bits)
	char *s;
	register char *bits;
	register unsigned short v;
{
	register int i, any = 0;
	register char c;

	if (bits && *bits == 8)
		printf("%s=%o", s, v);
	else
		printf("%s=%x", s, v);
	bits++;
	if (bits) {
		putchar('<');
		while (i = *bits++) {
			if (v & (1 << (i-1))) {
				if (any)
					putchar(',');
				any = 1;
				for (; (c = *bits) > 32; bits++)
					putchar(c);
			} else
				for (; *bits > 32; bits++)
					;
		}
		putchar('>');
	}
}

#define SNS(x) ((struct sockaddr_ns *) &(x))
struct sockaddr_ns *snstab[] = {
SNS(ridreq.ifr_addr), SNS(addreq.ifra_addr),
SNS(addreq.ifra_mask), SNS(addreq.ifra_broadaddr)};

xns_getaddr(addr, which)
char *addr;
{
	struct sockaddr_ns *sns = snstab[which];
	struct ns_addr ns_addr();

	sns->sns_family = AF_NS;
	sns->sns_len = sizeof(*sns);
	sns->sns_addr = ns_addr(addr);
	if (which == MASK)
		printf("Attempt to set XNS netmask will be ineffectual\n");
}

#define SISO(x) ((struct sockaddr_iso *) &(x))
struct sockaddr_iso *sisotab[] = {
SISO(iso_ridreq.ifr_Addr), SISO(iso_addreq.ifra_addr),
SISO(iso_addreq.ifra_mask), SISO(iso_addreq.ifra_dstaddr)};

iso_getaddr(addr, which)
char *addr;
{
	register struct sockaddr_iso *siso = sisotab[which];
	struct iso_addr *iso_addr();
	siso->siso_addr = *iso_addr(addr);

	if (which == MASK) {
		siso->siso_len = TSEL(siso) - (caddr_t)(siso);
		siso->siso_nlen = 0;
	} else {
		siso->siso_len = sizeof(*siso);
		siso->siso_family = AF_ISO;
	}
}

setnsellength(val)
	char *val;
{
	nsellength = atoi(val);
	if (nsellength < 0)
		errx(1, "Negative NSEL length is absurd");
	if (afp == 0 || afp->af_af != AF_ISO)
		errx(1, "Setting NSEL length valid only for iso");
}

fixnsel(s)
register struct sockaddr_iso *s;
{
	if (s->siso_family == 0)
		return;
	s->siso_tlen = nsellength;
}

adjust_nsellength()
{
	fixnsel(sisotab[RIDADDR]);
	fixnsel(sisotab[ADDR]);
	fixnsel(sisotab[DSTADDR]);
}

#ifdef INET6

#define SIN6(x) ((struct sockaddr_in6 *) &(x))
struct sockaddr_in6 *sin6tab[] = {
SIN6(inet6_ridreq.ifr_addr), SIN6(inet6_addreq.ifra_addr),
SIN6(inet6_addreq.ifra_mask), SIN6(inet6_addreq.ifra_broadaddr)};



inet6_getaddr(addr, which)
     char *addr;
{
  register struct sockaddr_in6 *sin6 = sin6tab[which];

  sin6->sin6_len = sizeof(*sin6);
  if (which != MASK)
    sin6->sin6_family = AF_INET6;

  if (ascii2addr(AF_INET6,addr,&sin6->sin6_addr)<0)
    {
      errx(1,"%s: bad value", addr);
    }
  /*
   * Put in hostname support?
   */
}
#endif /* INET6 */
