#       addkey - add, delete keys & get values with known keys
#
# Copyright (c) 1993, 1994, 1995, 1996, 1997  The TERENA Association
# Copyright (c) 1998                              RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF    
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
#
# $Id: addkey.pl,v 2.5 1999/01/11 09:08:27 roman Exp $
#
#	$RCSfile: addkey.pl,v $
#	$Revision: 2.5 $
#	$Author: roman $
#	$Date: 1999/01/11 09:08:27 $

require "dpr.pl";

################################################################
##  sub getvalues
################################################################
##
##  FUNCTIONAL DESCRIPTION:
##      returns hash value for given key taking into account 
##      the possible overflow (additional level of indirection)
##  FORMAL PARAMETERS:
##      *db: in
##           hash to look at (normally mapped  
##           to database index file via DB_File)  
##      $key: in
##           key to look up in the *db hash
##  RETURN VALUE:
##      $db{$key} - string containing list of comma separated 
##          offsets to database data file
##  DESIGN:
##      if hash value for given key contains predefined pattern
##      the rest of the value is treated as file extension of a
##      file containing the real value to be returned
##   
################################################################

sub getvalues {
    local(*db, $key) = @_;
    local($value)=$db{$key};
    
    if ($value =~ s/^$OVERFLOWPREFIXREGULAR//o) {
	my($overflowname) = $db[1] . $OVERFLOWEXTENSION . $value;
	if (open(OVERFLOWGET, "<".$overflowname)) {
	    $value=<OVERFLOWGET>;
	    close(OVERFLOWGET);
	}
	else {
	    &fatalerror("in getvalues: cannot open for getting values" .
			"$overflowname key: $key value: " .
			"$offsets code: $!");
	}
    }
    return $value;
}

################################################################
##  sub addkey
################################################################
##
##  FUNCTIONAL DESCRIPTION:
##      adds a list of offsets to a value of hash for given key 
##      taking into account possible overflow
##  FORMAL PARAMETERS:
##      *db: in/out
##          hash to be modified (normally mapped to database 
##          index file via DB_File)
##      $key: in
##          key to change offsets at
##      $offsets: in
##          list of offsets to be added to hash value at a 
##          given key
##  RETURN VALUE:
##      1 if key didn't already existed, 0 otherwise
##  SIDE EFFECTS:
##      operations on overflow file if necessary
##  DESIGN:
##      $db{$OVERFLOWKEY} contains current maximum overflow
##      file number.
##   
################################################################

sub addkey {
    local(*db, $key, $offsets)=@_;
    local($value)=$db{$key};
    local($length)=length($offsets)+length($key)+length($value)+4;
    my($overflowname);
    my($newval) = $offsets;
    $newval = $value . "\," . $offsets if ($value); 

#    &dpr("\$db = $db, \$key = $key, \$offsets = $offsets\n");
#    &dpr("\$db{\$key} = $db{$key}\n");
#    &dpr("\$length = $length\n");
#    &dpr("\$newval = $newval\n");
    
    return 1 if ((!$value) && ($length<$OVERFLOWSIZE) && ($db{$key}=$newval));
    
    if ($value=~ s/^$OVERFLOWPREFIXREGULAR//o) {
	
	$overflowname = $db[1] . $OVERFLOWEXTENSION . $value;
#	&dpr("\$overflowname = $overflowname\n");
	if (open(OVERFLOWADD, ">>".$overflowname)) {
	    print OVERFLOWADD "\,", $offsets or 
		&fatalerror("in addkey: cannot write $overflowname".
			    " key: $key value: $offsets code: $!");
	    close(OVERFLOWADD);
	    return 0;
	}
	else {
	    &fatalerror("in addkey: cannot open for addition " . 
			"$overflowname key: $key value: " .
			"$offsets code: $!");
	}
    }

    return 0 if (($length<$OVERFLOWSIZE) && ($db{$key}=$newval));
    
    local($filenumber) = ++$db{$OVERFLOWKEY};
    $overflowname = $db[1].$OVERFLOWEXTENSION.$filenumber;

    if (open(OVERFLOWADD, ">".$overflowname)) {
       print OVERFLOWADD $newval or
	   &fatalerror("in addkey: cannot write $overflowname " .
		       "key: $key value: $offsets code: $!");
       close(OVERFLOWADD);
       $db{$key}=$OVERFLOWPREFIX.$filenumber;
       return ($value) ? 0 : 1;
    }
    else {
       &fatalerror("in addkey: cannot open new $overflowname code: $!");
    }
    &fatalerror("in addkey: could not add $key with value $offsets");   
}


################################################################
##  sub delkey
################################################################
##
##  FUNCTIONAL DESCRIPTION:
##      deletes a list of offsets from a value of hash for a given
##      key taking into account that overflow file may not be 
##      needed anymmore
##  FORMAL PARAMETERS:
##      *db: in/out
##          hash to be modified (normally mapped to database
##          index file via DB_File)
##      $key: in
##          key to change offsets at
##      $offsets: in
##          list of offsets to be deleted from a hash value
##          at a given key
##  RETURN VALUE:
##      none
##  SIDE EFFECTS:
##      operations on overflow file if necessary
##  DESIGN:
##      {@description or none@}
##   
################################################################

sub delkey {
    local(*db, $key, $offsets)=@_;
    local($value)=$db{$key};
    local(@values);
    local(%mark)=();

    grep($mark{$_}++, split(/\,/, $offsets));

    if ($value=~ s/^$OVERFLOWPREFIXREGULAR//o) {
	my($overflowname) = $db[1] . $OVERFLOWEXTENSION . $value;
	if (open(OVERFLOWDEL, "+<".$overflowname)) {
	    my($start) = time();
	    if ((@values=grep(!$mark{$_}, split(/\,/, <OVERFLOWDEL>)))) {
		
		seek(OVERFLOWDEL, 0, 0);
		truncate(OVERFLOWDEL, 0);
		print OVERFLOWDEL join("\,", @values) or
		    &fatalerror("in addkey: cannot write $overflowname code: $!");
		close(OVERFLOWDEL);
	    }
	    else {
		delete $db{$key};
		close(OVERFLOWDEL);
		unlink($overflowname);
		my($elapsed) = time() - $start;
		my($msg) = "Overflow file $overflowname processed for $elapsed seconds";
		&syslog("AUDITLOG", $msg) if ($elapsed > 5);

	    }
	    return;
	}
	else {
	    &fatalerror("in delkey: cannot open for deletion $overflowname " .
			"key: $key value: $offsets code: $!");
	}
    }   
    if ((@values=grep(!$mark{$_}, split(/\,/, $value)))) {
	$db{$key}=join("\,", @values);
    }
    else {
	delete($db{$key});
    }
    return;
}

1;
