#ifdef	IPSO
static char rcsid[] = "$Header: /localsw/UNICOS/ssh-1.2.22.sdsc/RCS/ipopts_ok.c,v 1.3 1998-05-14 09:13:20-06 rrb LANL rrb $";
/*
 *  $Source: /localsw/UNICOS/ssh-1.2.22.sdsc/RCS/ipopts_ok.c,v $
 *  $Revision: 1.3 $
 *  $Date: 1998-05-14 09:13:20-06 $
 *
 *  Purpose:
 *	Scan the provided IP option byte-array and determine if the options
 *	are acceptable to ssh.  This routine is typically called to scan
 *	the options provided by the getsockopt call, for example:
 *
 *	    len = sizeof(op);
 *	    getsockopt (socket, IPPROTO_IP, IP_OPTIONS, (char *)op, &len);
 *	    if ( ipopts_ok (op, len)) { ...
 *
 *  Preconditions:	None.
 *  Postconditions:
 *	Return true if the options are acceptable, false if any provided
 *	option is unacceptable.
 *
 *  Invariants:
 *
 *  Author:	Randy Bremmer, March 1998
 *		Los Alamos National Laboratory
 *  Modification $Author: rrb $
 *
 *  Maintenance and modification 
 *	$Log: ipopts_ok.c,v $
 *	Revision 1.3  1998-05-14 09:13:20-06  rrb
 *	Fix definition of FLAGREG.
 *	Make argument "op" a "const".
 *
 *	Revision 1.2  1998-05-13 14:44:47-06  rrb
 *	Define TRUE, FALSE, and FLAGREG.
 *	Use these definitions.
 *	Make the whole module conditional on IPSO being defined.
 *
 *
 *
 *-------------------------*/

#include "includes.h"
#include <sys/param.h>

#define TRUE 1
#define FALSE 0

typedef int logical;
#define FLAGREG register logical

/*--------------------   Global declarations ------------------------*/
/*---------------- Within-module Global declarations ----------------*/

/*
 *	IP options lookup table.  Table entries include a text description
 *	and a flag which indicates whether ssh will allow the option.
 *	The table is in most-likely-to-occur order for a MLS UNICOS system.
 */
#define TBLEND -2

static struct {

    int		option;		/* option value */
    char*	label;		/* text description of the option */
    logical	allow;		/* Flag: !=0 indicates the option is allowed */

} ipopt_tbl[] = {
			{IPOPT_EOL, "EOL", TRUE},
#ifdef    IPOPT_CIPSO
			{IPOPT_CIPSO, "CIPSO", TRUE},
#endif
			{IPOPT_NOP, "no-op", TRUE},
#ifdef    IPOPT_BSO
			{IPOPT_BSO, "BSO", TRUE},
#endif
			{IPOPT_RR, "record route", FALSE},
			{IPOPT_TS, "timestamps", FALSE},
			{IPOPT_LSRR, "loose source route", FALSE},
			{IPOPT_SSRR, "strict source route", FALSE},
			{IPOPT_SATID, "satnet id", FALSE},
			{TBLEND, "unknown", FALSE},
};

/*--------------------- External declarations -----------------------*/
/*--------------------- Function declarations -----------------------*/

/*-----------------------------*/

int
ipopts_ok (op, len)

    const unsigned char* op;	/* pointer to byte-array containing IP options */
    int	len;			/* length in bytes of the op array. */
{
    int		ti;		/* table index */
    int		ilen;		/* individual option length */
    FLAGREG	ok = TRUE;	/* all is acceptable flag */
    int		olen = len;	/* total length, in bytes, of the opt array */
    int		opt;		/* option code.  See RFC 791. */
/*
 * Process options to determine if any are not allowed.
 */
    debug("Check IP options, total option len = %d", olen);
    while (olen > 0) {
        opt = (int)(*op);	/* Extract option code */
	for ( ti = 0; 		/* Find option code in option table */
	      (opt != ipopt_tbl[ti].option) && (ipopt_tbl[ti].option != TBLEND); 
	      ti++ );

	if ( opt != IPOPT_EOL ) debug(
		"IP option code : copy %d class %d number %d (%s)", 
		opt>>7, (opt>>5) & 3, opt & 0x37, ipopt_tbl[ti].label);

	ok = ok && ipopt_tbl[ti].allow;
	if ( ! ipopt_tbl[ti].allow ) {
	    ok = FALSE;
	    log_msg ("Detected disallowed IP option %s\n", ipopt_tbl[ti].label);
	}
/*
 *		EOL and NOP are only 1 byte in length and thus
 *		do not have a length byte, in which case the length
 *		is provided; otherwise, use the length byte (see RFC 791).
 */
        if ( opt == IPOPT_EOL || opt == IPOPT_NOP ) ilen = 1;
        else {
	    ilen = (int)op[IPOPT_OLEN];
	    if ( ilen < 2 ) {
                error("%s(%d) Invalid ip option \"%s\" length %d", 
			__FILE__, __LINE__, ipopt_tbl[ti].label, ilen);
		ok = FALSE;
                break;
	    }
	}
        if ( opt != IPOPT_EOL ) debug("%s option len=%d", 
					ipopt_tbl[ti].label, ilen);
/*
 *	    If individual options are to be interpreted, uncomment
 *	    the following line to prevent improper reference:
 */
/*	if (ilen > olen) ilen = olen;	*/
/*
 *		Move past the current option to the next option.
 */
        olen -= ilen;
        op += ilen;
    }
    return (ok);
}
#endif	IPSO
