/*
 *
 * fem.h:   Basic data structures for FEM-data retrieval
 *
 */

#ifndef FEM_H
#define FEM_H

#ifndef CRAY
typedef double REAL;
#define PRECISION   2
#else
typedef float REAL;
#define PRECISION   1
#endif

typedef enum { FALSE = 0, TRUE = 1 } bool;

static const int OK    =  1;
static const int ERROR = -1;
static const int NODATA= -2;

/*
 *
 * Nodal coordinate data structure
 *
 */

typedef struct {

  int  nid;            /* Node identifier/number */
  REAL x,y,z;          /* Cartesian coordinates  */

} Node;


/*
 *
 * Element topology data structure
 *
 */


typedef unsigned char Descr; /* General purpose 8-bit descriptor */

enum { 
  GEOM_ILLEGAL  =0,
  GEOM_POINT    =1,
  GEOM_LINE     =2,
  GEOM_TRIANGLE =3,
  GEOM_QUAD     =4,
  GEOM_TETRA    =5,
  GEOM_PYRAMID  =6,
  GEOM_WEDGE    =7,
  GEOM_HEXA     =8
  };

enum {
  ORDER_ILLEGAL    =0,
  ORDER_LINEAR     =1,
  ORDER_PARABOLIC  =2,
  ORDER_CUBIC      =3
  };

enum {
  STR_ILLEGAL   =0x00,    /* 0000 */
  STR_SOLID     =0x01,    /* 0001 */
  STR_SHELL     =0x02,    /* 0010 */
  STR_BEAM      =0x04     /* 0100 */
  };
  
enum {
  STR_NORMAL    =0x00,    /* 0000 */
  STR_VON_MISES =0x01,    /* 0001 */
  STR_MAX_SHEAR =0x02     /* 0010 */
  };

enum {
  STR_BOTTOM    =    1,
  STR_TOP       =32000
  };

typedef struct {

  int eid;            /* Element identifier/number            */
  int numnodes;       /* Number of nodes forming this element */
  int *nodelist;      /* List of nodes forming this element   */
  char *name;         /* Pointer to element name              */
  Descr geom;         /* Geometry descriptor                  */
  Descr order;        /* Order of element shape function      */
  Descr str_flag;     /* Stress recovery flag                 */

} Element;


/*
 *
 * Model data structure
 *
 */


typedef struct {

  Node *node;         /* List of nodal coordinates & nid's    */
  int numnodes;       /* Length of node-array                 */
  int maxnodeid;      /* Maximum node number in this Model    */

  Element *elem;      /* List of element topology data        */
  int numelems;       /* Length of element-array              */
  int maxelemid;      /* Maximum element number in this Model */
  
} Model;


/*
 *
 * Time increment data structure
 *
 */

typedef struct {

  int *step;          /* Time step tags                       */
  int *inc;           /* Time increment tags                  */
  REAL *time;         /* Time values of each increment        */
  int numincs;        /* Length of step/inc/time -arrays      */
  
} STime;


/*
 *
 * Nodal data structure
 *
 */

typedef struct {

  int   step;           /* Time step tag                                 */
  int   inc;            /* Time increment tag                            */
  REAL  time;           /* Time value                                    */
  int   *nid;           /* Node id:s for data                            */
  int   numnodes;       /* Length of nid-array; redundant, but clarifies */
  int   ndatacomp;      /* Number of data components per node            */
  REAL   **data;        /* Data array i.e. data for nodeid == nid[index] 
			   is stored as follows:

			   data[index][0] == 1st position
			   data[index][1] == 2nd position
			     :
			   data[index][ndatacomp-1] == last position    
			                                                 */
} NodalData;


#endif
