/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMSEQ.C - SEQUENCE functions
 *
 * $Revision:   1.4  $ $Date:   08 Jul 1992 17:37:10  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMSEQ.C_V  $
 * 
 *    Rev 1.4   08 Jul 1992 17:37:10   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.3   08 Jul 1992 16:54:44   gfoster
 * Fixed use of the /E and /W options.
 * 
 *    Rev 1.2   29 Jun 1992 21:04:14   gfoster
 * Fixed check for enumerated textual convention
 * syntax type.
 * 
 *    Rev 1.1   19 Jun 1992 16:29:02   gfoster
 * Copyright text was reformated.
 * 
 * Added check so that mismatch is not reported when
 * item in sequence has bad syntax.
 * 
 * Fixed error message for item not a child of row.
 * 
 *    Rev 1.0   27 May 1992 16:06:24   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <string.h>
#include <ctype.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/** addSEQname - add sequence
*
* call with:
*   pNa - new Sequence
*   pMod - containing module
*
* returns:
*   ptr to SEQor NULL if error
*/
    MIBSYM *
#ifdef __STDC__
addSEQname(STRTAB *pNa, MIBSYM *pMod)
#else
addSEQname(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pSeq;

    /* check if item already defined */
    pSym = pNa->pSym;
    if ((pSym != NULL) && (pSym->pMod == pMod)) {
        /* item already defined */
        /* check if forward reference */
        pSeq = pSym;
        if (pSym->usType != (MIBSYMFR|MIBSYMseq)) {
            yyerror("\"%s\" already defined in current module",
                    pSym->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }
    } else {
        /* item not defined */
        /* allocate new symbol */
        pSeq = newSEQ(pNa, pMod);
    }

    /* mark as SEQ, forward reference, and being defined */
    pSeq->usType |= MIBSYMseq | MIBSYMFR | MIBSYMDF;

    return(pSeq);

} /* addSEQname */


/** finishSEQ - finish definition of SEQUENCE
*
* NOTE: the checking of the sequence elements must be done
*       later after all the objects are defined
*
* call with:
*   pSeq - sequence
*
* returns:
*   ptr to sequence or NULL for error
*/
    MIBSYM *
#ifdef __STDC__
finishSEQ(MIBSYM *pSeq)
#else
finishSEQ(pSeq)
    MIBSYM *pSeq;
#endif /* __STDC__ */
{

    pSeq->usType = MIBSYMseq;

    return(pSeq);

} /* finishSEQ */


#define MIBSCKok 1              /* syntax matches */
#define MIBSCKcn 2              /* syntax is conflicting */
#define MIBSCKis 3              /* incompatible size */
#define MIBSCKir 4              /* incompatible range */
#define MIBSCKgr 5              /* gratuitous range */
#define MIBSCKgs 6              /* gratuitous size */
#define MIBSCKec 7              /* TC for enumerated with integer specified */
#define MIBSCKtc 8              /* seq item has base type, but obj has TC */


/** checkCompatiblity - check that syntax for sequence item
*                       matches syntax of object
*
* call with:
*   pI - item
*
* returns:
*   check result (SEE MIBSCKxxx)
*/
    USHORT
#ifdef __STDC__
checkCompatibility(MIBSEQI *pI)
#else
checkCompatibility(pI)
    MIBSEQI *pI;
#endif /* __STDC__ */
{
    MIBSYM *pOid;
    USHORT usRt;
    MIBSYN *pS;


    pOid = pI->pOid;

    /* check for bad syntax */
    if ((pOid->ut.oid.syn.usSyntax == MIBSYNbad) ||
            (pI->syn.usSyntax == MIBSYNbad))
        return(MIBSCKok);

    /* check for matching syntax */
    if (pOid->ut.oid.syn.usSyntax != pI->syn.usSyntax) {
        /* check for enumerated */
        if (pOid->ut.oid.rsyn.usSyntax == MIBSYNenum) {
            /* enumerated */
            if ((pI->syn.usSyntax == MIBSYNint) &&
                    (pI->syn.usSizeRange == MIBSRno)) {
                if (pOid->ut.oid.syn.usSyntax == MIBSYNenum)
                    /* syntax matches */
                    return(MIBSCKok);
                else;
                    /* TC for enum with integer specified */
                    return(MIBSCKec);
            } else {
                /* an error - conflicting syntax */
                return(MIBSCKcn);
            }
        }

        /* check for resolved match */
        if (pOid->ut.oid.rsyn.usSyntax != pI->syn.usSyntax) {
            /* an error - conflicting syntax */
            return(MIBSCKcn);
        } else {
            /* "resolved" syntax matches, so far */
            usRt = MIBSCKtc;
            pS = &(pOid->ut.oid.rsyn);
        }
    } else {
        usRt = MIBSCKok;
        pS = &(pOid->ut.oid.syn);
    }

    /* check for TC's */
    if (pS->usSyntax == MIBSYNtc) {
        /* textual convention */
        if (pS->usi.pTC != pI->syn.usi.pTC) {
            /* an error - conflicting syntax */
            return(MIBSCKcn);
        }
    }

    /* check if size/range specified for item */
    if (pI->syn.usSizeRange == MIBSRno) {
        /* everything OK */
        return(usRt);
    }

    /* seq item has size/range - check if compatible with object */
    if (pS->usSizeRange == pI->syn.usSizeRange) {
        switch (pS->usSizeRange) {
        case MIBSRfs:       /* fixed size */
        case MIBSRvs:       /* variable size */
            if ((pS->usr.usSize[0] == pI->syn.usr.usSize[0]) &&
                    (pS->usr.usSize[1] == pI->syn.usr.usSize[1])) {
                /* gratuitous size */
                if (!fNoSeqTcErr || fCheckSEQ)
                    return(MIBSCKgs);
                else
                    return(usRt);
            }
            break;
                
        case MIBSRpp:       /* positive to positive range */
        case MIBSRnp:       /* negative to positive range */
        case MIBSRnn:       /* negative to negative range */
            if ((pS->usr.ulRange[0] == pI->syn.usr.ulRange[0]) &&
                    (pS->usr.ulRange[1] == pI->syn.usr.ulRange[1])) {
                /* gratuitous range */
                if (!fNoSeqTcErr || fCheckSEQ)
                    return(MIBSCKgr);
                else
                    return(usRt);
            }
            break;
        }
    }

    /* return error */
    if ((pI->syn.usSizeRange == MIBSRfs) ||
            (pI->syn.usSizeRange == MIBSRvs)) {
        /* error - incompatible size */
        return(MIBSCKis);
    } else {
        /* error - incompatible range */
        return(MIBSCKir);
    }

} /* checkCompatibility */


/** checkSEQs - that that all SEQs for a module are defined
*               and that they have been referenced
*               Also check items in sequence to make sure
*               that they are defined and syntax matches.
*               And if they are not childern of the row
*               that is associated with the sequence, then that
*               the row that they are associated with has
*               the same indexs.
*
* call with:
*   pMod - module to check
*/
    VOID
#ifdef __STDC__
checkSEQs(MIBSYM *pMod)
#else
checkSEQs(pMod)
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSEQI *pI;
    MIBSYM *pOid;
    MIBSYM *pRow;
    MIBSYM *pPar;
    MIBINDX *pX1;
    MIBINDX *pX2;


    for (pT = pSeqGHL; pT != NULL; pT = pT->pNext) {
        if (pT->pMod != pMod)
            continue;

        if (!isupper(*(pT->pszName))) {
            yyerror("Name of sequence \"%s\" must start with uppercase letter",
                    pT->pszName);
        }

        if (pT->usType & MIBSYMFR) {
            yyerror("Sequence \"%s\" referenced but not defined",
                    pT->pszName);
        } else if (pT->ut.seq.cUse == 0) {
            yyerror("Sequence \"%s\" defined but not referenced",
                    pT->pszName);
        } else if (pT->ut.seq.pRow == NULL) {
            yyerror("Sequence \"%s\" not used to define a row",
                    pT->pszName);
        }
        pRow = pT->ut.seq.pRow;

        /* check that all items in sequence are objects and
            have syntax matching the object definitions */
        for (pI = pT->ut.seq.pSeqIL; pI != NULL; pI = pI->pNext) {
            pOid = pI->pOid;
            if (pOid->usType & MIBSYMFR) {
                /* object not defined, skip check */
                continue;
            }
            /* check if item an object */
            /* but first, skip objects with problems */
            if (pOid->ut.oid.usOType == MIBOTunk) {
                /* unknown or bad type -- so skip */
                continue;
            }

            if (pOid->ut.oid.usOType != MIBOTobj) {
                yyerror("Item \"%s\" in sequence \"%s\" is not an object",
                        pOid->pszName, pT->pszName);
                continue;
            }

            /* check syntax */
            switch(checkCompatibility(pI)) {
            case MIBSCKok:
                /* syntax matches */
                break;

            case MIBSCKcn:
                /* syntax is conflicting */
                yyerror("Item \"%s\" in sequence \"%s\" has conflicting syntax specified",
                        pOid->pszName, pT->pszName);
                break;

            case MIBSCKis:
                /* incompatible size */
                yyerror("Item \"%s\" in sequence \"%s\" has incompatible size specified",
                        pOid->pszName, pT->pszName);
                break;

            case MIBSCKir:
                /* incompatible range */
                yyerror("Item \"%s\" in sequence \"%s\" has incompatible range specified",
                        pOid->pszName, pT->pszName);
                break;

            case MIBSCKgr:
                /* gratuitous range */
                if (fCheckSEQ)
                    yywarning("Item \"%s\" in sequence \"%s\" has gratuitous range specified",
                        pOid->pszName, pT->pszName);
                break;

            case MIBSCKgs:
                /* gratuitous size */
                if (fCheckSEQ)
                    yywarning("Item \"%s\" in sequence \"%s\" has gratuitous size specified",
                        pOid->pszName, pT->pszName);
                break;

            case MIBSCKec:
                /* TC for enumerated with integer specified */
                if (!fNoSeqTcErr)
                    yyerror("Item \"%s\" in sequence \"%s\" has conflicting syntax specified",
                        pOid->pszName, pT->pszName);
                break;

            case MIBSCKtc:
                /* seq item has base type, but obj has TC */
                if (!fNoSeqTcErr)
                    yyerror("Item \"%s\" in sequence \"%s\" has conflicting syntax specified",
                            pOid->pszName, pT->pszName);
                break;

            default:
                /* an error */
                yyerror("checkSEQs: bad value from checkCompatibility");
            }

            /* check that item is in row */
            if (pRow == NULL)
                continue;
            
            if ((pPar = pOid->ut.oid.pPar) == NULL) {
                /* item has no parent */
                continue;
            } else if (pPar == pRow) {
                /* item in row using the sequence - nothing to do */
                continue;
            } else if (pPar->ut.oid.usOType != MIBOTrow) {
                /* item not in a row of a table */
                yywarning("Item \"%s\" in sequence \"%s\" is not a column for a row of a table",
                        pOid->pszName, pT->pszName);
                if (pOid->ut.oid.pDefSeq == NULL) {
                    /* item not in another sequence */
                    /* link to this sequence */
                    pOid->ut.oid.pDefSeq = pT;
                } else {
                    /* item in another sequence */
                    /* check that indexs in other sequence are
                        the same as this sequence */
                    if ((pOid->ut.oid.pDefSeq)->ut.seq.pRow == NULL)
                        /* sequence not attached to a row */
                        continue;
                    for (pX1 = pRow->ut.oid.pIndxL,
                            pX2 = ((pOid->ut.oid.pDefSeq)->ut.seq.pRow)->
                                    ut.oid.pIndxL;
                            (pX1 != NULL) && (pX2 != NULL);
                            pX1 = pX1->pNext, pX2 = pX2->pNext) {
                        if (pX1->usItype != pX2->usItype)
                            /* index not the same */
                            break;
                        if ((pX1->usItype == MIBITobj) ||
                                (pX1->usItype == MIBITnlobj)) {
                            /* object or no length object */
                            if (pX1->pOid != pX2->pOid)
                                /* index not the same */
                                break;
                        } else if (pX1->usItype == MIBITint) {
                            /* INTEGER */
                            if ((pX1->ulLow != pX2->ulLow) ||
                                    (pX1->ulHigh != pX2->ulHigh))
                                /* index not the same */
                                break;
                        } else if (pX1->usItype == MIBITfloct) {
                            /* fixed length OCTET STRING */
                            /* (size is in ulLow) */
                            if (pX1->ulLow != pX2->ulLow)
                                /* index not the same */
                                break;
                        }

                    }
                    if ((pX1 != NULL) || (pX2 != NULL)) {
                        /* indexes do not match */
                        yyerror("Sequence \"%s\" for row \"%s\" contains item \"%s\" used in sequence \"%s\" where indexes don't match",
                                pT->pszName, pRow->pszName,
                                pOid->pszName,
                                (pOid->ut.oid.pDefSeq)->pszName);
                    }
                }
            } else {
                /* item in row of another table */
                /* check if indexs equal */
                for (pX1 = pRow->ut.oid.pIndxL, pX2 = pPar->ut.oid.pIndxL;
                        (pX1 != NULL) && (pX2 != NULL);
                        pX1 = pX1->pNext, pX2 = pX2->pNext) {
                    if (pX1->usItype != pX2->usItype)
                        /* index not the same */
                        break;
                    if ((pX1->usItype == MIBITobj) ||
                            (pX1->usItype == MIBITnlobj)) {
                        /* object or no length object */
                        if (pX1->pOid != pX2->pOid)
                            /* index not the same */
                            break;
                    } else if (pX1->usItype == MIBITint) {
                        /* INTEGER */
                        if ((pX1->ulLow != pX2->ulLow) ||
                                (pX1->ulHigh != pX2->ulHigh))
                            /* index not the same */
                            break;
                    } else if (pX1->usItype == MIBITfloct) {
                        /* fixed length OCTET STRING */
                        /* (size is in ulLow) */
                        if (pX1->ulLow != pX2->ulLow)
                            /* index not the same */
                            break;
                    }
                }
                if ((pX1 != NULL) || (pX2 != NULL)) {
                    /* indexes do not match */
                    yyerror("Sequence \"%s\" for row \"%s\" contains item \"%s\" defined in row \"%s\" where indexes don't match",
                            pT->pszName, pRow->pszName,
                            pOid->pszName, pPar->pszName);
                }
            }
        }
    }

} /* checkSEQs */


/* end of file: SMSEQ.C */
