/* $Header: /home/vikas/netmgt/nocol/src/trapmon/RCS/trapmon.c,v 2.5 1994/05/16 02:16:34 vikas Exp $ */

/*
** DESCRIPTION
**	Listens for SNMP traps asynchronously and outputs information about
**	received traps in several possible formats.
**
**	For output to netmon, maintains a datafile with one EVENT per trap.
**	Appends new EVENTs to the end of the file.  Deletes EVENTs that are
**	more than 30 minutes old (by having two pointers running through the
**	file).
**
** Also see the DISCLAIMER file for important information.
*/

/*
** AUTHOR
**	David Wagner, wagner@jvnc.net, June 9 1992
**	I changed the snmptrapd program from Carnegie Mellon University
**	See their disclaimer attached below
*/

/*
 *
 * $Log: trapmon.c,v $
 * Revision 2.5  1994/05/16  02:16:34  vikas
 * Major rewrite. Now uses nocol_startup(). Consistent error messages.
 *
 * Revision 2.4  1994/01/10  20:58:15  aggarwal
 * Added VARUNITS and typecast in the gethostbyaddr.
 *
 * Revision 2.3  1993/11/04  07:39:22  aggarwal
 * Now shows type only for enterprise specific and also dumps
 * out the contents of the trap packets.
 *
 * Revision 2.2  1993/10/30  03:31:13  aggarwal
 * Tried to add the pdu->specific_type to the trapname.
 *
 * Revision 2.1  1993/09/21  19:50:27  aggarwal
 * Added NOCOL logging.
 *
 * Revision 2.0  1992/06/18  21:24:17  aggarwal
 * Had to send a pointer to the EVENT structure
 *
 * Revision 1.9  1992/06/12  00:51:12  aggarwal
 * Added enterprise specific trap identifier in the trap description.
 *
 * Revision 1.8  1992/06/11  07:21:14  aggarwal
 * Added 'usage' function.
 *
 * Revision 1.7  1992/06/11  06:44:51  aggarwal
 * Modified for the new t_desc structure.
 *
 * Revision 1.6  1992/06/10  21:27:47  wagner
 * Fixed the address and name fields of the event structure so that they
 * behave as expected now.
 *
 * Revision 1.1  1992/06/10  14:23:16  wagner
 * Initial revision
 *
 */

/*
 * snmptrapd.c - receive and log snmp traps
 *
 */
/***********************************************************
	Copyright 1989 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

/*  */

#ifndef lint
 static char rcsid[] = "$RCSfile: trapmon.c,v $ $Revision: 2.5 $ $Date: 1994/05/16 02:16:34 $" ;
#endif

#include "trapmon.h"

char		*prognm;			/* Program name */
static char	*datafile, *configfile;		/* nocol specific */

char *
uptime_string(timeticks, buf)
    register u_long timeticks;
    char *buf;
{
    int	seconds, minutes, hours, days;

    timeticks /= 100;
    days = timeticks / (60 * 60 * 24);
    timeticks %= (60 * 60 * 24);

    hours = timeticks / (60 * 60);
    timeticks %= (60 * 60);

    minutes = timeticks / 60;
    seconds = timeticks % 60;

    if (days == 0)
	sprintf(buf, "%d:%02d:%02d", hours, minutes, seconds);
    else if (days == 1)
	sprintf(buf, "%d day, %d:%02d:%02d", days, hours, minutes, seconds);
    else
	sprintf(buf, "%d days, %d:%02d:%02d", days, hours, minutes, seconds);

    return buf;
}

/*
 * Appends a new event to the end of the datafile. For an enterprise specific
 * trap, it adds the 'specific_type' of the trap for identification.
 */
add_event(addr, name, pdu)
    char *addr;					/* Address of trap sender */
    char *name;					/* Name of trap sender */
    struct snmp_pdu *pdu;			/* trap PDU */
{
    int fd;
    int trap = pdu->trap_type;			/* Trap number */
    struct tm *tm;
    time_t clock;
    EVENT v;

    bzero((char *)&v, sizeof(v));

    /*
     * In the following 'strncpy's, the null character at the end is
     * auto appended since to the bzero of the event struct and also
     * because we are doing a copy for 1 less than the size of the storage
     */
    strncpy(v.sender, prognm, sizeof(v.sender) - 1);	/* Set sender */
    strncpy(v.site.addr, addr, sizeof(v.site.addr) -1);	/* From address */
    strncpy(v.site.name, name, sizeof(v.site.name) -1);

    strncpy(v.var.name, trap_desc[trap].tname, sizeof(v.var.name) -1);
    if (trap == SNMP_TRAP_ENTERPRISESPECIFIC)	/* specific type */
    { 
	static char spdutype[16] ;		/* convert int to string */
	sprintf (spdutype, "/%d\0", pdu->specific_type) ;
	if ( (sizeof(v.var.name) - strlen(v.var.name)) > 5)
	  strncpy (v.var.name + strlen(v.var.name), spdutype, 4) ;
	else
	  strncpy (v.var.name + sizeof(v.var.name) - 5, spdutype, 4) ;
    }

    v.var.value = 0;				/* No value field */
    v.var.threshold = 0;			/* No threshold field */
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);

    v.severity = trap_desc[trap].tseverity;	/* Set the severity level */
    v.nocop = SETF_UPDOUN (v.nocop, trap_desc[trap].nocop);
    v.loglevel = trap_desc[trap].loglevel ;	/* logging level */

    clock = time(NULL);				/* Get the time of day... */
    tm = localtime(&clock);
    v.mon = tm->tm_mon + 1;			/* Don't forget to add 1! */
    v.day = tm->tm_mday;			/* Straightforward... */
    v.hour = tm->tm_hour;
    v.min = tm->tm_min;

    eventlog(&v);				/* Log the event */

    fd = open(datafile, O_WRONLY | O_APPEND);	/* Add the new EVENT */
    if (write(fd, &v, sizeof(v)) != sizeof(v)) {
	fprintf(stderr, "write() failed on %s\n", datafile);
	perror("addevent()");

	closeeventlog();
	exit(1);
    }
    close(fd);

    /* Store the time after which the trap event is to be deleted */
    die_at[numtraps++] = clock + TIME_TO_LIVE;	/* When to be reaped... */

    if (debug)
	printf("Added: sendr=%s sitenm=%s siteaddr=%s varnm=%s severity=%d mon=%d day=%d hr=%d min=%d die_at=%d\n",
	    v.sender, name, addr, v.var.name, v.severity, v.mon, v.day,
	    v.hour, v.min, die_at[numtraps - 1]);
}


int process_trap(op, session, reqid, pdu, magic)
    int op;
    struct snmp_session *session;
    int reqid;
    struct snmp_pdu *pdu;
    void *magic;
{
    char varbuf[512] ;				/* For variable_list */
    char buf[64];				/* For uptime_string() */
    struct hostent *he;				/* For gethostbyaddr() */
    char *dotted;				/* For inet_ntoa() */
    char *nameptr;				/* Name of trap sender */

    varbuf[0] = '\0';
    if (debug)
	printf("Entered process_trap...\n");

    if (op == RECEIVED_MESSAGE && pdu->command == TRP_REQ_MSG){
	he = gethostbyaddr((char *)&(pdu->agent_addr.sin_addr),
			   sizeof(unsigned long), AF_INET);
	if (he == NULL)
	    nameptr = "Unknown";		/* Error getting host name */
	else
	    nameptr = he->h_name;		/* Name of trap sender */
	dotted = inet_ntoa(pdu->agent_addr.sin_addr);/* Addr of sender */

	if (options & O_PRINT)			/* Log to stderr */
	    fprintf(stderr, "%s [%s]: %s Trap (%d) Uptime: %s %s\n",
		dotted,
		nameptr,
		trap_desc[pdu->trap_type].tname,
		pdu->specific_type,
		uptime_string(pdu->time, buf),
		trap_desc[pdu->trap_type].tseverity_str);

	if (debug)
	{
	    struct variable_list *vars;
	    char * pc = varbuf ;

            for(vars = pdu->variables; vars ; vars = vars->next_variable)
	      switch (vars->type)
	      {
		  int c ;
	       case ASN_INTEGER:
		  sprintf (pc, "%d \0", *(vars->val.integer)) ;
		  pc += strlen(pc) ;
		  break ;
               case ASN_OCTET_STR:
		  for (c = 0 ; c < vars->val_len ; ++c)
		    *pc++ = *(vars->val.string + c) ;
		  *pc++ = ' ';
		  break ;
	       default:
		  sprintf (pc, "(Unprintable type %d) \0", vars->type) ;
		  pc += strlen(pc) ;
		  break ;
	      }
	    sprintf (pc, "\n\0") ;
	    printf ("%s", varbuf);
	}

	if (options & O_LOG)			/* Log to syslog() */
	    syslog(LOG_WARNING, "%s [%s]: %s Trap (%d) Uptime: %s %s (%s)\n",
		dotted,
		nameptr,
		trap_desc[pdu->trap_type].tname,
		pdu->specific_type,
		uptime_string(pdu->time, buf),
		trap_desc[pdu->trap_type].tseverity_str,
	        varbuf);

	if (options & O_NOCOL)			/* Log EVENTs */
	    add_event(dotted,
	    nameptr,
	    pdu);
    } else if (op == TIMED_OUT) {		/* Not a trap! */
	fprintf(stderr, "Timeout: This shouldn't happen!\n");
	if (options & O_LOG)
	    syslog(LOG_WARNING, "Timeout: This shouldn't happen!\n");
    } /* end of if */

    if (debug) {
	printf("Done with process_trap.\n");
	fflush(stdout);
    }
}


/*+
** Strategy: I open the datafile twice, once read-only, and once write-only.
** Now I have two (independent) seek pointers for the single file.
** They then roam down the file, with the read pointer ahead of the
** write pointer.  Normally, the cycle is this:
**	(1) Read an EVENT with the read-only pointer, and advance it
**	(2) Write that EVENT with the write-only pointer, and advance it
** If, when reading an EVENT, we discover it needs to be reaped, then step (2)
** is not performed, i.e. only the read-pointer advances (and the write-pointer
** remains at the same location).
** There is one more detail, though.  After the whole file has been read and
** all the timed-out EVENTs have been reaped, the file must be truncated:
** it must be shortened by an amount proportional to the number of EVENTs
** reaped.
**
** The general loop, then, is this:
**	(1) Read an EVENT and advance the read-pointer.
**	(2) If this EVENT must be reaped, go to step 4.
**	(3) Otherwise, write the EVENT and advance the write-pointer.
**	(4) If end-of-file not reached yet, go back to 1.
**	(5) Truncate the file.
*/
int reap_events(fname)
    char *fname;				/* Name of EVENT file */
{
    int rd;					/* Read-only pointer */
    int wr;					/* Write-only pointer */
    int i;					/* Read index into die_at[] */
    int j;					/* Write index into die_at[] */
    EVENT v;					/* EVENT buffer */
    time_t death_date;				/* die_at[] buffer */
    time_t clock;				/* Current time */

    i = j = 0;
    clock = time(NULL);				/* Get the current time */
    if ((rd = open(fname, O_RDONLY)) < 0)	/* Init read pointer */
    {
	fprintf(stderr, "%s (reap_events): %s ", prognm, fname) ;
	perror ("read open");
	if (options & O_LOG)
	  syslog(LOG_WARNING, "(reap_events) Couldn't ropen datafile %s\n",
		 fname);

	closeeventlog();
	return (-1);
    }
    /* Init write pointer */
    if ((wr = open(fname, O_WRONLY, DATAFILE_MODE)) < 0)
    {
	fprintf(stderr, "%s (reap_events): %s ", prognm, fname) ;
	perror ("write open");
	if (options & O_LOG)
	  syslog(LOG_WARNING, "(reap_events) Couldn't wopen datafile %s\n",
		 fname);

	closeeventlog();
	return (-1);
    }	

    if (debug)
	printf("Entered reap_events() numtraps=%d clock=%d... ",
	    numtraps, clock);

    while (read(rd, &v, sizeof(v)) == sizeof(v))/* Read EVENTs one by one */
	if ((death_date = die_at[i++]) > clock) {	/* Reap this EVENT? */
	    write(wr, &v, sizeof(v));		/* No - keep this EVENT */
	    die_at[j++] = death_date;
	    if (debug)
		printf("kept, ");
	}
	else if (debug)			/* Yes - reap this EVENT */
	    printf("reaped, ");

    numtraps = j;
    ftruncate(wr, (off_t) j * sizeof(EVENT));	/* Shorten the file */

    if (debug)
	printf("truncated, ");

    close(rd);					/* Done with read pointer */
    close(wr);					/* Done with write pointer */

    if (debug) {
	printf("Done.\n");
	fflush(stdout);
    }
}

usage()
{
    fprintf(stderr,
	    "Usage: %s [-v (verbose)] [-p (errors on stderr)] ",
	    prognm);
    fprintf(stderr, "[-l (syslog output)] [-n (nocol output)]\n");
}


main(argc, argv)
    int	    argc;
    char    *argv[];
{
    struct snmp_session session, *ss;
    int	arg;
    int ch;
    int count, numfds, block;
    fd_set fdset;
    struct timeval timeout;

    options = 0;				/* Default: no logging */
    if ((prognm = strrchr(argv[0], '/')) == NULL) /* Get the basename.. */
	prognm = argv[0];			/* ..not the full path */
    else
	prognm++;

    while ((ch = getopt(argc, argv, "plnv")) != EOF)
	switch(ch) {
	    case 'p':				/* Print to stderr */
		options |= O_PRINT; break;
	    case 'l':				/* Log to syslog() */
		options |= O_LOG; break;
	    case 'n':				/* Create EVENTs for netmon */
		options |= O_NOCOL; break;
	    case 'd':				/* Print debugging output */
	    case 'v':				/* Print debugging output */
		debug = 1; break;
	    case '?':				/* Unrecognized switch */
	    default:
		usage() ;
		exit (1);
	} /* end of switch and while */

    if (options == 0) {		/* Gotta do something! */
	fprintf (stderr, "Reporting type not specified, setting to NOCOL.\n");
	options != O_NOCOL;
    }

    if (options & O_LOG)			/* Start up syslog() */
	init_syslog();

    if (options & O_NOCOL)			/* Make an empty EVENT file */
    {
	int fd;

	nocol_startup(&configfile, &datafile);
	if (debug) 
	{
	    fprintf(stderr, 
		    "prognm set to %s, datafile to %s.\n", 
		    prognm, datafile);
	    fflush(stderr);
	}

	unlink(datafile);
	fd = open(datafile, O_WRONLY|O_CREAT|O_TRUNC, DATAFILE_MODE);
	if (fd < 0) {
	    closeeventlog();

	    fprintf(stderr, "open() failed on %s\n", datafile);
	    perror(prognm);
	    exit(1);
	}
	close(fd);
	openeventlog();			/* Logging daemon */
    }	/* if nocol logging */

    if (debug)
	printf("options set to %d.\n", options);

    bzero((char *)&session, sizeof(struct snmp_session));
    session.peername = NULL;
    session.community = NULL;
    session.community_len = 0;
    session.retries = SNMP_DEFAULT_RETRIES;
    session.timeout = SNMP_DEFAULT_TIMEOUT;
    session.authenticator = NULL;
    session.callback = process_trap;		/* Defined above... */
    session.callback_magic = NULL;
    session.local_port = SNMP_TRAP_PORT;
    ss = snmp_open(&session);
    if (ss == NULL) {				/* Couldn't do snmp_open() */
	fprintf(stderr, "Couldn't open snmp session.\n");
	if (options & O_LOG)			/* Tell syslog() */
	  syslog(LOG_WARNING, "Couldn't open snmp session.\n");

	if (options & O_NOCOL)
	  closeeventlog();
	exit (1);
    }

    if (debug)
	printf("SNMP session opened successfully.\n");

    while (1) {					/* Main event-driven loop */
	if (options & O_NOCOL)
	    reap_events(datafile);		/* Delete timed-out EVENTs */
	numfds = 0;				/* Reset the list of file.. */
	FD_ZERO(&fdset);			/* ..descriptors */
	block = 0;				/* Don't block */
	timeout.tv_sec = (long) SELECT_TIMEOUT;	/* Only select() for a while */
	timeout.tv_usec = (long) 0;
	snmp_select_info(&numfds, &fdset, &timeout, &block);

	if (debug)
	    fprintf(stderr, "snmp_select_info()... ");

	count = select(numfds, &fdset, 0, 0, &timeout);

	if (debug)
	    fprintf(stderr, "select() == %d... ", count);

	if (count > 0)
	    snmp_read(&fdset);
	else if (count == 0)
	    snmp_timeout();
	else if (errno != EINTR) {
	    perror("select");
	    if (options & O_LOG)
		syslog(LOG_WARNING, "fatal select() error\n");

	    closeeventlog();
	    exit (1);
	} /* end of if */

	if (debug) {
	    fprintf(stderr, "Ok.\n");
	    fflush(stdout);
	}
    } /* end of while */

  /* NOTREACHED */
}

/*
 * All messages will be logged to the local0 facility and will be sent to
 * the console if syslog doesn't work.
 */
init_syslog()
{
    openlog("trapmon", LOG_CONS, LOG_LOCAL0);
    syslog(LOG_INFO, "Starting trapmon\n");
}
