/* $Header: /home/vikas/netmgt/nocol/src/noclog/RCS/noclogd.c,v 3.5 1994/05/16 02:01:40 vikas Exp $ */

/* Copyright 1994 Vikas Aggarwal, vikas@navya.com */

/*
 * AUTHOR
 *	Original version by David Wagner, wagner@phoenix.princeton.edu for
 *	JvNCnet.
 *
 *	Extensively modified and rewritten by Vikas Aggarwal, vikas@navya.com
 */

/*
 * DESCRIPTION
 *	A logging daemon for the nocol monitor programs; not unlike
 *	syslogd, it selectively records EVENT structures sent over
 *	a socket by the nocol programs.
 *
 *	The config file can also pipe the output to another program.
 *	Remember to use 'cat' since the EVENT strcuture will
 *	be piped to the executable and NOT sent over as command line
 *	argument.
 */

/*
 * BUGS
 *	Can't handle too many connections in a short period of time:
 *	should be able to handle (say) 100 per second just fine, though.
 */

/*
 * $Log: noclogd.c,v $
 * Revision 3.5  1994/05/16  02:01:40  vikas
 * Extensive cleanup for use with new lib nocol_startup()
 * call.
 *
 * Revision 3.4  1994/01/10  19:38:34  aggarwal
 * Added typecasts for all 'malloc()'
 *
 * Revision 3.3  1993/10/30  03:47:19  aggarwal
 * Small problems in pointer passing in authenticate_host(). Also
 * does a 'getsockopt()' to set the socket buffer size if possible.
 * Cleaned up code for auto closing a stream on error.
 *
 * Revision 3.2  1993/10/02  05:31:06  aggarwal
 * Moved the function to convert event to logstring into the nocol
 * library.
 *
 * Revision 3.1  1993/09/22  20:24:36  aggarwal
 * Deleted 'tag' from the print statement.
 * Not used in the nocol struct yet.
 *
 * Revision 3.0  1993/09/21  18:40:42  aggarwal
 * Deleted the code for the Unix socket. Tested, seems ready to go out.
 *
 * Revision 2.0  1993/09/18  21:41:38  aggarwal
 * This version 2.0 tries to use STREAM unix sockets and DGRAM inet
 * sockets (use of dgram unix sockets is not recommended due to
 * buffering and data overflow problems).
 *
 * However, the nature of the client logging is of the un-connected
 * type- since this didn't work in the stream connected environment,
 * this idea is being shelved.
 *
 * Revision 1.16  1993/09/16  23:45:28  aggarwal
 * Added macro to use 'pclose' instead of 'fclose' on receiving SIGHUP
 *
 * Revision 1.14  1993/09/14  21:32:36  aggarwal
 * Changed 'read' to recvfrom() for UDP sockets.
 * Also alarms for timeouts, and security for IP connections.
 *
 * Revision 1.6  1993/03/04  06:37:09  wagner
 * SOCK_STREAM became SOCK_DGRAM.  Started using select().  No more forking.
 *
 * Revision 1.3  1992/10/29  20:59:37  wagner
 * Added ability to pipe log output to a pipe.
 *
 * Revision 1.2  1992/10/29  20:06:51  wagner
 * Now reads a config file to determine where to log the EVENTs to.
 *
 * Revision 1.1  1992/10/29  18:23:02  wagner
 * Initial revision
 *
 */

#include "noclogd.h"

#define READ_TIMEOUT	5    	/* timeout for read() from sockets */

typedef struct node_s {
    char		*sender;	/* for comparing EVENT.sender */
    char		*path;		/* Filename in char format */
    FILE		*stream;	/* Opened stream */
    int			isapipe;	/* True if output is a pipe */
    int			loglevel ;   	/* Log level for this stream */
    struct node_s	*next;		/* Linked list */
} node;

char		*prognm;
int		debug;

static node	*loglist=NULL;
static int	mypid=0;
static char	*configfile, *errfile ;
static int	sighupflag=0, alarmflag=0 ;
static unsigned long  permithosts[50] ;		/* list of hosts */

void	sighuphandler(), alarm_intr();
char   *event_to_logstr();		/* format event in string format */

int main(argc, argv)
    int argc;
    char **argv;
{
    extern char	*optarg;
    extern int	optind;
    int		c ;

    /* For fatal(): currently no error condition */
    mypid = 0;
    errno = 0;

    /* Save the program name */
    if ((prognm = (char *)strrchr (argv[0], '/')) == NULL)
      prognm = argv[0] ; 			/* no path in program name */
    else
      prognm++ ;                                /* skip leading '/' */


    /* Parse the command line arguments; does no error checking */
    debug = 0;

    while ((c = getopt(argc, argv, "de:f:p:")) != EOF)
      switch(c)
      {
       case 'd':
	  debug++;
	  break;
       case 'e':
	  errfile = optarg ;
	  break ;
       case 'f':
	  configfile = optarg ;
	  break ;
       case '?':
       default:
	  fprintf(stderr, "%s: Unknown flag: %c\n", prognm, optarg);
	  fprintf(stderr, "Usage: %s [-d] [-f <config file>] ", prognm);
	  fprintf(stderr, "[-e <error filename>]\n");
	  exit (1);
      }

    fprintf(stderr,"%s (info): LOGHOST is defined to be '%s'\n",
	    prognm, NLOG_HOST);
    fprintf(stderr,"Make sure logging daemon runs on this host\n\n");

    if (errfile == NULL)		/* need this before daemon call */
      NLOG_ERRORFILE( (errfile = (char *)malloc(256)) ); /* default name */
		     
    if (debug)
    {
	fprintf(stderr, "%s: DEBUG  ON, server not daemon-izing\n",prognm);
#ifndef DEBUG
	fprintf(stderr,"%s: WARNING- program NOT compiled with DEBUG option\n",
		prognm);
#endif
    }
    else			/* Become a daemon */
      daemon();			/* this opens the error file also */


    /* Remember to do the nocol_startup() AFTER daemonizing */
    nocol_startup(&configfile, NULL);	/* datafile not really needed */

    if (debug)
      fprintf(stderr, "  configfile= '%s', errfile= '%s'\n\n",
	      configfile, errfile);

    /* Read the configuration file; good idea to do this after daemon() call */
    openlogfiles();

    /* Set up the signal handler */
    sighupflag = 0;
    (void) signal(SIGHUP, sighuphandler);

    /* And do all the socket stuff - this never returns */
    serve();

    /*NOTREACHED*/
}

/*
 * Opens a UDP logging socket, binds to it, and selects on it
 * until some data appears.  Then processes all data that comes
 * in and logs it into the appropriate file.  It never returns,
 * but dies on error.
 */
serve()
{
    int			inetfd, nfds,  sockbuffsize, optlen ;
    struct sockaddr_in	sin;
    struct servent	*sp;
    fd_set		readfds;

    /*
     * Open the network logging socket
     */
    if ((inetfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
	fatal("serve(): network socket() failed");

    optlen = 0;
    getsockopt(inetfd, /* level */SOL_SOCKET, /* optname */SO_RCVBUF, 
	       (char *)&sockbuffsize, &optlen);
#ifdef DEBUG
    if (debug)
      fprintf(stderr, "(debug) old inetfd RCVBUF size= %d\n", sockbuffsize);
#endif
    if (sockbuffsize < 65534)
    {
	sockbuffsize = 65534 ;
	setsockopt(inetfd, /* level */ SOL_SOCKET, /* optname */SO_RCVBUF, 
		   (char *)&sockbuffsize, sizeof sockbuffsize );
    }
    if (debug)
    {
	optlen = 0 ;
	getsockopt(inetfd, SOL_SOCKET, SO_RCVBUF, (char *)&sockbuffsize, 
		   &optlen);
	fprintf(stderr, 
		"(debug) inetfd RCVBUF size set to %d\n", sockbuffsize);
    }

    bzero((char *) &sin, sizeof (sin)) ;	/* important */
    sin.sin_family = AF_INET;
    /* Figure out what port number to use */
    if ((sp=getservbyname(prognm, "udp")) == NULL)
	sp = getservbyname(NLOG_SERVICE, "udp");
    if (sp == NULL)
	sin.sin_port = htons(NLOG_PORT);
    else
	sin.sin_port = sp->s_port;

    /* And do the bind() */
    if (bind(inetfd, (struct sockaddr *) &sin, sizeof(sin)) < 0)
	fatal("serve(): couldn't bind to network socket");

    if(debug)
      fprintf(stderr, "  INET socket fd= %d\n", inetfd);

    /* Start waiting for log messages */
    for (;;) {
	if (sighupflag)
	{
	    rereadconfig();
	    sighupflag = 0;
	}

	/* We're gonna check this after the select, so clear it now... */
	errno = 0;

	FD_ZERO(&readfds);
	FD_SET(inetfd, &readfds);
	nfds = select(FD_SETSIZE, &readfds, (fd_set *) NULL, (fd_set *) NULL,
			(struct timeval *) NULL);

	/*
	 * The select should only return with nothing to read from
	 * if we got a SIGHUP...
	 */
	if (errno == EINTR)
	    continue;
	else if (nfds <= 0)
	    fatal("serve(): select() failed");

	if (FD_ISSET(inetfd, &readfds))		/* Inet dgram socket */
	  readevent(inetfd);

    }
    /*NOTREACHED*/
}


/*
 * Reads in one EVENT structure and logs it to the appropriate stream,
 * based on the contents of loglist.  Openlogfiles() must have been
 * called first.  Dies upon error.
 */
readevent(fd)
     int fd;			/* socket file desc */
{
    register char *r, *s;
    int		n, len ;
    EVENT	v;
    char	from[sizeof(v.sender)+1]; 
    register node	*p;
    struct sockaddr_in	frominet;

    len = sizeof (frominet);

    /* Try to read an EVENT structure in */
    signal(SIGALRM, alarm_intr);
    alarmflag = 0 ;
    alarm(READ_TIMEOUT*10);

    n = recvfrom(fd, (char *)&v, sizeof(v), 0, 
		 (struct sockaddr *)&frominet, &len );
    if (n != sizeof(v))
    {
	if (alarmflag)
	  fprintf(stderr, "%s readevent: socket read timed out\n", prognm);
	else
	  fprintf(stderr,"%s readevent: socket read failed (incomplete)-- %s",
		  prognm, sys_errlist[errno]);

	alarm(0);	/* turn alarm off */
	return(1) ;	/* go back to waiting */
    }
    signal(SIGALRM, SIG_IGN);		/* ignore alarm */
    alarm(0) ;

#ifdef DEBUG
    if (debug)
      fprintf(stderr, "Recvd packet, sender '%s', loglevel '%d'\n",
	      v.sender, v.loglevel);
#endif

    /*
     * The rest of this should be fast, perhaps done in a forked child ?
     */
    if (! authenticate_host(&frominet))
      return(1) ;	      	/* illegal host */

    /* Make sure things aren't case sensitive */
    for (r=v.sender, s=from; *r; )
	*s++ = tolower(*r++);
    *s = '\0';

    /*
     * Check the linked list and log it to all applicable streams.
     * For speed, check,,,
     * 		log level first
     *		sender (takes longer since it is a string cmp)
     * Log to all streams at priority EVENT.loglevel and higher.
     */
    for (p=loglist; p; p=p->next)
      if ((int)v.loglevel <= p->loglevel)
	if (strcmp(p->sender, "*") == 0 || strstr(from, p->sender))
	{
	    /* Stream closed, try opening it.*/
	    if (!p->stream)
	    {
#ifdef DEBUG
		if (debug)
		  fprintf(stderr, "Opening stream '%s'\n", p->path);
#endif

		if (p->isapipe)
		  p->stream = popen(p->path, "w");
		else
		  p->stream = fopen(p->path, "a");

		if (! p->stream)   /* if the stream could not be reopened  */
		{
		    fprintf(stderr, "%s: Could not open stream `%s'- ",
			    prognm, p->path);
		    perror ("");
		    fflush(stderr);
		    continue ;
		}
	    }


	    if (fputs(event_to_logstr(&v), p->stream) == EOF) /* error */
	    {
		perror(p->path);
		fflush(stderr) ;
		p->isapipe ?  fclose(p->stream) : pclose(p->stream);
		p->stream = NULL ;	/* try reopening next time */
		continue ;		/* Next for() */
	    }
	    
	    fflush(p->stream);
	    
	    if (p->isapipe)		/* close and reopen every time ? */
	    {
		pclose(p->stream);
		p->stream = NULL;	/* reset */
	    }
	    
	    
	}	/*  end: if (stream closed, try reopening it) */

}	/* end readevent() */


/*+ 
 * Authenticate hosts. Check to see if the sender has an IP address in
 * the list from the config file. The argument is the socket file
 * descriptor which *must* be of the INET type - not a Unix type.
 *
 *	A = MASK & A
 *
 * Return 1 if OKAY, 0 if not
 */
authenticate_host(peer)
     struct sockaddr_in *peer ;	/* typecast from sockaddr to sockaddr_in  */
{
    struct sockaddr_in  s ;
    int i ;

    if (peer == NULL)
      return (0);

    for (i = 0; permithosts[i] != 0 ; ++i)
      if (permithosts[i] == (unsigned long) peer->sin_addr.s_addr)
	return(1) ;

    fprintf(stderr, "%s: Permission denied for host %s\n", 
	    prognm, inet_ntoa(peer->sin_addr));

    return (0);
}


/*+
 * Allocates space for a new node, initializes it's fields, and
 * inserts it into the beginning of loglist (since it is easier).
 * Dies upon error.
 */
node  *insert(sender, path)
    char *sender, *path;
{
    node	*new;
    int		i;
    char	*p;

    new = (node *) malloc(sizeof(node));
    if (new == NULL)
	fatal("insert(): out of memory");
    bzero(new, sizeof(*new));

    new->next = loglist;
    new->sender = (char *) malloc(strlen(sender)+1);
    new->path = (char *) malloc(strlen(path)+1);
    if (new->sender == NULL  ||  new->path == NULL)
      fatal("insert(): out of memory using malloc");
    for (p=new->sender; *sender; sender++)
	*p++ = tolower(*sender);
    *p++ = '\0';

    if (*path == '|')	/* pipe */
      strcpy(new->path, path+1) , new->isapipe = 1 ;
    else
      strcpy(new->path, path);

    loglist = new;	/* point to the new node */
    return(new);
}

/*
 * Reads in a configuration file and opens the log files named there
 * for output.  Also writes to pipes instead of a log file if requested.
 *
 *		PERMITHOSTS   addr  addr  addr
 *		EVENT.sender  min-log-severity  Log-filename
 *		EVENT.sender  min-log-severity	|execute-filename
 *
 * Dies upon failure.
 */
openlogfiles()
{
    char	line[1024], *progstr, *sevstr, *filestr, *p, *q;
    FILE	*config;
    int		level, linenum, i;
    node	*new;

    /* Sanity check */
    if (loglist)
	fatal("openlogfiles(): already a list of open files");
    if ((config = fopen(configfile, "r")) == NULL)
    {
	fprintf (stderr, "confile file '%s': ", configfile);
	fatal("openlogfiles(): couldn't fopen() ");
    }


    i = 0;
    for (linenum=0; fgets(line, sizeof(line)-1, config) != NULL; linenum++)
    {
	
	line[strlen(line)-1] = '\0';	/* Strip off '\n' at  end of line */

	if ((p=strchr(line, '#')) != NULL)	/* Get rid of comments */
	    *p = '\0';

	/* Parse the line into fields */
	if (*line == NULL  ||  (progstr=strtok(line, " \t")) == NULL)
	    continue;
	else if (strncasecmp("PERMITHOST", line, strlen("PERMITHOST")) == 0)
	{
	    while ((q = strtok(NULL, " \t")) != NULL)
	    {
		permithosts[i++] = inet_addr(q);
		if (debug)
		  fprintf(stderr, "Added %s to permithosts list\n", q);
	    } /* end while() */
	    continue ;
	}
	  
	else if ((sevstr=strtok(NULL, " \t")) == NULL) {
	    warning_int("config line %d: no severity string, ignoring...\n",
			linenum);
	    continue;
	} else if ((filestr=strtok(NULL, " \t")) == NULL) {
	    warning_int("config line %d: no logfile, ignoring...\n", linenum);
	    continue;
	}

	/* Figure out what severity we're at */
	level = -1;
	switch(tolower(sevstr[0])) {
	    case 'c': case '1': level = E_CRITICAL; break;
	    case 'e': case '2': level = E_ERROR; break;
	    case 'w': case '3': level = E_WARNING; break;
	    case 'i': case '4': level = E_INFO; break;
	}
	if (level == -1) {
	    warning_int("config line %d: bad severity, ignoring...\n", linenum);
	    continue;
	}

	/* Create and insert a new node into the linked list */
	new = insert(progstr, filestr);
	new->loglevel  = level ;
    }

    fclose(config);		/* close config file */
    permithosts[i] = 0 ;	/* null the last one */
}

/*
 * Reread the configuration file.  Called whenever SIGHUP is recieved.
 */
rereadconfig()
{
    time_t locclock;
    char *s;
    register node	*p, *q;

    locclock = time((time_t *)NULL);
    s = (char *)ctime(&locclock);
    *(strchr(s, '\n')) = '\0' ;		/* get rid of the newline */
    fprintf(stderr, "%s (%d) %s: Closing streams & Re-reading config file\n", 
	    prognm, mypid, s);
    for (p = loglist; p; p=q)
    {
	q = p->next;
	if (p->stream)
	{
	    fflush(p->stream);
	    p->isapipe ? pclose(p->stream) : fclose(p->stream);
	}

	if (p->sender != NULL)
	  free(p->sender);
	if (p->path != NULL)
	  free(p->path);
	free(p);
    }

    loglist = NULL;

    openlogfiles();
}

/*
 * Become a daemon: go into the background, reset our process group,
 * and clobber the current directory and stdin, stdout, stderr.
 * Ideas for what needs to be done more or less stolen from BSD sources.
 * Exits on failure, and complains vigorously!
 */
daemon()
{
    int		retval, fd, errfd;

    if ((retval=fork()) < 0)			/* Go into the background */
	fatal("daemon(): fork() failed");
    else if (retval)
	exit(0);

    if ((int) setsid() < 0)			/* Reset the process group */
	fatal("daemon(): setsid() failed");

    if (chdir(ETCDIR) < 0)			/* Clobber current directory */
      if (chdir("/") < 0)
	fatal("daemon(): chdir() failed");

    if ((fd = open("/dev/null", O_RDWR, 0)) < 0)  /* Clobber stdin,stdout.. */
      fatal ("daemon() open failed for /dev/null");
    if ((errfd = open(errfile, O_CREAT|O_APPEND|O_WRONLY, 0644)) < 0)
      fatal("daemon() open() failed for errorfile");
    if (dup2(fd, 0) < 0 || dup2(errfd, 1) < 0)
      fatal("daemon(): dup2() failed");

    /* Not much we can do if this dup2 fails - oh well, gotta ignore it */
    (void) dup2(errfd, 2);

    if ((fd > 2 && close(fd) < 0) || (errfd > 2 && close(errfd) < 0))
	fatal("daemon(): close() failed");

    signal(SIGALRM, SIG_IGN);		/* ignore alarms until desired */
}

/*
 * Outputs an error message to stderr.  Records the daemon's name and
 * PID, if available.  Prints out the error message corresponding to
 * errno, if errno is set.  Exits with return value 1.
 */
fatal(msg)
    char *msg;
{
    if (prognm)
	fprintf(stderr, "%s: fatal error. ", prognm);
    if (mypid)
	fprintf(stderr, "(%d) ", mypid);
    if (errno)
	perror(msg);
    else
	fprintf(stderr, "%s\n", msg);

    exit(1);
}



/*
 * Outputs a warning message to stderr.  Records the daemon's name and
 * PID, if available.  Prints out the error message corresponding to
 * errno, if errno is set.  Does not exit.
 *
 * Assumes that 'format' is a valid fprintf format string that requires
 * only one argument, the integer 'num'.
 */
warning_int(format, num)
    char *format;
    int num;
{
    if (prognm)
	fprintf(stderr, "%s: ", prognm);
    if (mypid)
	fprintf(stderr, "(%d) ", mypid);
    fprintf(stderr, format, num);
    if (errno)
	perror((char *) NULL);
    else
	fprintf(stderr, "\n");
}

/*
 * On receiving an alarm, simply set the flag and return
 */
void alarm_intr()
{
    alarmflag = 1 ;
}

/*
 * Sets the sighupflag whenever we get a SIGHUP signal.
 */
void sighuphandler()
{
    sighupflag = 1;
}

