/*
 * $Header: /home/vikas/netmgt/nocol/src/netconsole/RCS/filter.c,v 1.6 1994/11/29 20:40:34 vikas Exp $
 */

/*+ FUNCTION:
 ** A limited pattern filter to limit what is displayed on the NOCOL
 ** curses display.
 **
 ** The functions 'compile_pattern' and 'filter' allow NETCONSOLE to filter 
 ** out only the information that the user desires.  Logical AND and OR 
 ** is also incorporated in this feature, allowing input such as:
 **
 ** airport | 13:27 & Error (prints status of the 'airport' sites as well
 **                          any errors that occurred at 13:27)
 ** 
 ** 'compile_pattern' works like this:
 **
 ** A new argument is started every time an AND (&) operator is
 ** encountered.
 ** 
 ** A new row of arguments is started every time an OR (|) operator is
 ** encountered.
 **
 ** So, if the input is: THIS & THAT | YOU & ME | WHATEVER | A & B & C
 ** The pointers would be assigned as follows:
 **
 ** compiled[0][0] = "THIS"		compiled[0][1] = "THAT"
 ** compiled[1][0] = "YOU"		compiled[1][1] = "ME"
 ** compiled[2][0] = "WHATEVER"
 ** compiled[3][0] = "A"		compiled[3][1] = "B"	compiled[3][2] = "C"
 **
 ** 'filter' filters out the desired information by comparing the
 ** EVENT data structure against the arguments row by row, argument by
 ** argument.
 **
 ** Given the data above, 'filter' would first check arguments of
 ** row 0.  If THIS (args[0][0]) was not in the data, the whole row can
 ** be skipped because we know that (THIS & THAT) will be FALSE. 
 **
 ** 'filter' returns 1 if any of the rows of arguments match,
 ** otherwise it returns 0.
 **
 ** 	Initial version:  David Wagner, wagner@phoenix.princeton.edu
 **	Updates: Vikas Aggarwal, vikas@navya.com
 */

/*
 * $Log: filter.c,v $
 * Revision 1.6  1994/11/29  20:40:34  vikas
 * Updated as netconsole for v4.0
 *
 * Revision 1.5  1994/06/12  18:09:35  vikas
 * Deleted the 'options' variable and set debug/quiet/emode to
 * be simple integers instead.
 *
 * Revision 1.4  1994/05/16  01:59:12  vikas
 * Cleanup for new release.
 *
 * Revision 1.3  1994/01/18  17:57:53  aggarwal
 * Fixed bug in compile_pattern()  where the array 'compiled' was not
 * being terminated properly with a NULL in its rows and columns. Result
 * was that the compiled filter would try and match leftovers from old
 * filters
 *
 * Revision 1.2  1993/10/30  03:51:36  aggarwal
 * Changed the name from filter() to nocolfilter()
 *
 * Revision 1.1  1992/06/18  21:03:49  wagner
 * Initial revision
 *
 */

#include "netconsole.h"
#include <string.h>

#define MAXARGS		20			/* Max # of search words */
#define MAXTEXTLEN	1024			/* len of temp pattern */

void compile_pattern();
int nocolfilter();

static char *(compiled[MAXARGS][MAXARGS]);	/* Compiled pattern */
static char pattern_defined=0;			/* Is a pattern specified? */

#ifdef FILTERMAIN

#include <stdio.h>
#include <sys/file.h>
#include <arpa/inet.h>

#define	MAXFILTERLEN 	1024
#define FILENAME  "/nocol/data/ippingmon-data"	/* nocol data file */


int main(argc,argv)
     int argc;
     char *argv[];
{
    int i, j, fd;
    EVENT v;                                    /* defined in nocol.h        */
    char *p;
    char argline[MAXFILTERLEN];

    *argline = '\0';				/* Start argline[] off fresh */

    /* Let's get the arguments that we are going to search for */
    if (argc > 1)
      /* The arguments we need are on the command line - use them */
      for (i=1; i < argc; i++) {
	  /* Throw the entire command line into argline[] */
	  strncat(argline, argv[i], MAXFILTERLEN);
	  strncat(argline, " ", MAXFILTERLEN);
      } /* end of for */
    else {					/* Nothing useful in argv[] */
 	printf("arguments> ");			/* Prompt for arguments */

	for (i=0; i < MAXFILTERLEN; ) {		/* Build argline[] */
	    /*
	    ** Read one line & append it to argline[]
	    ** Don't forget to replace the NUL terminator ('\0')
	    ** with a space!
	    */
	    if (gets( &(argline[i]) ) == NULL)
	      break;
	    i = strlen(argline);
	    argline[i++] = ' ';
	}

	/* Now, we killed the last NUL terminator - put it back */
	argline[i] = '\0';
    }

    /* If "DISABLE" is one of the arguments, quit */
    if (strstr(argline,"DISABLE")) {
	printf("SEARCH DISABLED\n");
	exit(1);
    }

    compile_pattern(argline);			/* Parse argline[] */

    /* Open the file - if it doesn't exist, give error and exit */
    if ((fd = open(FILENAME, O_RDONLY)) == -1) {
	fprintf(stderr,"Error: main unable to open file %s\n", FILENAME);
	exit(1);
    }

    /* Read each EVENT structure in, one by one */
    while (read(fd, (char *) &v, sizeof(v)) == sizeof(v)) {
	/* Does this EVENT structure match the arguments? */
	if (nocolfilter(&v))	{
	    /*
	    ** It does!  Print it out
	    ** But do we want the extended format or the short format?
	    */
	    if (emode)
	      /* Extended format - EFMT, EFIELDS are defined in netconsole.h */
	      printf(EFMT, EFIELDS);
	    else
	      /* Short format - SFTM, SFIELDS are defined in netconsole.h */
	      printf(SFMT, SFIELDS);

	    printf("\n");			/* Don't forget the newline! */
	} /* end of if */
    } /* end of while */

    close(FILENAME);				/* All done, so close it */
} /* end of main */

#endif /* end of #ifdef FILTERMAIN */
 

	
/*+ 
** FUNCTION: compile_pattern  
**
** Copies the string of arguments 'pattern' to a temporary string,
** eliminating invalid characters and converting valid characters to
** lowercase.
**
** The locations of the arguments within 'pattern' are stored in the
** global static two-dimensional array of pointers 'compiled[row][col]'.  
**
** If an AND (&) is encountered, it is interpreted as a logical AND,
** and means that the argument to follow is to included in the current
** row (or group) of arguments.  Therefore, 'col' is increased by
** 1, while 'row' remains unchanged.
**
** If an OR (|) is encountered, it is interpreted as a logical OR,
** and means that the argument to follow is to be included in a new
** row (or group) of arguments.  Therefore, 'row' is increased by 1,
** and 'col' is set to zero (to start at the beginning of the row).
**
** Spaces and tabs are ignored, and any other characters are added to
** the end of the current argument.
**/


void compile_pattern(pattern)
     char *pattern;
{
    static char text[MAXTEXTLEN];		/* Cooked form of pattern */
    char *s, *t;
    int row, col;
    int was_conjunction;			/* Was previous word AND/OR? */

    if (strlen(pattern) > 0)
      pattern_defined = 1;			/* Enable filtering */
    else
      pattern_defined = 0;			/* Disable filtering */

    /*
    ** We don't want to see a conjunction right off the bat,
    ** so pretend we just saw one...  then we'll ignore any
    ** conjunctions that come before normal arguments
    */
    was_conjunction = 1;

    /* Run through the pattern one character at a time */
    t = text;
    s = pattern;
    row = col = 0;
    while (*s != '\0')
	switch(*s) {
	 case ' ': case '\t': 			/* skip white space */
	    s++; break;	   

	 case '|' :				/* OR */
	   /* If the last thing was a conjunction, ignore this one */
	   if (! was_conjunction) {		/* Begin a new row */
	       *t++ = '\0';
	       compiled[row][++col] = NULL ;	/* Terminating null */
	       compiled[++row][0] = ++t;
	       col = 0;

	       was_conjunction = 1;	       /* Remember we saw a conj. */
	   }
	    s++; break;
	      
	 case '&' :				/* AND */
	   /* If the last thing was a conjunction, ignore this one */
	   if (! was_conjunction) {
	       *t++ = '\0';			/* Goto next column */
	       compiled[row][++col] = t;

	       was_conjunction = 1;		/* Remember we saw a conj. */
	   }
	    s++; break;

	 default:				/* any normal character */
	    *t = tolower(*s);			/* Make everything lowercase */

	    /*
	    ** If the last thing was a conjunction, then this must
	    ** be the start of a normal argument - so save a pointer
	    ** to the beginning of this word in sargs[][]
	    **
	    ** Then, t will just skip over the rest of the characters
	    ** in the word
	    */
	    if (was_conjunction)
	      compiled[row][col] = t;
	    t++;
	    s++;

	    /* Remember this wasn't a conjunction */
	    was_conjunction = 0;
	    break;
	} /* end of switch */

    /* By the way, we're NOT in the while loop anymore */

    *t='\0';				/* Don't forget the NUL terminator */

    /* Deal with the case where the last thing we saw was a conjunction */
    if (was_conjunction)
      compiled[row][col]=NULL;

    /*
    ** Add a NULL terminator to our last row and also our list of rows
    ** so we know where the end is (remember, the arguments are global)
    */
    compiled[row][++col]=NULL;		/* terminate present row */
    compiled[++row][0]=NULL;		/* end of rows */

} /* end of compile_pattern */





/*+ 
** FUNCTION: nocolfilter
**
** Searches for arguments in the data structure 'v' by their
** respective groups as stored in 'args[row][argnum]'.
**
** Searching begins with the first row, argument by argument.  If an
** argument is not found in 'v' then searching stops in that row and
** moves to the next.
**
** If all arguments in a row are found in 'v' then this EVENT structure
** matches are filter condition and we may return TRUE immediately.
**/


int nocolfilter(vptr)
     EVENT *vptr;
{
    EVENT v;
    char line[1024];
    char *p;
    int row, col;

    if (!pattern_defined)			/* If no pattern specified, */
	return (1);				/* then don't filter anything */
    /*
    ** Copy the EVENT structure pointed by vptr into v
    ** This is done because the EFIELDS and SFIELDS #define's
    ** were set up assuming v would be a EVENT structure,
    ** not a pointer to one
    ** Yes, I know this qualifies as an ugly hack...
    */
    bcopy((char *) vptr, (char *) &v, sizeof(v));

    /*
    ** Throw everything in v into one big formatted line
    ** This make searching a lot easier
    */
    if (emode)
      /* Extended format - EFMT, EFIELDS are defined in netconsole.h */
      sprintf(line, EFMT, EFIELDS);
    else
      /* Short format - SFMT, SFIELDS are defined in netconsole.h */
      sprintf(line, SFMT, SFIELDS);

    for (p=line; *p != '\0'; p++)		/* Make everything lowercase */
      *p = tolower(*p);

    /* Loop through the rows one by one... */
    for (row=0; compiled[row][0] != NULL; row++) {
	/*
	** If all the arguments on this row match line[],
	** then return TRUE right away - this EVENT structure
	** should pass through the filter
	*/
	for (col=0; compiled[row][col] != NULL; col++)
	  if (! strstr(line, compiled[row][col]))
	    break;

	if (compiled[row][col] == NULL)
	  return(1);				/* Match found */

	/*
	** There were still arguments left on this row.
	** I guess line[] doesn't match this row - so let's try another!
	*/
    }

    return(0);					/* Match not found */
} /* end of filter */
