/*
** $Header: /nocol/src/tsmon/RCS/tsmon.c,v 2.0 1992/06/18 21:36:16 aggarwal Exp $
*/

/*
** DESCRIPTION
**	Monitors the number of modem lines in use on several terminal servers.
**	Watches the wtmp files output by those terminal servers to determine
**	how many lines are in use at each point in time.  Outputs to a file
**	readable by 'nocol'.
**
**	This program maintains a list of tty lines for each terminal server,
**	remembering who is currently logged on through that tty line or who
**	was last logged on.  This list is updated with every new wtmp entry,
**	and the number of lines in use is derived from it.  A configuration
** 	file which lists the terminal servers to monitor can be specified on
**	the command line.  Various levels of debugging output is available.
**
**	This program was derived from tslookup.c.  The man page contains more
**	information about tsmon.
*/

/*
** AUTHOR
**	David Wagner, wagner@jvnc.net, June 1992
*/

/*  Copyright 1992 JvNCnet

 Permission to use, copy, modify and distribute this software and its
 documentation for any purpose is hereby granted without fee, provided that
 the above copyright notice appear in all copies and that both that copyright
 notice and this permission notice appear in supporting documentation, and
 that the name of JvNCnet not be used in advertising or publicity pertaining
 to distribution of the software without specific, written prior permission.
 JvNCnet makes no representations about the suitability of this software for
 any purpose.  It is provided "as is" without express or implied warranty.

 JvNCnet DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL JvNCnet
 BE LIABLE FOR ANY DAMAGES WHATSOEVER, INCLUDING DAMAGES RESULTING FROM LOSS
 OF USE, DATA OR PROFITS, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/*
 * MODIFICATIONS:
 *
 * $Log: tsmon.c,v $
 * Revision 2.0  1992/06/18  21:36:16  aggarwal
 * Just to update the revision number
 *
 * Revision 1.6  1992/06/10  16:12:54  aggarwal
 * Changed the '-d' option to a '-i'
 *
 * Revision 1.5  1992/06/10  12:33:23  aggarwal
 * Cleaned up for release. Cosmetic changes only.
 *
 * Revision 1.4  1992/06/08  21:31:24  wagner
 * Got rid of unlink() call on the datafile - think that was confusing netmon.
 */

#include "tsmon.h"

main(argc, argv)
  char **argv;
{
  char progname[256], path[MAX_FILENAME_LEN];
  int i, c, sleep_interval, pass2debug;

  /* Set up defaults  */
  debug = pass2debug = Silent;
  strncpy(progname, argv[0], 256);
  strncpy(path, WTMP_PATH, MAX_FILENAME_LEN);
  *datafile = '\0';
  sleep_interval = INTERVAL;
  umask(002);

  /* Parse argv[] */
  while ((c = getopt(argc, argv, "wvVo:i:p:")) != EOF)
    switch (c) {
     case 'w':			/* show warnings */
	pass2debug = Warn; break;
     case 'v':			/* verbose output */
        pass2debug = PrintEachWTMP; break;
     case 'V':			/* very verbose debugging info */
	pass2debug = PrintUsers; break;
     case 'o':			/* alternate eventfile */
	strncpy(datafile, optarg, MAX_FILENAME_LEN); break;
     case 'i':			/* alternate sleep interval */
	sleep_interval = atoi(optarg); break;
     case 'p':			/* alternate path to wtmp files */
	strcpy(path, optarg); break;
     case '?':			/* unrecognized switch */
     default:
	fprintf(stderr,
		"Usage: %s [-w] [-v] [-V] [-o datafile] [-i sleep-interval] [-p wtmppath] configfile...\n",
	     progname);
	exit (1);
    } /* end of switch and while */

  /* No more switches - the rest must be names of configuration files */
  numwtmps = 0;
  for (; optind < argc; optind++)
      readwtmplist(argv[optind], path);

  if (numwtmps == 0) {
      fprintf(stderr, "%s: no wtmp files to process\n", progname);
      exit(1);
  }

  /* Set up the EVENT structures */
  initmsgs(progname);

  /* Now keep checking the wtmp files forever */
  while (1) {
      for (i=0; i<numwtmps; i++) {
	  dowtmp(i);
	  check(i);
      }
      if (debug >= PrintEvent)
	fprintf(stderr, "sleeping for %d seconds...\n\n", sleep_interval);
      dumpmsgs();

      /*
      ** We may have wanted debugging output only after the first pass
      ** If so, the local variable pass2debug will indicate the amount
      ** of debugging we want for the 2nd pass and afterward... so set
      ** the global variable debug (which all the other functions look
      ** at) to what pass2debug is
      */
      if (pass2debug > debug)
	debug = pass2debug;

      sleep(sleep_interval);
  }

  /*NOTREACHED*/
  exit(0);
}

/*
 * Adds the wtmp file named by "fname" to the "wtmps[]" list
 * and initializes that "wtmps[]" entry to refer to a server
 * called "sname" that has threshold array "t[]"
 */
initwtmp(fname, sname, t, path)
  char *fname;
  char *sname;
  int t[];
  char *path;
{
  WTMP *w;

  if (numwtmps >= MAX_WTMPS) {
      fprintf(stderr, "initwtmp(): too many wtmps - increase MAX_WTMPS\n");
      exit (1);
    }

  /* Set up a shorthand name for wtmps[numwtmps] */
  w = &(wtmps[numwtmps]);

  /* Initialize some important stuff to 0 */
  w->lastlen = 0;
  w->stat.inuse = 0;
  w->stat.numseen = 0;
  w->stat.warnings = 0;
  w->stat.seenreboot = 0;

  /* Copy filename and set NUL terminator */
  strncpy(w->fname, fname, sizeof(w->fname));
  w->fname[ sizeof(w->fname) - 1 ] = '\0';
  
  /* Copy full path and filename and set NUL terminator */
  strncpy(w->fullfname, path, sizeof(w->fullfname));
  if (path[strlen(path)-1] != '/')
    strncat(w->fullfname, "/", sizeof(w->fullfname));
  strncat(w->fullfname, fname, sizeof(w->fullfname));
  w->fname[ sizeof(w->fullfname) - 1 ] = '\0';

  /* Copy servername and set NUL terminator */
  strncpy(w->servername, sname, sizeof(w->servername));
  w->servername[ sizeof(w->servername) - 1 ] = '\0';

  /* Copy threshold array and remember that we added a new terminal server */
  w->threshold[E_ERROR] = t[E_ERROR];
  w->threshold[E_WARNING] = t[E_WARNING];
  numwtmps++;
}

/*
** Read a list of wtmp files from a config file called lname.
** path is the path to the wtmp files.
** Each line in the file should contain refer to only one terminal server.
** The format of each line should be as follows:
**	<modem server name> <wtmp filename> <warning> <error> # <comment>
** <wtmp filename> is the filename of the wtmp file.
** <modem server name> should have no spaces in it.
** Anything can go after a # - it will be ignored.
** Blank lines and lines containing only comments in them are allowed.
** Lines that are obviously invalid are ignored, though the error checking
** could be beefed up a bit.
** Anytime that the number of lines in use exceeds <warning>,
** a message (priority E_WARNING) will be sent to the netmon program,
** and if the number of lines in use becomes greater than <error>,
** a message of severity E_ERROR will be sent.
** Otherwise, the message will be sent with priority E_INFO.
** Note that there can never be a message with severity E_CRITICAL
*/
int readwtmplist(lname, path)
  char *lname;
  char *path;
{
  FILE *list;
  char wtmpfile[MAX_FILENAME_LEN], servname[MAX_SERVERNAME_LEN], line[1024];
  int matched, t[5];

  list = fopen(lname, "r");
  while (fgets(line, 1024, list) != NULL) {
      /* Remove comments */
      if (strchr(line, '#') != NULL)
	*(strchr(line, '#')) = '\0';

      /* Separate line into fields */
      matched = sscanf(line, "%s %s %d %d", servname, wtmpfile,
	     &(t[E_WARNING]), &(t[E_ERROR]));

      /* Sanity check on t[] */
      if (t[E_WARNING] > t[E_ERROR])
	continue;

      /* If this appears to be a valid entry, use it */
      if (matched == 4)
	initwtmp(wtmpfile, servname, t, path);
  }
  fclose(list);
}

/*
** Process the "i"th wtmp file, as listed in the "wtmps[]" structure
*/
dowtmp(i)
int i;
{
  int j, retval, numentriesread;
  STAT *st;

  /* Open the wtmp file  */
  if ((fd = open(wtmps[i].fullfname, O_RDONLY)) < 0) {
      fprintf(stderr, "dowtmp(): couldn't open %s", wtmps[i].fullfname);
      perror("");
      exit(1);
  }

  /* Go to where we left off last time, if possible */
  if (lseek(fd, 0, SEEK_END) < wtmps[i].lastlen) {
      /*
      ** The wtmp shrunk since the last time we looked at it.
      ** This could happen if the wtmp file were removed - as
      ** is often done periodically by the sysadmin or a cron job.
      ** This situation is not a problem - we just restart at the beginning
      ** of the new wtmp file, and retain state (i.e. don't change
      ** the STAT structure for this file, except for the lastlen field)
      */
      wtmps[i].lastlen = 0;
      if (debug >= Warn)
	fprintf(stderr, "\n\n%s has shrunk in size - but everything's ok\n\n",
		wtmps[i].fullfname);
  }
  lseek(fd, wtmps[i].lastlen, SEEK_SET);

  /* Reset some variables */
  st = &(wtmps[i].stat);
  st->warnings = 0;
  numentriesread = 0;

  /* Read in wtmp entries one at a time */
  while ((retval = getent()) > 0) {
      /* Update statistics */
      numentriesread++;

      /* Check for a reboot */
      if (isreboot(st)) {
	  st->inuse = 0;
	  st->seenreboot = 1;
	  if (debug >= PrintEachWTMP)
	    fprintf(stderr, "\n\n\tFound reboot - resetting inuse...\n\n");
	  continue;
      }

      /* Check for a login or a logout */
      if (islogin(st))
	st->inuse++;
      else if (islogout(st))
	st->inuse--;

      if (debug < PrintEachWTMP)
	continue;

      /* Print debugging info */
      printent(stderr, &buf);
      fprintf(stderr, "\t%d\n", st->inuse);
  } /* end of while */

  /* Check and make sure getent() didn't fail horribly */
  if (retval != 0) {
      /*
      ** getent() only read part of a wtmp entry - the file must be corrupted
      ** Give up the ghost and hope netmon notices
      */
      fprintf(stderr, "dowtmp(): %s's wtmp (%s) is corrupted\n",
	      wtmps[i].servername, wtmps[i].fullfname);
      exit(1);
  }

  /* Should we dump the entire seen[] array (and friends)? */
  if (debug >= PrintSEEN) {
    fprintf(stderr, "Inuse=%d,	Numseen=%d", st->inuse, st->numseen);
    for (j=0; j < st->numseen; j++) {
	fprintf(stderr, "\n\t\t%d ", st->isinuse[j]);
	printent(stderr, &(st->seen[j]));
    }
    fprintf(stderr, "\n");
  } else if (debug >= PrintUsers) {
    /*
    ** Print only the parts of the seen[] array that correspond to users
    ** who are currently logged in
    */
    fprintf(stderr, "Inuse=%d, Numseen=%d", st->inuse, st->numseen);
    for (j=0; j < st->numseen; j++)
	if (st->isinuse[j]) {
	    fprintf(stderr, "\n\t\t");
	    printent(stderr, &(st->seen[j]));
	}
    fprintf(stderr, "\n");
  }

  /* Done reading the wtmp file - now update some statistics and close */
  wtmps[i].lastlen = lseek(fd, (long) 0, SEEK_CUR);
  close(fd);

  /*
  ** Is this program having a lot of trouble interpreting the wtmp file?
  ** Use some arbitrary formula for the maximum number of acceptable warnings
  ** This should be changed at some time in the future
  */
  if (st->warnings > (0.5 * numentriesread + 1) && debug >= Warn)
    fprintf(stderr, "dowtmp(): %d/%d warnings - wtmp %s for %s inaccurate\n",
	    st->warnings, numentriesread,
	    wtmps[i].fullfname, wtmps[i].servername);

  return;
}

/*
** Set up the "eventlist[]"
*/
initmsgs(progpath)
  char *progpath;
{
  char *progname;
  int i;
  EVENT *v;

  /* Get zeroed-out memory for the eventlist */
  eventlist = (EVENT *) calloc(numwtmps, sizeof(EVENT));
  if (eventlist == NULL) {
      fprintf(stderr, "initmsgs(): out of memory\n");
      exit(1);
  }

  /* We don't want the full path to this program - just its name */
  progname = strrchr(progpath, '/');
  if (progname == NULL)
    progname = progpath;
  else
    progname++;

  /* Now set the name for the output datafile that netmon will look at */
  if (*datafile == '\0') {
      strcpy(datafile, DATADIR);
      strcat(datafile, "/");
      strcat(datafile, progname);
      strcat(datafile, OUTPUT_EXTENSION);
  }

  /* Set some fields of the EVENT structure that will never get changed */
  for (i=0; i<numwtmps; i++) {
      /* Set up an alias for the current EVENT structure */
      v = &(eventlist[i]);

      /* Copy the program name - and don't forget the NUL terminator */
      strncpy(v->sender, progname, sizeof(v->sender));
      v->sender[ sizeof(v->sender) - 1 ] = '\0';

      /* This might change, but till then, assume it's up */
      eventlist[i].nocop = n_UP;

      /* Copy the servername - and don't forget the NUL terminator */
      strncpy(v->site.name, wtmps[i].servername, sizeof(v->site.name));
      v->site.name[ sizeof(v->site.name) - 1 ] = '\0';

      /*
      ** Copy the filename (not the full path) into the site address
      ** This is kind of a hack...
      ** Oh, and don't forget the NUL terminator
      */
      strncpy(v->site.addr, wtmps[i].fname, sizeof(v->site.addr));
      v->site.addr[ sizeof(v->site.addr) - 1 ] = '\0';

      /* Copy the variable name - and don't forget the NUL terminator */
      strncpy(v->var.name, VARNAME, sizeof(v->var.name));
      v->var.name[ sizeof(v->var.name) - 1 ] = '\0';

      /* Set the units for the variable */
      strncpy(v->var.units, VARUNITS, sizeof(v->var.units));
      v->var.units[ sizeof(v->var.units) - 1 ] = '\0';
  } /* end of for */
}

/*+
** Check the "i"th entry in "wtmps[]" and update the EVENT structure for it
** Note that there can never be a message with severity E_CRITICAL
*/
check(i)
  int i;
{
  int this_threshold, level;
  struct timeval tp;
  struct timezone tz;
  struct tm *t;

  /* First find the severity */
  if (wtmps[i].stat.inuse < wtmps[i].threshold[E_WARNING]) {
      this_threshold = wtmps[i].threshold[E_WARNING];
      level = E_INFO;
  }
  else if (wtmps[i].stat.inuse < wtmps[i].threshold[E_ERROR]) {
      this_threshold = wtmps[i].threshold[E_WARNING];
      level = E_WARNING;
  }
  else {
      this_threshold = wtmps[i].threshold[E_ERROR];
      level = E_ERROR;
  }

  /* Now record that in the event structure */
  eventlist[i].var.value = wtmps[i].stat.inuse;
  eventlist[i].var.threshold = this_threshold;
  eventlist[i].severity = level;
  
  /*
  ** Fill in the time fields of the EVENT structure
  ** Use the current time instead of the timestamp on the last wtmp entry
  ** (the last wtmp entry could have a timestamp a long time in the past
  ** if there hasn't been much activity on it, but we still want to tell
  ** netmon that this program has been monitoring the wtmp closely)
  ** Note, too, that the month field ranges from 1..12 in the EVENT structure,
  ** but localtime returns the month in 0..11
  */
  gettimeofday(&tp, &tz);
  t = localtime(&((time_t) tp.tv_sec));
  eventlist[i].mon = t->tm_mon + 1;
  eventlist[i].day = t->tm_mday;
  eventlist[i].hour = t->tm_hour;
  eventlist[i].min = t->tm_min;

  /* Debugging info - show what event we just recorded */
  if (debug >= PrintEvent)
    fprintf(stderr, "---EVENT: server=%s val=%d thresh=%d lvl=%d, time is %d/%d %d:%d\n",
	  wtmps[i].servername, wtmps[i].stat.inuse, this_threshold, level,
	  eventlist[i].mon, eventlist[i].day, eventlist[i].hour,
	  eventlist[i].min);
}

/*+
** Write all the EVENT structures to the named file for netmon to peruse
**/
dumpmsgs()
{
  int byteswritten, fd;

  /* Open the eventfile with mode -rw-rw-r-- so netmon can read it */
  fd = open(datafile, O_WRONLY | O_CREAT | O_TRUNC, 0664);
  if (fd < 0) {
      fprintf(stderr, "dumpmsgs(): couldn't open eventfile %s", datafile);
      perror("");
      exit(1);
  }

  /* Now just dump the whole big list of EVENT structures using raw I/O */
  byteswritten = write(fd, eventlist, numwtmps * sizeof(EVENT));
  if (byteswritten != (numwtmps * sizeof(EVENT))) {
      /*
      ** Be naive about this - yes, I know the program could try again
      ** where the write() left off, but... this is so much simpler,
      ** and it hasn't failed yet
      */
      fprintf(stderr, "dumpmsgs(): couldn't dump to eventfile %s", datafile);
      perror("");
      exit(1);
  }
  close(fd);
}


/*
** Check if the current entry is a reboot.
*/
int isreboot(st)
  STAT *st;
{
  if (! lineq("~"))
    return 0;
  if (! nameq("")) {
      st->warnings++;
      if (debug < Warn)
	return 0;
      fprintf(stderr, "\nWarning: artificial (non-reboot) entry in wtmp:\n");
      printent(stderr, &buf);
      fprintf(stderr, "\nEntry ignored.\n");
      return 0;
  }

  st->numseen = 0;
  return 1;
}


/*
** Check if the current entry is a new login, and make sure
** that this isn't a case of the terminal server missing
** acknowledgements and seeming to log someone in
** several times.
** I also ignore any entry where someone new logs
** in on an already used line from a different host
** This case happens mostly because of the terminal server
** missing acknowledgements somewhere.
** Furthermore, this routine deals with the case where
** you get interleaved logins and logouts on one tty line.
** For example:
**	login  on tty01 by F. Bar from any.where.edu at 10:30
**	login  on tty01 by F. Bar from any.where.edu at 10:31
**	login  on tty01 by F. Bar from any.where.edu at 10:32
**	logout on tty01 by F. Bar from any.where.edu at 10:33
**	login  on tty01 by F. Bar from any.where.edu at 10:34
**      etc. etc.
** This situation could happen if a user logged on for only
** a minute or two (say to check his/her email) and then logged
** off while at the same time the terminal server misses
** acknowledgements.  This makes it look like F. Bar has logged 
** in and out several times in quick succession.
** (Even worse, if in the sequence above you leave out the "etc. etc."
** part, it would look like F. Bar logged in for 3 minutes, then
** logged out, then logged back in - but you would never see him
** logout until the next reboot - thus tty01 would be reported
** constantly in use till the next reboot, but this is a lie.)
** The final thing this routine does is to update the "seen[]"
** data structure.
*/
int islogin(st)
  STAT *st;
{
  int i;

  /* If the name field is blank, this surely isn't a login */
  if (nameq(""))
    return 0;

  /*
  ** Search through the "seen[]" data structure for a similar
  ** record
  */
  for (i=0; i < st->numseen; i++) {
    if (! lineq(st->seen[i].ut_line))
      continue;
    if (! st->isinuse[i]) {
	/*
	** This record in the "seen[]" data structure is
	** marked as being unused - check to see if we have
	** interleaved login/logouts or something like that
	*/
	if ((buf.ut_time - st->seen[i].ut_time) > THRESHOLD
	    || ! nameq(st->seen[i].ut_name)
	    || ! hosteq(st->seen[i].ut_host) )
	  /*
	  ** The current login entry in "buf" is perfectly valid
	  ** So now change to this record of the "seen[]" data structure
	  ** to being marked as in use, and return
	  */
	  break;
	else {
	    /*
	    ** The same guy just logged out a few seconds ago
	    ** Suspect multiple unacknowledged interleaved login/logout
	    ** signals from the modem server
	    ** Thus, this is most likely not a valid login entry, but
	    ** an artifact of the terminal server acknowledgement system
	    */
	    st->warnings++;
	    if (debug < Warn)
	      return 0;
	    fprintf(stderr, "\nAmbigous login/logout!\nLast logout:\t");
	    printent(stderr, &(st->seen[i]));
	    fprintf(stderr, "\nNew login:\t");
	    printent(stderr, &buf);
	    fprintf(stderr, "\nNew login ignored.\n");
	    return 0;
	}
    }

    /*
    ** We now know this record in the "seen[]" data structure
    ** is currently marked as in use
    ** If the name and host from this record match the name and
    ** host listed in "buf", then this is probably a repeat login
    ** entry which arises from the terminal server missing acknowledgements
    ** For example:
    **	  Login by John Doe from any.place.edu on tty01 at 10:30
    **	  Login by John Doe from any.place.edu on tty01 at 10:32
    **	  Login by John Doe from any.place.edu on tty01 at 10:33
    ** The last 2 entries above would exercise this code here,
    ** because they are repeat logins by the same person on the same line
    */
    if (nameq(st->seen[i].ut_name) && hosteq(st->seen[i].ut_host))
      return 0;

    /*
    ** This entry shows someone (with a different name or host)
    ** logging in on a line that is currently marked as in use
    ** This could arise for any number of reasons, but it seems
    ** safe to ignore the new entry (in "buf") and assume that it
    ** is not a valid login.
    ** This code would be reached on the second entry below:
    ** 	Login by Jane Smith from her.place.edu at 10:30 on tty01
    **  Login by Mr. Doe    from his.place.edu at 10:32 on tty01
    ** In this example, Mr. Doe's wtmp entry would be ignored.
    */
    st->warnings++;
    if (debug < Warn)
      return 0;
    fprintf(stderr, "\nConflict in islogin()!\nIs:\t");
    printent(stderr, &(st->seen[i]));
    fprintf(stderr, "\nNew:\t");
    printent(stderr, &buf);
    fprintf(stderr, "\nNew entry ignored.\n");
    return 0;
  }

  /*
  ** Ok, the current wtmp entry (in "buf") is a valid login.
  ** Better update the "seen[]" data structure.
  ** Also, if we've never seen this tty line before, add it
  ** on at the end and keep track of the increase of size of "seen[]"
  ** Finally, return <TRUE>
  */
  cpent(&(st->seen[i]), &buf);
  st->isinuse[i] = 1;
  if (i == st->numseen)
    st->numseen++;
  if (st->numseen < MAX_LINES)
    return 1;

  /* Woops - overflow in the seen[] structure! */
  fprintf(stderr, "islogin(): too many modem lines - increase MAX_LINES\n");
  exit (1);
}


/*
** Check if the current wtmp entry (in "buf") is a valid
** logout, and make sure that there was actually someone
** logged in on this line before (if this is a logout.)
*/
int islogout(st)
  STAT *st;
{
  int i;

  /* If the name field isn't blank, this surely isn't a logout entry... */
  if (! nameq(""))
    return 0;

  /*
  ** Search through the "seen[]" data structure for a record similar
  ** to the one in "buf"
  */
  for (i=0; i < st->numseen; i++) {
    if (! lineq(st->seen[i].ut_line))
      continue;

    /*
    ** If there is currently noone logged onto the line listed in "buf",
    ** then this surely isn't a valid logout.
    */
    if (! st->isinuse[i])
      return 0;

    /*
    ** If the tty line and host of this record of "seen[]" match the
    ** wtmp entry in "buf", and this record is listed as being in use,
    ** then "buf" must contain a perfectly valid logout entry
    */
    if (hosteq(st->seen[i].ut_host)) {
	st->isinuse[i] = 0;
	return 1;
    }

    /*
    ** The host field in the logout entry for this line is
    ** different from the host field in the login entry for this line.
    ** This case could arise as follows:
    **	Login  by John Doe     from his.place.edu on tty01 at 10:30
    **  Logout by [don't know] from her.place.edu on tty01 at 10:37
    ** Thus, the logout from her.place.edu would be ignored.
    */
    st->warnings++;
    if (debug < Warn)
      return 0;
    fprintf(stderr, "\nConflict!\nIs:\t");
    printent(stderr, &(st->seen[i]));
    fprintf(stderr, "\nNew:\t");
    printent(stderr, &buf);
    fprintf(stderr, "\nNew entry ignored.\n");
    return 0;
  }

  /*
  ** We've never seen this tty line before.
  ** So why are we getting a logout entry on it?
  ** Either things are really screwed up, or we're
  ** at the start of the wtmp file before a reboot;
  ** so we might have missed the login entry in this wtmp file.
  */
  st->warnings++;
  if (debug < Warn)
    return 0;
  fprintf(stderr, "\nLogout entry on a tty line without any logins!\n");
  printent(stderr, &buf);
  if (st->seenreboot)
    fprintf(stderr, "\nStrange, I didn't miss any reboots - ignored.\n");
  else
    fprintf(stderr, "\nProbably because I missed a reboot - Entry ignored.\n");
  return 0;
}
      

/*
** Copy the wtmp entry pointed to by "b" into the wtmp entry pointed to by "a"
*/
int cpent(a, b)
  struct utmp *a, *b;
{
  strncpy(a->ut_line, b->ut_line, LMAX);
  strncpy(a->ut_name, b->ut_name, NMAX);
  strncpy(a->ut_host, b->ut_host, HMAX);
  a->ut_time = b->ut_time;
}

/*
** Read an entry from the wtmp file and put it in the global wtmp buffer "buf"
** Returns 1 upon success, 0 upon end-of-file, and -1 upon failure
*/
int getent()
{
  int bytesread;

  bytesread = read(fd, &buf, sizeof(buf));
  if (bytesread == sizeof(buf))
    return 1;

  if (bytesread == 0)
    return 0;

  /* We only read part of a wtmp buffer - yuck! the file must be corrupted */
  return -1;
}

/*
** Print out the wtmp entry pointed to by "a" onto the stream "f"
** This is mostly here for debugging purposes
*/
int printent(f, a)
  FILE *f;
  struct utmp *a;
{
  char *ct;

  ct = ctime(&((time_t) a->ut_time));
  fprintf(f, "%-*.*s %-*.*s %-*.*s %10.10s %5.5s",
	  NMAX, NMAX, a->ut_name,
	  LMAX, LMAX, a->ut_line,
	  HMAX, HMAX, a->ut_host,
	  ct, ct+11);
}
