/*
**	$Header: /nocol/src/pingmon/RCS/pingmon.c,v 2.0 1992/06/18 21:18:16 aggarwal Exp $
*/

/*+
**
** DESCRIPTION
**
**	Runs the "ping" program to monitor a site. Marks a site as UP
**	DOWN or UNKNOWN. It writes directly to the 'pingoutput' file
**	instead of to a temporary file. Uses raw i/o calls.
**
**	This program opens the 'data file' supplied by the user and then
**	creates the 'pingoutput' file. It then directly reads and writes
**	from this file. On recieving a HUP signal, it rescans the 'data
**	file' supplied by the user and continues.
**
**
** AUTHOR
**	Vikas Aggarwal, October 16, 1989
**
*/

/*  Copyright 1992 JvNCnet

 Permission to use, copy, modify and distribute this software and its
 documentation for any purpose is hereby granted without fee, provided that
 the above copyright notice appear in all copies and that both that copyright
 notice and this permission notice appear in supporting documentation, and
 that the name of JvNCnet not be used in advertising or publicity pertaining
 to distribution of the software without specific, written prior permission.
 JvNCnet makes no representations about the suitability of this software for
 any purpose.  It is provided "as is" without express or implied warranty.

 JvNCnet DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL JvNCnet
 BE LIABLE FOR ANY DAMAGES WHATSOEVER, INCLUDING DAMAGES RESULTING FROM LOSS
 OF USE, DATA OR PROFITS, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/


/*
 * MODIFICATIONS
 *
 * $Log: pingmon.c,v $
 * Revision 2.0  1992/06/18  21:18:16  aggarwal
 * Checked sleep logic. Cleaned up for release.
 *
 * Revision 1.11  1992/05/13  16:13:59  aggarwal
 * Altered for automatic output datafile, etc depending on its own
 * runtime name.
 *
 * Revision 1.6  90/06/04  14:27:18  aggarwal
 * Moved get_ppid() to the poll_sites module. This was done so that
 * pingmon rescans the watchdog.pid file just before sending out the
 * SIGUSR1 signal (didn't make sense to send out a signal to a process
 * which might have restarted since the pid was first read).
 * 
 * Revision 1.5  90/04/16  18:12:21  aggarwal
 * Added stuff so that the config file can have comments.
 * Also can now send a SIGUSR1 to the program whose pid
 * is stored in 'program.pid' file.
 * 
 * Revision 1.4  90/03/15  15:27:22  aggarwal
 * Closed the fdout if it was not zero (if getting a SIGHUP signal, then
 * it should close the open output file before reopening it).
 * 
 * Revision 1.3  90/03/09  12:52:02  aggarwal
 * Major changes so that it can read in SIGNAL progname to send a
 * signal to another process when a site goes to CRITICAL the first
 * time.
 * 
 * Revision 1.1  89/12/13  14:16:09  aggarwal
 * Initial revision
 * 
 */ 
#ifndef lint
 static char rcsid[] = "$RCSfile: pingmon.c,v $ $Revision: 2.0 $ $Date: 1992/06/18 21:18:16 $" ;
#endif

#include "pingmon.h"			/* program specific defines	*/

#include <strings.h>			/* For strcat() definitions	*/
#include <sys/file.h>
#include <signal.h>			/* For signal numbers		*/
#include <setjmp.h>


/*+ 
** Global variables
**/

jmp_buf	env;				/* For longjmp and setjmp 	*/

main (ac, av)
     int ac;
     char **av;
{
    extern char	*prognm ;			/* in pingmon.h	 */
    extern int debug ;
    extern char *optarg;
    extern int optind;
    int	fdout = 0;			/* File desc for output data file */
    register int c ;
    time_t starttm, polltime ;
    void *done(), *restart();			/* For the signals	*/

    prognm = av[0] ;				/* Save the program name */

#ifdef SENDER
    sender = SENDER ;
#else						/* delete the directory name */
    if ((sender = (char *)strrchr (prognm , '/')) == NULL)
      sender = prognm ;				/* no path in program name */
    else
      sender++ ;				/* skip leading '/' */
#endif

    /* the output data filename */    
    sprintf(datafile,  "%s/%s%s\0",
	    DATADIR,  sender, PINGOUTPUTEXT); 
    sprintf(ipnodes, "%s\0", NODESFILE) ;	/* the name of IP hosts file */

    ping = PING ;

    while ((c = getopt(ac, av, "do:p:")) != EOF)
      switch (c)
      {
       case 'd':
	  debug++ ;
	  break ;
       case 'o':				/* output datafile */
	  sprintf(datafile, "%s\0", optarg) ;
	  break ;
       case 'p':				/* ping command to use */
	  ping = (char *)malloc (strlen(optarg) + 1) ;
	  strcpy(ping, optarg);
	  break ;
       case '?':
       default:
	  fprintf (stderr, "%s: Unknown flag: %c\n", prognm, optarg);
	  help() ;
	  goto Cleanup ;
      }

    switch ( ac - optind )
    {
     case 0:					/* default input file */
	break ;
     case 1:
	sprintf(ipnodes, "%s\0", av[optind]) ;
	break ;
     default:
	fprintf (stderr, "%s Error: Too many 'hosts' files\n\n", prognm);
	help() ;
	goto Cleanup;
    }

    if (access(ping, F_OK | X_OK) != 0)	/* check if PING is executable	*/
    {
	perror (ping);
	done();
    }

    if (standalone(prognm) == -1) 	/* Kill prev running process	*/
    {
	fprintf(stderr, "%s: Error in standalone...exiting\n", prognm);
	exit (1);
    }

    if (debug)
      fprintf(stderr,
	      "(debug) %s: PING= '%s', NODESFILE= '%s', DATAFILE= '%s'\n",
	      prognm, ping, ipnodes, datafile) ;
    
    setjmp (env);			/* Save the environment in case	*/

    signal (SIGQUIT, done);		/* Delete pid file while dying	*/
    signal (SIGTERM, done);
    signal (SIGINT, done);
    signal (SIGHUP, restart);		/* Re-read the ipnodes file	*/

    if (getenv(DEBUG_FLG) != NULL)      /* see ifdebug mode desired     */
      debug = debug ^ 1 ;

    umask (002);			/* write access to the group	*/
    if (fdout != 0)			/* Say, recovering from longjmp	*/
      close(fdout);

    if ( (fdout = open(datafile, O_RDWR | O_CREAT | O_TRUNC, 0664)) < 0)
    {
	fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
	perror (datafile);
	goto Cleanup ;
    }

    if (init_sites(fdout, ipnodes) == -1 )
      goto Cleanup ;

    /* poll_sites makes one complete pass over the list of nodes */
    while (1)			      		/* forever */
    {
	starttm = time((time_t *)NULL) ;	/* time started this cycle */
	if (poll_sites(fdout) == -1)		/* Polling error */
	  break ;

	if ( (polltime = time((time_t *)NULL) - starttm) < POLLINTERVAL)
	  sleep ((unsigned)(POLLINTERVAL - polltime));
    }
    /* HERE ONLY IF ERROR */
 Cleanup:
    done();

}	/***************** End of main *******************/
    
/*+ 
** FUNCTION:
** 	Brief usage
**/
help ()
{
    static char usage[] = " [-d (debug)]  [-o <output file>] [-p <ping command>] [ipnodes file]\n";

    fprintf(stderr, "\nUSAGE: %s %s\n\n", prognm, usage);
    fprintf(stderr,"\tThis program finds out the status of sites by ");
    fprintf(stderr,"sending out an ICMP ping request and writing\n");
    fprintf(stderr,"\tthe output into the file %s.\n", datafile);

    fprintf(stderr,"\tBy default, the list of nodes to monitor is %s\n", 
	    ipnodes);
    fprintf(stderr,"\tbut can be changed on the command line.\n");

    fprintf(stderr,"\tThe 'node-file' format is:\n");
    fprintf(stderr,"\t\t <node> <ip-address>  [TEST]\n");
    fprintf(stderr,"\t\t SIGNAL <program name>\n\n");
    fprintf(stderr,"\tThe program writes its pid in %s.pid and \n",prognm);
    fprintf(stderr,"\tif a new process starts, it kills the earlier one.\n");
    fprintf(stderr,"\tIf a HUP signal is sent to the process, it rescans ");
    fprintf(stderr,"the nodes file. \n");
    fprintf(stderr,"\tThe SIGNAL keyword sends a SIGUSR1 signal to the pid\n");
    fprintf(stderr,"\tstored in the <program name>.pid whenever a site goes\n");
    fprintf(stderr,"\tCRITICAL the first time\n\n");
    fprintf(stderr,"\nTo run in debug mode, do a 'setenv %s 1'\n",DEBUG_FLG);
    return (1);
}


/*
** FUNCTION
**
**	init_sites
**
**	This function writes to the LSTFILE. All sites in the NODESFILE
**	file are set to UNKNOWN status.
**
**	Careful while using 'localtime': the calue of the month varies from
**	0 - 11 and hence has to be incremented for the correct month.
*/

init_sites(fdout, ipnodes)
     int fdout ;			/* Output file descriptor	*/
     char *ipnodes;			/* Filename of the ipnodes file	*/
{
    extern char sigtoprog[];		/* In pingmon.h			*/
    extern char *sender ;
    FILE *p_nodes ;
    EVENT v;				/* Defined in NOCOL.H		*/
    char record[MAXLINE];
    struct tm *loctime ;
    time_t locclock ;			/* Careful, don't use 'long'	*/

    if ((p_nodes = fopen(ipnodes, "r")) == NULL)
    {
	fprintf(stderr, "%s error (init_sites) ", prognm) ;
	perror (ipnodes);
	return (-1);
    }

    /*
     * Fill in the static data stuff
     */
    bzero (&v, sizeof(v)) ;

    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;	v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;		v.min = loctime->tm_min;

    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.name, VARNM, sizeof (v.var.name) - 1);
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.threshold = PING_THRES ;		/* num of packets lost	*/

    v.nocop = SETF_UPDOUN (v.nocop, n_UNKNOWN);	/* Set all to UNKNOWN	*/
    v.severity = E_INFO ;

    while(fgetline(p_nodes,record,MAXLINE) > 0 ) 
    {
#ifdef IP
	u_long inet_addr() ;
#endif
	char w1[MAXLINE], w2[MAXLINE], w3[MAXLINE];	/* Confg words	*/
	int rc;						/* return code	*/

	v.nocop = 0 ;				/* Init options to zero	*/
	*w1 = *w2 = *w3 = NULL ;
	rc = sscanf(record,"%s %s %s", w1, w2, w3);
	if (rc == 0 || *w1 == NULL || *w1 == '#')  /* Comment or blank 	*/
	  continue;

	if (strcmp(w1, "signal") == 0 || strcmp(w1, "SIGNAL") == 0)
	{
	    strcpy(sigtoprog, w2) ;		/* Prog to get signal	*/
	    strcat(sigtoprog, ".pid");		/* append suffix	*/
	    continue ;
	}

	strncpy(v.site.name, w1, sizeof(v.site.name));
	strncpy(v.site.addr, w2, sizeof(v.site.addr));	/* no checks */

#ifdef	IP		/* For IP, check the IP address */

	if (inet_addr(w2) == -1)	/* bad address */
	{
	    fprintf(stderr,
		    "(%s): Error in address '%s' for site '%s', ignoring\n",
		    prognm, w2, w1);
	    continue ;
	}
#endif
	if (*w3 != NULL)			/* Some other keyword	*/
	{
	    if (strcmp(w3, "test") == 0 || strcmp(w3, "TEST") == 0)
	      v.nocop = v.nocop | n_TEST ;
	    else
	      fprintf(stderr, "%s: Ignoring unknown keyword- %s\n", prognm,w3);
	}

	if (write (fdout, (char *)&v, sizeof(v)) != sizeof(v))
	{
	    fprintf(stderr, "%s (write): %s\n", prognm, sys_errlist[errno]);
	    exit (1) ;
	}

    }					/* end: while			*/
    fclose (p_nodes);			/* Not needed any more		*/    
    return(1);				/* All OK			*/
}		/* end:  init_sites()		*/


/*+ 		restart
** FUNCTION:
**
** 	If a SIGHUP is sent, it rescans the ipnodes file and starts all
** over without exiting from the program.
**/

void *
  restart ()
{
    longjmp (env, 1);
}

/*+ 		done
** FUNCTION:
**
** Delete the PID and the data file.
** Called just before exiting. Only if error. Exits since it can be called
** on recieving a signal.
**/
void *
  done ()
{
    char pidfile[MAXLINE] ;

    fprintf (stderr, "%s: removing data, pid file.... ", prognm);
    sprintf (pidfile, "%s.pid\0", prognm);
    unlink (pidfile);				/* remove the PID file	*/
    unlink (datafile);				/* delete the data file */
    fprintf (stderr, "Done\n");
    exit (1);
}
