/*
 * $Header: /nocol/src/nsmon/RCS/poll_sites.c,v 1.1 1992/06/11 05:06:12 aggarwal Exp $
 *
 * poll_sites.c for Nameserver monitor (nsmon)
 * 	makes one pass over the entire data file
 * 	adapted from poll_sites.c of pingmon 1.15
 *
 *
 * $Log: poll_sites.c,v $
 * Revision 1.1  1992/06/11  05:06:12  aggarwal
 * Initial revision
 *
 */

/* Copyright 1992 JvNCnet, Princeton */

/*+ 
 * INCLUDE FILES
 */

#include "nocol.h"			/* common structures */
#include "nsmon.h"			/* program specific defines */

#include <signal.h>
#include <sys/file.h>
#include <arpa/nameser.h>


/*
 * Macro to escalate the severity of a site.
 * Change E_CRITICAL to E_ERROR if you don't want this program to put
 * events in the critical state.
 */
#define ESC_SEVERITY(sev) ((sev == E_CRITICAL)? E_CRITICAL : (sev - 1))

/*
 * The level trasition at which a signal is sent to the 'watchdog' program.
 * This is called before increasing the severity in the structure, so if
 * you want that a signal should be sent when it changes from WARN -> ERROR,
 * then, check for sev = WARN. Don't set true if 'sev' is CRIT since the
 * state will not transition anymore and you don't want to send a signal
 * in each pass- only when it first changed state.
 */
#define AT_SIGNAL_LEVEL(sev)  ((sev <= E_WARNING && sev != E_CRITICAL) ? 1 : 0)


/* #defines for finish_status */
#define REACHED_EOF 1
#define READ_ERROR 2

poll_sites(fdout)
     int fdout;				/* Descriptors to open files	*/
{
  EVENT v;			    	/* described in nocol.h		*/
  struct tm *ltime ;    
  time_t locclock ;			/* careful: don't use 'long'	*/
  long status;		       		/* site status			*/
  int bufsize;				/* for reading in file */
  int sigpid;				/* PID of program to get signal	*/
  char querydata[MAXLINE];
  struct sockaddr_in sockad;

  if ( lseek(fdout, (off_t)0, L_SET) == -1) { /* rewind the file	*/
    perror (prognm);
    return (-1);
  }

  /* 
   * until end of the file or erroneous data... one entire pass
   */

  while ( (bufsize = read (fdout, (char *)&v, sizeof(v))) == sizeof(v) ) {
    if (QUERYTYPE == T_SOA)
      strcpy(querydata, QUERYDATA);
    else {
      sockad.sin_addr.s_addr = inet_addr(v.site.addr);
      sprintf(querydata,"%d.%d.%d.%d.in-addr.arpa",
        sockad.sin_addr.S_un.S_un_b.s_b4,
        sockad.sin_addr.S_un.S_un_b.s_b3,
        sockad.sin_addr.S_un.S_un_b.s_b2,
        sockad.sin_addr.S_un.S_un_b.s_b1);
    }
    switch(nsmon(v.site.addr, querydata, QUERYCLASS, QUERYTYPE,
        QUERYTIMEOUT, QUERY_AA_ONLY, debug)) {
      case ALL_OK: case NOT_AUTHORITATIVE:
        status = 1; break;			/* '1' for all OK */
      case ERROR: default:
        status = 0; break;
    } /* switch */
    locclock = time((time_t *)NULL);	/* Grab the time	*/
    ltime = localtime ((long *)&locclock);
    /* Now determine what data to modify in the structure..	*/

    switch (status) {
      case 1:
	v.var.value = 1 ;		/* change the old status	*/
	v.nocop = SETF_UPDOUN(v.nocop, n_UP) ;
	v.severity = E_INFO ;
	v.mon = ltime->tm_mon + 1;	v.day = ltime->tm_mday;
	v.hour = ltime->tm_hour;	v.min = ltime->tm_min;
	break;
	       
      case 0:				/* escalate the severity	*/
        if (!(v.nocop & n_DOWN))	/* recent change of state	*/
        {
          v.mon = ltime->tm_mon + 1 ;
          v.day = ltime->tm_mday;
          v.hour = ltime->tm_hour;
          v.min = ltime->tm_min;
        }
        if (AT_SIGNAL_LEVEL(v.severity))
          if ((sigpid = get_ppid(sigtoprog)) > 0)
          {
            kill(sigpid, SIGUSR1);  /*..send out signal to watchdog */
            if (debug)
              fprintf(stderr, "%s: sent SIGUSR1 to %d\n", prognm, sigpid);
          }

	/* escalate using defined macro	*/        
        v.severity = ESC_SEVERITY(v.severity) ;
        v.var.value = 0 ;		/* store current state	*/
	v.nocop = SETF_UPDOUN (v.nocop, n_DOWN);
        break ;
    }					/* end:  switch		*/
     
    /* Now rewind to start of present record and write to the file	*/
     
    lseek(fdout,-(off_t)sizeof(v), L_INCR);
    write (fdout, (char *)&v, sizeof(v));
     
   }		/* end of:    while (read..)	*/
    
  /**** Now determine why we broke out of the above loop *****/
    
  if (bufsize == 0)			/* reached end of the file	*/
    return (1);
  else {				/* error in output data file 	*/
    fprintf (stderr, "%s: Insufficient read data in output file", prognm);
    return (-1);
  }
}	/* end of:  poll_sites		*/

/*+ 		get_ppid()
** FUNCTION:
** 	Get the pid of the process which recieves the SIGUSR1 signal.
** Simply opens the filename passed to it and reads the pid 
** from the file. Returns a zero if it gets an invalid value else it
** returns the pid of the process.
**/
get_ppid(program)
     char *program;
{
    extern int debug ;
    int fd, pid ;
    char buffer[MAXLINE];

    if (program == NULL)
    {
      if (debug)
	fprintf(stderr, "(debug) %s: (get_ppid) no program name supplied\n",
		prognm);
      return(0);
    }
    if ((fd = open(program, O_RDONLY)) < 0)
      return(0);

    if (read(fd, buffer, MAXLINE) < 0)
    {
	if (debug)
	  fprintf(stderr, 
		  "(debug) %s: (get_ppid) 'read' %s- %s\n", 
		  prognm, program, sys_errlist[errno]);
	pid = 0;
    }
    else
      sscanf(buffer, "%d", &pid);

    close(fd);
    return(pid);
}			/* end: get_ppid */
