/*
 * $Header: /nocol/src/nsmon/RCS/main.c,v 1.2 1992/06/14 23:38:16 aggarwal Exp $
 *
 * DESCRIPTION: name-server monitor
 *   This program opens the data file supplied by the user  and then
 *   creates a 'nsmon-output' file for use with netmon.  It then
 *   directly reads and writes from this file.  If it gets SIGHUP, it
 *   rescans the data file before continuing.
 *
 *   Derived in part from the pingmon code, RCS v1.11
 *
 *   S. Spencer Sun, Princeton Univ. / JvNCnet, June 1992
 *
 *
 * MODIFICATIONS
 *
 * $Log: main.c,v $
 * Revision 1.2  1992/06/14  23:38:16  aggarwal
 * Had to increment 'sender' after 'strrchr'
 *
 * Revision 1.1  1992/06/11  05:02:00  aggarwal
 * Initial revision
 *
 *
 */

/*  Copyright 1992 JvNCnet

 Permission to use, copy, modify and distribute this software and its
 documentation for any purpose is hereby granted without fee, provided that
 the above copyright notice appear in all copies and that both that copyright
 notice and this permission notice appear in supporting documentation, and
 that the name of JvNCnet not be used in advertising or publicity pertaining
 to distribution of the software without specific, written prior permission.
 JvNCnet makes no representations about the suitability of this software for
 any purpose.  It is provided "as is" without express or implied warranty.

 JvNCnet DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL JvNCnet
 BE LIABLE FOR ANY DAMAGES WHATSOEVER, INCLUDING DAMAGES RESULTING FROM LOSS
 OF USE, DATA OR PROFITS, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/*  */

#ifndef lint
 static char rcsid[] = "$RCSfile: main.c,v $ $Revision: 1.2 $ $Date: 1992/06/14 23:38:16 $" ;
#endif

#include "nocol.h"			/*	common structures	*/

#define GLOBALS				/* for global variables */
#include "nsmon.h"			/* program specific defines	*/
#undef GLOBALS

#include <strings.h>			/* For strcat() definitions	*/
#include <sys/file.h>
#include <signal.h>			/* For signal numbers		*/
#include <setjmp.h>


jmp_buf	env;				/* For longjmp and setjmp 	*/

main (ac, av)
     int ac;
     char **av;
{
    extern char	*prognm ;			/* in pingmon.h	 */
    extern int debug ;
    extern char *optarg;
    extern int optind;
    int	fdout = 0;			/* File desc for output data file */
    register int c ;
    time_t starttm, polltime ;
    void *done(), *restart();			/* For the signals	*/

    prognm = av[0] ;				/* Save the program name */

#ifdef SENDER
    sender = SENDER ;
#else						/* delete the directory name */
    if ((sender = (char *)strrchr (prognm , '/')) == NULL)
      sender = prognm ;				/* no path in program name */
    else
      sender++ ;				/* skip leading '/' */
#endif

    /* the output data filename */    
    sprintf(datafile,  "%s/%s%s\0",
	    DATADIR,  sender, NSMONOUTPUTEXT); 
    sprintf(ipnodes, "%s\0", NODESFILE) ;	/* the name of config file */

    while ((c = getopt(ac, av, "do:")) != EOF)
      switch (c) {
        case 'd':
	  debug++ ;
	  break ;
        case 'o':				/* output datafile */
	  sprintf(datafile, "%s\0", optarg) ;
	  break ;
        case '?':
        default:
	  fprintf (stderr, "%s: Unknown flag: %c\n", prognm, optarg);
	  help() ;
	  goto Cleanup ;
      }

    switch (ac - optind) {
      case 0:					/* default input file */
	break ;
      case 1:
	sprintf(ipnodes, "%s\0", av[optind]) ;
	break ;
      default:
	fprintf (stderr, "%s Error: Too many 'hosts' files\n\n", prognm);
	help() ;
	goto Cleanup;
    }

    if (standalone(prognm) == -1) { 	/* Kill prev running process	*/
      fprintf(stderr, "%s: Error in standalone...exiting\n", prognm);
      exit (1);
    }

    if (debug)
      fprintf(stderr,
	"(debug) %s: NODESFILE= '%s', DATAFILE= '%s'\n",
	  prognm, ipnodes, datafile) ;
    
    setjmp (env);			/* Save the environment in case	*/

    signal (SIGQUIT, done);		/* Delete pid file while dying	*/
    signal (SIGTERM, done);
    signal (SIGINT, done);
    signal (SIGHUP, restart);		/* Re-read the ipnodes file	*/

    if (getenv(DEBUG_FLG) != NULL)      /* see ifdebug mode desired     */
      debug = debug ^ 1 ;

    umask (002);			/* write access to the group	*/
    if (fdout != 0)			/* Say, recovering from longjmp	*/
      close(fdout);

    if ( (fdout = open(datafile, O_RDWR | O_CREAT | O_TRUNC, 0664)) < 0) {
      fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
      perror (datafile);
      goto Cleanup ;
    }

    if (init_sites(fdout, ipnodes) == -1 )
      goto Cleanup ;

    /* poll_sites makes one complete pass over the list of nodes */
    while (1)			      		/* forever */
    {
	starttm = time((time_t *)NULL) ;	/* time started this cycle */
	if (poll_sites(fdout) == -1)		/* Polling error */
	  break ;

	if ( (polltime = time((time_t *)NULL) - starttm) < POLLINTERVAL)
	  sleep ((unsigned)(POLLINTERVAL - polltime));
    }

    /* HERE ONLY IF ERROR */
  Cleanup:
    done();

}	/***************** End of main *******************/
    
/*+ 
** FUNCTION:
** 	Brief usage
**/
help ()
{
    static char usage[] = " [-d (debug)] [-o <output file>] <ipnodes file>\n";

    fprintf(stderr, "\nUSAGE: %s %s\n\n", prognm, usage);
    fprintf(stderr, "\tThis program sends queries to nameservers to verify\n");
    fprintf(stderr, "\ttheir operational status.\n");

    fprintf(stderr,"\tBy default, the list of nodes to monitor is %s\n", 
	    ipnodes);
    fprintf(stderr,"\tbut can be changed on the command line.\n");

    fprintf(stderr,"\tThe 'node-file' format is:\n");
    fprintf(stderr,"\t\t <node> <ip-address>  [TEST]\n");
    fprintf(stderr,"\t\t SIGNAL <program name>\n\n");
    fprintf(stderr,"\tThe program writes its pid in %s.pid and \n",prognm);
    fprintf(stderr,"\tif a new process starts, it kills the earlier one.\n");
    fprintf(stderr,"\tIf a HUP signal is sent to the process, it rescans ");
    fprintf(stderr,"the nodes file. \n");
    fprintf(stderr,"\tThe SIGNAL keyword sends a SIGUSR1 signal to the pid\n");
    fprintf(stderr,"\tstored in the <program name>.pid whenever a nameserver\n");
    fprintf(stderr,"\tgoes CRITICAL the first time\n\n");
    return (1);
}


/*
** FUNCTION
**
**	init_sites
**
**	This function writes to the LSTFILE. All sites in the NODESFILE
**	file are set to UNINIT status.
**
**	Careful while using 'localtime': the calue of the month varies from
**	0 - 11 and hence has to be incremented for the correct month.
*/

init_sites(fdout, ipnodes)
     int fdout ;			/* Output file descriptor	*/
     char *ipnodes;			/* Filename of the ipnodes file	*/
{
    extern char sigtoprog[];		/* In pingmon.h			*/
    extern char *sender ;
    FILE *p_nodes ;
    EVENT v;				/* Defined in NOCOL.H		*/
    char record[MAXLINE];
    struct tm *loctime ;
    time_t locclock ;			/* Careful, don't use 'long'	*/

    if ((p_nodes = fopen(ipnodes, "r")) == NULL)
    {
	fprintf(stderr, "%s error (init_sites) ", prognm) ;
	perror (ipnodes);
	return (-1);
    }

    /*
     * Fill in the static data stuff
     */
    bzero (&v, sizeof(v)) ;
    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;	v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;		v.min = loctime->tm_min;

    /*
     * in the following strncpy's, the NULL is already appended because
     * of the bzero, and copying one less than size of the arrays.
     */
    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.name, VARNM, sizeof (v.var.name) - 1);
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.value = 0 ; v.var.threshold = 0 ;	/* threshold not used */
    v.nocop = SETF_UPDOUN (v.nocop, n_UNKNOWN);	/* Set all to UNKNOWN	*/
    v.severity = E_INFO ;

    while(fgetline(p_nodes,record,MAXLINE) > 0 ) 
    {
	u_long inet_addr() ;
	char w1[MAXLINE], w2[MAXLINE], w3[MAXLINE];	/* Confg words	*/
	int rc;						/* return code	*/

	v.nocop = 0 ;				/* Init options to zero	*/
	*w1 = *w2 = *w3 = NULL ;
	rc = sscanf(record,"%s %s %s", w1, w2, w3);
	if (rc == 0 || *w1 == NULL || *w1 == '#')  /* Comment or blank 	*/
	  continue;

	if (strcmp(w1, "signal") == 0 || strcmp(w1, "SIGNAL") == 0)
	{
	    strcpy(sigtoprog, w2) ;		/* Prog to get signal	*/
	    strcat(sigtoprog, ".pid");		/* append suffix	*/
	    continue ;
	}

	strncpy(v.site.name, w1, sizeof(v.site.name));
	strncpy(v.site.addr, w2, sizeof(v.site.addr));	/* no checks */

	if (inet_addr(w2) == -1)	/* bad address */
	{
	    fprintf(stderr,
		    "(%s): Error in address '%s' for site '%s', ignoring\n",
		    prognm, w2, w1);
	    continue ;
	}

	if (*w3 != NULL)			/* Some other keyword	*/
	{
	    if (strcmp(w3, "test") == 0 || strcmp(w3, "TEST") == 0)
	      v.nocop = v.nocop | n_TEST ;
	    else
	      fprintf(stderr, "%s: Ignoring unknown keyword- %s\n", prognm,w3);
	}

	write (fdout, (char *)&v, sizeof(v)) ;
    }					/* end: while			*/
    fclose (p_nodes);			/* Not needed any more		*/    
    return(1);				/* All OK			*/
}		/* end:  init_sites()		*/


/*+ 		restart
** FUNCTION:
**
** 	If a SIGHUP is sent, it rescans the ipnodes file and starts all
** over without exiting from the program.
**/

void  *restart ()
{
    longjmp (env, 1);
}


/*+ 		done
** FUNCTION:
**
** Delete the PID and the data file.
** Called just before exiting. Only if error. Exits since it can be called
** on recieving a signal.
**/
void *
  done ()
{
    char pidfile[MAXLINE] ;

    fprintf (stderr, "%s: removing data, pid file.... ", prognm);
    sprintf (pidfile, "%s.pid\0", prognm);
    unlink (pidfile);				/* remove the PID file	*/
    unlink (datafile);				/* delete the data file */
    fprintf (stderr, "Done\n");
    exit (1);
}
