/*
 * $Header: /noc/network/netlog/src/RCS/utils.c,v 2.2 1992/06/01 19:43:08 aggarwal Exp $
 */

/* Copyright 1992 JvNCnet, Princeton University */

/*+ 
** Utility functions:
**
**	get_reply: prompt and get user's response.
**	copy_using_open_files_desc: copies over files using file-descriptors
**	fd_getline: similar to 'fgets', but works with an OPEN file desc
**	create_blank_file: creates a blank file with specified mode
**	lock_file, unlock_file: do file locking (netlog style)
**	get_file_id: get a file's UID, GID, MODE
**	tackon_dir: tack on dirname if filename doesn't have a '/'
**	
** 
**/

#ifndef lint
 static char rcsid[] = "$RCSfile: utils.c,v $ $Revision: 2.2 $ $Date: 1992/06/01 19:43:08 $" ;
#endif

#include "netlog.h"
#ifdef NFS
#include <fcntl.h>				/* for lock_file()	*/
#endif
#include <signal.h>

extern char *sys_errlist[] ;
extern int errno ;

/*+ 		get_reply
** FUNCTION:
** 	This function prompts a user and gets a response from him. The
** 'prompt' & 'default' are supplied to the function - it returns a
** pointer to a string which is the 'reply'.
**/

char *get_reply (prompt, deflt, response_type)
     char *prompt;
     char *deflt ;
     int  response_type ;		/* types defined in utils.h */
{
    static char reply[MAXLINE];
    char *r ;				/* temp pointer */
    int nodefault = 0, invalid = 0;

    reply[sizeof(reply) - 1] = '\0' ;		/* terminate with a NULL */

    if (deflt == NULL || *deflt == NULL)
      nodefault = 1 ;				/* No default value	*/

    printf("%s ", prompt);
    if (nodefault)
      printf(": ");
    else
      printf("[%s]: ", deflt) ;
    if (strlen(prompt) > 70)				/* long string	*/
      putchar('\n');

 again:
    r = (char *)reply ;
    gets(reply) ;
    if ( *reply == NULL )
    {
	if ( nodefault )
	{
	    printf("Invalid NULL response!! Enter again: ");
	    goto again;
	}
	else	/* default has been supplied */
	{
	    strncpy(reply, deflt, sizeof(reply) - 1) ;
	    return (reply) ;
	}
    }
    
    /*
     * Now check the user's response against the desired response_type
     */

    if (response_type == C_ANY)
      return (reply);

    invalid = 0;
    while (!invalid && *r)
    {
	register int ch_type = classify (*r++) ;

	if ( (response_type & ch_type) == 0)
	{
	    fprintf (stderr, "Invalid reply. ");
	    if ( isprint(*(--r)) )
		fprintf(stderr, "(character '%c' illegal)\n", *r);
	    fprintf(stderr, " Enter again: ");
	    invalid = 1 ;
	    break ;
	}
    }	/* end while */

    if (invalid)
      goto again;
    else
      return (reply);

}	/* end get_reply	*/

/*
 * Return class of character 'c' - one of C_ALPHA, C_DIGIT, ... Used
 * by get_reply()
 */
classify (c)
  int c ;
{

    if (isalpha(c))
      return (C_ALPHA) ;

    else if (isdigit(c))
      return (C_DIGIT) ;

    else if (isspace(c))
      return (C_SPACE) ;

    else if (ispunct(c))
      return (C_PUNCT);

    else
      return (C_ANY);

}	/* end classify */
      
/*  */

/*+ 		copy_using_open_files_desc
** FUNCTION:
** 	Copy over from one open file descriptor to the other.
**/
copy_using_open_files_desc(from_fd, to_fd)
     int from_fd, to_fd ;		/* open file descriptors */
{
    char *buf ;
    int nread, blksize ;
    struct stat	stb ;

    if (fstat(from_fd, &stb) == 0 && stb.st_blksize > 0)
      blksize = stb.st_blksize;
    else
      blksize = BUFSIZ;
    buf = (char *)malloc(blksize);
    if (buf == NULL) 
    {
	Fprintf("copy_from_open_files_desc: ERROR- ");
	perror("malloc") ;
	return (-1) ;
    }

    while ((nread = read(from_fd, buf, blksize)) > 0)
      if (write(to_fd, buf, nread) == -1)
      {
	  Fprintf("copy_from_open_files_desc: ERROR- ");
	  perror("write") ;
	  return(-1);
      }
    free(buf) ;
    return (0) ;		/* All okay */
}

/*+		fd_getline()
** FUNCTION:
**
**	Analogous to the stdio function 'fgets' but returns the number
**	read from an OPEN file descriptor instead of a file pointer.
**	Leaves the '\n' character and appends a null '\0' at end of string.
**/

fd_getline(fd, s, lim)
     int fd, lim ;			/* Max limit to read in */
     char *s;				/* ptr to storage area for string */
{
    register int i;		   	/* Temporary index		*/
    char c = 0 ;
    register char *pc = &c;		/* Fast pointer to read char	*/
    
    i=0;
    while (--lim > 0 && read(fd, pc, 1) == 1  && c != '\n') 
      s[i++] = c;
    if ( c == '\n' )
      s[i++] = c;

    s[i] = '\0';

    return(i);
}		/* end:   fd_getline() */


/*+ 		create_blank_file
** FUNCTION:
** 	Creates a blank file with the specified mask. Returns a -1 if
** cannot for any reason and does a perror. 
**
** This function is useful if one is doing a COPY since it creates a
** file with the required mode and copy preserves that mode.
**/
create_blank_file(file, mode)
     char *file;
     int mode ;
{
    int fd ;

    if ((fd = creat(file, mode)) == -1)
    {			
	Fprintf ("create_blank_file- ");
	perror(file) ;
	return(-1) ;
    }
    close(fd) ;
    return (1);
}			/* end: create_blank_file	*/


/*  */


/*+ 		lock_file
 ** FUNCTION:
 ** 	Locks entire file as "read" or "write". Needs the file
 ** descriptor, the type of lock desired ("r" or "w"), whether it should
 ** try to remove the lock forcibly after waitsecs.
 ** If NFS is not defined, 'flock' is used instead of 'fcntl'.
 ** Also gets the PID of the locking process.
 **
 ** Returns '-1' if couldn't lock file anyway.
 +*/

lock_file(fd, ltype, force, waitsecs)
     int fd, force, waitsecs ;
     char *ltype;
{
    void wakeup() ;				/* On getting ALARM  */
    int lockvalue;				/* make things easier	*/
#ifdef NFS					/* In NFS files		*/
    struct flock lk ;				/* To lock the file	*/
    
    if ( *ltype == 'w' || *ltype == 'W')
      lk.l_type = F_WRLCK;
    else
      lk.l_type = F_RDLCK ;
    lk.l_whence = 0, lk.l_start = 0 ;		/* from start of file	*/
    lk.l_len = 0 ;				/* lock entire file	*/

    if(fcntl(fd, F_SETLK, &lk) == -1)
    {
	struct flock glk ;	/* for getting info about the lock  */

	fcntl(fd, F_GETLK, &glk) ;	/* Get description of lock  */
	if (glk.l_type == F_UNLCK)	/* Its now unlocked	*/
	  ;				/* so continue */
	else
	{
	    fprintf(stderr, "Netlog (fcntl): ") ;
	    fprintf(stderr, "file locked by another (pid %ld), waiting...\n",
		    (long)glk.l_pid);

	}

	signal(SIGALRM, wakeup);		/* Wait for waittime secs */
	alarm(waitsecs);			/* ... and retry	*/
	if (fcntl(fd, F_SETLKW, &lk) == -1)	/* second try at locking */
	{				       	/* ... fails on SIGALRM */
	    perror("lock_file");
	    if (force)			/* Unlock  */
	    {
		fprintf(stderr, "forcibly removing lock\n");
		if (unlock_file(fd) == -1)
		  return(-1);
		else
		  return( fcntl(fd, F_SETLK, &lk) );	/* Return what hap */
	    }
	    else
	      return(-1) ;		/* Couldn't lock, and not force */

	}      	/* end second try at locking */
    }		/* end if( fcntl failed ) */


#else						/* In BSD 4.3 systems	*/

    if ( *ltype == 'w' || *ltype == 'W')	
      lockvalue = LOCK_EX ;
    else
      lockvalue = LOCK_SH ;			/* shared lock for read	*/
    
    if (flock(fd, lockvalue | LOCK_NB) == -1 )	/* see if locked already */
    {
	if (errno == EWOULDBLOCK)		/* Would block 		*/
	{				
	    fprintf(stderr,
		    "Netlog (flock): file locked by another, waiting...\n");
	}
	else					/* Some other error 	*/
	{
	    perror("lock_file");
	    return(-1);
	}

	signal(SIGALRM, wakeup);		/* Wait for waittime secs */
	alarm(waitsecs);			/* ... and retry	*/

	if (flock (fd, lockvalue) == -1)	/* Second try, with waiting */
	{					/* ... fails on SIGALRM */
	    perror("lock_file");
	    return(-1) ;
	}
    }		/* end if LOCK_NB	*/

#endif
    return(0) ;

}	/* end: lock_file	*/


/*+ 		unlock_file
** FUNCTION:
** 	Unlocks file.
**/

unlock_file(fd)
     int fd;
{
#ifdef NFS					/* In NFS files		*/
	struct flock lk ;			/* To lock the file	*/

	lk.l_type = F_UNLCK ;
	lk.l_whence = 0, lk.l_start = 0 ;	/* from start of file	*/
	lk.l_len = 0 ;				/* unlock entire file	*/
	if (fcntl(fd, F_SETLK, &lk) == -1)
	{
	    perror("(unlock_file) fcntl");
	    return(-1) ;
	}
#else
	if (flock(fd, LOCK_UN))			/* In BSD 4.3 systems	*/
	{
	    perror("(unlock_file) flock");
	    return(-1) ;
	}
#endif
	return(0) ;
} 	/* end: unlock_file	*/

/*+ 
** FUNCTION:
** 	Just wakeup from the ALARM signal. Do nothing.
**/
void wakeup()
{
    signal(SIGALRM, SIG_DFL);		/* restore default action */
    if (debug)
      fprintf(stderr, "(debug) wakeup: reset SIGALRM to SIG_DFL") ;
}


/*+ 		get_file_id
** FUNCTION:
** 	Get UID, GID and MODE of the filename passed to it. Needs ptrs
** which are filled in by the values. Returns -1 on error. If NULL's
** passed to it, then doesn't fill those in (thus can get only one of
** the uid, gid or mode).
**/
get_file_id (fname, fuid, fgid, fmode)
     char *fname ;
     uid_t *fuid ;
     gid_t *fgid ;
     u_short *fmode ;
{

    struct stat buf ;

    if (fname == NULL)
      return (-1) ;			/* goof off */

    if (stat(fname, &buf) != 0)		/* error */
    {
	fprintf(stderr, "(get_file_id) stat error for %s: %s\n", 
		fname, sys_errlist[errno]) ;
	return (-1) ;
    }

    if (fuid != NULL)
      *fuid = buf.st_uid ;

    if (fgid != NULL)
      *fgid = buf.st_gid ;

    if (fmode != NULL)
      *fmode = buf.st_mode ;

    return (1) ;

}

/*+ 		tackon_dir
** FUNCTION:
** 	Tacks on dir to the filename specified IF it does not have
** any '/' in its name. Returns pointer to a buffer space obtained by malloc()
** so in effect it is doing a 'strdup' function.
**/
char *
tackon_dir(pdir, pfil)
     char *pdir, *pfil ;			/* filename */
{
    static char qfil[BUFSIZ] ;
    char *newstr ;

    if (strchr(pfil, '/') == NULL)		/* pathname not specified */
      sprintf(qfil,"%s/%s\0", pdir, pfil) ;
    else
      sprintf(qfil, "%s\0", pfil) ;		/* full pathname already */


    if ( (newstr = (char *)malloc (strlen(qfil) + 1)) == NULL)
    {
	perror("malloc in tackon_dir") ;
	return (NULL);
    }

    return (strcpy (newstr, qfil) ) ;
}					/* end: tackon_dir */

