/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The Benchmark Control Protocol - handle the connections to the (remote)
 * demon processes.
 */

#include <general.h>
#include <ipc.h>
#include <ipc_pres.h>
#include <netinet/in.h>


/*
 * Exports:
 *	hostname2addr(host: char *; a_addr: bcpaddr_t *) -> OK/NOTOK
 *	host_add_connect(host: char *; a_addr: bcpaddr_t *) -> OK/NOTOK
 *
 *	host_close_all()
 *	host_mask(a_mask: int *; a_nfd: int *)
 *			Create a mask to use with select
 *	host_mask2sock(mask: int) -> int *
 *	host_mask2addr(mask: int) -> bcpaddr_t
 *	host_clearclockdiff() -> int
 *	host_getclockdiff() -> int
 *	host_clockdiff(from: bcpaddr_t; time: double) -> double
 *	host_lookup(addr: bcpaddr_t) -> a_socket: int *
 *	host_namelookup()
 *	host_enter()
 *	host_remove(addr:bcpaddr_t)
 *
 * Internal:
 *	open_connection(addr: bcpaddr_t) -> OK/NOTOK
 */

/*  Implementation dependent #include's */

#ifndef _NETDB_
#define _NETDB_
#include <netdb.h>
#endif _NETDB_

#ifndef AF_INET
#include <sys/socket.h>
#endif 

/*  Local data */

#define MAXHOST	NOFILE	/* Can't possibly exceed this value when using sockets */

struct hostdescr {
    char 	*hd_name;
    int		hd_socket;
    bcpaddr_t	hd_addr;
    double	hd_clockdiff;
};

static int nhost = 0;
static struct hostdescr hosts[MAXHOST];

void host_print();

/*  */

int hostname2addr(host, a_addr)
    char *host;
    bcpaddr_t *a_addr;
{
    bcpaddr_t addr;
    struct hostent *remote;

    tprintf("hostname2addr(0x%x, 0x%x)\n", host, a_addr);
    dprintf("hostname2addr: %s\n", host);

    remote = gethostbyname(host);

    if (remote == NULL) {
#ifdef notdef
	/* error code in h_errno */
	switch (h_errno) {
	case HOST_NOT_FOUND:
	    fprintf(stderr, "Unknown host: %s\n", host);
	    break;
	case TRY_AGAIN:
	    fprintf(stderr, "Temporary error looking up: %s\n", host);
	    break;
	case NO_RECOVERY:
	    fprintf(stderr, "Error looking up: %s\n", host);
	    break;
	case NO_ADDRESS:
	    fprintf(stderr, "%s doesn't have an IP address\n", host);
	    break;
	}
#endif notdef
	eprintf(EF_3, COMMUNICATION,
		"Name lookup on remote host failed", host);
  	return NOTOK;
    }
    if (remote->h_addrtype != AF_INET) {
	eprintf(EF_3, COMMUNICATION,
		"Name lookup on remote host failed - no internet address",
		host);
	return NOTOK;
    }

    addr.ba_addr = BA_SYSTEM;
    addr.ba_index = *(long *)remote->h_addr;
    addr.ba_index = htonl(addr.ba_index);
    dprintf("hostname2addr: 0x%x:0x%x\n", addr.ba_addr, addr.ba_index);
    *a_addr = addr;
    return OK;
} /* hostname2addr */

/*  */

int host_add_connect(host, a_addr)
    char *host;
    bcpaddr_t *a_addr;
{
    bcpaddr_t addr;
    int *a_sock;

    tprintf("host_add_connect(0x%X, 0x%x)\n", host, a_addr);
    dprintf("host_add_connect to %s\n", host);
    
    if (host_namelookup(host, &addr) == NOTOK) {
	if (hostname2addr(host, &addr) == NOTOK) {
	    dprintf(EF_IN3, INTERNAL, "hostname2addr", "host_add_connect");
	    return NOTOK;
	}
    }

    a_sock = host_lookup(addr);
    if (a_sock != NULL && *a_sock != NOTOK) {
	/* already entered and connected */
	dprintf("Already entered and connected\n");
	*a_addr = addr;
	return OK;
    }
	
    (void)host_enter(host, addr);

    if (open_connection(addr) == NOTOK) {
	eprintf("Failed conecting to demon on %s\n", host);
	dprintf(EF_IN4, COMMUNICATION, PROTOCOL, "open_connection",
		"host_add_connect");
	(void)host_remove(addr);
	return NOTOK;
    }
    *a_addr = addr;
    return OK;
} /* host_add_connect */
    
/*  */ 
    
int host_close_all()
{
    int i;

    tprintf("host_close_all()\n");
    
    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL) {
	    if (hosts[i].hd_socket != NOTOK)
		(void)close(hosts[i].hd_socket);
	    hosts[i].hd_socket = NOTOK;
	    free(hosts[i].hd_name);
	    hosts[i].hd_name = NULL;
	}
    return OK;
} /* host_close_all */
	    
/*  */

int host_mask(a_mask, a_nfd)
    int *a_mask, *a_nfd;
{
    int mask = 0, nfd = 0;
    int i;
    
    tprintf("host_mask(0x%x, 0x%x)\n", a_mask, a_nfd);

    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && hosts[i].hd_socket != NOTOK) {
	    mask |= 1 << hosts[i].hd_socket;
	    if (hosts[i].hd_socket >= nfd)
		nfd = hosts[i].hd_socket + 1;
	}

    *a_mask = mask;
    *a_nfd = nfd;
    return OK;
} /* host_mask */

/*  */

int *host_mask2sock(mask)
    int mask;
{
    int i;
    
    tprintf("host_mask2sock(0x%x)\n", mask);

    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && hosts[i].hd_socket != NOTOK
	    && mask & (1 << hosts[i].hd_socket)) {
	    return &hosts[i].hd_socket;
	}

    eprintf(EF_IN3, INTERNAL, "no match on mask", "host_mask2sock");
    return NULL;
} /* host_mask2sock */

/*  */

bcpaddr_t host_mask2addr(mask)
    int mask;
{
    int i;
    bcpaddr_t addr;
    
    tprintf("host_mask2addr(0x%x)\n", mask);

    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && hosts[i].hd_socket != NOTOK
	    && mask & (1 << hosts[i].hd_socket)) {
	    return hosts[i].hd_addr;
	}

    eprintf(EF_IN3, INTERNAL, "no match on mask", "host_mask2addr");
    addr.ba_addr = 0; addr.ba_index = 0;
    return addr;
} /* host_mask2addr */

/*  */

int host_clearclockdiff()
{
    int i;
    
    for (i = 0; i < nhost; i++) 
	hosts[i].hd_clockdiff = 0;
} /* host_clearclockdiff */

/*  */

#define GET_DELAY	5.0	/* seconds */
#define MAX_TIME	1	/* seconds */

int host_getclockdiff()
{
    msg_t msgs, *msg = &msgs;
    int i;
    double delay = GET_DELAY;
    int max_time = MAX_TIME;
    struct timeval tv;
    struct timezone dummy;
    int len;
    double localtime, remotetime;
    
    tprintf("host_getclockdiff()\n");

    for (i = 0; i < nhost; i++) {
	if (hosts[i].hd_name == NULL)
	    continue;

	printf("Will check clock difference with %s in %.1f seconds ...\n",
	       hosts[i].hd_name, delay);
	bench_delay(delay);
	msg->msg_to = hosts[i].hd_addr;
	msg->msg_code = MSG_TIMEQ;
	msg->msg_datalen = 0;
	if (send_msg(msg) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "send_msg", "host_getclockdiff");
	    goto abort;
	}
	if (await_recv_msg(msg, max_time) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "await_recv_msg", "host_getclockdiff");
	    goto abort;
	}
	switch (msg->msg_code) {
	case MSG_TIME:
	    break;
	case MSG_TIMEOUT:
	    eprintf("Failed receiving the time from %s in %d seconds\n",
		    hosts[i].hd_name, max_time);
	    goto abort;
	default:
	    eprintf("Received a %s message from %s when waiting for\n",
		    code2str(msg->msg_code), bcpaddr2str(msg->msg_from));
	    eprintf("\ta MSG_TIME from %s!\n", bcpaddr2str(hosts[i].hd_addr));
	    goto abort;
	}
	/* compute difference */
	
	if (gettimeofday(&tv, &dummy) == NOTOK) {
	    eprintf(EF_SYSCALL, INTERNAL, "gettimeofday", "host_getclockdiff",
		    getsyserr());
	    goto abort;
	}
	len = msg->msg_datalen;
	localtime = tv.tv_sec + ((double)tv.tv_usec)/1000000;
	remotetime = str2time(msg->msg_data, &len);
	if (msg->msg_datalen)
	    free(msg->msg_data);
	hosts[i].hd_clockdiff = remotetime - localtime;

	printf("Clock difference for %s set to %.3f (%.3f - %.3f)\n",
	       hosts[i].hd_name, hosts[i].hd_clockdiff, remotetime, localtime);
    }

    return OK;

 abort:
    eprintf("\tUnable to find clock differences for the hosts!\n");
    (void)host_clearclockdiff();
    return NOTOK;

} /* host_getclockdiff */
    
/*  */

double host_clockdiff(from, time)
    bcpaddr_t from;
    double time;
{
    bcpaddr_t src;
    int i;

    tprintf("host_clockdiff(%s, %.3f)\n", bcpaddr2str(from), time);

    if (route_lookup(from, &src) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "route_lookup", "host_clockdiff");
	eprintf("\tNo route to %s\n", bcpaddr2str(from));
	return time;
    }

    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && addrcmp(hosts[i].hd_addr, src) == 0) {
	    return time - hosts[i].hd_clockdiff;
	}

    eprintf(EF_IN3, INTERNAL, "Not in host table", "host_clockdiff");
    eprintf("\t%s not found in host table\n", bcpaddr2str(src));
    
    return time;
} /* host_clockdiff */

/*  */

static void host_print()
{
    int i;

    printf("Host table ...\n");
    for (i = 0; i < nhost; i++) {
	if (hosts[i].hd_name != NULL) {
	    printf("\t%d: name %s, addr %s, socket %d, clockdiff %.3f\n",
		   i, hosts[i].hd_name, bcpaddr2str(hosts[i].hd_addr),
		   hosts[i].hd_socket, hosts[i].hd_clockdiff);
	}
    }
} /* host_print */

/*  */

static int host_namelookup(host, a_addr)
    char *host;
    bcpaddr_t *a_addr;
{
    int i;

    tprintf("host_namelookup(0x%x,0x%x)\n", host, a_addr);
    
    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && strcmp(hosts[i].hd_name, host) == 0) {
	    *a_addr = hosts[i].hd_addr;
	    return OK;
	}
    dprintf("host_namelookup: not found\n");
    return NOTOK;
} /* host_namelookup */

/*  */

int addrcmp(a1, a2)
    bcpaddr_t a1, a2;
{
#ifdef notdef
    if (a1.ba_addr != BA_SYSTEM) {
	eprintf(EF_IN3, INTERNAL, "addrcmp", "ipc_remote.c - not BA_SYSTEM");
	return NOTOK;
    }
    if (a2.ba_addr != BA_SYSTEM) {
	eprintf(EF_IN3, INTERNAL, "addrcmp", "ipc_remote.c - not BA_SYSTEM");
	return NOTOK;
    }
#endif notdef    
    return !(a1.ba_addr == a2.ba_addr && a1.ba_index == a2.ba_index);
} /* addrcmp */

/*  */

int *host_lookup(addr)
    bcpaddr_t addr;
{
    int i;

    tprintf("host_lookup(0x%x:0x%x)\n", addr.ba_addr, addr.ba_index);
    
    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && addrcmp(hosts[i].hd_addr, addr) == 0) {
	    return &hosts[i].hd_socket;
	}
    dprintf("host_lookup: not found\n");
    return NULL;
} /* host_lookup */

/*  */

static int host_enter(host, addr)
    char *host;
    bcpaddr_t addr;
{
    int i;
    bcpaddr_t dummy;

    tprintf("host_enter(0x%x, 0x%x:0x%x)\n",
	    host, addr.ba_addr, addr.ba_index);
    dprintf("host_enter: %s\n", host);

    /* initialize if necessery */
    if (nhost < MAXHOST) {
	for (i = nhost; i < MAXHOST; i++) {
	    hosts[i].hd_name = NULL;
	    hosts[i].hd_socket = NOTOK;
	    hosts[i].hd_clockdiff = 0;
	}
	nhost = MAXHOST;
    }
    
    if (host_namelookup(host, &dummy) == OK) {
	/* already entered  */
	dprintf("Host already entered\n");
	return OK;
    }

    for (i = 0; i < nhost; i++) {
	if (hosts[i].hd_name == NULL) {
	    /* found free slot */
	    hosts[i].hd_name = (char *)malloc(strlen(host)+1);
	    if (hosts[i].hd_name == NULL) {
		eprintf(EF_IN3, INTERNAL, RESOURCE, "host_enter - malloc");
		return NOTOK;
	    }
	    strcpy(hosts[i].hd_name, host);
	    hosts[i].hd_addr = addr;
	    hosts[i].hd_socket = NOTOK;

	    if (Debug)
		host_print();
	    
	    return OK;
	}
    }
    eprintf(EF_IN3, INTERNAL, RESOURCE, "host_enter");
    return NOTOK;
} /* host_enter */

/*  */ 
    
static int host_remove(addr)
    bcpaddr_t addr;
{
    int i;

    tprintf("host_remove(0x%x: %d)\n", addr.ba_addr, addr.ba_index);

    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && addrcmp(hosts[i].hd_addr, addr) == 0) {
	    if (hosts[i].hd_socket != NOTOK)
		(void)close(hosts[i].hd_socket);
	    hosts[i].hd_socket = NOTOK;
	    free(hosts[i].hd_name);
	    hosts[i].hd_name = NULL;
	    return OK;
	}
    dprintf("host_remove: not here\n");
    return NOTOK;
} /* host_remove */
	    
/*  */

static int open_connection(addr)
    bcpaddr_t addr;
{
    int i, *a_socket;

    tprintf("open_connection(0x%x:0x%x)\n", addr.ba_addr, addr.ba_index);

    if ((a_socket = host_lookup(addr)) == NULL) {
	eprintf(EF_IN3, INTERNAL, "addr not found", "open_connection");
	return NOTOK;
    }

    if (*a_socket != NOTOK)
	(void)close(*a_socket); 	/* and reopen below */

    *a_socket = transport_connect_server(addr);
    if (*a_socket == NOTOK) {
	dprintf(EF_IN3, INTERNAL, "transport_connect_server",
		"open_connection");
	return NOTOK;
    }
    for (i = 0; i < nhost; i++)
	if (hosts[i].hd_name != NULL && addrcmp(hosts[i].hd_addr, addr) == 0) {
	    hosts[i].hd_socket = *a_socket;	/* no operation! */
	    return OK;
	}
    eprintf(EF_IN3, INTERNAL, "inconsistencies", "host descriptor database");
    return NOTOK;
} /* host_lookup */

