/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The benchmark io routines providing input and output to
 * files (block buffered), virtual memory and memory.
 */

/*
 * Exports:
 * 	iohdl_t *iohdl_create(tag: int; filename: char *) 
 * 	iohdl_print(ioh: iohdl_t *) 
 */

#include <general.h>

#include <sys/stat.h>
#include <sys/errno.h>
#include <sys/file.h>
#ifdef SYSV
#include <fcntl.h>
#endif SYSV

/* Forwards */

int file_open();
int file_destroy();
int file_get();
int file_put();
int file_size();

int mem_open();
int mem_destroy();
int mem_get();
int mem_put();
int mem_size();

int vm_open();
int vm_destroy();
int vm_get();
int vm_put();
int vm_size();


iohdl_t *iohdl_create(tag, filename)
    int 	tag;
    char *filename;
{
    iohdl_t *iohdl;

    
    tprintf("iohdl_create(%d, 0x%x)\n", tag, filename);
    
    if ((iohdl = (iohdl_t *)malloc(sizeof(iohdl_t))) == NULL) {
	eprintf(EF_IN4X, INTERNAL, RESOURCE, "iohdl_create",
	       "\nMalloc returns NULL");
	    return NULL;
    }
    switch (tag) {
    case IOT_FILE:
	if (filename == NULL) {
	    eprintf(EF_IN4X, INTERNAL, "Bad parameter", "iohdl_create",
		   "\nNo file name for file type IO");
	    return NULL;
	}
	iohdl->io_filename = (char *)malloc(strlen(filename) + 1);
	if (iohdl->io_filename == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "iohdl_create");
	    return NULL;
	}
	strcpy(iohdl->io_filename, filename);
	iohdl->io_open = file_open;
	iohdl->io_destroy = file_destroy;
	iohdl->io_get = file_get;
	iohdl->io_put = file_put;
	iohdl->io_size = file_size;
	break;
    case IOT_MEM:
	iohdl->io_filename = NULL;
	iohdl->io_open = mem_open;
	iohdl->io_destroy = mem_destroy;
	iohdl->io_get = mem_get;
	iohdl->io_put = mem_put;
	iohdl->io_size = mem_size;
	break;
    case IOT_VM:
	iohdl->io_filename = NULL;
	iohdl->io_open = vm_open;
	iohdl->io_destroy = vm_destroy;
	iohdl->io_get = vm_get;
	iohdl->io_put = vm_put;
	iohdl->io_size = vm_size;
	break;
    default:
	eprintf(EF_IN4X, INTERNAL, "Bad parameter", "iohdl_create",
	       "\nBad type of IO");
	return NULL;
    }
    iohdl->io_tag = tag;
    iohdl->io_itemsize = sizeof(char);
    iohdl->io_fid = -1;
    iohdl->io_buffer = iohdl->io_bufptr = NULL;		/* set when opened */
    iohdl->io_bufsize = 0;

    return iohdl;
} /* iohdl_create */

iohdl_print(iohdl)
    iohdl_t *iohdl;
{
    if (iohdl == NULLIO) {
	eprintf(EF_IN4X, INTERNAL, "Bad parameter", "iohdl_print",
	       "\nNULL io handle");
    	return;
    }

    switch (iohdl->io_tag) {
    case IOT_FILE:
	fprintf(stderr, "io handle: FILE %s (%d)\n",
		iohdl->io_filename, iohdl->io_fid);
	break;
    case IOT_MEM:
	fprintf(stderr, "io handle: MEMORY\n");
	break;
    case IOT_VM:
	fprintf(stderr, "io handle: VIRTUAL MEMORY\n");
	break;
    }
    fprintf(stderr, "\tdirection %s, itemsize %d\n",
	    (iohdl->io_direction == source) ? "source" : "dest",
	    iohdl->io_itemsize);
    fprintf(stderr, "\tbuffer 0x%x, bufsize %d bufptr 0x%x\n",
	    iohdl->io_buffer,
	    iohdl->io_bufsize,
	    iohdl->io_bufptr);
    fprintf(stderr, "\Routines: 0x%x, 0x%x, 0x%x, 0x%x, 0x%x\n",
	    iohdl->io_open,
	    iohdl->io_put,
	    iohdl->io_get,
	    iohdl->io_destroy,
	    iohdl->io_size);
} /* iohdl_print */


/*  */

io_print(iohdl)
     iohdl_t *iohdl;
{
  printf("io_print(0x%x) ", iohdl);
  if (iohdl == NULLIO)
    return;

  switch (iohdl->io_tag) {
  case IOT_MEM: printf("IOT_MEM "); break;
  case IOT_VM: printf("IOT_VM "); break;
  case IOT_FILE: printf("IOT_FILE %s (0x%x) fid %d ", 
			iohdl->io_filename, iohdl->io_filename,
			iohdl->io_fid); 
		break;
	      }
  switch (iohdl->io_direction) {
  case source: printf("SRC\n"); break;
  case destination: printf("DEST\n"); break;
  default: printf("(nodir)\n"); break;
	     }
  printf("\titemsize %d, buffer 0x%x, bufsize %d, bufptr 0x%x\n",
	 iohdl->io_itemsize, iohdl->io_buffer, iohdl->io_bufsize,
	 iohdl->io_bufptr);
  switch (iohdl->io_tag) {
  case IOT_FILE:
    check(iohdl->io_open, file_open, "file_open");
    check(iohdl->io_destroy, file_destroy, "file_destroy");
    check(iohdl->io_get, file_get, "file_get");
    check(iohdl->io_put, file_put, "file_put");
    check(iohdl->io_size, file_size, "file_size");
    break;
  case IOT_MEM:
    check(iohdl->io_filename, NULL, "NULL filename");
    check(iohdl->io_open, mem_open, "mem_open");
    check(iohdl->io_destroy, mem_destroy, "mem_destroy");
    check(iohdl->io_get, mem_get, "mem_get");
    check(iohdl->io_put, mem_put, "mem_put");
    check(iohdl->io_size, mem_size, "mem_size");
	break;
  case IOT_VM:
    check(iohdl->io_filename, NULL, "NULL filename");
    check(iohdl->io_open, vm_open, "vm_open");
    check(iohdl->io_destroy, vm_destroy, "vm_destroy");
    check(iohdl->io_get, vm_get, "vm_get");
    check(iohdl->io_put, vm_put, "vm_put");
    check(iohdl->io_size, vm_size, "vm_size");
    break;
  default:
    eprintf(EF_IN4X, INTERNAL, "Bad parameter", "io_print",
	       "\nBad type of IO");
    }

} /* io_print */

check(a, b, str)
     char *a, *b, *str;
{
  if (a != b)
    printf("io_print: not same; %s\n", str);
}

/*  */

/*
 * File I/O routines
 */

file_open(iohdl, direction, bufsize)
    iohdl_t *iohdl;
    direction_t direction;
    long bufsize;
{
    struct stat st;
    
    
    tprintf("file_open(0x%x, %d, %d)\n", iohdl, direction, bufsize);

    if (Debug)
      io_print(iohdl);

    if (iohdl->io_fid != -1) {
	eprintf(EF_IN3, INTERNAL, "file has already been opened", "file_open");
	return NOTOK;
    }
    if (direction == source) {
	iohdl->io_fid =  open(iohdl->io_filename, O_RDONLY);
    } else {
#ifdef notdef
	if (stat(iohdl->io_filename, &st) == NOTOK) {
	    if (errno != ENOENT) {
		eprintf(EF_IN4X, INTERNAL, "stat()", "file_open",
		       getsyserr());
		return NOTOK;
	    }
	} else {
	    if (unlink(iohdl->io_filename) == NOTOK) {
		eprintf(EF_IN4X, INTERNAL, "unlink()", "file_open",
		       getsyserr());
		return NOTOK;
	    }
	}
#endif notdef		
	iohdl->io_fid = open(iohdl->io_filename, O_WRONLY | O_CREAT | O_TRUNC,
			     0600);
    }
    if (iohdl->io_fid == NOTOK) {
	eprintf(EF_IN4X, SPECIFICATION, "file can't be opened", getsyserr(),
	       iohdl->io_filename);
	iohdl->io_fid = -1;
	return NOTOK;
    }

    /* setup buffers */

    iohdl->io_bufsize = bufsize;
    iohdl->io_buffer = (char *)malloc(bufsize);
    if (iohdl->io_buffer == NULL) {
	eprintf(EF_IN4, INTERNAL, RESOURCE, "file_open",
	       "Malloc returned NULL");
	return NOTOK;
    }
    iohdl->io_bufptr = iohdl->io_buffer;
    
    iohdl->io_direction = direction;
    return OK;
} /* file_open */

file_destroy(iohdl)
    iohdl_t *iohdl;
{
    
    tprintf("file_destroy(0x%x)\n", iohdl);
    if (iohdl->io_fid == -1) {
	eprintf(EF_IN3, INTERNAL, "file isn't open", "file_destroy");
	return NOTOK;
    }
    if (iohdl->io_filename != NULL)
	free(iohdl->io_filename);
    if (iohdl->io_buffer != NULL)
	free(iohdl->io_buffer);
    iohdl->io_buffer = iohdl->io_bufptr = NULL;
    iohdl->io_bufsize = 0;
    if (close(iohdl->io_fid) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "Closing the file failed",
		"file_destroy", getsyserr());
        free((char *)iohdl);
	return NOTOK;
    }
    free((char *)iohdl);
    return OK;
} /* file_destroy */

file_put(iohdl, amount)
    iohdl_t *iohdl;
    int amount;
{
    
    tprintf("file_put(0x%x, %d)\n", iohdl, amount);
    if (iohdl->io_direction != destination) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "file_put",
	       "\nio handle only source");
    	return NOTOK;
    }
    if (iohdl->io_fid == -1) {
	eprintf(EF_IN4, INTERNAL, "File not open", "file_put", "");
	return NOTOK;
    }
    if (amount*iohdl->io_itemsize > iohdl->io_bufsize) {
	eprintf(EF_IN4X, INTERNAL, "Too much data in write buffer", "file_put",
	       "Ignored!");
    }
	
    if (write(iohdl->io_fid, iohdl->io_buffer,
	      amount*iohdl->io_itemsize) == NOTOK) {
	eprintf(EF_IN4, INTERNAL, "Write failed", "file_put", getsyserr());
	return NOTOK;
    }

    return OK;
} /* file_put */

file_get(iohdl, amount)
    iohdl_t *iohdl;
    int amount;
{
    int cc;
    
    
    tprintf("file_get(0x%x, %d)\n", iohdl, amount);
    if (iohdl->io_direction != source) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "file_get",
	       "\nio handle only destination");
    	return NOTOK;
    }
    if (iohdl->io_fid == -1) {
	eprintf(EF_IN4, INTERNAL, "File not open", "file_get", "");
	return NOTOK;
    }
    if (amount*iohdl->io_itemsize > iohdl->io_bufsize) {
	eprintf(EF_IN4X, INTERNAL, "Too much data into read buffer", "file_get",
	       "Ignored!");
    }

    cc = read(iohdl->io_fid, iohdl->io_buffer, amount*iohdl->io_itemsize);
    if (cc == NOTOK) {
	eprintf(EF_IN4, INTERNAL, "Read failed", "file_get", getsyserr());
	return NOTOK;
    }
    if (cc != amount*iohdl->io_itemsize) {
	eprintf(EF_IN4X, INTERNAL, "Reached end-of-file", "file_get",
	       "Ignored!");
    }
	
    return OK;
} /* file_get */

int file_size(iohdl)
    iohdl_t *iohdl;
{
    struct stat st;
    
    
    tprintf("file_size(0x%x)\n", iohdl);

    if (stat(iohdl->io_filename, &st) == NOTOK) {
	if (errno == ENOENT)
	    return SIZE_UNLIMIT;
	
	eprintf(EF_IN4, INTERNAL, "stat call", "file_size",
	       getsyserr());
    	return NOTOK;
    }
	
    return (int)st.st_size;
} /* file_size */

/*  */

/*
 * Memory I/O routines
 */

mem_open(iohdl, direction, bufsize)
    iohdl_t *iohdl;
    direction_t direction;
    long bufsize;
{
    
    tprintf("mem_open(0x%x, %d, %d)\n", iohdl, direction, bufsize);

    iohdl->io_bufsize = bufsize;
    iohdl->io_buffer = (char *)malloc(bufsize);
    if (iohdl->io_buffer == NULL) {
	eprintf(EF_IN4, INTERNAL, "Resource shortage", "mem_open",
	       "Malloc returned NULL");
	return NOTOK;
    }
    iohdl->io_bufptr = iohdl->io_buffer;

    iohdl->io_direction = direction;
    return OK;
} /* mem_open */

mem_destroy(iohdl)
    iohdl_t *iohdl;
{
    
    tprintf("mem_destroy(0x%x)\n", iohdl);
    if (iohdl->io_buffer != NULL)
	free(iohdl->io_buffer);
    free((char *)iohdl);

    return OK;
} /* mem_destroy */

mem_put(iohdl, amount)
    iohdl_t *iohdl;
    int amount;
{
    
    tprintf("mem_put(0x%x, %d)\n", iohdl, amount);
    if (iohdl->io_direction != destination) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "mem_put",
	       "\nio handle only source");
    	return NOTOK;
    }
    if (amount*iohdl->io_itemsize > iohdl->io_bufsize) {
	eprintf(EF_IN4X, INTERNAL, "Too much data in write buffer", "mem_put",
	       "Ignored!");
    }

    
    iohdl->io_bufptr = iohdl->io_buffer;
    return OK;
} /* mem_put */

mem_get(iohdl, amount)
    iohdl_t *iohdl;
    int amount;
{
    
    tprintf("mem_get(0x%x, %d)\n", iohdl, amount);
    if (iohdl->io_direction != source) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "mem_get",
	       "\nio handle only destination");
    	return NOTOK;
    }
    if (amount*iohdl->io_itemsize > iohdl->io_bufsize) {
	eprintf(EF_IN4X, INTERNAL, "Too much data into read buffer", "mem_get",
	       "Ignored!");
    }
    iohdl->io_bufptr = iohdl->io_buffer;
    return OK;
} /* mem_get */

int mem_size(iohdl)
    iohdl_t *iohdl;
{
    
    tprintf("mem_size(0x%x)\n", iohdl);

    return SIZE_UNLIMIT;
} /* mem_size */

/*  */

/*
 * Virtual memory I/O routines
 *
 * Note: the bufsize parameter must be the complete size of the data to be
 * transferred (i.e. maxmsgsize*iterations*type2size).
 */

vm_open(iohdl, direction, bufsize)
    iohdl_t *iohdl;
    direction_t direction;
    long bufsize;
{
    
    tprintf("vm_open(0x%x, %d, %d)\n", iohdl, direction, bufsize);

    iohdl->io_bufsize = bufsize;
    iohdl->io_buffer = (char *)malloc(bufsize);
    if (iohdl->io_buffer == NULL) {
	eprintf(EF_IN4, INTERNAL, "Resource shortage", "vm_open",
	       "Malloc returned NULL");
	return NOTOK;
    }
    iohdl->io_bufptr = iohdl->io_buffer;

    iohdl->io_direction = direction;
    return OK;
} /* vm_open */

vm_destroy(iohdl)
    iohdl_t *iohdl;
{
    
    tprintf("vm_destroy(0x%x)\n", iohdl);
    if (iohdl->io_buffer != NULL)
	free(iohdl->io_buffer);
    free((char *)iohdl);

    return OK;
} /* vm_destroy */

vm_put(iohdl, amount)
    iohdl_t *iohdl;
    int amount;
{
    
    tprintf("vm_put(0x%x, %d)\n", iohdl, amount);
    if (iohdl->io_direction != destination) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "vm_put",
	       "\nio handle only source");
    	return NOTOK;
    }
    if (amount*iohdl->io_itemsize > iohdl->io_bufsize) {
	eprintf(EF_IN4, INTERNAL, "Too much data in write buffer", "vm_put",
	       "Not ignored!");
	return NOTOK;
    }

    iohdl->io_bufptr += amount*iohdl->io_itemsize;
    return OK;
} /* vm_put */

vm_get(iohdl, amount)
    iohdl_t *iohdl;
    int amount;
{
    tprintf("vm_get(0x%x, %d)\n", iohdl, amount);
    
    if (iohdl->io_direction != source) {
	eprintf(EF_IN4, INTERNAL, "Bad parameter", "vm_get",
	       "\nio handle only destination");
    	return NOTOK;
    }
    if (amount*iohdl->io_itemsize > iohdl->io_bufsize) {
	eprintf(EF_IN4, INTERNAL, "Too much data into read buffer", "vm_get",
	       "Not ignored!");
	return NOTOK;
    }

    iohdl->io_bufptr += amount*iohdl->io_itemsize;
    return OK;
} /* vm_get */

int vm_size(iohdl)
    iohdl_t *iohdl;
{

    tprintf("vm_size(0x%x)\n", iohdl);

    return SIZE_UNLIMIT;
} /* vm_size */

    


