/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Delay handling - delays are used when a procedure should sleep some
 * (a short amount of) time inbetween performing the service
 * primitves.
 *
 * The flag USE_DELAY will enable the use of busy-wait (a loop)
 * instead of a kernel timer. The advantage of USE_DELAY is that
 * delays smaller then the clock granularity can be made, and
 * the disadvantage is that is has to be calibrated for each machine
 * type.
 * The calibration is done using the variable calibrate.
 */
#include <general.h>

#define USE_DELAY


/*
 * Exports:
 */
void bench_delay();

#ifdef USE_DELAY
/*  VAX */
/* From /sys/vax/machparam.h */
#ifdef vax
static int cpuspeed = 1;

#define	DELAY(n)	{ register int N = cpuspeed * (n); while (--N > 0); }
/* Default value is 1000000 */
static int calibrate = 1000000;
#endif vax

/*  SUN */
/* From /usr/sys/sun/param.h */
#ifdef sun

#ifdef mc68020
static int cpudelay = 3;
/* Default value is 1000000 */
static int calibrate = 777605;		/* For Sun3/60 - 20Mhz */
#endif mc68020
#ifdef sparc
static int cpudelay = 5;
/* Default value is 1000000 */
static int calibrate = 1000000;
#endif sparc

/*
 * Delay units are in microseconds.
 */
#define	DELAY(n)	\
{ \
	extern int cpudelay; \
	register int N = (((n)<<4) >> cpudelay); \
 \
	while (--N > 0) ; \
}
#endif sun

/*  */
#endif USE_DELAY

void bench_delay(time)
    double time;
{
#ifdef DELAY
    int dtime;
#else    
    struct itimerval it;
#endif
    
    tprintf("bench_delay(%.3f)\n", time);
    
#ifdef DELAY
    dtime = (int)(time*calibrate);

    DELAY(dtime);
#else    
    it.it_interval.tv_sec = it.it_interval.tv_usec = 0;
    it.it_value.tv_sec = (int)time;
    it.it_value.tv_usec = (time - (double)it.it_value.tv_sec)*1000000 + .5;


    dprintf("bench_delay: s = %d, us = %d\n", it.it_value.tv_sec,
		it.it_value.tv_usec);

#ifdef notdef    
    if (timerisset(&it.it_value)) {
	if (setitimer(0, &it, NULL) == NOTOK) {
	    eprintf(EF_SYSCALL, INTERNAL, "setitimer call", "bench_delay",
		    getsyserr());
	}
	
    	sigpause(oldmask);	/* Always returns NOTOK */
    }
#endif notdef

#ifdef DNIX
    eprintf(EF_IN3, "Can't handle delaying","Not implemented",
	    "DNIX");
#else    
    if (select(0, NULL, NULL, NULL, &it.it_value) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "select", "bench_delay",
		getsyserr());
    }
#endif    
#endif !DELAY
    
} /* bench_delay */
