/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Header file for benchmarks on the UDP protocol
 */

#ifndef _UDP_
#define _UDP_

#ifndef __SOCKET__
#define __SOCKET__
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif __SOCKET__

/* A channel is a socket number and the server address, plus sequence
 * numbers used for detecting last datagrams */
typedef struct {
    int 		sock;
    struct sockaddr_in 	addr;
    char send_seq, recv_seq;
    } channel_t;

struct server_t {
    int sock;		/* a socket being listened on */
};


#include "../protoaddrs/portaddr.h"

/*
 * Error handling
 */

typedef int error_t;
#define failed(errind)	(*errind == NOTOK)

#define report_error(errind,str)	udp_report_error(errind,str)


#define InitClient()		{};

#define ConnRequest(addr,a_ch,errind) \
    udp_conn_request(addr,a_ch,errind)

#define CreateChannel(addr,a_ch,errind) \
    udp_create_channel(addr,a_ch,errind)

#define DiscRequest(ch, errind) \
    udp_disc_request(ch, errind)

#define DestroyChannel(ch, errind)	DiscRequest(ch, errind)

#define AwaitConnInd(serverp, a_ch, errind) \
    udp_await_conn_ind(serverp, a_ch, errind)


#define AwaitDiscInd(ch, errind) \
    udp_await_disc_ind(ch, errind)

#define CreateServer(aa_server, aa_addr, errind) \
    				udp_create_server(aa_server, aa_addr, errind)
#define DestroyServer(a_server, errind) \
    				udp_destroy_server(a_server, errind)

/* Can send a receive a maximum of 8k bytes */

#define DataRequest(channel,buffer,amount, errind) \
{ pprintf("DataRequest: address %d:0x%x:%d, number %d\n", \
	  (channel).addr.sin_family, \
	  (channel).addr.sin_addr.s_addr, (channel).addr.sin_port, \
	  0xff & (channel).send_seq); \
  buffer[0] = (channel).send_seq++; \
  if (sendto((channel).sock, buffer, amount, 0, &(channel).addr, \
	     sizeof((channel).addr)) == NOTOK) { \
      eprintf(EF_SYSCALL, COMMUNICATION, "sendto", \
	      "DataRequest", getsyserr()); \
      *errind = NOTOK; \
  } else \
      *errind = OK;					\
}

#define AwaitDataIndication(channel, buffer, amount, errind)	\
{ 				\
      int left = amount;	 		\
      int got, fromlen = sizeof((channel).addr); \
      struct sockaddr from; \
  				\
      pprintf("AwaitDataIndication(expecting number %d)\n", \
	      (channel).recv_seq); \
      *errind = OK;			\
      while (left > 0) {			\
	  got = recvfrom((channel).sock, buffer, left, 0, &(channel).addr, &fromlen); \
	  if (got == NOTOK) { /* error */		\
	      eprintf(EF_SYSCALL, COMMUNICATION, "recvfrom", \
		      "AwaitDataIndication", getsyserr()); \
	      *errind = NOTOK;			\
	      break;			\
	  } 				\
	  if (got == 0) { /* end-of-file */		\
	      errno = 0; 	\
	      *errind = NOTOK;	\
	      break;			\
	  }				\
	  left -= got;			\
      }				\
      if (buffer[0] != (channel).recv_seq++) { \
	      eprintf(EF_IN3, COMMUNICATION, "Out of sequence", \
		      "AwaitDataIndication"); \
	      eprintf("\tExpected sequence number %d, received %d\n", \
		      (channel).recv_seq - 1, buffer[0]); \
	      errno = 0; \
	      *errind = NOTOK;	\
      } \
}

#define BulkGetDataReq		DataRequest
#define BulkPutDataReq  	DataRequest
#define BulkGetAwaitDataInd	AwaitDataIndication
#define BulkPutAwaitDataInd	AwaitDataIndication

#define RRSendRequest 		BulkPutDataReq
#define RRAwaitResponseInd	BulkGetAwaitDataInd
#define RRAwaitRequestInd	BulkPutAwaitDataInd
#define RRSendResponse		BulkGetDataReq

/*
 * Synchronize with the sender to avoid buffering anomalies (the
 * sender starting before this receiver and the receiver just retrieves
 * data from the buffers!)
 */

#define BulkGetStart(ch, errind) \
{	\
    char buffer[1];	/* small buffer */	\
    DataRequest(ch, buffer, 1, errind);	\
}

#define AwaitBulkGetStart(ch, errind)	\
{	\
    char buffer[1];	/* small buffer */	\
    AwaitDataIndication(ch, buffer, 1, errind);	\
}

#define BulkGetStop(ch, errind) 	\
	    			{*errind = OK; }

#define AwaitBulkGetStop(ch, errind)	\
	    			{*errind = OK; }



#define BulkPutStart(ch, errind)	\
	    			{*errind = OK; }

#define AwaitBulkPutStart(ch, errind)\
	    			{*errind = OK; }

#define BulkPutStop(ch, errind)	\
{	\
    char buffer[1];	/* small buffer */	\
    AwaitDataIndication(ch, buffer, 1, errind);	\
}

#define AwaitBulkPutStop(ch, errind)	\
{	\
    char buffer[1];	/* small buffer */	\
    DataRequest(ch, buffer, 1, errind);	\
}


#define QueryCall(addrp, srcbuf, srclen, dstbuf, dstlen, errind) \
{ \
  channel_t ch; \
 \
  ConnRequest(addrp, &ch, errind); \
  if (!failed(errind)) { \
    RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind); \
    if (!failed(errind)) { \
      DiscRequest(ch, errind); \
    } \
  } \
}

#define QueryAwaitCallInd(serverp, dstbuf, dstlen, chp, errind) \
{ \
    AwaitConnInd(serverp, chp, errind); \
	if (!failed(errind)) { \
	RPCAwaitCallInd(*(chp), dstbuf, dstlen, errind); \
    } \
}

#define QueryReturn(ch, srcbuf, srclen, errind) \
{ \
    RPCReturn(ch, srcbuf, srclen, errind); \
    if (!failed(errind)) { \
	AwaitDiscInd(ch, errind); \
    } \
}

#define RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind) \
{ \
      DataRequest(ch, srcbuf, srclen, errind); \
      if (!failed(errind)) { \
	     AwaitDataIndication(ch, dstbuf, dstlen, errind); \
	     if (failed(errind)) { \
		report_error(errind, "AwaitDataIndication in RPCCall"); \
		return NOTOK; \
	     } \
      } \
}

#define RPCAwaitCallInd(ch, dstbuf, dstlen, errind) \
{ \
    AwaitDataIndication(ch, dstbuf, dstlen, errind); \
}

#define RPCReturn(ch, srcbuf, srclen, errind) \
{ \
    DataRequest(ch, srcbuf, srclen, errind); \
}
#endif _UDP_
