/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * TCP protocol specific code
 */

#include <general.h>
#include <protospec.h>

/*  */

tcp_conn_request(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    
    pprintf("tcp_conn_request(0x%x, 0x%x, 0x%x)\n",	addr, a_ch, errind);
    
    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "socket call", "tcp_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    if (connect(s, addr, sizeof(*addr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "connect call", "tcp_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    *a_ch = s;
    *errind = OK;
} /* tcp_conn_request */

/*  */


tcp_create_channel(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    
    pprintf("tcp_create_channel(0x%x, 0x%x, 0x%x)\n",	addr, a_ch, errind);
    
    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "socket call", "tcp_create_channel",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    *a_ch = s;
    *errind = OK;
} /* tcp_create_channel */

#ifdef notdef
tcp_conn_channel(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    pprintf("tcp_conn_channel(%d, 0x%x)\n", ch, errind);

    if (connect(ch, addr, sizeof(*addr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "connect call", "tcp_conn_channel",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    *errind = OK;
} /* tcp_conn_channel */
#endif notdef

/*  */

tcp_await_conn_ind(server, a_ch, errind)
    struct server_t *server;	
    channel_t *a_ch;
    error_t *errind;
{
    struct sockaddr_in otheraddr;
    int len = sizeof(otheraddr), so;

    
    pprintf("tcp_await_conn_ind(server %d)\n", *server);

    if ((so = accept(*server, &otheraddr, &len)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "accept call", "tcp_await_conn_ind",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    *a_ch = so;
    *errind = OK;
} /* tcp_await_conn_ind */

/*  */

tcp_disc_request(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    pprintf("tcp_disc_request(%d, 0x%x)\n", ch, errind);
    
    if (close(ch)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close call", "tcp_disc_request",
	       getsyserr());
  	*errind = NOTOK;
	return;
    }
    *errind = OK;
} /* tcp_disc_request */

/*  */

tcp_await_disc_ind(ch, errind)
    channel_t ch;
    error_t *errind;
{
    char buf;	/* very small buffer */
    int cc;	/* Not char to work with int constant EOF */

    
    pprintf("tcp_await_disc_ind(ch %d, 0x%x)\n", ch, errind);
    
    cc = read(ch, &buf, 1);
    if (cc == NOTOK) {
	if (errno == ECONNRESET)
	    cc = 0;		/* Other end was fast to close */
	else {
	    int tmp = errno;
	    eprintf(EF_SYSCALL, COMMUNICATION, "read on a tcp socket",
		    "tcp_await_disc_ind", getsyserr());
	    (void)close(ch);
	    errno = tmp;
	    *errind = NOTOK;
	    return;
	}
    }
    if (cc != 0) {
	eprintf(EF_SYSCALL, COMMUNICATION, "receiving data",
		"tcp_await_disc_ind",
		"\nShould get a 0 i.e. EOF = DiscIndication");
    }
    if (close(ch)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close call", "tcp_await_disc_ind",
	       getsyserr());
    }

    *errind = OK;
} /* tcp_await_disc_ind */

/*  */

tcp_create_server(aa_server, aa_addr, errind)
    struct server_t **aa_server;
    struct address_t **aa_addr;
    error_t   *errind;
{

    int s;
    struct sockaddr_in myaddr;
    int len = sizeof(myaddr);
    
    
    pprintf("tcp_create_server(0x%x, 0x%x, 0x%x)\n",
		aa_server, aa_addr, errind);

    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call", "tcp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    myaddr.sin_family = AF_INET;
    myaddr.sin_port = 0;
    myaddr.sin_addr.s_addr = INADDR_ANY;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "tcp_create_server",
	       getsyserr());
	return NOTOK;
    }
    
    if (listen(s, 5) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "listen call", "tcp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    if (getsockname(s, &myaddr, &len) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "tcp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    if (len != sizeof(myaddr)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "tcp_create_server",
	       "\nBad length on returned address");
	*errind = NOTOK;
	return;
    }
    if (get_myipaddr(&myaddr.sin_addr) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "init_myipaddr", "CreateServer");
	*errind = NOTOK;
	return;
    }

    pprintf("Created socket %d with address: 0x%x:%d\n",
		s, myaddr.sin_addr.s_addr, myaddr.sin_port);

    *aa_server = (struct server_t *)malloc(sizeof(struct server_t));
    if (*aa_server == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	return;
    }
    *aa_addr = (struct address_t *)malloc(sizeof(struct address_t));
    if (*aa_addr == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	free((char *)*aa_server);
	return;
    }
    (*aa_server)->sock = s;
    **aa_addr = myaddr;
    *errind = OK;
} /* tcp_create_server */

/*  */

tcp_destroy_server(a_server, errind)
    struct server_t *a_server;
    error_t   *errind;
{
    
    pprintf("tcp_destroy_server(0x%x, 0x%x)\n", a_server, errind);

    if (close(*a_server) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close", "tcp_destroy_server",
	       getsyserr());
	free((char *)a_server);
	*errind = NOTOK;
	return;
    }

    free((char *)a_server);
	
    *errind = OK;
} /* tcp_destroy_server */

/*  */

tcp_report_error(err, str)
    error_t *err;
    char *str;
{
    
    pprintf("tcp_report_error(0x%x)\n", err);
    
    if (err == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "Error", str,
	       getsyserr());
    } else {
	eprintf(EF_SYSCALL, COMMUNICATION, "TCP protocol", str,
	       getsyserr());
    }
} /* tcp_report_error */

