/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * bgprio5: sum the results of all procedures belonging
 * to the same part of the composite benchmark. Check that
 * the first part starts last and stops first (i.e. all
 * but the first part run as background load)
 */

/*
 *
 * PrintResponder switch added: /8911 PerG
 *
 */

#include <general.h>
#include <math.h>       /* need HUGE */

#define BACKGROUND_PRIO 5

extern int PrintResponder;

/*  */

/* VARARGS1 */
bgprio5(req, p1, p2, p3, p4)
    int req;
{
    tprintf("bgprio5(%d)\n", req);

    switch (req) {
    case 0:
	if (result_init(p1,p2,p3,p4) == NOTOK)
	    return NOTOK;
	return bgprio_init(p1, p2, p3, p4);
    case 1:
	return result_handle(p1,p2,p3,p4);
    case 2:
	return result_print(p1,p2,p3,p4);
    default:
	eprintf(EF_IN4, INTERNAL, PARAMETER, "Illegal req",
		"bgprio5 filter");
	return NOTOK;
    }
    /* NOTREACHED */
} /* bgprio5 */

static int nresult = 0;
static statistics_t *sums = NULL, *sqsums = NULL, *mins = NULL, *maxs = NULL;
static statistics_t *rsums = NULL, *rsqsums = NULL, *rmins = NULL,
		    *rmaxs = NULL;  /* responder results */

static result_init()
{
    tprintf("result_init()\n");
    
    nresult = 0;
    if (sums != NULL)
	free((char *)sums);
    if (sqsums != NULL)
	free((char *)sqsums);
    if (mins != NULL)
	free((char *)mins);
    if (maxs != NULL)
	free((char *)maxs);
    if (rsums != NULL)
 	free((char *)rsums);
    if (rsqsums != NULL)
 	free((char *)rsqsums);
    if (rmins != NULL)
	free((char *)rmins);
    if (rmaxs != NULL)
	free((char *)rmaxs);
    
    sums = sqsums = mins = maxs = NULL;
    rsums = rsqsums = rmins = rmaxs = NULL;

} /* result_init */

/*  */

static int result_handle(nstats, stats, rstats, sp)
    int nstats;
    statistics_t stats[];
    statistics_t rstats[];
    struct spec *sp;
{
    statistics_t ress, *res = &ress;
    statistics_t respress, *respres = &respress;
    int i;
    int ret = OK;
    
    tprintf("result_handle(%d, 0x%x, 0x%x, 0x%x)\n",
	    nstats, stats, rstats, sp);

    if (nresult == 0) {
	if (bench_isbasic(spec_bench(sp)))
	    nresult = 1;
	else {
	    struct part *pt;
	    
	    if ((nresult = comp_number(spec_bench(sp))) == NP_UNSPEC) {
		nresult = 0;
		pt = comp_parts(spec_bench(sp));
		while (pt != NULLPT) {
		    nresult++;
		    pt = part_next(pt);
		}
		comp_number(spec_bench(sp)) = nresult;
	    }
	    dprintf("result_handle: nresult set to %d\n", nresult);
	}
	sums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (sums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	sqsums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (sqsums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	mins = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (mins == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	maxs = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (maxs == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
        rsums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rsums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
        rsqsums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rsqsums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	rmins = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rmins == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	rmaxs = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rmaxs == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	
	for (i = 0; i < nresult; i++) {
	    InitStats(&sums[i], (double)0);
	    InitStats(&sqsums[i], (double)0);
	    InitStats(&mins[i], (double)HUGE);
	    InitStats(&maxs[i], (double)-HUGE);
            InitStats(&rsums[i], (double)0);
	    InitStats(&rsqsums[i], (double)0);
	    InitStats(&rmins[i], (double)HUGE);
	    InitStats(&rmaxs[i], (double)-HUGE);
	}
    }
    if (nstats == 0) {
	dprintf(EF_IN3, INTERNAL, "No result to handle", "result_handle");
	return NOTOK;
    }
    if (nstats == 1) {
	*res = stats[0];
	*respres = rstats[0];
	/* update statistics */
	result_update(res, &sums[0], &mins[0], &maxs[0], &sqsums[0]);
        result_update(respres, &rsums[0], &rmins[0], &rmaxs[0], &rsqsums[0]);
    } else {
	int i, part;
	struct part *pt;
	double minstart, maxstop;
	double firststart, firststop;
	statistics_t *st = stats;
        statistics_t *rst = rstats;

	if (bench_isbasic(spec_bench(sp))) {
	    eprintf(EF_IN3, INTERNAL, PARAMETER,
		    "result_update - basic benchmark and many stats");
	    return NOTOK;
	}
	pt = comp_parts(spec_bench(sp));
	for (part = 0; part < nresult; part++, pt = part_next(pt)) {
	    dprintf("result_update: part %d, %d replicas\n", part,
		    part_number(pt));
	    TotalStatistics(part_number(pt), st, res);
	    /* check constraint */
	    if (part == 0) {
		minstart =  st[0].st_starttime;
		maxstop = st[0].st_stoptime;
		for (i = 1; i < part_number(pt); i++) {
		    if (st[i].st_starttime < minstart)
			minstart =  st[i].st_starttime;
		    if (st[i].st_stoptime > maxstop)
			maxstop = st[i].st_stoptime;
		}
		firststart = minstart;
		firststop = maxstop;
	    } else {
		double maxstart, minstop;
		
		maxstart =  st[0].st_starttime;
		minstop = st[0].st_stoptime;
		for (i = 1; i < part_number(pt); i++) {
		    if (st[i].st_starttime > maxstart)
			maxstart =  st[i].st_starttime;
		    if (st[i].st_stoptime < minstop)
			minstop = st[i].st_stoptime;
		}
		if (maxstart > firststart) {
		    eprintf("Part %d started after part 0: %f\n",
			    part, maxstart - firststart);
		    ret = NOTOK;
		}
		if (minstop < firststop) {
		    eprintf("Part %d stopped before part 0: %f\n",
			    part, firststop - minstop);
		    ret = NOTOK;
		}
	    }
	    st = st + part_number(pt);
	}

	if (ret == NOTOK)
	    return ret;
	
	pt = comp_parts(spec_bench(sp));
	st = stats;
	for (part = 0; part < nresult; part++, pt = part_next(pt)) {
	    dprintf("result_update: part %d, %d replicas\n", part,
		    part_number(pt));
	    TotalStatistics(part_number(pt), st, res);
	    minstart =  st[0].st_starttime;
	    maxstop = st[0].st_stoptime;
	    for (i = 1; i < part_number(pt); i++) {
		if (st[i].st_starttime < minstart)
		    minstart =  st[i].st_starttime;
		if (st[i].st_stoptime > maxstop)
		    maxstop = st[i].st_stoptime;
	    }
	    res->st_starttime /= part_number(pt);
	    res->st_stoptime /= part_number(pt);
	    res->st_elapsed = 1000*(maxstop - minstart);

	    /* And the responders */
	    TotalStatistics(part_number(pt), rst, respres);
	    minstart =  rst[0].st_starttime;
	    maxstop = rst[0].st_stoptime;
	    for (i = 1; i < part_number(pt); i++) {
		if (rst[i].st_starttime < minstart)
		    minstart =  rst[i].st_starttime;
		if (rst[i].st_stoptime > maxstop)
		    maxstop = rst[i].st_stoptime;
	    }
	    respres->st_starttime /= part_number(pt);
	    respres->st_stoptime /= part_number(pt);
	    respres->st_elapsed = 1000*(maxstop - minstart);
	    /* update statistics */
	    result_update(res, &sums[part], &mins[part],
			  &maxs[part], &sqsums[part]);

            result_update(respres, &rsums[part], &rmins[part],
			  &rmaxs[part], &rsqsums[part]);
    
	    st = st + part_number(pt);
	    rst = rst + part_number(pt);
	}
    }

    return OK;
} /* result_handle */

/*  */

static int result_update(result, sum, min, max, sqsum)
    statistics_t *result, *sum, *min, *max, *sqsum;
{
    tprintf("result_update(0x%x, 0x%x, 0x%x, 0x%x, 0x%x)\n",
	    result, sum, min, max, sqsum);
    
    SumtoStats(result, sum);
    MintoStats(result, min);
    MaxtoStats(result, max);
    SqsumtoStats(result, sqsum);
} /* result_update */

/*  */

static int result_print(sp, nfailure)
    struct spec *sp;
    int nfailure;
{
    int i;
    int nrun;
    
    tprintf("result_print(0x%x, %d)\n", sp, nfailure);
  
    nrun = spec_number(sp);
    printf("Result for %d runs:\n", nrun);
    if (nfailure != 0)
	printf("Failure percentage: %.2f%%\n",
	       100*(double)nfailure/nrun);

    for (i = 0; i < nresult; i++) {
        printf("Initiator %d:\n", i);
	stats_print("", &sums[i], &mins[i], &maxs[i], &sqsums[i],
		    nrun - nfailure);
	if (PrintResponder) {
	    printf("Responder %d:\n", i);
	    stats_print("r", &rsums[i], &rmins[i], &rmaxs[i], &rsqsums[i],
			nrun - nfailure);
	}
    }
} /* result_print */

/*  */

static int bgprio_init(be)
    struct bench *be;
{
    int i, nproc;
    struct part *pt;

    tprintf("bgprio_init(0x%x)\n", be);

    if (bench_isbasic(be)) {
	eprintf(EF_IN3, INTERNAL, PARAMETER, "bgprio_init");
	return NOTOK;
    }
    nproc = bench_numprocesses(be);

    pt = comp_parts(be);

    pt = part_next(pt);		/* skip foreground processes */
    while (pt != NULLPT) {
	basic_setbackground(part_bench(pt));
#if BACKGROUND_PRIO != 0
	basic_setpriority(part_bench(pt), BACKGROUND_PRIO);
#endif	
	pt = part_next(pt);
    }
    return OK;
} /* bgprio_init */

	
    
