/////////////////////////////////////////////////////////////////
// RESTART.C - 1993, Jeffrey M. Perkel
// Restart is a utility that allows the user to quit or restart
// Windows.
//
// Usage:  RESTART [/R | /E]
//               "R" -- Restart
//               "E" -- Exit
/////////////////////////////////////////////////////////////////

#include <windows.h>
#include <string.h>
#include <ctype.h>      // For definition of TOLOWER (INT C)
#include "restart.h"
                    
// Global Variables

static HINSTANCE ghInstance;
static char szAppName[] = "Restart";
static int Action, Confirm;
static BOOL bSuccess, bAnswer;
static LPSTR lpCommandLine;

// Exported Functions

BOOL FAR PASCAL _export MainDlgProc (HWND hDlg, unsigned message, WORD
        wParam, LONG lParam);
BOOL FAR PASCAL _export AboutDlgProc (HWND hAboutDlg, UINT message, UINT
        wParam, LONG lParam);

// Internal Functions

BOOL ParseCmdLine (LPSTR lpCommandLine);
BOOL MakeItSo (int Action);
BOOL AreYouSure (HWND hDlg, int Action);
int ErrorHandler (HWND hwnd, int iError);

// Program Entry Point

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
        LPSTR lpCmdLine, int nCmdShow)
        {
        BOOL bCmdLineParam = FALSE;
        
        ghInstance = hInstance;
        lpCommandLine = lpCmdLine;
        
        if (!hPrevInstance)
                {
                WNDCLASS wndclass;
                
                wndclass.style = CS_HREDRAW | CS_VREDRAW;
                wndclass.lpfnWndProc = DefDlgProc;
                wndclass.cbClsExtra = 0;
                wndclass.cbWndExtra = DLGWINDOWEXTRA;
                wndclass.hInstance = ghInstance;
                wndclass.hIcon = LoadIcon (NULL, IDI_APPLICATION);
                wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
                wndclass.hbrBackground = GetStockObject (WHITE_BRUSH);
                wndclass.lpszMenuName = NULL;
                wndclass.lpszClassName = szAppName;                
        
                if (!RegisterClass (&wndclass)) {
                        ErrorHandler (NULL, ERR_WNDREGISTRATIONFAIL);
                        return FALSE;
                        }
                }
        
// Determine if RESTART was invoked using command line parameters

        bCmdLineParam = (ParseCmdLine (lpCommandLine));
        if (bCmdLineParam == TRUE) {
                bAnswer = AreYouSure ((HWND) NULL, Action);
                if (bAnswer == TRUE) MakeItSo (Action);
                else return 1;
                }
                
        return DialogBoxParam (ghInstance, szAppName, NULL, (DLGPROC)
                MakeProcInstance ((FARPROC) MainDlgProc, ghInstance),
                (LONG) nCmdShow);
}

///////////////////////////////////////////////////////////////
// MainDlgProc - Dialog Procedure for the Main Program Window
///////////////////////////////////////////////////////////////

BOOL FAR PASCAL _export MainDlgProc (HWND hDlg, unsigned message, WORD
        wParam, LONG lParam)
{
        switch (message)
                {
                case WM_INITDIALOG:
                        ShowWindow (hDlg, LOWORD(lParam));
                        SendDlgItemMessage (hDlg, IDD_CONFIRM_ON,
                                BM_SETCHECK, TRUE, 0L);
                        break;
                
                case WM_CLOSE:
                        EndDialog (hDlg, FALSE);
                        break;
                
                case WM_COMMAND:
                        switch (wParam)
                                {
                              case ID_OK:
                                {
                                Confirm = (IsDlgButtonChecked (hDlg,
                                    IDD_CONFIRM_ON)) ? 1 : 0;
                                Action = (IsDlgButtonChecked (hDlg,
                                    IDD_RESTART)) ? EW_RESTARTWINDOWS :
                                    (IsDlgButtonChecked (hDlg, IDD_EXIT)) ?
                                    NULL : NOCHECK;
                                if (Action == NOCHECK) 
                                        {
                                        MessageBeep(0);
                                        break;
                                        }
                                if (Confirm)
                                        {
                                        bAnswer = AreYouSure (hDlg, Action);
                                        if (!bAnswer) break;
                                        }
                                bSuccess = MakeItSo (Action);
                                if (!bSuccess) {
                                        ErrorHandler (hDlg,
                                                ERR_APPREFUSETERMINATE);
                                        }
                               
                                break;
                               }
                              case ID_CANCEL:
                                        EndDialog (hDlg, FALSE);
                                        break;
                                        
                              case ID_ABOUT:
                                        {
                                        FARPROC lpfnAboutDlgProc;
                                        
                                        lpfnAboutDlgProc = 
                                                MakeProcInstance((FARPROC)
                                                AboutDlgProc, ghInstance); 
                                        DialogBox(ghInstance, 
                                                MAKEINTRESOURCE (AboutBox),
                                                hDlg, (DLGPROC) 
                                                lpfnAboutDlgProc);
                                        FreeProcInstance (
                                                lpfnAboutDlgProc);
                                        break;         
                                        }
                                break;
                                } 
                                break;       
                              default:
                                        return FALSE;
                                }
                                
                        return TRUE;
                        }
                        
                        
///////////////////////////////////////////
// AboutDlgProc - About... box proceedure
///////////////////////////////////////////

BOOL FAR PASCAL _export AboutDlgProc (HWND hAboutDlg, UINT message,
        UINT wParam, LONG lParam)
        
{
        switch (message)
                {
                case WM_INITDIALOG:
                        return TRUE;
                        
                case WM_COMMAND:
                        switch (wParam)
                                {
                                case ID_OK:
                                        EndDialog (hAboutDlg, 0);
                                        return TRUE;
                                }
                        break;
                }
        return FALSE;
}
/////////////////////////////////////////////////////////////
// MakeItSo - This function is the "meat" of the program.
/////////////////////////////////////////////////////////////                                        

BOOL MakeItSo (int Action)
{
        BOOL bResult;
            
        bResult = ExitWindows (Action, 0);
        return bResult;
}                                                  

/////////////////////////////////////////////////////////////
// AreYouSure - Confirm the user's response.
/////////////////////////////////////////////////////////////

BOOL AreYouSure (HWND hDlg, int Action)
{
        char szBuffer[30];
        short nAnswer;
        HWND hwnd;
        
        wsprintf (szBuffer, "%s the Windows System?", (LPSTR) (Action == 
            EW_RESTARTWINDOWS ? "Restart" : "Exit"));
        nAnswer = MessageBox (hDlg, szBuffer,
            "Confirmation", MB_YESNOCANCEL | MB_ICONQUESTION);
        
        if ((nAnswer == IDNO) || (nAnswer == IDCANCEL)) return FALSE;
        else return TRUE;
}

/////////////////////////////////////////////////////////////
// ParseCmdLine - Parses the command line
/////////////////////////////////////////////////////////////

BOOL ParseCmdLine (LPSTR lpCommandLine) {
        
        while ((*lpCommandLine == '/') || (*lpCommandLine == '-')) {
           lpCommandLine++;
                switch (tolower (*lpCommandLine)) {
                   case 'r':
                        Action = EW_RESTARTWINDOWS;
                        return TRUE;
                        break;
                   case 'e':
                        Action = NULL;
                        return TRUE;
                        break;         
                   default:
                        ErrorHandler (NULL, ERR_BADCMDLINESWITCH);
                        return FALSE;
                }      
        }
        if (*lpCommandLine == 0) return FALSE;
}

/////////////////////////////////////////////////////////////
// Error Handler Routine
// Parameters:  hwnd - handle to the parent window
//              iError - error code
/////////////////////////////////////////////////////////////

int ErrorHandler (HWND hwnd, int iError)
{
        char szBuffer[50];
        
        switch (iError) {
                case ERR_BADCMDLINESWITCH:
                        strcpy (szBuffer, "Invalid Command Line Switch!");
                        break;
                
                case ERR_WNDREGISTRATIONFAIL:
                        strcpy (szBuffer, "RegisterClass() failed!");
                        break;
                
                case ERR_APPREFUSETERMINATE:
                        strcpy (szBuffer, 
                        "One or more applications refused to terminate!");
                        break;        
                default:
                        strcpy (szBuffer, "Error!");
                        
                }
        
        return MessageBox (hwnd, szBuffer, "Error!", MB_ICONEXCLAMATION |
                MB_OK); 
}                
