/***************************************************
 * bitcode.c : code bytes for non-8-bit-clean lines
 *
 * Copyright (C) 1992 J.Lepp{j{rvi
 ***************************************************/

#include "system.h"
#include "bitcode.h"

/*
 * BitCode() codes a buffer according to mode.
 * Note that the result buffer is larger than
 * than the source (at max. by a factor of
 * 1.333..). The return value is the exact number
 * of significant bytes in the result buffer.
 *
 * mode : combination of M_BIT5 and M_BIT7
 *        M_BIT5 : bit 5 always set (= no ctrl codes)
 *        M_BIT7 : do not use bit 7
 *        note : M_NONE (0) does not perform any
 *        operation, not even copy bytes as they are
 *
 *
 * source : source buffer
 * result : result buffer (must be 1.333 x source size)
 * nbytes : # of bytes to code (in source)
 *
 * returns : # of bytes in the result buffer after coding
 */

int BitCode(mode,source,result,nbytes)
char *source, *result;
int mode, nbytes;
{
    register unsigned char c;
    register unsigned char *s = (unsigned char *) source;
    register unsigned char *r = (unsigned char *) result;
    unsigned char *_result = r;

    if (mode == (M_BIT5 | M_BIT7))
        for (; nbytes > 0 ;nbytes -= 3)
        {
            c   = s[2] & 0xc0;
            c >>= 2;
            c  |= s[1] & 0xc0;
            c >>= 2;
            c  |= s[0] & 0xc0;
            c >>= 2;
            c  += 32;

            *r++ = c;
            *r++ = (*s++ & 0x3f) + 32;
            *r++ = (*s++ & 0x3f) + 32;
            *r++ = (*s++ & 0x3f) + 32;
        }

    else if (mode == M_BIT5)
        for (; nbytes > 0 ;nbytes -= 7)
        {
            result = (char *) r++;

            c    = (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;
            c   <<= 1;
            c   |= (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;
            c   <<= 1;
            c   |= (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;
            c   <<= 1;
            c   |= (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;
            c   <<= 1;
            c   |= (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;
            c   <<= 1;
            c   |= (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;
            c   <<= 1;
            c   |= (*s & 0x20) ? 1 : 0;
            *r++ = *s++ | 0x20;

            *result = (c & 0x20 ? 0x80 : 0) | c | 0x20;
        }

    else if (mode == M_BIT7)
        for (; nbytes > 0 ;nbytes -= 7)
        {

            result = (char *) r++;

            c    = *s & 0x80;
            *r++ = *s++ & 0x7f;
            c  >>= 1;
            c   |= *s & 0x80;
            *r++ = *s++ & 0x7f;
            c  >>= 1;
            c   |= *s & 0x80;
            *r++ = *s++ & 0x7f;
            c  >>= 1;
            c   |= *s & 0x80;
            *r++ = *s++ & 0x7f;
            c  >>= 1;
            c   |= *s & 0x80;
            *r++ = *s++ & 0x7f;
            c  >>= 1;
            c   |= *s & 0x80;
            *r++ = *s++ & 0x7f;
            c  >>= 1;
            c   |= *s & 0x80;
            *r++ = *s++ & 0x7f;

            *result = c >> 1;
        }

    return((r - _result) + nbytes);
}

/*
 * BitDecode() decodes a buffer build by BitCode().
 * Note that with BitDecode() source may equal to
 * result since the buffer shrinks.
 *
 * mode : mode used to code the buffer
 * source : buffer to be decoded
 * result : buffer to place decoded data
 * nbytes : # of bytes of coded data
 *
 * returns : # of bytes in result
 */

int BitDecode(mode,source,result,nbytes)
char *source, *result;
int mode, nbytes;
{
    register unsigned char c;
    register unsigned char *s = (unsigned char *) source;
    register unsigned char *r = (unsigned char *) result;
    unsigned char *_result = r;

    if (mode == (M_BIT5 | M_BIT7))
        for (; nbytes > 0 ;nbytes -= 4)
        {
            c    = *s++ - 32;
            c  <<= 2;
            *r++ = (*s++ - 32) | (c & 0xc0);
            c  <<= 2;
            *r++ = (*s++ - 32) | (c & 0xc0);
            c  <<= 2;
            *r++ = (*s++ - 32) | (c & 0xc0);
        }

    else if (mode == M_BIT5)
        for (; nbytes > 0 ;nbytes -= 8)
        {
            c    = *s++ & ~0x20;
            c   |= c & 0x80 ? 0x20 : 0;

            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
            c  <<= 1;
            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
            c  <<= 1;
            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
            c  <<= 1;
            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
            c  <<= 1;
            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
            c  <<= 1;
            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
            c  <<= 1;
            *r++ = (*s++ & ~0x20) | (c & 0x40 ? 0x20 : 0);
        }

    else if (mode == M_BIT7)
        for (; nbytes > 0 ;nbytes -= 8)
        {
            c    = *s++;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
            c  >>= 1;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
            c  >>= 1;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
            c  >>= 1;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
            c  >>= 1;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
            c  >>= 1;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
            c  >>= 1;
            *r++ = *s++ | (c & 0x01 ? 0x80 : 0);
        }

    return((r - _result) + nbytes);
}
