unit SendReceiveTests;

interface
uses
{$IFDEF FPC}
  fpcunit,
  testregistry,
{$ELSE}
  TestFramework,
  TestExtensions,
{$ENDIF}
  BTJMSInterfaces,
  BTJMSConnection,
  BTJMSClient;

const
  MSG_COUNT = 100;

type

  TCoreTests = class(TTestCase, IMessageListener)
  private
    Connection: TBTJMSConnection;
    Session: ISession;
    Producer: IMessageProducer;
    Consumer: IMessageConsumer;
    Destination: IDestination;
    CorrelationID: string;

    MsgCount: Integer;
    Started: Int64;

    function Elapsed: Int64;

    procedure ReceiveMain;

    procedure OnJMSTextMessage(Sender: TObject; JMSMessage: ITextMessage);
    procedure OnJMSBytesMessage(Sender: TObject; JMSMessage: IBytesMessage);
    procedure OnMessage(Message: IMessage);

  protected
    procedure AssignCommAdapter; virtual; abstract;
  public
    procedure SetUp; override;
    procedure TearDown; override;

  published
    procedure TestTextMessages; virtual;
    procedure TestBytesMessages; virtual;

  end;

  TSendReceiveIndyTests = class(TCoreTests)
  protected
    procedure AssignCommAdapter; override;
  end;

  TSendReceiveSynapseTests = class(TCoreTests)
  protected
    procedure AssignCommAdapter; override;
  end;

  TSendReceiveIcs6Tests = class(TCoreTests)
  protected
    procedure AssignCommAdapter; override;
  end;

  TSendReceiveScktCompTests = class(TCoreTests)
  protected
    procedure AssignCommAdapter; override;
  end;

implementation
uses

  BTCommAdapterIndy, BTCommAdapterSynapse,
  Forms, DateUtils, SysUtils;

const
  QUEUE_DUNIT = '/queue/unittests';
  TEXT_DUNIT = 'This is test message %d';
  BYTES_DUNIT = #0 + #1 + #0 + #2 + #0 + #3 + '%d';

  { TSendReceiveTests }

function MaxMsec: Integer;
begin
  Result := 6 * MSG_COUNT + 1000;
end;

function TestInfo: string;
begin
  Result := Format('Send %d messages and receive them in less than %d msec',
    [MSG_COUNT, MaxMsec]);
end;

function TCoreTests.Elapsed: Int64;
begin
  Result := MilliSecondOfTheYear(Now) - Started;
end;

procedure TCoreTests.OnJMSTextMessage(Sender: TObject;
  JMSMessage: ITextMessage);
begin
  if JMSMessage.Text = Format(TEXT_DUNIT, [MsgCount + 1]) then
  begin
    if JMSMessage.CorrelationID = CorrelationID then
      Inc(MsgCount)
    else
      Fail('Wrong CorrelationID');
  end else begin
    Fail('Wrong Sequence: ' + TEXT_DUNIT);
  end;
end;

procedure TCoreTests.OnJMSBytesMessage(Sender: TObject;
  JMSMessage: IBytesMessage);
begin
  if JMSMessage.Content = Format(BYTES_DUNIT, [MsgCount + 1]) then
  begin
    if JMSMessage.CorrelationID = CorrelationID then
      Inc(MsgCount)
    else
      Fail('Wrong CorrelationID');
  end else begin
     // Fail('Wrong Sequence: ' + IntToStr(MsgCount));
  end;
end;

procedure TCoreTests.OnMessage(Message: IMessage);
begin
  if Supports(Message, ITextMessage) then
  begin
    OnJMSTextMessage(Self, ITextMessage(Message));
  end
  else if Supports(Message, IBytesMessage) then
  begin
    OnJMSBytesMessage(Self, IBytesMessage(Message));
  end;
end;

procedure TCoreTests.SetUp;

begin
  inherited;

  MsgCount := 0;
  Started := MilliSecondOfTheYear(Now);

  // Create the connection.
  Connection := TBTJMSConnection.Create(TBTJMSClient.Create(nil));
  AssignCommAdapter;
  Connection.Start;

  // Create the session
  Session := Connection.CreateSession(False, amAutoAcknowledge);
  Destination := Session.CreateQueue(QUEUE_DUNIT);

  // Create the producer.
  Producer := Session.CreateProducer(Destination);

  // Create the consumer.
  Consumer := Session.CreateConsumer(Destination);
  Consumer.MessageListener := Self;

end;

procedure TCoreTests.TearDown;
begin
  inherited;

  Connection.Close;
end;

procedure TCoreTests.TestBytesMessages;
var
  CID: TGUID;
  I: Integer;
  Message: IBytesMessage;
begin
  // Create unique CorrelationID
  CreateGUID(CID);
  CorrelationID := GUIDToString(CID);

  for I := 1 to MSG_COUNT do
  begin
    Message := Session.CreateBytesMessage;
    Message.CorrelationID := CorrelationID;
    Message.Content := Format(BYTES_DUNIT, [I]);
    Producer.Send(Message);
  end;

  ReceiveMain;

end;

procedure TCoreTests.TestTextMessages;
var
  CID: TGUID;
  I: Integer;
  Message: ITextMessage;
begin
  // Create unique CorrelationID
  CreateGUID(CID);
  CorrelationID := GUIDToString(CID);

  for I := 1 to MSG_COUNT do
  begin
    Message := Session.CreateTextMessage(Format(TEXT_DUNIT, [I]));
    Message.CorrelationID := CorrelationID;
    Producer.Send(Message);
  end;

  ReceiveMain;
end;

procedure TCoreTests.ReceiveMain;
begin
  while (Elapsed < MaxMsec) do
  begin
    Application.ProcessMessages;
  end;

  CheckEquals(MSG_COUNT, MsgCount);
end;

{ TSendReceiveIndyTests }

procedure TSendReceiveIndyTests.AssignCommAdapter;
begin
  Connection.Transport.CommAdapter := TBTCommAdapterIndy.Create(nil);
end;

{ TSendReceiveSynapseTests }

procedure TSendReceiveSynapseTests.AssignCommAdapter;
begin
  Connection.Transport.CommAdapter := TBTCommAdapterSynapse.Create(nil);
end;

{ TSendReceiveIcs6Tests }

procedure TSendReceiveIcs6Tests.AssignCommAdapter;
begin
  // Conn.CommAdapter := TBTCommAdapterIcs6.Create(Conn);
end;

{ TSendReceiveScktCompTests }

procedure TSendReceiveScktCompTests.AssignCommAdapter;
begin
  // Conn.CommAdapter := TBTCommAdapterScktComp.Create(Conn);
end;

initialization
  RegisterTests(TestInfo, [TSendReceiveIndyTests.Suite]);
  RegisterTests(TestInfo, [TSendReceiveSynapseTests.Suite]);
  (* RegisterTests('Receive and send ' + IntToStr(MSG_COUNT) + ' messages',
    [TSendReceiveIcs6Tests.Suite]);
  RegisterTests('Receive and send ' + IntToStr(MSG_COUNT) + ' messages',
    [TSendReceiveScktCompTests.Suite]); *)
end.

