/*

  Copyright (C) 2003-2005 Smardec. All rights reserved.

  http://www.smardec.com

*/

package com.smardec.asc.gradient;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.ColorModel;


/**
 * <code>QuickGradientPaint</code> is a <code>Paint</code> object that draws gradient fills
 * much more faster than standard's Java <code>java.awt.GradientPaint</code> class.
 */
public class QuickGradientPaint implements Paint {
    private Color startColor, endColor;
    private boolean isVertical;

    /**
     * Creates <code>QuickGradientPaint</code> object with specified colors and direction.
     *
     * @param startColor The start color for gradient fill
     * @param endColor   The end color for gradient fill
     * @param isVertical <code>true</code> if vertical gradient is painted,
     *                   <code>false</code> if horizontal gradient is painted
     */
    public QuickGradientPaint(Color startColor, Color endColor, boolean isVertical) {
        this.startColor = startColor;
        this.endColor = endColor;
        this.isVertical = isVertical;
    }

    /**
     * Creates and returns a <code>PaintContext</code> used to
     * generate the color pattern.
     *
     * @param cm           the {@link ColorModel} that receives the
     *                     <code>Paint</code> data. This is used only as a hint
     * @param deviceBounds the device space bounding box of the graphics primitive being rendered
     * @param userBounds   the user space bounding box of the graphics primitive being rendered
     * @param xform        the {@link AffineTransform} from user space into device space
     * @param hints        the hint that the context object uses to choose between rendering alternatives
     * @return the <code>PaintContext</code> for generating color patterns.
     * @see PaintContext
     */
    public synchronized PaintContext createContext(ColorModel cm, Rectangle deviceBounds, Rectangle2D userBounds, AffineTransform xform, RenderingHints hints) {
        return new QuickGradientPaintContext(cm, deviceBounds, startColor, endColor, isVertical);
    }

    /**
     * Returns the type of transparency.
     *
     * @return The field type of is transparency, which is either <code>OPAQUE</code> or <code>TRANSLUCENT</code>.
     */
    public int getTransparency() {
        int a1 = startColor.getAlpha();
        int a2 = endColor.getAlpha();
        return (((a1 & a2) == 0xff) ? OPAQUE : TRANSLUCENT);
    }
}
