// Syntax.cpp: implementation of the LSyntax* classes.
//

#include "Stdafx.h"
#include "Genfun.h"
#include "Syntax.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// LSyntaxItem Class
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

LSyntaxItem::LSyntaxItem()
{
	o_wName			= 0;				// name resource id
	o_dwTextColor	= 0x000000;			// text color
	o_dwBkgdColor	= 0xffffff;			// background color
	o_lfWeight		= FW_NORMAL;		// font weight
	o_lfItalic		= 0x00;				// italic
	o_lfUnderline	= 0x00;				// underline
	o_lfStrikeOut	= 0x00;				// strikeout

	m_pTextBrush	= 0;				// text brush
	m_pPen			= 0;				// triangles pen
	m_dwFont		= 0;				// font signature

	m_pFont			= 0;				// font
	m_nyChar		= 0;				// character height
	m_nyLead		= 0;				// external leading
	m_nxOverhang	= 0;				// overhang
	m_nxDot			= 0;				// dot character width
	m_nxBlank		= 0;				// blank character width
	m_nxParagraph	= 0;				// paragraph character width
	ZeroMemory (&m_nx, sizeof (m_nx));	// character widths
}

LSyntaxItem::~LSyntaxItem()
{
}

/*======================================================================*/
/* Dedicated pGetSyntax constructor										*/
/*======================================================================*/

LSyntaxItem::LSyntaxItem(
	WORD	wName,					// name resource id
	DWORD	dwTextColor,			// text color
	DWORD	dwBkgdColor,			// background color
	LONG	lfWeight,				// font weight
	BYTE	lfItalic				// italic
	)
{
	o_wName			= wName;		// name resource id
	o_dwTextColor	= dwTextColor;	// text color
	o_dwBkgdColor	= dwBkgdColor;	// background color
	o_lfWeight		= lfWeight;		// font weight
	o_lfItalic		= lfItalic;		// italic
	o_lfUnderline	= 0x00;			// underline
	o_lfStrikeOut	= 0x00;			// strikeout
	o_bUppercase	= FALSE;		// uppercase

	m_pTextBrush	= 0;			// text brush
	m_pPen			= 0;			// triangles pen
	m_dwFont		= 0;			// font signature

	m_pFont			= 0;			// font
	m_nyChar		= 0;			// character height
	m_nyLead		= 0;			// external leading
	m_nxOverhang	= 0;			// overhang
	m_nxDot			= 0;			// dot character width
	m_nxBlank		= 0;			// blank character width
	m_nxParagraph	= 0;			// paragraph character width
	ZeroMemory (&m_nx, sizeof (m_nx)); // character widths
}

/*======================================================================*/
/* Load: load syntax item from registry									*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void LSyntaxItem::Load (
	HKEY hItem					// registry key
	)
{
	o_dwTextColor	= GetDword (hItem, _T("TextColor"));
	o_dwBkgdColor	= GetDword (hItem, _T("BkgdColor"));
	o_lfWeight		= GetDword (hItem, _T("Weight"));
	o_lfItalic		= (BYTE) GetDword (hItem, _T("Italic"));
	o_lfUnderline	= (BYTE) GetDword (hItem, _T("Underline"));
	o_lfStrikeOut	= (BYTE) GetDword (hItem, _T("StrikeOut"));
	o_bUppercase	= (BOOL) GetDword (hItem, _T("Uppercase"));
}

/*======================================================================*/
/* Store: store syntax item into registry								*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void LSyntaxItem::Store (
	HKEY hItem					// registry key
	)
{
	SetDword (hItem, _T("TextColor"), o_dwTextColor);
	SetDword (hItem, _T("BkgdColor"), o_dwBkgdColor);
	SetDword (hItem, _T("Weight"), o_lfWeight);
	SetDword (hItem, _T("Italic"), o_lfItalic);
	SetDword (hItem, _T("StrikeOut"), o_lfStrikeOut);
	SetDword (hItem, _T("Underline"), o_lfUnderline);
	SetDword (hItem, _T("Uppercase"), o_bUppercase);
}

/*======================================================================*/
/* GetWidthArray: get character widths in the ascii range				*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void LSyntaxItem::GetWidthArray (
	CDC* pDC,					// display context
	TEXTMETRIC* ptm				// text metrics
	)
{
	pDC->GetCharWidth (0, 255, m_nx);
	for (int n = 0; n < 256; n++)
		m_nx[n] -= ptm->tmOverhang;

	m_nxDot = m_nx[(unsigned char)''];
	m_nxBlank = m_nx[(unsigned char)' '];
	m_nxParagraph = m_nx[(unsigned char)''];
}

//////////////////////////////////////////////////////////////////////
// LSyntax Class
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

LSyntax::LSyntax()
{
	o_csName = _T("");				// syntax library name
	o_csFile = _T("");				// syntax library file
	o_csTypes = _T("");				// file types handled
	o_dwShading = 0;				// shading delta for nesting levels
	m_hHan = 0;						// syntax library handle
	m_nHan = -1;					// in-library syntax handle
	m_pInitSyntax = 0;				// InitSyntax address
	m_pGetSyntax = 0;				// GetSyntax address
	m_pGetToken = 0;				// GetToken address
	o_csFont = _T("Courier New");	// base font name
	o_lfCharSet = DEFAULT_CHARSET;	// character set
	o_nSize = 10;					// font size
	o_lfWeight = 400;				// weight
	o_lfItalic = '\0';				// italic
	o_lfUnderline = '\0';			// underline
	o_lfStrikeOut = '\0';			// strike out
	m_nxChar = 0;					// average char width
	m_nyChar = 0;					// average char height

	o_csPFont = _T("Courier New");	// printer font name
	o_lfPCharSet = DEFAULT_CHARSET;	// character set
	o_nPSize = 10;					// font size
	o_lfPWeight = 400;				// weight
	o_lfPItalic = '\0';				// italic
	o_lfPUnderline = '\0';			// underline
	o_lfPStrikeOut = '\0';			// strike out
	o_bMirror = TRUE;				// mirror screen font
}

LSyntax::~LSyntax()
{
}

LSyntax::LSyntax (LSyntax& S)
{
	o_csName		= S.o_csName;		// syntax library name
	o_csFile		= S.o_csFile;		// syntax library file
	o_csTypes		= S.o_csTypes;		// file types handled
	o_dwShading		= S.o_dwShading;	// shading delta for nesting levels
	m_hHan			= S.m_hHan;			// syntax library handle
	m_nHan			= S.m_nHan;			// in-library syntax handle
	m_pInitSyntax	= S.m_pInitSyntax;	// InitSyntax address
	m_pGetSyntax	= S.m_pGetSyntax;	// GetSyntax address
	m_pGetToken		= S.m_pGetToken;	// GetToken address

	o_csFont		= S.o_csFont;		// font name
	o_lfCharSet		= S.o_lfCharSet;	// character set
	o_nSize			= S.o_nSize;		// font size
	o_lfWeight		= S.o_lfWeight;		// weight
	o_lfItalic		= S.o_lfItalic;		// italic
	o_lfUnderline	= S.o_lfUnderline;	// underline
	o_lfStrikeOut	= S.o_lfStrikeOut;	// strike out
	m_nxChar		= S.m_nxChar;		// average char width
	m_nyChar		= S.m_nyChar;		// average char height

	o_csPFont		= S.o_csPFont;		// printer font name
	o_lfPCharSet	= S.o_lfPCharSet;	// character set
	o_nPSize		= S.o_nPSize;		// font size
	o_lfPWeight		= S.o_lfPWeight;	// weight
	o_lfPItalic		= S.o_lfPItalic;	// italic
	o_lfPUnderline	= S.o_lfPUnderline;	// underline
	o_lfPStrikeOut	= S.o_lfPStrikeOut;	// strike out
	o_bMirror		= S.o_bMirror;		// mirror screen font

	o_SIA.RemoveAll();
	for (int n = 0; n <= S.o_SIA.GetUpperBound(); n++) {
		LSyntaxItem SI = S.o_SIA.GetAt(n);
		o_SIA.Add(SI);
		}
};

void LSyntax::operator= (LSyntax& S)
{
	o_csName		= S.o_csName;		// syntax library name
	o_csFile		= S.o_csFile;		// syntax library file
	o_csTypes		= S.o_csTypes;		// file types handled
	o_dwShading		= S.o_dwShading;	// shading delta for nesting levels
	m_hHan			= S.m_hHan;			// syntax library handle
	m_nHan			= S.m_nHan;			// in-library syntax handle
	m_pInitSyntax	= S.m_pInitSyntax;	// InitSyntax address
	m_pGetSyntax	= S.m_pGetSyntax;	// GetSyntax address
	m_pGetToken		= S.m_pGetToken;	// GetToken address

	o_csFont		= S.o_csFont;		// font name
	o_lfCharSet		= S.o_lfCharSet;	// character set
	o_nSize			= S.o_nSize;		// font size
	o_lfWeight		= S.o_lfWeight;		// weight
	o_lfItalic		= S.o_lfItalic;		// italic
	o_lfUnderline	= S.o_lfUnderline;	// underline
	o_lfStrikeOut	= S.o_lfStrikeOut;	// strike out
	m_nxChar		= S.m_nxChar;		// average char width
	m_nyChar		= S.m_nyChar;		// average char height

	o_csPFont		= S.o_csPFont;		// printer font name
	o_lfPCharSet	= S.o_lfPCharSet;	// character set
	o_nPSize		= S.o_nPSize;		// font size
	o_lfPWeight		= S.o_lfPWeight;	// weight
	o_lfPItalic		= S.o_lfPItalic;	// italic
	o_lfPUnderline	= S.o_lfPUnderline;	// underline
	o_lfPStrikeOut	= S.o_lfPStrikeOut;	// strike out
	o_bMirror		= S.o_bMirror;		// mirror screen font

	o_SIA.RemoveAll();
	for (int n = 0; n <= S.o_SIA.GetUpperBound(); n++) {
		LSyntaxItem SI = S.o_SIA.GetAt(n);
		o_SIA.Add(SI);
		}
};

void LSyntax::Copy (LSyntax* pS)
{
	o_csName		= pS->o_csName;			// syntax library name
	o_csFile		= pS->o_csFile;			// syntax library file
	o_csTypes		= pS->o_csTypes;		// file types handled
	o_dwShading		= pS->o_dwShading;		// shading delta for nesting levels
	m_hHan			= pS->m_hHan;			// syntax library handle
	m_nHan			= pS->m_nHan;			// in-library syntax handle
	m_pInitSyntax	= pS->m_pInitSyntax;	// InitSyntax address
	m_pGetSyntax	= pS->m_pGetSyntax;		// GetSyntax address
	m_pGetToken		= pS->m_pGetToken;		// GetToken address

	o_csFont		= pS->o_csFont;			// font name
	o_lfCharSet		= pS->o_lfCharSet;		// character set
	o_nSize			= pS->o_nSize;			// font size
	o_lfWeight		= pS->o_lfWeight;		// weight
	o_lfItalic		= pS->o_lfItalic;		// italic
	o_lfUnderline	= pS->o_lfUnderline;	// underline
	o_lfStrikeOut	= pS->o_lfStrikeOut;	// strike out
	m_nxChar		= pS->m_nxChar;			// average char width
	m_nyChar		= pS->m_nyChar;			// average char height

	o_csPFont		= pS->o_csPFont;		// printer font name
	o_lfPCharSet	= pS->o_lfPCharSet;		// character set
	o_nPSize		= pS->o_nPSize;			// font size
	o_lfPWeight		= pS->o_lfPWeight;		// weight
	o_lfPItalic		= pS->o_lfPItalic;		// italic
	o_lfPUnderline	= pS->o_lfPUnderline;	// underline
	o_lfPStrikeOut	= pS->o_lfPStrikeOut;	// strike out
	o_bMirror		= pS->o_bMirror;		// mirror screen font

	o_SIA.RemoveAll();
	for (int n = 0; n <= pS->o_SIA.GetUpperBound(); n++) {
		LSyntaxItem SI = pS->o_SIA.GetAt(n);
		o_SIA.Add(SI);
		}
};

/*======================================================================*/
/* Load: load syntax from registry										*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void LSyntax::Load (
	HKEY hSyntax					// registry key
	)
{
	o_SIA.RemoveAll();				// reinitialize list

	/********************/
	/* load syntax data */
	/********************/

	o_csName = GetCstring (hSyntax, _T("DllName"));
	o_csFile = GetCstring (hSyntax, _T("DllFile"));
	int nFile = o_csFile.GetLength();
	if (nFile == 0) {
		o_csFile = GetCstring (hSyntax, _T("DllPath"));
		int nFile = o_csFile.GetLength();
		LPTSTR lpszFile = o_csFile.GetBuffer(0);
		for (int n = nFile; n > 0;)
			if (*(lpszFile+(--n)) == '\\') {
				o_csFile = lpszFile+(++n);
				break;
				}
		SetLpstr (hSyntax, _T("DllFile"), o_csFile.GetBuffer(0));
		RegDeleteValue (hSyntax, _T("DllPath"));
		}

	o_csTypes = GetCstring (hSyntax, _T("FileTypes"));
	o_dwShading = GetDword (hSyntax, _T("Shading"));

	o_csFont = GetCstring (hSyntax, _T("FontName"));
	o_lfCharSet = (BYTE) GetDword (hSyntax, _T("FontCharSet"));
	o_nSize = GetDword (hSyntax, _T("FontSize"));
	o_lfWeight = GetDword (hSyntax, _T("FontWeight"));
	o_lfItalic = (BYTE) GetDword (hSyntax, _T("FontIsItalic"));
	o_lfUnderline = (BYTE) GetDword (hSyntax, _T("FontIsUnderlined"));
	o_lfStrikeOut = (BYTE) GetDword (hSyntax, _T("FontIsStrikedOut"));

	o_csPFont = GetCstring (hSyntax, _T("PrinterFontName"));
	o_lfPCharSet = (BYTE) GetDword (hSyntax, _T("PrinterFontCharSet"));
	o_nPSize = GetDword (hSyntax, _T("PrinterFontSize"));
	o_lfPWeight	= GetDword (hSyntax, _T("PrinterFontWeight"));
	o_lfPItalic	= (BYTE) GetDword (hSyntax, _T("PrinterFontIsItalic"));
	o_lfPUnderline = (BYTE) GetDword (hSyntax, _T("PrinterFontIsUnderlined"));
	o_lfPStrikeOut = (BYTE) GetDword (hSyntax, _T("PrinterFontIsStrikedOut"));
	o_bMirror = (BOOL) GetDword (hSyntax, _T("MirrorScreenFont"), 1);

	/***********************************/
	/* enumerate and load syntax items */
	/***********************************/

	DWORD dw = 0;
	TCHAR szItem[32];
	DWORD dwItem = sizeof(szItem)/sizeof(TCHAR);

	while (ERROR_SUCCESS == RegEnumKey (hSyntax, dw, szItem, dwItem)) {

		HKEY hItem = 0;
		RegOpenKeyEx (hSyntax, szItem, NULL, KEY_ALL_ACCESS, &hItem);

		LSyntaxItem SI;
		SI.o_wName = atoi (szItem);
		if (hItem) SI.Load (hItem);
		o_SIA.SetAtGrow (dw++, SI);

		if (hItem) RegCloseKey (hItem);
		}
}

/*======================================================================*/
/* Store: store syntax to registry										*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void LSyntax::Store (
	HKEY hSyntax					// registry key
	)
{
	/*********************/
	/* store syntax data */
	/*********************/

	SetLpstr (hSyntax, _T("DllName"), o_csName.GetBuffer(0));
	SetLpstr (hSyntax, _T("DllFile"), o_csFile.GetBuffer(0));
	SetLpstr (hSyntax, _T("FileTypes"), o_csTypes.GetBuffer(0));
	SetDword (hSyntax, _T("Shading"), o_dwShading);

	SetLpstr (hSyntax, _T("FontName"), o_csFont.GetBuffer(0));
	SetDword (hSyntax, _T("FontCharSet"), o_lfCharSet);
	SetDword (hSyntax, _T("FontSize"), o_nSize);
	SetDword (hSyntax, _T("FontWeight"), o_lfWeight);
	SetDword (hSyntax, _T("FontIsItalic"), o_lfItalic);
	SetDword (hSyntax, _T("FontIsUnderlined"), o_lfUnderline);
	SetDword (hSyntax, _T("FontIsStrikedOut"), o_lfStrikeOut);

	SetLpstr (hSyntax, _T("PrinterFontName"), o_csPFont.GetBuffer(0));
	SetDword (hSyntax, _T("PrinterFontCharSet"), o_lfPCharSet);
	SetDword (hSyntax, _T("PrinterFontSize"), o_nPSize);
	SetDword (hSyntax, _T("PrinterFontWeight"), o_lfPWeight);
	SetDword (hSyntax, _T("PrinterFontIsItalic"), o_lfPItalic);
	SetDword (hSyntax, _T("PrinterFontIsUnderlined"), o_lfPUnderline);
	SetDword (hSyntax, _T("PrinterFontIsStrikedOut"), o_lfPStrikeOut);
	SetDword (hSyntax, _T("MirrorScreenFont"), o_bMirror);

	/************************************/
	/* enumerate and store syntax items */
	/************************************/

	for (int n = 0; n <= o_SIA.GetUpperBound(); n++) {

		HKEY hItem = 0;
		DWORD dwDisposition;
		CString cs; cs.Format ("%2d", o_SIA[n].o_wName);

		RegCreateKeyEx (hSyntax, cs.GetBuffer(0), NULL,
			TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
			&hItem, &dwDisposition);

		if (hItem) {
			o_SIA[n].Store (hItem);
			RegCloseKey (hItem);
			}
		}
}

/*======================================================================*/
/* FreeResources: free syntax resources									*/
/* return:																*/
/*		void															*/
/*======================================================================*/

void LSyntax::FreeResources (
	void						// no argument
	)
{
	for (int n = 0; n <= o_SIA.GetUpperBound(); n++) {

		LSyntaxItem SI = o_SIA.GetAt (n);

		/*************************/
		/*  free brushes and pen */
		/*************************/

		if (SI.m_pTextBrush) {
			SI.m_pTextBrush->DeleteObject();
			delete SI.m_pTextBrush;
			SI.m_pTextBrush = 0;
			}

		if (SI.m_pPen) {
			SI.m_pPen->DeleteObject();
			delete SI.m_pPen;
			SI.m_pPen = 0;
			}

		/*************/
		/* free font */
		/*************/

		if (SI.m_pFont) {
			SI.m_pFont->DeleteObject();
			delete SI.m_pFont;
			SI.m_pFont = 0;
			}

		o_SIA.SetAt (n, SI);
		}

	/******************/
	/* unload library */
	/******************/

	if (m_hHan) {
		FreeLibrary (m_hHan);
		m_hHan = 0;
		m_nHan = -1;
		}
	m_pInitSyntax = 0;
	m_pGetSyntax = 0;
	m_pGetToken = 0;
}

/*======================================================================*/
/* AllocateResources: allocate syntax resources							*/
/* return:																*/
/*		0						// ok									*/
/*		SYN_ERRLOADLIB			// could not load library				*/
/*		SYN_ERRPROCADD			// library procedure not found			*/
/*		SYN_ERRINITLIB			// could not initialize library			*/
/*		SYN_ERRALLOC			// memory allocation error				*/
/*======================================================================*/

int LSyntax::AllocateResources (
	DWORD dwTextColor,			// default text color
	DWORD dwBkgdColor,			// default background color
	CDC* pDC					// display context
	)
{
	TEXTMETRIC tm;				// text metrics
	BOOL bFreeDC;				// free DC
	LOGFONT lf;					// logical font
	int nError;					// return code
	CDC DC;						// working DC

	if (pDC == NULL) {
		pDC = &DC;
		pDC->CreateCompatibleDC (NULL);
		bFreeDC = TRUE;
		}
	else
		bFreeDC = FALSE;

	m_nxChar = 0;				// will keep maximum value
	m_nyChar = 0;				// will keep maximum value

	/****************/
	/* load library */
	/****************/

	TCHAR szDllPath[MAX_PATH];
	GetModuleFileName (GetModuleHandle ("Editeur.ocx"), szDllPath, MAX_PATH);
	int nPath;					// offset to directory
	int nRadical;				// offset to file name
	int nSuffix;				// offset to file suffix
	SplitPath (szDllPath, &nPath, &nRadical, &nSuffix);
	nmCat (szDllPath, MAX_PATH, nRadical, o_csFile.GetBuffer(0), o_csFile.GetLength());

	m_hHan = LoadLibrary (szDllPath);
	if (m_hHan == 0)
		return SYN_ERRLOADLIB;

	m_pInitSyntax = GetProcAddress (m_hHan, (LPCSTR) "InitSyntax");
	if (m_pInitSyntax == 0) {
		FreeLibrary (m_hHan);
		return SYN_ERRPROCADD;
		}

	m_pGetSyntax = GetProcAddress (m_hHan, (LPCSTR) "GetSyntax");
	if (m_pGetSyntax == 0) {
		FreeLibrary (m_hHan);
		m_pInitSyntax = 0;
		return SYN_ERRPROCADD;
		}

	m_pGetToken = GetProcAddress (m_hHan, (LPCSTR) "GetToken");
	if (m_pGetToken == 0) {
		FreeLibrary (m_hHan);
		m_pInitSyntax = 0;
		m_pGetSyntax = 0;
		return SYN_ERRPROCADD;
		}

	INITPROC pInitSyntax = (INITPROC) m_pInitSyntax;
	m_nHan = (*pInitSyntax) (-1, o_csName);
	if (m_nHan < 0) {
		FreeLibrary (m_hHan);
		m_pInitSyntax = 0;
		m_pGetSyntax = 0;
		m_pGetToken = 0;
		return SYN_ERRINITLIB;
		}

	try {

		/***********************************/
		/* set up colors - selected syntax */
		/***********************************/

		LSyntaxItem SI = o_SIA.GetAt(0);

		if (SI.o_dwTextColor > 0xFFFFFF) {
			SI.o_dwTextColor &= 0xFF000000;
			SI.o_dwTextColor |= GetSysColor (COLOR_HIGHLIGHTTEXT);
			}

		if (SI.o_dwBkgdColor > 0xFFFFFF) {
			SI.o_dwBkgdColor &= 0xFF000000;
			SI.o_dwBkgdColor |= GetSysColor (COLOR_HIGHLIGHT);
			}

		o_SIA.SetAt (0, SI);

		/**********************************/
		/* set up colors - other syntaxes */
		/**********************************/

		for (int n = 1; n <= o_SIA.GetUpperBound(); n++) {

			LSyntaxItem SI = o_SIA.GetAt(n);

			if (SI.o_dwTextColor > 0xFFFFFF) {
				SI.o_dwTextColor &= 0xFF000000;
				SI.o_dwTextColor |= dwTextColor;
				}

			if (SI.o_dwBkgdColor > 0xFFFFFF) {
				SI.o_dwBkgdColor &= 0xFF000000;
				SI.o_dwBkgdColor |= dwBkgdColor;
				}

			o_SIA.SetAt (n, SI);
			}

		/******************************************/
		/* set up other ressources - all syntaxes */
		/******************************************/

		for (n = 0; n <= o_SIA.GetUpperBound(); n++) {

			LSyntaxItem SI = o_SIA.GetAt(n);

			/**************************/
			/* set up brushes and pen */
			/**************************/

			SI.m_pTextBrush = new CBrush;
			if (SI.m_pTextBrush == 0)
				AfxThrowMemoryException();
			if (SI.m_pTextBrush->CreateSolidBrush (SI.o_dwTextColor & 0xFFFFFF) == 0) {
				delete SI.m_pTextBrush;
				SI.m_pTextBrush = 0;
				AfxThrowMemoryException();
				}

			SI.m_pPen = new CPen;
			if (SI.m_pPen == 0)
				AfxThrowMemoryException();
			if (SI.m_pPen->CreatePen (PS_SOLID, 1, SI.o_dwBkgdColor & 0xFFFFFF) == 0) {
				delete SI.m_pPen;
				SI.m_pPen = 0;
				AfxThrowMemoryException();
				}

			/***************/
			/* set up font */
			/***************/

			ZeroMemory (&lf, sizeof (LOGFONT));
			lstrcpy (lf.lfFaceName, o_csFont);
			lf.lfCharSet = o_lfCharSet;
			lf.lfHeight = o_nSize * 10;
			lf.lfWidth = 0;
			lf.lfWeight = max (o_lfWeight, SI.o_lfWeight);
			lf.lfItalic = max (o_lfItalic, SI.o_lfItalic);
			lf.lfUnderline = max (o_lfUnderline, SI.o_lfUnderline);
			lf.lfStrikeOut = max (o_lfStrikeOut, SI.o_lfStrikeOut);

			SI.m_pFont = new CFont;
			if (SI.m_pFont == 0)
				AfxThrowMemoryException();
			if (SI.m_pFont->CreatePointFontIndirect (&lf, pDC) == 0) {
				delete SI.m_pFont;
				SI.m_pFont = 0;
				AfxThrowMemoryException();
				}

			SI.m_dwFont = lf.lfWeight;
			if (lf.lfItalic)
				SI.m_dwFont |= 0x40000000;
			if (lf.lfUnderline)
				SI.m_dwFont |= 0x20000000;
			if (lf.lfStrikeOut)
				SI.m_dwFont |= 0x10000000;

			/***************/
			/* get metrics */
			/***************/

			CFont* pOldFont = (CFont*) pDC->SelectObject (SI.m_pFont);
			pDC->GetTextMetrics (&tm);
			SI.GetWidthArray (pDC, &tm);
			pDC->SelectObject (pOldFont);

			m_bTrueType = (BOOL) (tm.tmPitchAndFamily & TMPF_TRUETYPE);
			SI.m_nyChar = tm.tmExternalLeading + tm.tmHeight;
			SI.m_nyLead = tm.tmExternalLeading;
			SI.m_nxOverhang = tm.tmOverhang;
			m_nxChar = max (m_nxChar, tm.tmAveCharWidth);
			m_nyChar = max (m_nyChar, SI.m_nyChar);

			o_SIA.SetAt (n, SI);
			}

		/*****************************************************************/
		/* change external leading depending on overall character height */
		/*****************************************************************/

		for (n = 0; n <= o_SIA.GetUpperBound(); n++)
			o_SIA[n].m_nyLead += m_nyChar - o_SIA[n].m_nyChar;

		nError = 0;				// success
		}

	/***********************************************/
	/* free all resources if not totally allocated */
	/***********************************************/

	catch (CMemoryException *e) {
		FreeResources();
		e->Delete();
		nError = SYN_ERRALLOC;	// failure
		}

	if (bFreeDC)
		pDC->DeleteDC();		// delete worker DC

	return nError;
}

/*======================================================================*/
/* AllocatePrintResources: allocate printer syntax resources			*/
/* return:																*/
/*		0						// ok									*/
/*		SYN_ERRLOADLIB			// could not load library				*/
/*		SYN_ERRPROCADD			// library procedure not found			*/
/*		SYN_ERRINITLIB			// could not initialize library			*/
/*		SYN_ERRALLOC			// memory allocation error				*/
/*======================================================================*/

int LSyntax::AllocatePrintResources (
	DWORD dwTextColor,			// default text color
	DWORD dwBkgdColor,			// default background color
	CDC* pDC					// display context
	)
{
	TEXTMETRIC tm;				// text metrics
	LOGFONT lf;					// logical font
	int nError;					// return code
	CDC DC;

	m_nxChar = 0;				// will keep maximum value
	m_nyChar = 0;				// will keep maximum value

	/****************/
	/* load library */
	/****************/

	TCHAR szDllPath[MAX_PATH];
	GetModuleFileName (GetModuleHandle ("Editeur.ocx"), szDllPath, MAX_PATH);
	int nPath;					// offset to directory
	int nRadical;				// offset to file name
	int nSuffix;				// offset to file suffix
	SplitPath (szDllPath, &nPath, &nRadical, &nSuffix);
	nmCat (szDllPath, MAX_PATH, nRadical, o_csFile.GetBuffer(0), o_csFile.GetLength());

	m_hHan = LoadLibrary (szDllPath);
	if (m_hHan == 0)
		return SYN_ERRLOADLIB;

	m_pInitSyntax = GetProcAddress (m_hHan, (LPCSTR) "InitSyntax");
	if (m_pInitSyntax == 0) {
		FreeLibrary (m_hHan);
		return SYN_ERRPROCADD;
		}

	m_pGetSyntax = GetProcAddress (m_hHan, (LPCSTR) "GetSyntax");
	if (m_pGetSyntax == 0) {
		FreeLibrary (m_hHan);
		m_pInitSyntax = 0;
		return SYN_ERRPROCADD;
		}

	m_pGetToken = GetProcAddress (m_hHan, (LPCSTR) "GetToken");
	if (m_pGetToken == 0) {
		FreeLibrary (m_hHan);
		m_pInitSyntax = 0;
		m_pGetSyntax = 0;
		return SYN_ERRPROCADD;
		}

	INITPROC pInitSyntax = (INITPROC) m_pInitSyntax;
	m_nHan = (*pInitSyntax) (-1, o_csName);
	if (m_nHan < 0) {
		FreeLibrary (m_hHan);
		m_pInitSyntax = 0;
		m_pGetSyntax = 0;
		m_pGetToken = 0;
		return SYN_ERRINITLIB;
		}

	try {

		/***********************************/
		/* set up colors - selected syntax */
		/***********************************/

		LSyntaxItem SI = o_SIA.GetAt(0);

		if (SI.o_dwTextColor > 0xFFFFFF) {
			SI.o_dwTextColor &= 0xFF000000;
			SI.o_dwTextColor |= GetSysColor (COLOR_HIGHLIGHTTEXT);
			}

		if (SI.o_dwBkgdColor > 0xFFFFFF) {
			SI.o_dwBkgdColor &= 0xFF000000;
			SI.o_dwBkgdColor |= GetSysColor (COLOR_HIGHLIGHT);
			}

		o_SIA.SetAt (0, SI);

		/**********************************/
		/* set up colors - other syntaxes */
		/**********************************/

		for (int n = 1; n <= o_SIA.GetUpperBound(); n++) {

			LSyntaxItem SI = o_SIA.GetAt(n);

			if (SI.o_dwTextColor > 0xFFFFFF) {
				SI.o_dwTextColor &= 0xFF000000;
				SI.o_dwTextColor |= dwTextColor;
				}

			if (SI.o_dwBkgdColor > 0xFFFFFF) {
				SI.o_dwBkgdColor &= 0xFF000000;
				SI.o_dwBkgdColor |= dwBkgdColor;
				}

			o_SIA.SetAt (n, SI);
			}

		/******************************************/
		/* set up other ressources - all syntaxes */
		/******************************************/

		for (n = 0; n <= o_SIA.GetUpperBound(); n++) {

			LSyntaxItem SI = o_SIA.GetAt(n);

			/**************************/
			/* set up brushes and pen */
			/**************************/

			SI.m_pTextBrush = new CBrush;
			if (SI.m_pTextBrush == 0)
				AfxThrowMemoryException();
			if (SI.m_pTextBrush->CreateSolidBrush (SI.o_dwTextColor & 0xFFFFFF) == 0) {
				delete SI.m_pTextBrush;
				SI.m_pTextBrush = 0;
				AfxThrowMemoryException();
				}

			SI.m_pPen = new CPen;
			if (SI.m_pPen == 0)
				AfxThrowMemoryException();
			if (SI.m_pPen->CreatePen (PS_SOLID, 1, SI.o_dwBkgdColor & 0xFFFFFF) == 0) {
				delete SI.m_pPen;
				SI.m_pPen = 0;
				AfxThrowMemoryException();
				}

			/***************/
			/* set up font */
			/***************/

			ZeroMemory (&lf, sizeof (LOGFONT));

			if (o_bMirror) {
				lstrcpy (lf.lfFaceName, o_csFont);
				lf.lfCharSet = o_lfCharSet;
				lf.lfHeight = o_nSize * 10;
				lf.lfWidth = 0;
				lf.lfWeight = max (o_lfWeight, SI.o_lfWeight);
				lf.lfItalic = max (o_lfItalic, SI.o_lfItalic);
				lf.lfUnderline = max (o_lfUnderline, SI.o_lfUnderline);
				lf.lfStrikeOut = max (o_lfStrikeOut, SI.o_lfStrikeOut);
				}

			else {
				lstrcpy (lf.lfFaceName, o_csPFont);
				lf.lfCharSet = o_lfPCharSet;
				lf.lfHeight = o_nPSize * 10;
				lf.lfWidth = 0;
				lf.lfWeight = max (o_lfPWeight, SI.o_lfWeight);
				lf.lfItalic = max (o_lfPItalic, SI.o_lfItalic);
				lf.lfUnderline = max (o_lfPUnderline, SI.o_lfUnderline);
				lf.lfStrikeOut = max (o_lfPStrikeOut, SI.o_lfStrikeOut);
				}

			SI.m_pFont = new CFont;
			if (SI.m_pFont == 0)
				AfxThrowMemoryException();
			if (SI.m_pFont->CreatePointFontIndirect (&lf, pDC) == 0) {
				delete SI.m_pFont;
				SI.m_pFont = 0;
				AfxThrowMemoryException();
				}

			SI.m_dwFont = lf.lfWeight;
			if (lf.lfItalic)
				SI.m_dwFont |= 0x40000000;
			if (lf.lfUnderline)
				SI.m_dwFont |= 0x20000000;
			if (lf.lfStrikeOut)
				SI.m_dwFont |= 0x10000000;

			/***************/
			/* get metrics */
			/***************/

			CFont* pOldFont = (CFont*) pDC->SelectObject (SI.m_pFont);
			pDC->GetTextMetrics (&tm);
			SI.GetWidthArray (pDC, &tm);
			pDC->SelectObject (pOldFont);

			m_bTrueType = (BOOL) (tm.tmPitchAndFamily & TMPF_TRUETYPE);
			SI.m_nyChar = tm.tmExternalLeading + tm.tmHeight;
			SI.m_nyLead = tm.tmExternalLeading;
			SI.m_nxOverhang = tm.tmOverhang;
			m_nxChar = max (m_nxChar, tm.tmAveCharWidth);
			m_nyChar = max (m_nyChar, SI.m_nyChar);

			o_SIA.SetAt (n, SI);
			}

		/*****************************************************************/
		/* change external leading depending on overall character height */
		/*****************************************************************/

		for (n = 0; n <= o_SIA.GetUpperBound(); n++)
			o_SIA[n].m_nyLead += m_nyChar - o_SIA[n].m_nyChar;

		nError = 0;				// success
		}

	/***********************************************/
	/* free all resources if not totally allocated */
	/***********************************************/

	catch (CMemoryException *e) {
		FreeResources();
		e->Delete();
		nError = SYN_ERRALLOC;	// failure
		}

	return nError;
}

/*======================================================================*/
/* Reset: reset syntax attributes										*/
/* return:																*/
/*		TRUE					// success								*/
/*		FALSE					// failure								*/
/*======================================================================*/

BOOL LSyntax::Reset (
	DWORD dwTextColor,			// default text color
	DWORD dwBkgdColor			// default background color
	)
{
	HKEY hSoftKey = 0;			// Software key
	HKEY hEditKey = 0;			// Editeur key
	HKEY hOcxKey = 0;			// Ocx key
	HKEY hSyntKey = 0;			// Syntax key
	HKEY hTargKey = 0;			// Target key

	try {

		/***************************************/
		/* find current user syntaxes root key */
		/***************************************/

		DWORD dwDisposition;	// REG_CREATED_NEW_KEY or REG_OPENED_EXISTING_KEY 

		int rc = RegCreateKeyEx (HKEY_CURRENT_USER, _T("Software"), NULL,
			TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
			&hSoftKey, &dwDisposition);

		if (rc == 0) rc = RegCreateKeyEx (hSoftKey, _T("Editeur"), NULL,
			TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
			&hEditKey, &dwDisposition);

		if (rc == 0) rc = RegCreateKeyEx (hEditKey, _T("Ocx"), NULL,
			TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
			&hOcxKey, &dwDisposition);

		if (rc == 0) rc = RegCreateKeyEx (hOcxKey, _T("SyntaxChange"), NULL,
			TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
			&hSyntKey, &dwDisposition);

		if (rc == 0) rc = RegCreateKeyEx (hSyntKey, o_csName.GetBuffer(0), NULL,
			TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
			&hTargKey, &dwDisposition);

		/************************/
		/* reset and get syntax */
		/************************/

		LSyntax T;

		if (rc == 0) {

			INITPROC pInitSyntax = (INITPROC) m_pInitSyntax;
			m_nHan = (*pInitSyntax) (m_nHan, o_csName);
			if (m_nHan < 0) return FALSE;

			GETPROC pGetSyntax = (GETPROC) m_pGetSyntax;
			(*pGetSyntax) (m_nHan, o_csName, o_csFile, hSyntKey);
			T.Load (hTargKey);
			}

		/************/
		/* clean up */
		/************/

		if (hTargKey) RegCloseKey (hTargKey);
		if (hSyntKey) RegCloseKey (hSyntKey);
		if (hOcxKey) RegCloseKey (hOcxKey);
		if (hEditKey) RegCloseKey (hEditKey);
		if (hSoftKey) {
			RegFlushKey (hSoftKey);
			RegCloseKey (hSoftKey);
			}

		if (rc) return FALSE;

		/***************************/
		/* reset syntax attributes */
		/***************************/

		o_csTypes = T.o_csTypes;
		o_dwShading = T.o_dwShading;
		o_csFont = T.o_csFont;
		o_lfCharSet = T.o_lfCharSet;
		o_nSize = T.o_nSize;
		o_csPFont = T.o_csPFont;
		o_lfPCharSet = T.o_lfPCharSet;
		o_nPSize = T.o_nPSize;

		LSyntaxItem SI = o_SIA.GetAt(0);
		LSyntaxItem TI = T.o_SIA.GetAt(0);

		SI.o_dwTextColor = TI.o_dwTextColor;		// text color
		if (SI.o_dwTextColor > 0xFFFFFF) {
			SI.o_dwTextColor &= 0xFF000000;
			SI.o_dwTextColor |= GetSysColor (COLOR_HIGHLIGHTTEXT);
			}

		SI.o_dwBkgdColor = TI.o_dwBkgdColor;		// background color
		if (SI.o_dwBkgdColor > 0xFFFFFF) {
			SI.o_dwBkgdColor &= 0xFF000000;
			SI.o_dwBkgdColor |= GetSysColor (COLOR_HIGHLIGHT);
			}

		SI.o_lfWeight		= TI.o_lfWeight;		// font weight
		SI.o_lfItalic		= TI.o_lfItalic;		// italic
		SI.o_lfUnderline	= TI.o_lfUnderline;		// underline
		SI.o_lfStrikeOut	= TI.o_lfStrikeOut;		// strike out
		SI.o_bUppercase		= TI.o_bUppercase;		// uppercase

		o_SIA.SetAt (0, SI);

		for (int n = 1; n <= T.o_SIA.GetUpperBound(); n++) {

			LSyntaxItem SI = o_SIA.GetAt(n);
			LSyntaxItem TI = T.o_SIA.GetAt(n);

			SI.o_dwTextColor = TI.o_dwTextColor;						// text color
			if (SI.o_dwTextColor > 0xFFFFFF) {
				SI.o_dwTextColor &= 0xFF000000;
				SI.o_dwTextColor |= dwTextColor;
				}

			SI.o_dwBkgdColor = TI.o_dwBkgdColor;						// background color
			if (SI.o_dwBkgdColor > 0xFFFFFF) {
				SI.o_dwBkgdColor &= 0xFF000000;
				SI.o_dwBkgdColor |= dwBkgdColor;
				}

			SI.o_lfWeight = max (o_lfWeight, TI.o_lfWeight);			// font weight
			SI.o_lfItalic = max (o_lfItalic, TI.o_lfItalic);			// italic
			SI.o_lfUnderline = max (o_lfUnderline, TI.o_lfUnderline);	// underline
			SI.o_lfStrikeOut = max (o_lfStrikeOut, TI.o_lfStrikeOut);	// strike out

			o_SIA.SetAt (n, SI);
			}

		return TRUE;
		}

	catch (CMemoryException *e) {
		e->Delete();
		}

	return FALSE;
}

/*======================================================================*/
/* UpdateResources: change brushes and fonts							*/
/* return:																*/
/*		TRUE					// ok									*/
/*		FALSE					// ko - new operator failure			*/
/*======================================================================*/

BOOL LSyntax::UpdateResources (
	void						// no argument
	)
{
	TEXTMETRIC tm;				// text metrics
	LOGFONT lf;					// logical font
	BOOL rc;					// return code
	CDC DC;						// worker DC

	DC.CreateCompatibleDC (NULL);

	m_nxChar = 0;				// will keep maximum value
	m_nyChar = 0;				// will keep maximum value

	try {

		/************************/
		/* for all syntax items */
		/************************/

		for (int n = 0; n <= o_SIA.GetUpperBound(); n++) {

			LSyntaxItem SI = o_SIA.GetAt(n);

			/**************************/
			/* change brushes and pen */
			/**************************/

			SI.m_pTextBrush->DeleteObject();
			if (SI.m_pTextBrush->CreateSolidBrush (SI.o_dwTextColor & 0xFFFFFF) == 0) {
				delete SI.m_pTextBrush;
				SI.m_pTextBrush = 0;
				AfxThrowMemoryException();
				}

			SI.m_pPen->DeleteObject();
			if (SI.m_pPen->CreatePen (PS_SOLID, 1, SI.o_dwBkgdColor & 0xFFFFFF) == 0) {
				delete SI.m_pPen;
				SI.m_pPen = 0;
				AfxThrowMemoryException();
				}

			/***************/
			/* change font */
			/***************/

			ZeroMemory (&lf, sizeof (LOGFONT));
			lstrcpy (lf.lfFaceName, o_csFont);
			lf.lfCharSet = o_lfCharSet;
			lf.lfHeight = o_nSize * 10;
			lf.lfWidth = 0;
			lf.lfWeight = max (o_lfWeight, SI.o_lfWeight);
			lf.lfItalic = max (o_lfItalic, SI.o_lfItalic);
			lf.lfUnderline = max (o_lfUnderline, SI.o_lfUnderline);
			lf.lfStrikeOut = max (o_lfStrikeOut, SI.o_lfStrikeOut);

			SI.m_pFont->DeleteObject();
			if (SI.m_pFont->CreatePointFontIndirect (&lf, &DC) == 0) {
				delete SI.m_pFont;
				SI.m_pFont = 0;
				AfxThrowMemoryException();
				}

			SI.m_dwFont = lf.lfWeight;
			if (lf.lfItalic)
				SI.m_dwFont |= 0x40000000;
			if (lf.lfUnderline)
				SI.m_dwFont |= 0x20000000;
			if (lf.lfStrikeOut)
				SI.m_dwFont |= 0x10000000;

			/***************/
			/* get metrics */
			/***************/

			CFont* pOldFont = (CFont*) DC.SelectObject (SI.m_pFont);
			DC.GetTextMetrics (&tm);
			SI.GetWidthArray (&DC, &tm);
			DC.SelectObject (pOldFont);

			m_bTrueType = (BOOL) (tm.tmPitchAndFamily & TMPF_TRUETYPE);
			SI.m_nyChar = tm.tmExternalLeading + tm.tmHeight;
			SI.m_nyLead = tm.tmExternalLeading;
			SI.m_nxOverhang = tm.tmOverhang;
			m_nxChar = max (m_nxChar, tm.tmAveCharWidth);
			m_nyChar = max (m_nyChar, SI.m_nyChar);

			o_SIA.SetAt (n, SI);
			}

		/*****************************************************************/
		/* change external leading depending on overall character height */
		/*****************************************************************/

		for (n = 0; n <= o_SIA.GetUpperBound(); n++)
			o_SIA[n].m_nyLead += m_nyChar - o_SIA[n].m_nyChar;

		rc = TRUE;				// success
		}

	/***********************************************/
	/* free all resources if not totally allocated */
	/***********************************************/

	catch (CMemoryException *e) {
		FreeResources();
		e->Delete();
		rc = FALSE;				// failure
		}

	DC.DeleteDC();				// delete worker DC

	return rc;
}

//////////////////////////////////////////////////////////////////////
// LSyntaxArray Class
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

LSyntaxArray::LSyntaxArray()
{
}

LSyntaxArray::~LSyntaxArray()
{
}

/*======================================================================*/
/* Load: load syntax array from registry								*/
/* return:																*/
/*		void															*/
/*======================================================================*/

BOOL LSyntaxArray::Load (
	DWORD& dwTextColor,			// default text color
	DWORD& dwBkgdColor,			// default background color
	LPCTSTR szSyntKey			// key to load from
	)
{
	for (int n = 0; n <= o_SA.GetUpperBound(); n++)
		o_SA[n].o_SIA.RemoveAll();

	o_SA.RemoveAll();			// initialize list

	HKEY hSoftKey = 0;			// Software key
	HKEY hEditKey = 0;			// Editeur key
	HKEY hOcxKey = 0;			// Ocx key
	HKEY hSyntKey = 0;			// Syntaxes key
	HKEY hDefKey = 0;			// Default key

	/***************************************/
	/* find current user syntaxes root key */
	/***************************************/

	int rc = RegOpenKeyEx (HKEY_CURRENT_USER, _T("Software"), NULL, KEY_ALL_ACCESS, &hSoftKey);
	if (rc == 0) rc = RegOpenKeyEx (hSoftKey, _T("Editeur"), NULL, KEY_ALL_ACCESS, &hEditKey);
	if (rc == 0) rc = RegOpenKeyEx (hEditKey, _T("Ocx"), NULL, KEY_ALL_ACCESS, &hOcxKey);
	if (rc == 0) rc = RegOpenKeyEx (hOcxKey, szSyntKey, NULL, KEY_ALL_ACCESS, &hSyntKey);
	if (rc == 0) rc = RegOpenKeyEx (hSyntKey, _T("Default"), NULL, KEY_ALL_ACCESS, &hDefKey);

	/***************************/
	/* load default color data */
	/***************************/

	dwTextColor = GetDword (hSyntKey, _T("TextColor"), 0xFF000000);
	dwBkgdColor = GetDword (hSyntKey, _T("BkgdColor"), 0xFFFFFFFF);

	/**************************************/
	/* none - load available syntax dll's */
	/**************************************/

	if (rc) {

		/************/
		/* clean up */
		/************/

		if (hDefKey) {
			RegCloseKey (hDefKey);
			hDefKey = 0;
			}
		if (hSyntKey) {
			RegCloseKey (hSyntKey);
			hSyntKey = 0;
			}
		if (hOcxKey) {
			RegCloseKey (hOcxKey);
			hOcxKey = 0;
			}
		if (hEditKey) {
			RegCloseKey (hEditKey);
			hEditKey = 0;
			}
		if (hSoftKey) {
			RegCloseKey (hSoftKey);
			hSoftKey = 0;
			}

		/*************************/
		/* try to open .reg file */
		/*************************/

		TCHAR szRegPath[MAX_PATH];
		GetModuleFileName (NULL, szRegPath, MAX_PATH);
		LPTSTR lp = rChr (szRegPath, '\\');

		try {

			if (!*lp)
				AfxThrowUserException();

			lstrcpy (lp, "\\Editeur.reg");

			/*****************/
			/* open the file */
			/*****************/

			HANDLE hFile = CreateFile (szRegPath,
				GENERIC_READ, FILE_SHARE_READ, NULL,
				OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);

			if (hFile == INVALID_HANDLE_VALUE)
				AfxThrowUserException();

			/*********************************/
			/* allocate a buffer for reading */
			/*********************************/

			long lCurrentPos = _llseek ((HFILE) hFile, 0, 1);
			long lFileLength = _llseek ((HFILE) hFile, 0, 2);
			_llseek ((HFILE) hFile, lCurrentPos, 0);
			if (lFileLength == 0) {
				CloseHandle (hFile);
				AfxThrowUserException();
				}

			HGLOBAL hBuf;					// buffer handle
			LPTSTR lpBuf;					// buffer pointer
			DWORD dwToRead;					// length to read
			DWORD dwRead;					// length actually read

			dwToRead = (DWORD) lFileLength;
			hBuf = GlobalAlloc (GHND, dwToRead + 1);
			if (hBuf == NULL) {
				CloseHandle (hFile);
				AfxThrowUserException();
				}

			/**********************************/
			/* access to the buffer allocated */
			/**********************************/

			lpBuf = (LPTSTR) GlobalLock (hBuf);
			if (lpBuf == (LPTSTR) 0) {
				GlobalFree (hBuf);
				CloseHandle (hFile);
				AfxThrowUserException();
				}

			/***************************/
			/* read and close the file */
			/***************************/

			if (!ReadFile (hFile, lpBuf, dwToRead * sizeof (TCHAR), &dwRead, NULL) ||
				dwRead != dwToRead) {
					CloseHandle (hFile);
					AfxThrowUserException();
				}

			CloseHandle (hFile);

			/**********************/
			/* process file lines */
			/**********************/

			int lChar;			// beginning-of-line character
			int nChar;			// number of end-of-line characters

			for (; *lpBuf; lpBuf += lChar + nChar) {

				/************/
				/* get line */
				/************/

				for (lChar = 0;; lChar++) {

					/*************************************************/
					/* stay on null character in case of end-of-file */
					/*************************************************/

					if (*(lpBuf + lChar) == 0) {
						nChar = 0;
						break;
						}

					/**********************************************************/
					/* delimit line and point next one in case of end-of-line */
					/**********************************************************/

					if (*(lpBuf + lChar) == '\n') {
						nChar = 1;
						if (lChar > 0) {
							if (*(lpBuf + --lChar) == '\r')
								nChar = 2;
							else
								lChar++;
							}
						break;
						}
					}

				if (lChar == 0)
					continue;

				/****************/
				/* process line */
				/****************/

				LPCTSTR lpszSubKeys;	// chain of subkeys
				LPTSTR lpszValueName;	// address of name of value to query

				switch (*lpBuf) {
				case '[':
					lpszSubKeys = 1 + Chr (lpBuf, '\\');
					*(Chr (lpBuf, ']')) = 0;
					break;
				case '"':
					lpszValueName = 1 + lpBuf;
					lp = Chr (lpszValueName, '"');
					*lp = '\0';
					lp += 2;

					switch (*lp) {
					case '"':
						{
						LPBYTE lpbData;			// address of data buffer
						DWORD cbData;			// data buffer size

						lpbData = (LPBYTE) (1 + lp);
						lp = Chr (1 + lp, '"');
						*lp = '\0';
						cbData = (LPBYTE) lp - lpbData;

						if (WriteValue (
							HKEY_CURRENT_USER,	// root key
							lpszSubKeys,		// chain of subkeys
							lpszValueName,		// address of name of value to query
							REG_SZ,				// value type
							lpbData,			// address of data buffer
							cbData				// data buffer size
							))
								AfxThrowUserException();
						}
						break;
					case 'd':
						{
						BYTE bDataR[4];			// data buffer
						BYTE bData[4];			// data buffer

						lp += 6;
						hexbin ((LPBYTE) lp, 8, (LPBYTE) &bDataR);
						bData[0] = bDataR[3];	// move
						bData[1] = bDataR[2];	// to
						bData[2] = bDataR[1];	// little
						bData[3] = bDataR[0];	// endian

						if (WriteValue (
							HKEY_CURRENT_USER,	// root key
							lpszSubKeys,		// chain of subkeys
							lpszValueName,		// address of name of value to query
							REG_DWORD,			// value type
							(LPBYTE) &bData,	// address of data buffer
							4					// data buffer size
							))
								AfxThrowUserException();
						}
						break;
						}
					break;
					}
				}

			/***********************/
			/* done - destroy file */
			/***********************/

			DeleteFile (szRegPath);

			/****************************************************/
			/* try again finding current user syntaxes root key */
			/****************************************************/

			rc = RegOpenKeyEx (HKEY_CURRENT_USER, _T("Software"), NULL, KEY_ALL_ACCESS, &hSoftKey);
			if (rc == 0) rc = RegOpenKeyEx (hSoftKey, _T("Editeur"), NULL, KEY_ALL_ACCESS, &hEditKey);
			if (rc == 0) rc = RegOpenKeyEx (hEditKey, _T("Ocx"), NULL, KEY_ALL_ACCESS, &hOcxKey);
			if (rc == 0) rc = RegOpenKeyEx (hOcxKey, szSyntKey, NULL, KEY_ALL_ACCESS, &hSyntKey);
			if (rc == 0) rc = RegOpenKeyEx (hSyntKey, _T("Default"), NULL, KEY_ALL_ACCESS, &hDefKey);
			}

		catch (CUserException *e) {
			e->Delete();

			/***************************************/
			/* find current user syntaxes root key */
			/***************************************/

			DWORD dwDisposition;		// REG_CREATED_NEW_KEY or REG_OPENED_EXISTING_KEY 

			rc = RegCreateKeyEx (HKEY_CURRENT_USER, _T("Software"), NULL,
				TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
				&hSoftKey, &dwDisposition);

			if (rc == 0) rc = RegCreateKeyEx (hSoftKey, _T("Editeur"), NULL,
				TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
				&hEditKey, &dwDisposition);

			if (rc == 0) rc = RegCreateKeyEx (hEditKey, _T("Ocx"), NULL,
				TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
				&hOcxKey, &dwDisposition);

			if (rc == 0) rc = RegCreateKeyEx (hOcxKey, szSyntKey, NULL,
				TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
				&hSyntKey, &dwDisposition);

			if (rc)
				return FALSE;

			/*****************************/
			/* move to Editeur directory */
			/*****************************/

			TCHAR szSavDir[MAX_PATH];
			GetCurrentDirectory (MAX_PATH, szSavDir);

			TCHAR szDir[MAX_PATH];
			GetModuleFileName (NULL, szDir, MAX_PATH);
			*(rChr (szDir, '\\') + 1) = '\0';
			SetCurrentDirectory (szDir);

			/**************************/
			/* process default syntax */
			/**************************/

			TCHAR szFile[MAX_PATH];
			lstrcpy (szFile, _T("Default.ini"));
			*(rChr (szFile, '.')) = '\0';
			AddSyntax (szFile, dwTextColor, dwBkgdColor, hSyntKey);

			/**************************/
			/* process other syntaxes */
			/**************************/

			WIN32_FIND_DATA FFD;
			HANDLE hSearch = FindFirstFile (_T("*.ini"), &FFD);
			if (hSearch != INVALID_HANDLE_VALUE) do {
				if (FFD.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
					continue;			// not a file
				if (0 == lstrcmp (_T("Default.ini"), FFD.cFileName))
					continue;			// not a file
				lstrcpy (szFile, FFD.cFileName);
				*(rChr (szFile, '.')) = '\0';
				AddSyntax (szFile, dwTextColor, dwBkgdColor, hSyntKey);
				} while (FindNextFile (hSearch, &FFD));

			SetCurrentDirectory (szSavDir);

			/*************************************/
			/* exit if default syntax not loaded */
			/*************************************/

			rc = RegOpenKeyEx (hSyntKey, _T("Default"), NULL, KEY_ALL_ACCESS, &hDefKey);

			return (rc == 0);
			}
		}

	if (rc == 0) {

		/**************************/
		/* process default syntax */
		/**************************/

		DWORD dw = 0;
		TCHAR szSyntax[32];
		DWORD dwSyntax = sizeof(szSyntax)/sizeof(TCHAR);

		while (ERROR_SUCCESS == RegEnumKey (hSyntKey, dw++, szSyntax, dwSyntax)) {

			HKEY hSyntax = 0;
			RegOpenKeyEx (hSyntKey, szSyntax, NULL, KEY_ALL_ACCESS, &hSyntax);

			if (0 == lstrcmp (_T("Default"), szSyntax)) {
				LSyntax S;
				S.o_csName = szSyntax;

				if (hSyntax) {
					S.Load (hSyntax);
					o_SA.Add (S);
					RegCloseKey (hSyntax);
					break;
					}
				}

			RegCloseKey (hSyntax);
			}

		/**************************/
		/* process other syntaxes */
		/**************************/

		dw = 0;
		while (ERROR_SUCCESS == RegEnumKey (hSyntKey, dw++, szSyntax, dwSyntax)) {

			HKEY hSyntax = 0;
			RegOpenKeyEx (hSyntKey, szSyntax, NULL, KEY_ALL_ACCESS, &hSyntax);

			if (0 != lstrcmp (_T("Default"), szSyntax)) {
				LSyntax S;
				S.o_csName = szSyntax;

				if (hSyntax) {
					S.Load (hSyntax);
					o_SA.Add (S);
					}
				}

			RegCloseKey (hSyntax);
			}
		}

	/************/
	/* clean up */
	/************/

	if (hDefKey) RegCloseKey (hDefKey);
	if (hSyntKey) RegCloseKey (hSyntKey);
	if (hOcxKey) RegCloseKey (hOcxKey);
	if (hEditKey) RegCloseKey (hEditKey);
	if (hSoftKey) RegCloseKey (hSoftKey);

	return (rc == 0);
}

/*======================================================================*/
/* Store: store syntax array into registry								*/
/* return:																*/
/*		TRUE					// success								*/
/*		FALSE					// failure								*/
/*======================================================================*/

void LSyntaxArray::Store (
	DWORD dwTextColor,			// default text color
	DWORD dwBkgdColor,			// default background color
	LPCTSTR szSyntKey,			// key to store to
	BOOL bFlush					// flush to registry
	)
{
	HKEY hSoftKey = 0;			// Software key
	HKEY hEditKey = 0;			// Editeur key
	HKEY hOcxKey = 0;			// Ocx key
	HKEY hSyntKey = 0;			// Syntaxes key

	/***************************************/
	/* find current user syntaxes root key */
	/***************************************/

	DWORD dwDisposition;		// REG_CREATED_NEW_KEY or REG_OPENED_EXISTING_KEY 

	int rc = RegCreateKeyEx (HKEY_CURRENT_USER, _T("Software"), NULL,
		TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
		&hSoftKey, &dwDisposition);

	if (rc == 0) rc = RegCreateKeyEx (hSoftKey, _T("Editeur"), NULL,
		TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
		&hEditKey, &dwDisposition);

	if (rc == 0) rc = RegCreateKeyEx (hEditKey, _T("Ocx"), NULL,
		TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
		&hOcxKey, &dwDisposition);

	if (rc == 0) RegDeleteKey (hOcxKey, szSyntKey);

	if (rc == 0) rc = RegCreateKeyEx (hOcxKey, szSyntKey, NULL,
		TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
		&hSyntKey, &dwDisposition);

	if (rc == 0) {

		/****************************/
		/* store default color data */
		/****************************/

		SetDword (hSyntKey, _T("TextColor"), dwTextColor);
		SetDword (hSyntKey, _T("BkgdColor"), dwBkgdColor);

		/*******************************/
		/* enumerate and load syntaxes */
		/*******************************/

		for (int n = 0; n <= o_SA.GetUpperBound(); n++) {

			HKEY hSyntax = 0;

			if (rc == 0) rc = RegCreateKeyEx (hSyntKey, o_SA[n].o_csName.GetBuffer(0), NULL,
				TEXT(""), REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL,
				&hSyntax, &dwDisposition);

			if (hSyntax) {
				o_SA[n].Store (hSyntax);
				RegCloseKey (hSyntax);
				}
			}
		}

	/************/
	/* clean up */
	/************/

	if (hSyntKey) RegCloseKey (hSyntKey);
	if (hOcxKey) RegCloseKey (hOcxKey);
	if (hEditKey) RegCloseKey (hEditKey);
	if (hSoftKey) {
		if (bFlush) RegFlushKey (hSoftKey);
		RegCloseKey (hSoftKey);
		}
}

/*======================================================================*/
/* AddSyntax: add syntax to syntax list									*/
/* return:																*/
/*		0						// ok									*/
/*		GEN_ERROPEN				// could not open file					*/
/*		GEN_ERRALLOC			// could not allocate memory			*/
/*		GEN_ERRLOCK				// could not lock memory				*/
/*		GEN_ERRREAD				// could not read file					*/
/*		GEN_ERRSEC				// dll section not found				*/
/*		SYN_ERRLOADLIB			// could not load library				*/
/*		SYN_ERRPROCADD			// library procedure not found			*/
/*		SYN_ERRINITLIB			// could not initialize library			*/
/*======================================================================*/

int LSyntaxArray::AddSyntax (
	LPCTSTR	lpszName,			// syntax name
	DWORD dwTextColor,			// default text color
	DWORD dwBkgdColor,			// default background color
	HKEY hSyntKey				// syntax key
	)
{
	LSyntax S;					// syntax object
	HGLOBAL hBuf;				// handle to profile data
	LPTSTR pBuf;				// pointer to profile data
	int nCharBeg;				// offset to current profile line
	int nNewLine;				// offset to next line
	int nLef;					// left part length
	LPTSTR pRig;				// pointer to right part
	int nRig;					// right part length
	int nError;					// error code

	/*****************************************************/
	/* read .ini file and get syntax processing dll name */
	/*****************************************************/

	TCHAR szPath[MAX_PATH];
	GetModuleFileName (NULL, szPath, MAX_PATH);
	int nPath;					// offset to directory
	int nRadical;				// offset to file name
	int nSuffix;				// offset to file suffix
	SplitPath (szPath, &nPath, &nRadical, &nSuffix);
	int nOff = nmCat (szPath, MAX_PATH, nRadical, lpszName, lstrlen (lpszName));
	nmCat (szPath, CharSize(szPath), nOff, _T(".ini"), 4);

	nError = FileRead (szPath, &hBuf, &pBuf);
	if (nError) return nError;

	nError = FindProfileSection (_T("Rules"), TRUE, pBuf, &nCharBeg);
	if (nError) return nError;

	while ((nNewLine = GetProfileLine (pBuf + nCharBeg, &nLef, &pRig, &nRig)) != 0) {
		if (niCmp (pBuf + nCharBeg, _T("Dll"), 3) == 0)
			break;
		nCharBeg += nNewLine;
		}

	TCHAR szFile[MAX_PATH];		// file name
	nmCpy (szFile, CharSize(szFile), pRig, nRig);

	GlobalUnlock (hBuf);
	GlobalFree (hBuf);

	/****************/
	/* load library */
	/****************/

	S.o_csName = lpszName;
	S.o_csFile = szFile;

	nmCat (szPath, MAX_PATH, nRadical, S.o_csFile.GetBuffer(0), S.o_csFile.GetLength());

	S.m_hHan = LoadLibrary (szPath);
	if (S.m_hHan == 0)
		return SYN_ERRLOADLIB;

	S.m_pInitSyntax = GetProcAddress (S.m_hHan, (LPCSTR) "InitSyntax");
	if (S.m_pInitSyntax == 0) {
		FreeLibrary (S.m_hHan);
		return SYN_ERRPROCADD;
		}

	S.m_pGetSyntax = GetProcAddress (S.m_hHan, (LPCSTR) "GetSyntax");
	if (S.m_pGetSyntax == 0) {
		FreeLibrary (S.m_hHan);
		S.m_pInitSyntax = 0;
		return SYN_ERRPROCADD;
		}

	S.m_pGetToken = GetProcAddress (S.m_hHan, (LPCSTR) "GetToken");
	if (S.m_pGetToken == 0) {
		FreeLibrary (S.m_hHan);
		S.m_pInitSyntax = 0;
		S.m_pGetSyntax = 0;
		return SYN_ERRPROCADD;
		}

	INITPROC pInitSyntax = (INITPROC) S.m_pInitSyntax;
	S.m_nHan = (*pInitSyntax) (-1, lpszName);
	if (S.m_nHan < 0) {
		FreeLibrary (S.m_hHan);
		S.m_pInitSyntax = 0;
		S.m_pGetSyntax = 0;
		S.m_pGetToken = 0;
		return SYN_ERRINITLIB;
		}

	GETPROC pGetSyntax = (GETPROC) S.m_pGetSyntax;
	(*pGetSyntax) (S.m_nHan, lpszName, szFile, hSyntKey);

	HKEY hSyntax = 0;
	RegOpenKeyEx (hSyntKey, lpszName, NULL, KEY_ALL_ACCESS, &hSyntax);
	
	if (hSyntax) {
		LSyntax T; T.Load (hSyntax);
		RegCloseKey (hSyntax);
		T.o_csName		= lpszName;			// syntax library name
		T.o_csFile		= szFile;			// syntax library file
		T.m_hHan		= S.m_hHan;			// syntax library handle
		T.m_nHan		= S.m_nHan;			// in-library syntax handle
		T.m_pInitSyntax	= S.m_pInitSyntax;	// InitSyntax address
		T.m_pGetSyntax	= S.m_pGetSyntax;	// GetSyntax address
		T.m_pGetToken	= S.m_pGetToken;	// GetToken address
		o_SA.Add (T);
		return o_SA[GetCount()-1].AllocateResources (dwTextColor, dwBkgdColor, NULL);
		}

	return 0;
}

/*======================================================================*/
/* FindSyntax: find syntax based on file type							*/
/* return:																*/
/*		LSyntax&				// syntax reference found				*/
/*======================================================================*/

LSyntax* LSyntaxArray::FindSyntax (
	LPCTSTR lpszName			// file name whose syntax is to be seeked
	)
{
	/****************************************************/
	/* search syntax libraries array for file extension */
	/****************************************************/

	LPTSTR lpszExt = rChr (lpszName, '.');

	if (lpszExt++) {

		for (int n = 0; n <= o_SA.GetUpperBound(); n++) {

			CString csTypes = o_SA[n].o_csTypes;

			LPTSTR lp1, lp2;

			for (lp1 = csTypes.GetBuffer(0), lp2 = lp1; *lp2; lp2++) {
				if (*lp2 == '*' || *lp2 == '.')
					continue;
				if (*lp2 == ';')
					*lp1 = '\0';
				else
					*lp1 = *lp2;
				lp1++;
				}
			*lp1++ = 0;
			*lp1 = 0;

			for (LPTSTR lp = csTypes.GetBuffer(0); *lp; lp += lstrlen(lp)+1)
				if (lstrlen(lpszExt) == lstrlen(lp) &&
					niCmp (lp, lpszExt, lstrlen(lp)) == 0)
						return &o_SA[n];
			}
		}

	/*************************/
	/* return default syntax */
	/*************************/

	return &o_SA[0];
}
