/*
 * Copyright (c) 2002-2007 TeamDev Ltd. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * The complete licence text can be found at
 * http://www.teamdev.com/winpack/license.jsf
 */

import teamdev.jxcapture.DrawableCapture;
import teamdev.jxcapture.RegionCapture;
import teamdev.jxcapture.controllers.SelectionController;
import teamdev.jxcapture.controllers.RegionController;
import teamdev.jxcapture.events.CaptureAdapter;
import teamdev.jxcapture.painters.SelectionPainter;
import teamdev.jxcapture.painters.RegionPainter;
import teamdev.jxcapture.positioners.DiagonalPositioner;
import teamdev.jxcapture.positioners.Positioner;
import teamdev.jxcapture.toolkit.DialogComponent;
import teamdev.jxcapture.toolkit.DrawableArea;
import teamdev.jxcapture.toolkit.Positions;

import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.border.Border;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

/**
 * The example demonstrates how to set the specified dialog for the capture operation.
 * The dialog displays the information like color of the point under mouse cursor.
 *
 * @author Ikryanov Vladimir
 */
public class HelperDialogSample {
    static {
        try {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static void main(String[] args) {
        SelectionPainter painter = new RegionPainter();
        SelectionController controller = new RegionController(painter);
        DrawableCapture captureOperation = new RegionCapture(controller);
        captureOperation.addCaptureListener(new CaptureAdapter() {
            public void cancel() {
                System.out.println("Cancel.");
            }

            public void complete(BufferedImage image) {
                try {
                    ImageIO.write(image, "PNG", new File("C:/region.png"));
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        });
        final MyDialogComponent dialogComponent = new MyDialogComponent();

        final DrawableArea drawableArea = captureOperation.getDrawableArea();
        // setting the cursor image for the capture operation
        drawableArea.setCursor(new Cursor(Cursor.CROSSHAIR_CURSOR));
        // setting the dialog for the capture operation
        drawableArea.setDialogComponent(dialogComponent);

        Component component = drawableArea.getComponent();
        component.addMouseMotionListener(new MouseMotionAdapter() {
            public void mouseMoved(MouseEvent e) {
                Point point = e.getPoint();
                BufferedImage image = drawableArea.getBackgroundImage();
                int rgb = image.getRGB(point.x, point.y);
                Color color = new Color(rgb);
                dialogComponent.setColor(color);
            }
        });
        // executing the capture operation
        captureOperation.execute();
    }

    /**
     * Represents the custom dialog that will be shown during capture operation.
     */
    private static class MyDialogComponent implements DialogComponent {
        private static final int BORDER_WIDTH = 40;
        private JLabel pixelColor = new JLabel(" R: 000, G: 000, B: 000", JLabel.LEFT);

        public MyDialogComponent() {
        }

        public JComponent createComponent() {
            // create mouse panel
            JPanel mousePanel = new JPanel(new BorderLayout());
            mousePanel.add(new JLabel(" Left - Click and drag to select an area ", JLabel.LEFT), BorderLayout.CENTER);
            Border lineBorder = BorderFactory.createLineBorder(Color.GRAY);
            mousePanel.setBorder(BorderFactory.createTitledBorder(lineBorder, " Mouse "));
            // create keyboard panel
            JPanel keyboardPanel = new JPanel(new BorderLayout());
            keyboardPanel.add(new JLabel(" Esc - Abort operation   ", JLabel.LEFT), BorderLayout.CENTER);
            keyboardPanel.setBorder(BorderFactory.createTitledBorder(lineBorder, " Keyboard "));
            // create pixel color panel
            JPanel pixelColorPanel = new JPanel(new BorderLayout());
            pixelColorPanel.add(pixelColor, BorderLayout.CENTER);
            pixelColorPanel.setBorder(BorderFactory.createTitledBorder(lineBorder, " Pixel Color "));
            // create content panel
            JPanel contentPanel = new JPanel(new GridBagLayout());
            contentPanel.setBorder(BorderFactory.createRaisedBevelBorder());
            contentPanel.add(mousePanel, new GridBagConstraints(0, 0, 1, 1, 1.0, 0.0,
                    GridBagConstraints.NORTHWEST, GridBagConstraints.HORIZONTAL, new Insets(5, 5, 5, 5), 0, 0));
            contentPanel.add(keyboardPanel, new GridBagConstraints(0, 1, 1, 1, 1.0, 0.0,
                    GridBagConstraints.NORTHWEST, GridBagConstraints.HORIZONTAL, new Insets(0, 5, 5, 5), 0, 0));
            contentPanel.add(pixelColorPanel, new GridBagConstraints(0, 2, 1, 1, 1.0, 0.0,
                    GridBagConstraints.NORTHWEST, GridBagConstraints.HORIZONTAL, new Insets(0, 5, 5, 5), 0, 0));
            return contentPanel;
        }

        public void setColor(Color color) {
            pixelColor.setText(" R: " + color.getRed() + ", G: " + color.getGreen() + ", B: " + color.getBlue());
        }

        public int getInvisibleBorder() {
            return BORDER_WIDTH;
        }

        public int getLocation() {
            return Positions.TOP_LEFT;
        }

        public Positioner getPositioner() {
            return new DiagonalPositioner();
        }
    }
}
