/*
 * New editor name:  tde, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991
 *
 * This modification of Douglas Thomson's code is released into the
 * public domain, Frank Davis.  You may distribute it freely.
 *
 * This file contains all the structure declarations and defines common
 *  to all the editor modules.
 */

#define MAX_KEYS           206

#define MAX_COLS            80  /* widest screen ever used */
#define MAX_LINES           24  /* highest screen ever used */
#define BUFF_SIZE          256  /* buffer size for lines */
#define MAX_LINE_LENGTH    255  /* longest line allowed in file */
#define UNDO_MAX            12  /* maximum number of lines in undo buffer */

#define ERROR           (-1)          /* abnormal termination */
#define OK              0             /* normal termination */
#define TRUE            1             /* logical true */
#define FALSE           0             /* logical false */

#define RTURN           262           /* Return key = 262 */
#define CONTROL_Z       26            /* Control z  = 26 or DOS eof character */
#define ESC             258           /* Escape key = 258 */

/*
 * The following defines are used by the "error" function, to indicate
 *  how serious the error is.
 */
#define WARNING 1               /* user must acknowledge, editor continues */
#define FATAL   2               /* editor aborts - very rare! */

/*
 * define the type of block marked by user
 */
#define NOTMARKED  0            /* block type undefined */
#define BOX        1            /* block marked by row and column */
#define LINE       2            /* block marked by begin and end lines */

#define MOVE       1
#define DELETE     2
#define COPY       3
#define KOPY       4
#define FILL       5
#define OVERLAY    6

#define LOCAL      1
#define NOT_LOCAL  2
#define GLOBAL     3

#define IGNORE     1
#define MATCH      2

#define PROMPT     1
#define NOPROMPT   2

#define FORWARD    1
#define BACKWARD   2

#define BEGIN      1
#define END        2

#define EXIST       0
#define WRITE       2
#define READ        4
#define READ_WRITE  6

/*
 * possible answers to various questions - see get_yn, get_ynaq and get_oa
 */
#define A_YES       1
#define A_NO        2
#define A_ALWAYS    3
#define A_QUIT      4
#define A_ABORT     5
#define A_OVERWRITE 6
#define A_APPEND    7

/*
 * The following defines specify which video attributes give desired
 *  effects on different display devices.
 * REVERSE is supposed to be reverse video - a different background color,
 *  so that even a blank space can be identified.
 * HIGH is supposed to quickly draw the user's eye to the relevant part of
 *  the screen, either for a message or for matched text in find/replace.
 * NORMAL is supposed to be something pleasant to look at for the main
 *  body of the text.
 * These defines may not be optimal for all types of display. Eventually
 *  the user should be allowed to select which attribute is used where.
 */
#define VIDEO_INT      0x10

#define HERC_REVERSE   0x70
#define HERC_UNDER     0x01
#define HERC_NORMAL    0x07
#define HERC_HIGH      0x0F

#define COLOR_HEAD     0x4b
#define COLOR_TEXT     0x07
#define COLOR_MODE     0x17
#define COLOR_BLOCK    0x7f
#define COLOR_MESSAGE  0x0f
#define COLOR_HELP     0x1A
#define COLOR_DIAG     0x1A
#define COLOR_EOF      0x09

#define COLOR_80       3
#define MONO_80        7

#define VGA             3
#define EGA             2
#define CGA             1
#define MDA             0

/*
 * Some systems (like the PC) require a special kind of pointer for
 *  arrays or structures larger than 64K.  For Microsoft C compilers, I handle
 *  the special pointer arithmetic internally.  See the functions in tdeasm.c
 *  for those special pointer routines.
 */
#ifdef __TURBOC__
   typedef char huge *text_ptr;
#elif __MSC__
   typedef char far * text_ptr;
#else
   typedef char *text_ptr;
#endif

struct vcfg {
   int color;
   int rescan;
   int mode;
   int far *videomem;
};


typedef struct {
   int  pattern_length;
   int  search_case;
   int  search_defined;
   unsigned char pattern[MAX_COLS];
   char skip_forward[256];
   char skip_backward[256];
} boyer_moore_type;


/*
 * "mode_infos" contain the editor mode variables.  The configuration
 *  utility modifies this structure to custimize the start-up tde
 *  configuration
 */
typedef struct {
   int color_scheme;            /* color to start out with */
   int insert;                  /* in insert mode? */
   int indent;                  /* in auto-indent mode? */
   int tab_size;                /* characters between tab stops */
   int backup;                  /* copy file to .bak? */
   int sdel;                    /* stream delete mode */
   int enh_kbd;                 /* type of keyboard */
   char *eof;                   /* message to display at end of file */
} mode_infos;


/*
 * "displays" contain all the status information about what attributes are
 *  used for what purposes, which attribute is currently set, and so on.
 * The editor only knows about one physical screen.
 */
typedef struct {
    int line;                   /* actual line cursor currently on */
    int col;                    /* actual column cursor currently in */
    int nlines;                 /* lines on display device */
    int ncols;                  /* columns on display device */
    int line_length;            /* length of longest line */
    int mode_line;              /* line to display editor modes - fmd */
    int head_color;             /* file header color */
    int text_color;             /* text area color */
    int mode_color;             /* mode line color - footer */
    int block_color;            /* hilited area of blocked region */
    int message_color;          /* color of editor messages */
    int help_color;             /* color of help screen */
    int diag_color;             /* color for diagnostics in mode line */
    int eof_color;              /* color for end of file line */
    int adapter;                /* VGA, EGA, CGA, or MDA? */
    unsigned int overw_cursor;  /* hi part = top scan line, lo part = bottom */
    unsigned int insert_cursor; /* hi part = top scan line, lo part = bottom */
    char far *display_address;  /* address of display memory */
} displays;

/*
 * "status_infos" contain all the editor status information that is
 *  global to the entire editor (i.e. not dependent on the file or
 *  window)
 */
typedef struct {
    struct s_windows *current_window;  /* current active window */
    struct s_file_infos *current_file; /* current active file */
    struct s_file_infos *file_list; /* all active files */
    struct s_windows *window_list; /* all active windows */
    int  window_count;           /* number of windows - displayed and hidden */
    int  file_count;             /* number of files currently open */
    int  next_file_number;      /* number to assign next open file */
    text_ptr start_mem;         /* first char in main text buffer */
    text_ptr end_mem;           /* last char in main text buffer used+1 */
    text_ptr temp_end;          /* temporary end_mem marker */
    text_ptr max_mem;           /* last char available for storage (+1) */
    int  marked;                 /* has block been marked? */
    int  prompt_line;            /* line to display cursor */
    int  prompt_col;             /* column to display cursor */
    struct s_windows *marked_window;   /* pointer to window with marked block */
    struct s_file_infos *marked_file;  /* pointer to file w/ marked block */
    char rw_name[MAX_COLS];      /* name of last file read or written */
    char pattern[MAX_COLS];      /* last search pattern */
    char subst[MAX_COLS];        /* last substitute text */
    int  replace_flag;           /* prompt or noprompt b4 replacing */
    int  overlap;                /* overlap between pages for page up etc */
    text_ptr buff_line;          /* address of line in line_buff */
    int  copied;                 /* has line been copied to file? */
    char line_buff[BUFF_SIZE+2]; /* for currently edited line */
    int  command;                /* function of key just entered */
    int  key_pressed;
    int  stop;                   /* stop indicator */
    int  undo_head;                        /* pointer to last line in buffer */
    char undo_buffer[UNDO_MAX][BUFF_SIZE]; /* room for last 12 changed lines */
} status_infos;

/*
 * "file_infos" contain all the information unique to a given file
 */
typedef struct s_file_infos {
    text_ptr start_text;        /* first char in file */
    text_ptr end_text;          /* last char in file (+1) */
    long length;                /* number of lines in file */
    int modified;               /* file has been modified since save? */
    int dirty;                  /* file in window modified? */
    int new_file;               /* is current file new? */
    char file_name[MAX_COLS];   /* name of current file being edited */
    int block_type;             /* block type - line or box */
    text_ptr block_start;       /* beginning block position */
    text_ptr block_end;         /* ending block position */
    int  block_bc;              /* beginning column */
    long block_br;              /* beginning row */
    int  block_ec;              /* ending column */
    long block_er;              /* ending row */
    int  file_no;               /* file number */
    int  ref_count;             /* no. of windows referring to file */
    int  next_window_letter;    /* letter to assign to next window */
    int  file_attrib;           /* file attributes (rwx etc) */
    struct s_file_infos *next;  /* next file in doubly linked list */
    struct s_file_infos *prev;  /* previous file in doubly linked list */
} file_infos;

/*
 * "windows" contain all the information that is unique to a given
 *  window.
 */
typedef struct s_windows {
    file_infos *file_info;      /* file in window */
    text_ptr cursor;            /* start of line containing cursor */
    int ccol;                   /* column cursor logically in */
    int rcol;                   /* column cursor actually in */
    int bcol;                   /* base column to start display */
    int cline;                  /* line cursor logically in */
    long rline;                 /* real line cursor in */
    int top_line;               /* top line in window */
    int bottom_line;            /* bottom line in window */
    int page;                   /* no. of lines to scroll for one page */
    int visible;                /* window hidden or visible */
    int letter;                 /* window letter */
    struct s_windows *next;     /* next window in doubly linked list */
    struct s_windows *prev;     /* previous window in doubly linked list */
} windows;

/*
 *  Use a dispatch array-structure to determine function of key pressed
 *  by user.  Initialize it in "default.h".  Read in a configuration
 *  file to change the key functions as desired by user, coming soon. ;*)
 */

typedef struct {
   unsigned char func;
} DISPATCH_TABLE;
