
/****************************************************************************
*
*   PROGRAM:  Modified GENERIC.C used to created ROUNDED.EXE
*
*   PURPOSE:  Generic template for Windows applications by Fran Finnegan
*
*   FUNCTIONS:
*
*       WinMain() - calls initialization function, processes message loop
*       InitApplication() - initializes window data and registers window
*       InitInstance() - saves instance handle and creates main window
*       MainWndProc() - processes messages
*       UpdateCorner() - update covered window(s) and then update corner
*       About() - processes messages for "About" dialog box
*
*   COMMENTS:
*
*       Windows can have several copies of your application running at the
*       same time.  The variable mhInst keeps track of which instance this
*       application is so that processing will be to the correct window.
*
*   COPYRIGHT (c) 1993 Finnegan O'Malley & Company Inc.  All Rights Reserved.
*
****************************************************************************/

#include "windows.h"    // required for all Windows applications
#include "generic.h"    // specific to this program

#define WNDMAIN_W       320             // Main window width
#define WNDMAIN_H       240             // Main window height

static  HANDLE          mhInst;         // current instance

static  HWND            mhWndMain;      // Main
                                        // owns:
static  HWND            mhWndClient;    // Client (Edit)
                                        // owns:
static  HWND            mhWndUL;        // Upper Left
static  HWND            mhWndUR;        // Upper Right
static  HWND            mhWndLL;        // Lower Left
static  HWND            mhWndStatus;    // Status
static  HWND            mhWndLR;        // Lower Right

static  void            UpdateCorner( HWND );

/****************************************************************************
*
*   FUNCTION:  WinMain( HANDLE, HANDLE, LPSTR, int )
*
*   PURPOSE:  Calls initialization function, processes message loop
*
*   COMMENTS:
*
*       Windows recognizes this function by name as the initial entry point
*       for the program.  This function calls the application initialization
*       routine, if no other instance of the program is running, and always
*       calls the instance initialization routine.  It then executes a message
*       retrieval and dispatch loop that is the top-level control structure
*       for the remainder of execution.  The loop is terminated when a WM_QUIT
*       message is received, at which time this function exits the application
*       instance by returning the value passed by PostQuitMessage().
*
*       If this function must abort before entering the message loop, it
*       returns the conventional value FALSE.
*
****************************************************************************/

extern  int PASCAL      WinMain
(
        HANDLE          ahInstance,     // current instance
        HANDLE          ahPrevInstance, // previous instance
        LPSTR           alpsCmdLine,    // command line
        int             aiCmdShow       // show-window type (open/icon)
)
{
auto    MSG             aMsg;           // message

if (!ahPrevInstance)                    // Other instances of app running?
    if (!InitApplication( ahInstance )) // Initialize shared things
        return FALSE;                   // Exits if unable to initialize

// Perform initializations that apply to a specific instance
if (!InitInstance( ahInstance, aiCmdShow ))
    return FALSE;

// Acquire and dispatch messages until a WM_QUIT message is received.
while (GetMessage( &aMsg,       // message structure
        0,                      // handle of window receiving the message
        WM_NULL,                // lowest message to examine
        WM_NULL ))              // highest message to examine
    {
    TranslateMessage( &aMsg);   // Translates virtual key codes
    DispatchMessage( &aMsg);    // Dispatches message to window
    }
return 0;
}

/****************************************************************************
*
*   FUNCTION:  InitApplication( HANDLE )
*
*   PURPOSE:  Initializes window data and registers window class
*
*   COMMENTS:
*
*       This function is called at initialization time only if no other
*       instances of the application are running.  This function performs
*       initialization tasks that can be done once for any number of running
*       instances.
*
*       In this case, we initialize a window class by filling out a data
*       structure of type WNDCLASS and calling the Windows RegisterClass()
*       function.  Since all instances of this application use the same window
*       class, we only need to do this when the first instance is initialized.
*
****************************************************************************/

extern  BOOL            InitApplication
(
        HANDLE          ahInstance      // current instance
)
{
auto    WNDCLASS        aWc;

// Fill in window class structure with parameters that describe the
// main window.
aWc.style = 0;                          // Class style(s).
aWc.lpfnWndProc = MainWndProc;          // Function to retrieve messages for
                                        // all windows of this class.
aWc.cbClsExtra = 0;                     // No per-class extra data.
aWc.cbWndExtra = sizeof( WORD );        // Per-window corner-updated flag.
aWc.hInstance = ahInstance;             // Application that owns the class.
aWc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
aWc.hCursor = LoadCursor( NULL, IDC_ARROW );
aWc.hbrBackground = COLOR_BTNFACE + 1;  // COLOR_BTNFACE: normally light gray.
aWc.lpszMenuName = NULL;                // Name of menu resource in .RC file.
aWc.lpszClassName = "Rounded";          // Name used in call to CreateWindow.

// Register the window class and return success/failure code.
return RegisterClass( &aWc );
}

/****************************************************************************
*
*   FUNCTION:  InitInstance( HANDLE, int )
*
*   PURPOSE:  Saves instance handle and creates main window
*
*   COMMENTS:
*
*       This function is called at initialization time for every instance of
*       this application.  This function performs initialization tasks that
*       cannot be shared by multiple instances.
*
*       In this case, we save the instance handle in a static variable and
*       create and display the main program window.
*
****************************************************************************/

extern  BOOL            InitInstance
(
        HANDLE          ahInstance,     // Current instance identifier.
        int             aiCmdShow       // Param for first ShowWindow() call.
)
{
auto    RECT            aRt;

// Save the instance handle in static variable, which will be used in
// many subsequence calls from this application to Windows.
mhInst = ahInstance;

// Create a main window for this application instance.
mhWndMain = CreateWindow(
        "Rounded",                      // See RegisterClass() call.
        "Rounded Window Corners",       // Text for window title bar.
        WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,
        CW_USEDEFAULT,                  // Default horizontal position.
        CW_USEDEFAULT,                  // Default vertical position.
        WNDMAIN_W,                      // Width.
        WNDMAIN_H,                      // Height.
        0,                              // Overlapped windows have no parent.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

GetWindowRect( mhWndMain, &aRt );

mhWndClient = CreateWindow(
        "Edit",                         // Standard "Edit" class.
        "\r\n\r\nDemo by:\r\n\r\n\tFran Finnegan"
        "\r\n\tFinnegan O'Malley & Company Inc."
        "\r\n\tSan Francisco, California"
        "\r\n\r\nSo you thought this was impossible!\r\n",
        WS_POPUP | WS_BORDER | WS_VSCROLL |
                ES_AUTOVSCROLL | ES_MULTILINE | ES_WANTRETURN,
        aRt.left             - 20,      // Horizontal position.
        aRt.top              + 19,      // Vertical position.
        aRt.right - aRt.left + 40,      // Width.
        aRt.bottom - aRt.top - 38,      // Height.
        mhWndMain,                      // Owner.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

mhWndUL = CreateWindow(
        "Rounded",                      // See RegisterClass() call.
        "1",                            // Text for window title bar.
        WS_POPUP,                       // Window style.
        aRt.left             - 20,      // Horizontal position.
        aRt.top                  ,      // Vertical position.
                               20,      // Width.
                               20,      // Height.
        mhWndClient,                    // Owner.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

mhWndUR = CreateWindow(
        "Rounded",                      // See RegisterClass() call.
        "2",                            // Text for window title bar.
        WS_POPUP,                       // Window style.
        aRt.right                ,      // Horizontal position.
        aRt.top                  ,      // Vertical position.
                               20,      // Width.
                               20,      // Height.
        mhWndClient,                    // Owner.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

mhWndLL = CreateWindow(
        "Rounded",                      // See RegisterClass() call.
        "3",                            // Text for window title bar.
        WS_POPUP,                       // Window style.
        aRt.left             - 20,      // Horizontal position.
        aRt.bottom           - 20,      // Vertical position.
                               20,      // Width.
                               20,      // Height.
        mhWndClient,                    // Owner.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

mhWndStatus = CreateWindow(
        "Rounded",                      // See RegisterClass() call.
        "Status",                       // Text for window title bar.
        WS_POPUP | WS_BORDER,           // Window style.
        aRt.left                 ,      // Horizontal position.
        aRt.bottom - aRt.top - 20,      // Vertical position.
        aRt.right - aRt.left     ,      // Width.
                               20,      // Height.
        mhWndClient,                    // Owner.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

mhWndLR = CreateWindow(
        "Rounded",                      // See RegisterClass() call.
        "4",                            // Text for window title bar.
        WS_POPUP,                       // Window style.
        aRt.right                ,      // Horizontal position.
        aRt.bottom           - 20,      // Vertical position.
                               20,      // Width.
                               20,      // Height.
        mhWndClient,                    // Owner.
        0,                              // The window class menu.
        mhInst,                         // This instance owns this window.
        NULL                            // Pointer not needed.
        );

// If windows could not be created, return "failure"
if (!mhWndMain
||  !mhWndClient
||  !mhWndUL
||  !mhWndUR
||  !mhWndLL
||  !mhWndStatus
||  !mhWndLR)
    return FALSE;

//  {       // debugging output
//  auto    char            aszText[ 128 ];
//
//  wsprintf( aszText,
//          "%X\tMain\r\n"
//          "%X\tClient\r\n"
//          "%X\tUL\r\n"
//          "%X\tUR\r\n"
//          "%X\tLL\r\n"
//          "%X\tStatus\r\n"
//          "%X\tLR\r\n",
//          mhWndMain,
//          mhWndClient,
//          mhWndUL,
//          mhWndUR,
//          mhWndLL,
//          mhWndStatus,
//          mhWndLR );
//  SetWindowText( mhWndClient, aszText );
//  }

// Make the windows visible; update the client areas; and return "success"
ShowWindow( mhWndMain, aiCmdShow );
return TRUE;
}

/****************************************************************************
*
*   FUNCTION:  MainWndProc( HWND, UINT, WPARAM, LPARAM )
*
*   PURPOSE:  Processes messages
*
*   MESSAGES:
*
*       WM_COMMAND    - application menu (About dialog box)
*       WM_DESTROY    - destroy window
*       etc. (see Windows Q&A column)
*
*   COMMENTS:
*
*       To process the IDM_ABOUT message, call MakeProcInstance() to get the
*       current instance address of the About() function.  Then call Dialog
*       box which will create the box according to the information in your
*       generic.rc file and turn control over to the About() function.  When
*       it returns, free the intance address.
*
****************************************************************************/

extern  long FAR PASCAL MainWndProc
(
        HWND            ahWnd,          // window handle
        UINT            auiMessage,     // type of message
        WPARAM          awParam,        // additional information
        LPARAM          alParam         // additional information
)
{
auto    RECT            aRt;
auto    char            aszText[ 64 ];

switch (auiMessage)
    {
//  case WM_COMMAND:            // message: command from application menu
//      if (awParam == IDM_ABOUT)
//          {
//          auto    FARPROC alpProcAbout;   // pointer to the "About" function
//
//          alpProcAbout = MakeProcInstance( About, mhInst );
//          DialogBox( mhInst,          // current instance
//                  "AboutBox",         // resource to use
//                  ahWnd,              // parent handle
//                  alpProcAbout );     // About() instance address
//
//          FreeProcInstance( alpProcAbout );
//          break;
//          }
//      else                            // Lets Windows process it
//          return DefWindowProc( ahWnd, WM_COMMAND, awParam, alParam );

    case WM_CREATE:
        SetWindowWord( ahWnd, 0, FALSE );   // turn off corner-updated flag
        break;

    case WM_DESTROY:            // message: window being destroyed
        if (ahWnd == mhWndMain)
            PostQuitMessage( 0 );
        break;

    case WM_CLOSE:
        if (IsWindow( mhWndUL     ))  DestroyWindow( mhWndUL     );
        if (IsWindow( mhWndUR     ))  DestroyWindow( mhWndUR     );
        if (IsWindow( mhWndLL     ))  DestroyWindow( mhWndLL     );
        if (IsWindow( mhWndStatus ))  DestroyWindow( mhWndStatus );
        if (IsWindow( mhWndLR     ))  DestroyWindow( mhWndLR     );
    //  DestroyWindow( mhWndClient );   // visuals better if destroyed last
        DestroyWindow( mhWndMain   );   // this will destroy mhWndClient last
        break;

    case WM_GETMINMAXINFO:
        // lpmmi = (MINMAXINFO FAR*)alParam;    // address of structure
        ((MINMAXINFO FAR*)alParam)->ptMinTrackSize.x = 0;
        ((MINMAXINFO FAR*)alParam)->ptMinTrackSize.y = 0;
        break;

    case WM_ACTIVATEAPP:
        // fActive = (BOOL)awParam;     // activation/deactivation flag
        // htask = (HTASK)LOWORD( alParam );    // task handle
        SendMessage( mhWndMain, WM_NCACTIVATE, awParam, 0L );
        break;

    case WM_SETFOCUS:
        // hwnd = (HWND)awParam;        // handle of window losing focus
        if (ahWnd != mhWndMain
        &&  IsWindow( mhWndClient ))
            {
            SetFocus( mhWndClient );
            SendMessage( mhWndMain, WM_NCACTIVATE, TRUE, 0L );
            }
        break;

    case WM_KILLFOCUS:
        // hwndGetFocus = (HWND)awParam;  // handle of window receiving focus
        if (ahWnd   == mhWndMain
        && (awParam == mhWndClient
        ||  IsChild( mhWndClient, (HWND)awParam )))
            SendMessage( mhWndMain, WM_NCACTIVATE, TRUE, 0L );
        break;

    case WM_MOUSEACTIVATE:
        // hwndTopLevel = (HWND)awParam;        // handle of top-level parent
        // wHitTestCode = LOWORD( alParam );    // hit-test code
        // wMsg = HIWORD( alParam );            // mouse-message identifier
        if (ahWnd != mhWndMain
        &&  GetFocus() == mhWndClient)
            return MA_NOACTIVATEANDEAT;         // prevents flashing
        else
            return DefWindowProc( ahWnd, WM_MOUSEACTIVATE, awParam, alParam );

    case WM_MOVE:
        // xPos = (int)LOWORD( alParam );   // horz position of client area
        // yPos = (int)HIWORD( alParam );   // vert position of client area
        if (ahWnd == mhWndMain
        && !IsIconic( mhWndMain )
        &&  IsWindow( mhWndClient ))
            {
            auto    int             aiXPos;
            auto    int             aiYPos;

            GetWindowRect( mhWndMain, &aRt );   // need window's, not client's
            aiXPos = aRt.left;  // horizontal position of window area
            aiYPos = aRt.top;   // vertical position of window area

            SetWindowPos( mhWndUL    , 0,
                    aiXPos - 20       , aiYPos                 , 0, 0,
                    SWP_NOZORDER | SWP_NOSIZE | SWP_NOACTIVATE );
            SetWindowPos( mhWndUR    , 0,
                    aiXPos + WNDMAIN_W, aiYPos                 , 0, 0,
                    SWP_NOZORDER | SWP_NOSIZE | SWP_NOACTIVATE );
            SetWindowPos( mhWndClient, 0,
                    aiXPos - 20       , aiYPos + 19            , 0, 0,
                    SWP_NOZORDER | SWP_NOSIZE                  );
            SetWindowPos( mhWndLL    , 0,
                    aiXPos - 20       , aiYPos + WNDMAIN_H - 20, 0, 0,
                    SWP_NOZORDER | SWP_NOSIZE | SWP_NOACTIVATE );
            SetWindowPos( mhWndStatus, 0,
                    aiXPos            , aiYPos + WNDMAIN_H - 20, 0, 0,
                    SWP_NOZORDER | SWP_NOSIZE | SWP_NOACTIVATE );
            SetWindowPos( mhWndLR    , 0,
                    aiXPos + WNDMAIN_W, aiYPos + WNDMAIN_H - 20, 0, 0,
                    SWP_NOZORDER | SWP_NOSIZE | SWP_NOACTIVATE );

            // Client (Edit) show
            ShowWindow( mhWndClient, SW_SHOW );  // visually better if first

            // Upper Left corner
            UpdateCorner( mhWndUL );

            // Upper Right corner
            UpdateCorner( mhWndUR );

            // Lower Left corner
            UpdateCorner( mhWndLL );

            // Status show
            ShowWindow( mhWndStatus, SW_SHOWNA );

            // Lower Right corner
            UpdateCorner( mhWndLR );

            // Status and Client (Edit) updates
            UpdateWindow( mhWndStatus );
            UpdateWindow( mhWndClient );    // last, since this may take time
            }
        break;

    case WM_PAINT:
        {
        auto    HDC             ahDc;
        auto    PAINTSTRUCT     aPs;

        ahDc = BeginPaint( ahWnd, &aPs );
        if (ahWnd == mhWndStatus)       // Status window
            {
            auto    HPEN            ahPen;

            GetClientRect( ahWnd, &aRt );

            ahPen = CreatePen( PS_SOLID, 1,
                                GetSysColor( COLOR_BTNSHADOW ) );
            ahPen = SelectObject( ahDc, ahPen );
            MoveTo( ahDc, 2             , 1              );  // top
            LineTo( ahDc, aRt.right  - 3, 1              );  // horizontal
            MoveTo( ahDc, 2             , 2              );  // left
            LineTo( ahDc, 2             , aRt.bottom - 2 );  // vertical
            ahPen = SelectObject( ahDc, ahPen );
            DeleteObject( ahPen );

            ahPen = CreatePen( PS_SOLID, 1,
                                GetSysColor( COLOR_BTNHIGHLIGHT ) );
            ahPen = SelectObject( ahDc, ahPen );
            MoveTo( ahDc, aRt.right  - 3, 2              );  // right
            LineTo( ahDc, aRt.right  - 3, aRt.bottom - 2 );  // vertical
            MoveTo( ahDc, 3             , aRt.bottom - 2 );  // bottom
            LineTo( ahDc, aRt.right  - 2, aRt.bottom - 2 );  // horizontal
            ahPen = SelectObject( ahDc, ahPen );
            DeleteObject( ahPen );

            SetTextColor( ahDc, GetSysColor( COLOR_BTNTEXT ) );
            SetBkMode( ahDc, TRANSPARENT );
            GetWindowText( ahWnd, aszText, sizeof( aszText ) );
            TextOut( ahDc, 4, 0, aszText, lstrlen( aszText ) );
            }
        EndPaint( ahWnd, &aPs );

        if (ahWnd == mhWndMain)     // if painting Main, Client was destroyed
            {
            if (!IsWindow( mhWndClient ))
                PostMessage( mhWndMain, WM_CLOSE, 0, 0L );
            }
        else    // a potential corner, so test the corner-updated flag:
            if (GetWindowWord( ahWnd, 0 ))      // if corner was updated,
                UpdateCorner( ahWnd );          // it must now be re-updated
        break;
        }

    case WM_ERASEBKGND:
        // hdc = (HDC)awParam;  // device-context handle
        if (ahWnd == mhWndMain)         // visually best to not paint Main
            return 0;
        GetWindowText( ahWnd, aszText, sizeof( aszText ) );
        switch (((int *)aszText)[0])    // test for a corner window
            {
            case '1':   // Upper Left
            case '2':   // Upper Right
            case '3':   // Lower Left
            case '4':   // Lower Right
                DrawIcon( (HDC)awParam, 0, 0, LoadIcon( mhInst,
                        MAKEINTRESOURCE( aszText[0] - ('1' - 1) ) ) );
                return 0;
            }
        // otherwise, fall through
    default:                    // Passes it on if unproccessed
        return DefWindowProc( ahWnd, auiMessage, awParam, alParam );
    }
return 0;
}

/****************************************************************************
*
*   FUNCTION:  UpdateCorner( HWND )
*
*   PURPOSE:  Update covered window(s) and then update corner
*
*   COMMENTS:
*
*       Update the covered window(s) by checking the window(s) under each
*       corner of a corner window.  Since our corners may overlap our
*       Client (Edit) window by a pixel, adjust the corners vertically
*       inward by one pixel beforehand.  Once the window(s) are updated,
*       update the corner window.  Turn off the corner-updated flag
*       beforehand, and turn it on afterward.
*
****************************************************************************/

static  void            UpdateCorner
(
        HWND            ahWndCorner
)
{
auto    RECT            aRt;
auto    POINT           aPt;
auto    HWND            ahWndCovered;

SetWindowWord( ahWndCorner, 0, FALSE );  // turn off corner-updated flag
GetWindowRect( ahWndCorner, &aRt );
SetWindowPos( ahWndCorner, 0, 0, 0, 0, 0, SWP_NOZORDER | SWP_NOMOVE |
        SWP_NOSIZE | SWP_HIDEWINDOW | SWP_NOACTIVATE );  // hide corner

aPt.x = aRt.left      ; // upper left of corner
aPt.y = aRt.top    + 1;
if (ahWndCovered = WindowFromPoint( aPt ))
    {
    UpdateWindow( ahWndCovered );
    if (ahWndCovered = GetParent( ahWndCovered ))
        UpdateWindow( ahWndCovered );
    }

aPt.x = aRt.right  - 1; // upper right of corner
aPt.y = aRt.top    + 1;
if (ahWndCovered = WindowFromPoint( aPt ))
    {
    UpdateWindow( ahWndCovered );
    if (ahWndCovered = GetParent( ahWndCovered ))
        UpdateWindow( ahWndCovered );
    }

aPt.x = aRt.left      ; // lower left of corner
aPt.y = aRt.bottom - 2;
if (ahWndCovered = WindowFromPoint( aPt ))
    {
    UpdateWindow( ahWndCovered );
    if (ahWndCovered = GetParent( ahWndCovered ))
        UpdateWindow( ahWndCovered );
    }

aPt.x = aRt.right  - 1; // lower right of corner
aPt.y = aRt.bottom - 2;
if (ahWndCovered = WindowFromPoint( aPt ))
    {
    UpdateWindow( ahWndCovered );
    if (ahWndCovered = GetParent( ahWndCovered ))
        UpdateWindow( ahWndCovered );
    }

SetWindowPos( ahWndCorner, 0, 0, 0, 0, 0, SWP_NOZORDER | SWP_NOMOVE |
        SWP_NOSIZE | SWP_SHOWWINDOW | SWP_NOACTIVATE );  // show corner
UpdateWindow( ahWndCorner );    // corner is now painted over covered window
SetWindowWord( ahWndCorner, 0, TRUE );  // turn on corner-updated flag
}

/****************************************************************************
*
*   FUNCTION:  About( HWND, unsigned, WORD, LONG )
*
*   PURPOSE:  Processes messages for "About" dialog box
*
*   MESSAGES:
*
*       WM_INITDIALOG - initialize dialog box
*       WM_COMMAND    - Input received
*
*   COMMENTS:
*
*       No initialization is needed for this particular dialog box, but TRUE
*       must be returned to Windows.
*
*       Wait for user to click on "Ok" button, then close the dialog box.
*
****************************************************************************/

extern  BOOL FAR PASCAL About
(
        HWND            ahDlg,          // window handle of the dialog box
        unsigned        auiMessage,     // type of message
        WORD            awParam,        // message-specific information
        LONG            alParam
)
{
//  switch (auiMessage)
//      {
//      case WM_INITDIALOG:                 // message: initialize dialog box
//          return TRUE;
//
//      case WM_COMMAND:                    // message: received a command
//          if (awParam == IDOK             // "OK" box selected?
//          ||  awParam == IDCANCEL)        // System menu close command?
//              {
//              EndDialog( ahDlg, TRUE );   // Exits the dialog box
//              return TRUE;
//              }
//          break;
//      }
return FALSE;                           // Didn't process a message
}

