/*******************************************************************

    wormtype.h

    This file contains the global type definitions for the
    Windows Sockets Wormhole project.

********************************************************************/


#ifndef _WORMTYPE_H_
#define _WORMTYPE_H_


//
//  Some useful types missing from 16-bit Windows.
//

#ifndef WIN32

typedef char            CHAR;
typedef int             INT;
typedef unsigned long   ULONG;
typedef ULONG         * PULONG;
typedef ULONG         * LPULONG;

#endif  // !WIN32


//
//  Socket-specific types.
//

typedef INT             SOCKERR;        // A socket error code.
typedef WORD            PORT;           // A socket port number.
typedef WORD            SOCKEVENT;      // An asynchronous socket event.


//
//  Packet structures.  Since these go out on the wire,
//  they're packed to a known boundary.
//

#ifndef RC_INVOKED
#pragma pack(4)
#endif  // !RC_INVOKED

typedef struct _PACKET_FILE
{
    DWORD       type;                   // Must be PACKET_TYPE_FILE.
    DWORD       xid;                    // Transaction ID.
    DWORD       cbFile;                 // File size.
    CHAR        szFile[MAX_PATH];       // The file's name.

} PACKET_FILE, FAR * LPPACKET_FILE;


typedef struct _PACKET_PORT
{
    DWORD       type;                   // Must be PACKET_TYPE_PORT.
    DWORD       xid;                    // Transaction ID.
    PORT        port;                   // Port number for data transfer.

} PACKET_PORT, FAR * LPPACKET_PORT;


typedef struct _PACKET_NACK
{
    DWORD       type;                   // Must be PACKET_TYPE_NACK.
    DWORD       xid;                    // Transaction ID.
    DWORD       reason;                 // Failure reason (not yet implemented).

} PACKET_NACK, FAR * LPPACKET_NACK;

#ifndef RC_INVOKED
#pragma pack()
#endif  // !RC_INVOKED


//
//  Potential states for client-side windows.
//

typedef enum _CLIENT_STATE
{
    Embryonic,                          // Newly create client window.
    Idle,                               // Idle, no socket activity.
    WaitingForPort,                     // Waiting for PORT command from server.
    Transferring,                       // File transfer in progress.
    Disconnected                        // Connection terminated.

} CLIENT_STATE;


//
//  Client-side per-window data.  A pointer to this structure is
//  kept at offset GWL_CLIENT in the client-side windows.
//

typedef struct _CLIENT_DATA
{
    DWORD        dwType;                // Type of window (WINDOW_TYPE_CLIENT).
    DWORD        xid;                   // Transaction ID for this connection.
    CLIENT_STATE state;                 // Current state.
    SOCKET       sReply;                // Socket waiting for reply from srv.
    SOCKET       sData;                 // Open data socket for transfer.
    LONG         timeout;               // Timeout counter;
    LONG         timeNext;              // Value for next timeout.
    IN_ADDR      inetServer;            // Target server's IP address.
    CHAR         szServer[MAX_HOST];    // Target server's host name.
    HFILE        hFile;                 // Handle to open file.
    DWORD        cbFile;                // File size.
    DWORD        cbSent;                // Count of bytes sent so far.
    CHAR         szFile[MAX_PATH];      // Name of file being sent.
    BYTE         bBuffer[BUFFER_SIZE];  // File transfer buffer.

} CLIENT_DATA, FAR * LPCLIENT_DATA;

#define CLIENTPTR(w)            ((LPCLIENT_DATA)GetWindowLong((w), GWL_CLIENT))
#define IS_CLIENT_WINDOW(w)     (CLIENTPTR(w)->dwType == WINDOW_TYPE_CLIENT)


//
//  Server-side per-window data.  A pointer to this structure is
//  kept at offset GWL_SERVER in the server-side windows.
//

typedef struct _SERVER_DATA
{
    DWORD       dwType;                 // Type of window (WINDOW_TYPE_SERVER).
    DWORD       xid;                    // Transaction ID for this connection.
    HFILE       hFile;                  // Handle to open file.
    SOCKET      sData;                  // Open data socket for transfer.
    IN_ADDR     inetClient;             // Client's IP address.
    PORT        portClient;             // Client's UDP socket port.
    PORT        portServer;             // Server's TCP socket port.
    DWORD       cbReceived;             // Count of bytes received so far.
    LONG        timeout;                // Timeout counter.
    CHAR        szFile[MAX_PATH];       // Name of file being received.
    BYTE        bBuffer[BUFFER_SIZE];   // File transfer buffer.

} SERVER_DATA, FAR * LPSERVER_DATA;

#define SERVERPTR(w)            ((LPSERVER_DATA)GetWindowLong((w), GWL_SERVER))
#define IS_SERVER_WINDOW(w)     (SERVERPTR(w)->dwType == WINDOW_TYPE_SERVER)


#endif  // _WORMTYPE_H_

