/*******************************************************************

    utility.c

    This file contains routines of general utility.

********************************************************************/


#include "wormhole.h"


//
//  Public functions.
//

/*******************************************************************

    NAME:       MsgBox

    SYNOPSIS:   A printf-like interface to MessageBox.

    ENTRY:      hwndParent - The "owning" parent window.

                fuType - A set of MB_* flags.

                pszFormat - A printf-like format string.

                ... - Other printf-like arguments as needed.

    RETURNS:    INT - The result of the MessageBox API.

********************************************************************/
INT MsgBox( HWND  hwndParent,
            UINT  fuType,
            LPSTR pszFormat,
            ... )
{
    CHAR    szOutput[MAX_PRINTF_OUTPUT];
    va_list ArgList;

    va_start( ArgList, pszFormat );
    wvsprintf( szOutput, pszFormat, ArgList );
    va_end( ArgList );

    return MessageBox( hwndParent, szOutput, pszAppName, fuType );

}   // MsgBox


/*******************************************************************

    NAME:       WinPrintf

    SYNOPSIS:   A printf-like interface to TextOut

    ENTRY:      hdc - Display context for the text.

                row - Starting row (* tmHeight).

                col - Starting column (* tmAveCharWidth).

                pszFormat - A printf-like format string.

                ... - Other printf-like arguments as needed.

********************************************************************/
VOID WinPrintf( HDC   hdc,
                INT   row,
                INT   col,
                LPSTR pszFormat,
                ... )
{
    CHAR    szOutput[MAX_PRINTF_OUTPUT];
    INT     cbOutput;
    va_list ArgList;

    va_start( ArgList, pszFormat );
    cbOutput = wvsprintf( szOutput, pszFormat, ArgList );
    va_end( ArgList );

    TextOut( hdc,
             col * tmAveCharWidth,
             row * tmHeight,
             szOutput,
             cbOutput );

}   // WinPrintf


/*******************************************************************

    NAME:       SockerrToString

    SYNOPSIS:   Maps a socket error (like WSAEINTR) to a displayable
                form (like "Interrupted system call").

    ENTRY:      serr - The error to map.

    RETURNS:    LPSTR - The displayable form of the error.  Will be
                    "Unknown" for unknown errors.

********************************************************************/
LPSTR SockerrToString( SOCKERR serr )
{
    switch( serr )
    {
    case WSAENAMETOOLONG :
        return "Name too long";

    case WSANOTINITIALISED :
        return "Not initialized";

    case WSASYSNOTREADY :
        return "System not ready";

    case WSAVERNOTSUPPORTED :
        return "Version is not supported";

    case WSAESHUTDOWN :
        return "Can't send after socket shutdown";

    case WSAEINTR :
        return "Interrupted system call";

    case WSAHOST_NOT_FOUND :
        return "Host not found";

    case WSATRY_AGAIN :
        return "Try again";

    case WSANO_RECOVERY :
        return "Non-recoverable error";

    case WSANO_DATA :
        return "No data record available";

    case WSAEBADF :
        return "Bad file number";

    case WSAEWOULDBLOCK :
        return "Operation would block";

    case WSAEINPROGRESS :
        return "Operation now in progress";

    case WSAEALREADY :
        return "Operation already in progress";

    case WSAEFAULT :
        return "Bad address";

    case WSAEDESTADDRREQ :
        return "Destination address required";

    case WSAEMSGSIZE :
        return "Message too long";

    case WSAEPFNOSUPPORT :
        return "Protocol family not supported";

    case WSAENOTEMPTY :
        return "Directory not empty";

    case WSAEPROCLIM :
        return "EPROCLIM returned";

    case WSAEUSERS :
        return "EUSERS returned";

    case WSAEDQUOT :
        return "Disk quota exceeded";

    case WSAESTALE :
        return "ESTALE returned";

    case WSAEINVAL :
        return "Invalid argument";

    case WSAEMFILE :
        return "Too many open files";

    case WSAEACCES :
        return "Access denied";

    case WSAELOOP :
        return "Too many levels of symbolic links";

    case WSAEREMOTE :
        return "The object is remote";

    case WSAENOTSOCK :
        return "Socket operation on non-socket";

    case WSAEADDRNOTAVAIL :
        return "Can't assign requested address";

    case WSAEADDRINUSE :
        return "Address already in use";

    case WSAEAFNOSUPPORT :
        return "Address family not supported by protocol family";

    case WSAESOCKTNOSUPPORT :
        return "Socket type not supported";

    case WSAEPROTONOSUPPORT :
        return "Protocol not supported";

    case WSAENOBUFS :
        return "No buffer space is supported";

    case WSAETIMEDOUT :
        return "Connection timed out";

    case WSAEISCONN :
        return "Socket is already connected";

    case WSAENOTCONN :
        return "Socket is not connected";

    case WSAENOPROTOOPT :
        return "Bad protocol option";

    case WSAECONNRESET :
        return "Connection reset by peer";

    case WSAECONNABORTED :
        return "Software caused connection abort";

    case WSAENETDOWN :
        return "Network is down";

    case WSAENETRESET :
        return "Network was reset";

    case WSAECONNREFUSED :
        return "Connection refused";

    case WSAEHOSTDOWN :
        return "Host is down";

    case WSAEHOSTUNREACH :
        return "Host is unreachable";

    case WSAEPROTOTYPE :
        return "Protocol is wrong type for socket";

    case WSAEOPNOTSUPP :
        return "Operation not supported on socket";

    case WSAENETUNREACH :
        return "ICMP network unreachable";

    case WSAETOOMANYREFS :
        return "Too many references";

    default :
        return "Unknown";
    }
}


/*******************************************************************

    NAME:       ResetSocket

    SYNOPSIS:   Performs a "hard" close on the given socket.

    ENTRY:      sock - The socket to close.

    RETURNS:    SOCKERR - 0 if successful, !0 if not.

********************************************************************/
SOCKERR ResetSocket( SOCKET sock )
{
    LINGER linger;

    if( sock == INVALID_SOCKET )
    {
        //
        //  Ignore invalid sockets.
        //

        return 0;
    }

    //
    //  Enable linger with a timeout of zero.  This will
    //  force the hard close when we call closesocket().
    //
    //  We ignore the error return from setsockopt.  If it
    //  fails, we'll just try to close the socket anyway.
    //

    linger.l_onoff  = TRUE;
    linger.l_linger = 0;

    setsockopt( sock,
                SOL_SOCKET,
                SO_LINGER,
                (CHAR FAR *)&linger,
                sizeof(linger) );

    //
    //  Close the socket.
    //

    return closesocket( sock );

}   // ResetSocket


/*******************************************************************

    NAME:       CreateSocket

    SYNOPSIS:   Creates a data socket for the specified address & port.

    ENTRY:      psock - Will receive the new socket ID if successful.

                type - The socket type (SOCK_DGRAM or SOCK_STREAM).

                address - The IP address for the socket.

                port - The port for the socket.

    RETURNS:    SOCKERR - 0 if successful, !0 if not.

********************************************************************/
SOCKERR CreateSocket( SOCKET FAR * psock,
                      INT          type,
                      ULONG        address,
                      PORT         port )
{
    SOCKET  sNew;
    SOCKERR serr;

    //
    //  Create the socket.
    //

    sNew = socket( PF_INET, type, 0 );
    serr = ( sNew == INVALID_SOCKET ) ? WSAGetLastError() : 0;

    if( serr == 0 )
    {
        SOCKADDR_IN sockAddr;

        //
        //  Bind an address to the socket.
        //

        sockAddr.sin_family      = AF_INET;
        sockAddr.sin_addr.s_addr = address;
        sockAddr.sin_port        = port;

        if( bind( sNew, (SOCKADDR FAR *)&sockAddr, sizeof(sockAddr) ) != 0 )
        {
            serr = WSAGetLastError();
        }
    }

    if( serr != 0 )
    {
        ResetSocket( sNew );
        sNew = INVALID_SOCKET;
    }

    *psock = sNew;

    return serr;

}   // CreateSocket

