/*******************************************************************

    server.c

    This file contains routines for displaying and managing the
    server-side MDI child windows.

********************************************************************/


#include "wormhole.h"


//
//  Private prototypes.
//

VOID Server_OnClose( HWND hwnd );

VOID Server_OnPaint( HWND hwnd );

VOID Server_OnTimer( HWND hwnd,
                     UINT id );

VOID Server_OnSocketSelect( HWND      hwnd,
                            SOCKET    sock,
                            SOCKERR   serr,
                            SOCKEVENT sevent );

VOID Server_DestroyConnection( HWND          hwnd,
                               LPSERVER_DATA pServerData );


//
//  Public functions.
//

/*******************************************************************

    NAME:       Server_WndProc

    SYNOPSIS:   Window procedure for the server-side MDI child windows.

    ENTRY:      hwnd - Window handle.

                nMessage - The message.

                wParam - The first message parameter.

                lParam - The second message parameter.

    RETURNS:    LRESULT - Depends on the actual message.

********************************************************************/
LRESULT CALLBACK Server_WndProc( HWND   hwnd,
                                 UINT   nMessage,
                                 WPARAM wParam,
                                 LPARAM lParam )
{
    switch( nMessage )
    {
        HANDLE_MSG( hwnd, WM_CLOSE,         Server_OnClose        );
        HANDLE_MSG( hwnd, WM_PAINT,         Server_OnPaint        );
        HANDLE_MSG( hwnd, WM_TIMER,         Server_OnTimer        );
        HANDLE_MSG( hwnd, WM_SOCKET_SELECT, Server_OnSocketSelect );
    }

    return DefMDIChildProc( hwnd, nMessage, wParam, lParam );

}   // Server_WndProc


/*******************************************************************

    NAME:       Server_CreateNew

    SYNOPSIS:   Creates a new server-side MDI child window.

    ENTRY:      pServerData - Points to a SERVER_DATA structure
                    containing data for this new session.

    RETURNS:    HWND - Handle to the new child window.

********************************************************************/
HWND Server_CreateNew( LPSERVER_DATA pServerData )
{
    MDICREATESTRUCT mcs;
    HWND            hwnd;

    mcs.szClass = pszServerClass;
    mcs.szTitle = inet_ntoa( pServerData->inetClient );
    mcs.hOwner  = hInst;
    mcs.x       = CW_USEDEFAULT;
    mcs.y       = CW_USEDEFAULT;
    mcs.cx      = CW_USEDEFAULT;
    mcs.cy      = CW_USEDEFAULT;
    mcs.style   = 0;

    hwnd = FORWARD_WM_MDICREATE( hwndMDIClient,
                                 (LPMDICREATESTRUCT)&mcs,
                                 SendMessage );

    if( hwnd == NULL )
    {
        return NULL;
    }

    if( WSAAsyncSelect( pServerData->sData,
                        hwnd,
                        WM_SOCKET_SELECT,
                        FD_ACCEPT ) != 0 )
    {
        //
        //  Cannot initiate an async select on the data socket.
        //

        FORWARD_WM_MDIDESTROY( hwndMDIClient, hwnd, SendMessage );
        return NULL;
    }

    //
    //  Success!
    //

    SetWindowLong( hwnd, GWL_SERVER, (LONG)pServerData );
    ShowWindow( hwnd, SW_SHOW );

    return hwnd;

}   // Server_CreateNew


//
//  Private functions.
//

/*******************************************************************

    NAME:       Server_OnClose

    SYNOPSIS:   Handles WM_CLOSE messages.

    ENTRY:      hwnd - Window handle.

********************************************************************/
VOID Server_OnClose( HWND hwnd )
{
    LPSERVER_DATA pServerData;

    pServerData = SERVERPTR(hwnd);

    if( pServerData != NULL )
    {
        //
        //  This window has server data.  Close any open
        //  socket/file handles before freeing the buffer.
        //

        Server_DestroyConnection( NULL, pServerData );
        GlobalFreePtr( pServerData );
        SetWindowLong( hwnd, GWL_SERVER, 0L );
    }

    FORWARD_WM_CLOSE( hwnd, DefMDIChildProc );

}   // Server_OnClose


/*******************************************************************

    NAME:       Server_OnPaint

    SYNOPSIS:   Handles WM_PAINT messages.

    ENTRY:      hwnd - Window handle.

********************************************************************/
VOID Server_OnPaint( HWND hwnd )
{
    PAINTSTRUCT   psPaint;
    LPSERVER_DATA pServerData;
    HDC           hdc;

    hdc = BeginPaint( hwnd, &psPaint );

    pServerData = SERVERPTR(hwnd);

    if( pServerData != NULL )
    {
        WinPrintf( hdc, 1,  1, "Client:" );
        WinPrintf( hdc, 1, 20, "%s", inet_ntoa( pServerData->inetClient ) );

        WinPrintf( hdc, 2,  1, "Receiving:" );
        WinPrintf( hdc, 2, 20, "%s", pServerData->szFile );

        WinPrintf( hdc, 3,  1, "Bytes Received:" );
        WinPrintf( hdc, 3, 20, "%lu", pServerData->cbReceived );
    }

    EndPaint( hwnd, &psPaint );

}   // Server_OnPaint


/*******************************************************************

    NAME:       Server_OnTimer

    SYNOPSIS:   Handles WM_TIMER messages.

    ENTRY:      hwnd - Window handle.

                id - Timer ID.

********************************************************************/
VOID Server_OnTimer( HWND hwnd,
                     UINT id )
{
    LPSERVER_DATA pServerData;

    //
    //  Get the server data associated with this window.  If
    //  there is no data associated yet, ignore the timer.
    //

    pServerData = SERVERPTR(hwnd);

    if( pServerData == NULL )
    {
        return;
    }

    //
    //  Update timeout counter.  If we've timed-out, drop the
    //  session.
    //

    if( pServerData->timeout++ >= TIMEOUT_SERVER )
    {
        Server_DestroyConnection( hwnd, pServerData );
    }

}   // Server_OnTimer


/*******************************************************************

    NAME:       Server_OnSocketSelect

    SYNOPSIS:   Handles WM_SOCKET_SELECT messages.

    ENTRY:      hwnd - Window handle.

                sock - The socket that generated the message.

                serr - A socket error code.

                sevent - One of the FD_* events that occurred.

********************************************************************/
VOID Server_OnSocketSelect( HWND      hwnd,
                            SOCKET    sock,
                            SOCKERR   serr,
                            SOCKEVENT sevent )
{
    LPSERVER_DATA pServerData;
    INT           cbBuffer;
    SOCKADDR_IN   addr;
    INT           cbAddr;
    HDC           hdc;

    pServerData = SERVERPTR(hwnd);

    if( pServerData == NULL )
    {
        return;
    }

    if( serr != 0 )
    {
        //
        //  Socket error.
        //

        Server_DestroyConnection( hwnd, pServerData );
        return;
    }

    //
    //  Interpret the socket event.
    //

    switch( sevent )
    {
    case FD_CLOSE :
        //
        //  Client closed the connection.
        //

        serr = WSAENOTCONN;
        break;

    case FD_ACCEPT :
        //
        //  The client finally opened our data socket.  Accept
        //  the connection, then wait for data from the socket.
        //

        cbAddr = sizeof(addr);
        sock   = accept( sock, (SOCKADDR FAR *)&addr, &cbAddr );

        if( sock == INVALID_SOCKET )
        {
            serr = WSAGetLastError();
            break;
        }

        ResetSocket( pServerData->sData );
        pServerData->sData = sock;

        if( WSAAsyncSelect( sock,
                            hwnd,
                            WM_SOCKET_SELECT,
                            FD_READ | FD_CLOSE ) != 0 )
        {
            serr = WSAGetLastError();
            break;
        }

        //
        //  Fall through to try to read first block of data.
        //

    case FD_READ :
        //
        //  Data is available on the socket.
        //

        cbBuffer = recv( pServerData->sData,
                         (CHAR FAR *)pServerData->bBuffer,
                         sizeof(pServerData->bBuffer),
                         0 );

        if( cbBuffer == SOCKET_ERROR )
        {
            serr = WSAGetLastError();
            break;
        }

        if( cbBuffer == 0 )
        {
            //
            //  End of file.
            //

            break;
        }

        //
        //  Write the buffer to the file.
        //

        if( _lwrite( pServerData->hFile,
                     (CHAR FAR *)pServerData->bBuffer,
                     cbBuffer ) == HFILE_ERROR )
        {
            //
            //  File write error.  Make up a socket error to
            //  force the cleanup code to get invoked.
            //

            serr = WSAENOBUFS;
            break;
        }

        //
        //  Update the displayed transfer statistics.
        //

        pServerData->cbReceived += cbBuffer;
        hdc = GetDC( hwnd );
        WinPrintf( hdc, 3, 20, "%lu", pServerData->cbReceived );
        ReleaseDC( hwnd, hdc );
        break;

    default :
        //
        //  Who knows.  Ignore it.
        //

        return;
    }

    //
    //  At this point, there are three interesting values for serr:
    //
    //          0              - everything OK
    //          WSAEWOULDBLOCK - tried to recv(), but no data available
    //          anything else  - something tragic occurred
    //
    //  If the error is 0 or WSAEWOULDBLOCK, then restart the timeout
    //  counter and wait for another message.  Otherwise, disconnect
    //  the session and abort the transfer.
    //

    if( ( serr == 0 ) || ( serr == WSAEWOULDBLOCK ) )
    {
        pServerData->timeout = 0;
    }
    else
    {
        Server_DestroyConnection( hwnd, pServerData );
    }

}   // Server_OnSocketSelect


/*******************************************************************

    NAME:       Server_DestroyConnection

    SYNOPSIS:   Destroys an existing connection.

    ENTRY:      hwnd - The server-side window handle.  If !NULL,
                    then destroy the window.

                pServerData - Points to a SERVER_DATA structure
                    describing the connection.

********************************************************************/
VOID Server_DestroyConnection( HWND          hwnd,
                               LPSERVER_DATA pServerData )
{
    closesocket( pServerData->sData );
    pServerData->sData = INVALID_SOCKET;

    _lclose( pServerData->hFile );
    pServerData->hFile = HFILE_ERROR;

    if( hwnd != NULL )
    {
        FORWARD_WM_MDIDESTROY( hwndMDIClient, hwnd, SendMessage );
    }

}   // Server_DestroyConnection

