/*******************************************************************

    frame.c

    This file contains routines for displaying and managing the
    MDI frame window.

********************************************************************/


#include "wormhole.h"


//
//  Private constants.
//
#define DESIRED_WINSOCK_VERSION         0x0101  // we'd like winsock ver 1.1...
#define MINIMUM_WINSOCK_VERSION         0x0001  // ...but we'll take ver 1.0
#define FRAME_TIMER_ID                  1


//
//  Private prototypes.
//

VOID Frame_OnCommand( HWND hwnd,
                      INT  id,
                      HWND hwndCtl,
                      UINT codeNotify );

BOOL Frame_OnCreate( HWND               hwnd,
                     CREATESTRUCT FAR * pCreateStruct );

VOID Frame_OnDestroy( HWND hwnd );

VOID Frame_OnTimer( HWND hwnd,
                    UINT id );

VOID Frame_OnSocketSelect( HWND      hwnd,
                           SOCKET    sock,
                           SOCKERR   serr,
                           SOCKEVENT sevent );

LRESULT Frame_MDIMessageForwarder( HWND   hwnd,
                                   UINT   nMessage,
                                   WPARAM wParam,
                                   LPARAM lParam );

VOID Frame_NewConnection( VOID );

BOOL Frame_SendPortCommand( SOCKET        sock,
                            LPSOCKADDR_IN paddr,
                            LPSERVER_DATA pServerData );

BOOL Frame_SendNackCommand( SOCKET        sock,
                            LPSOCKADDR_IN paddr,
                            DWORD         xid );

HWND Frame_FindServerWindow( LPSOCKADDR_IN paddr,
                             DWORD         xid );

LPSERVER_DATA Frame_CreateServerData( LPPACKET_FILE ppacket,
                                      LPSOCKADDR_IN paddr );

LPCLIENT_DATA Frame_CreateClientData( LPSTR     pszName,
                                      LPIN_ADDR paddr );


//
//  Public functions.
//

/*******************************************************************

    NAME:       Frame_WndProc

    SYNOPSIS:   Window procedure for the MDI frame window.

    ENTRY:      hwnd - Window handle.

                nMessage - The message.

                wParam - The first message parameter.

                lParam - The second message parameter.

    RETURNS:    LRESULT - Depends on the actual message.

********************************************************************/
LRESULT CALLBACK Frame_WndProc( HWND   hwnd,
                                UINT   nMessage,
                                WPARAM wParam,
                                LPARAM lParam )
{
    switch( nMessage )
    {
        HANDLE_MSG( hwnd, WM_COMMAND,       Frame_OnCommand      );
        HANDLE_MSG( hwnd, WM_CREATE,        Frame_OnCreate       );
        HANDLE_MSG( hwnd, WM_DESTROY,       Frame_OnDestroy      );
        HANDLE_MSG( hwnd, WM_TIMER,         Frame_OnTimer        );
        HANDLE_MSG( hwnd, WM_SOCKET_SELECT, Frame_OnSocketSelect );
    }

    return DefFrameProc( hwnd, hwndMDIClient, nMessage, wParam, lParam );

}   // Frame_WndProc


//
//  Private functions.
//

/*******************************************************************

    NAME:       Frame_OnCommand

    SYNOPSIS:   Handles WM_COMMAND messages.

    ENTRY:      hwnd - Window handle.

                id - Identifies the menu/control/accelerator.

                hwndCtl - Identifies the control sending the command.

                codeNotify - A notification code.  Will be zero for
                    menus, one for accelerators.

********************************************************************/
VOID Frame_OnCommand( HWND hwnd,
                      INT  id,
                      HWND hwndCtl,
                      UINT codeNotify )
{
    switch( id )
    {
    case IDM_CONNECTION_NEW :
        Frame_NewConnection();
        break;

    case IDM_CONNECTION_EXIT :
        PostMessage( hwnd, WM_CLOSE, 0, 0 );
        break;

    case IDM_WINDOW_CASCADE :
        FORWARD_WM_MDICASCADE( hwndMDIClient, 0, SendMessage );
        break;

    case IDM_WINDOW_TILE_VERTICALLY :
        FORWARD_WM_MDITILE( hwndMDIClient, MDITILE_VERTICAL, SendMessage );
        break;

    case IDM_WINDOW_TILE_HORIZONTALLY :
        FORWARD_WM_MDITILE( hwndMDIClient, MDITILE_HORIZONTAL, SendMessage );
        break;

    case IDM_WINDOW_ARRANGE_ICONS :
        FORWARD_WM_MDIICONARRANGE( hwndMDIClient, SendMessage );
        break;

    case IDM_HELP_ABOUT :
        AboutDialog( hwnd );
        break;

    default :
        FORWARD_WM_COMMAND( hwnd,
                            id,
                            hwndCtl,
                            codeNotify,
                            Frame_MDIMessageForwarder );
        break;
    }

}   // Frame_OnCommand


/*******************************************************************

    NAME:       Frame_OnCreate

    SYNOPSIS:   Handles WM_CREATE messages.

    ENTRY:      hwnd - Window handle.

                pCreateStruct - Contains window creation parameters.

    RETURNS:    BOOL - TRUE if window created OK, FALSE otherwise.

********************************************************************/
BOOL Frame_OnCreate( HWND               hwnd,
                     CREATESTRUCT FAR * pCreateStruct )
{
    CLIENTCREATESTRUCT ccs;
    WSADATA            wsadata;
    SOCKERR            serr;
    HDC                hdc;
    TEXTMETRIC         tm;

    //
    //  Initialize the sockets library.
    //

    serr = WSAStartup( DESIRED_WINSOCK_VERSION, &wsadata );

    if( serr != 0 )
    {
        MsgBox( NULL,
                MB_ICONSTOP | MB_OK,
                "Cannot initialize socket library, error %d: %s",
                serr,
                SockerrToString( serr ) );

        return FALSE;
    }

    if( wsadata.wVersion < MINIMUM_WINSOCK_VERSION )
    {
        MsgBox( NULL,
                MB_ICONSTOP | MB_OK,
                "Windows Sockets version %02X.%02X, I need at least %02X.%02X",
                LOBYTE(wsadata.wVersion),
                HIBYTE(wsadata.wVersion),
                LOBYTE(MINIMUM_WINSOCK_VERSION),
                HIBYTE(MINIMUM_WINSOCK_VERSION) );

        return FALSE;
    }

    //
    //  Create the command socket.
    //

    serr = CreateSocket( &sCommand,
                         SOCK_DGRAM,
                         htonl( INADDR_ANY ),
                         htons( WORMHOLE_COMMAND_PORT ) );

    if( serr == 0 )
    {
        if( WSAAsyncSelect( sCommand,
                            hwnd,
                            WM_SOCKET_SELECT,
                            FD_READ ) != 0 )
        {
            serr = WSAGetLastError();
        }
    }

    if( serr != 0 )
    {
        MsgBox( NULL,
                MB_ICONSTOP | MB_OK,
                "Cannot create command socket, error %d: %s",
                serr,
                SockerrToString( serr ) );

        return FALSE;
    }

    //
    //  Create the MDI client window.
    //

    ccs.hWindowMenu  = GetSubMenu( GetMenu( hwnd ), 1 );
    ccs.idFirstChild = IDM_WINDOW_FIRST_CHILD;

    hwndMDIClient = CreateWindow( pszMDIClientClass,
                                  NULL,
                                  WS_CHILD | WS_CLIPCHILDREN
                                      | WS_VSCROLL | WS_HSCROLL,
                                  0, 0,
                                  0, 0,
                                  hwnd,
                                  0,
                                  hInst,
                                  (LPSTR)&ccs );

    if( hwndMDIClient == NULL )
    {
        return FALSE;
    }

    ShowWindow( hwndMDIClient, SW_SHOW );

    //
    //  Create the timeout timer.
    //

    if( SetTimer( hwnd, FRAME_TIMER_ID, 1000, NULL ) == 0 )
    {
        MsgBox( NULL,
                MB_ICONSTOP | MB_OK,
                "Cannot create timeout timer" );

        return FALSE;
    }

    //
    //  Get textmetric data.
    //

    hdc = GetDC( hwnd );
    GetTextMetrics( hdc, &tm );
    ReleaseDC( hwnd, hdc );

    tmAveCharWidth = (INT)tm.tmAveCharWidth;
    tmHeight       = (INT)tm.tmHeight;

    //
    //  Success!
    //

    return TRUE;

}   // Frame_OnCreate


/*******************************************************************

    NAME:       Frame_OnDestroy

    SYNOPSIS:   Handles WM_DESTROY messages.

    ENTRY:      hwnd - Window handle.

********************************************************************/
VOID Frame_OnDestroy( HWND hwnd )
{
    //
    //  Disconnect from the sockets library before terminating.
    //

    WSACleanup();
    PostQuitMessage( 0 );

}   // Frame_OnDestroy


/*******************************************************************

    NAME:       Frame_OnTimer

    SYNOPSIS:   Handles WM_TIMER messages.  The MDI frame window
                doesn't actually need timer messages, it just
                forwards them to each child window.  This is so
                we don't need to constantly create & destroy timers.

    ENTRY:      hwnd - Window handle.

                id - Timer ID.

********************************************************************/
VOID Frame_OnTimer( HWND hwnd,
                    UINT id )
{
    HWND hwndChild;

    hwndChild = GetWindow( hwndMDIClient, GW_CHILD );

    while( hwndChild != NULL )
    {
        if( GetWindow( hwndChild, GW_OWNER ) == NULL )
        {
            FORWARD_WM_TIMER( hwndChild, id, PostMessage );
        }

        hwndChild = GetWindow( hwndChild, GW_HWNDNEXT );
    }

}   // Frame_OnTimer


/*******************************************************************

    NAME:       Frame_OnSocketSelect

    SYNOPSIS:   Handles WM_SOCKET_SELECT messages.

    ENTRY:      hwnd - Window handle.

                sock - The socket that generated the message.

                serr - A socket error code.

                sevent - One of the FD_* events that occurred.

********************************************************************/
VOID Frame_OnSocketSelect( HWND      hwnd,
                           SOCKET    sock,
                           SOCKERR   serr,
                           SOCKEVENT sevent )
{
    INT           cbRead;
    PACKET_FILE   filePacket;
    SOCKADDR_IN   addrClient;
    INT           cbAddrClient;
    LPSERVER_DATA pServerData = NULL;
    HWND          hwndServer  = NULL;

    if( ( sock != sCommand ) || ( sevent != FD_READ ) || ( serr != 0 ) )
    {
        //
        //  Unknown command or unknown event or socket error.  Ignore it.
        //

        return;
    }

    //
    //  We got a command packet.  Read & interpret.
    //

    cbAddrClient = sizeof(addrClient);

    cbRead = recvfrom( sock,
                       (CHAR FAR *)&filePacket,
                       sizeof(filePacket),
                       0,
                       (SOCKADDR FAR *)&addrClient,
                       &cbAddrClient );

    if( ( cbRead != sizeof(filePacket) ) ||
        ( filePacket.type != htonl( PACKET_TYPE_FILE ) ) )
    {
        //
        //  Garbage or partial packet.  Ignore it.
        //

        return;
    }

    //
    //  See if a server-side window already exists for this
    //  particular address/xid pair.
    //

    hwndServer = Frame_FindServerWindow( &addrClient, filePacket.xid );

    if( hwndServer != NULL )
    {
        //
        //  A server-side window already exists for this client.
        //  This probably indicates that a previous FILE command
        //  was lost in a jumble of network traffic.  So, we'll
        //  send another.  If this fails, such is life in the
        //  wild and wonderful world of datagrams.
        //

        Frame_SendPortCommand( sock, &addrClient, SERVERPTR(hwndServer) );
        return;
    }

    //
    //  Create the server-side data.
    //

    pServerData = Frame_CreateServerData( &filePacket, &addrClient );

    if( pServerData == NULL )
    {
        goto FatalExit;
    }

    //
    //  Create a server-side child window.
    //

    hwndServer = Server_CreateNew( pServerData );

    if( hwndServer != NULL )
    {
        //
        //  Send the port back to the client.
        //

        if( Frame_SendPortCommand( sock, &addrClient, pServerData ) )
        {
            //
            //  Success!
            //

            return;
        }
    }

    //
    //  Either we couldn't create the server-side child window
    //  or we failed to send the port command.  Cleanup & return.
    //

    if( pServerData != NULL )
    {
        ResetSocket( pServerData->sData );
        _lclose( pServerData->hFile );
        GlobalFreePtr( pServerData );
    }

    if( hwndServer != NULL )
    {
        FORWARD_WM_MDIDESTROY( hwndMDIClient, hwndServer, SendMessage );
    }

FatalExit:

    //
    //  Send a NACK packet to the client.
    //

    Frame_SendNackCommand( sock, &addrClient, filePacket.xid );

}   // Frame_OnSocketSelect


/*******************************************************************

    NAME:       Frame_MDIMessageForwarder

    SYNOPSIS:   Forwards messages to DefFrameProc.  This is usually
                used as the last parameter to a FORWARD_WM_* macro.

    ENTRY:      hwnd - Window handle.

                nMessage - The message.

                wParam - The first message parameter.

                lParam - The second message parameter.

    RETURNS:    LRESULT - Depends on the actual message.

********************************************************************/
LRESULT Frame_MDIMessageForwarder( HWND   hwnd,
                                   UINT   nMessage,
                                   WPARAM wParam,
                                   LPARAM lParam )
{
    return DefFrameProc( hwnd, hwndMDIClient, nMessage, wParam, lParam );

}   // Frame_MDIMessageForwarder


/*******************************************************************

    NAME:       Frame_NewConnection

    SYNOPSIS:   Prompts the user for a host name, then establishes
                a connection to that host.

********************************************************************/
VOID Frame_NewConnection( VOID )
{
    CHAR          szHostName[MAX_HOST];
    IN_ADDR       addrHost;
    LPCLIENT_DATA pClientData = NULL;
    HWND          hwndClient  = NULL;

    //
    //  Prompt the user for a new host.
    //

    if( !NewHostDialog( hwndMDIFrame, szHostName, &addrHost ) )
    {
        return;
    }

    //
    //  Create the client-side data.
    //

    pClientData = Frame_CreateClientData( szHostName, &addrHost );

    if( pClientData == NULL )
    {
        return;
    }

    //
    //  Create the client-side child window.
    //

    hwndClient = Client_CreateNew( pClientData );

    if( hwndClient == NULL )
    {
        GlobalFreePtr( pClientData );
        return;
    }

}   // Frame_NewConnection


/*******************************************************************

    NAME:       Frame_SendPortCommand

    SYNOPSIS:   Sends a PORT command back to the client.  This reply
                contains the port number of the TCP socket to be used
                for the data transfer.

    ENTRY:      sock - The socket to send the command on.

                paddr - Points to a SOCKADDR_IN structure containing
                    the client's target address & port.

                pServerData - Points to a SERVER_DATA structure
                    describing the connection.

    RETURNS:    BOOL - TRUE if command sent OK, FALSE otherwise.

********************************************************************/
BOOL Frame_SendPortCommand( SOCKET        sock,
                            LPSOCKADDR_IN paddr,
                            LPSERVER_DATA pServerData )
{
    PACKET_PORT portPacket;
    INT         cbWritten;

    //
    //  Send the port back to the client.
    //

    portPacket.type = htonl( PACKET_TYPE_PORT );
    portPacket.xid  = pServerData->xid;
    portPacket.port = pServerData->portServer;

    cbWritten = sendto( sock,
                        (CHAR FAR *)&portPacket,
                        sizeof(portPacket),
                        0,
                        (SOCKADDR FAR *)paddr,
                        sizeof(SOCKADDR_IN) );

    return cbWritten == sizeof(portPacket);

}   // Frame_SendPortCommand


/*******************************************************************

    NAME:       Frame_SendNackCommand

    SYNOPSIS:   Sends a NACK command back to the client.  This reply
                indicates that a FILE command was rejected.

    ENTRY:      sock - The socket to send the command on.

                paddr - Points to a SOCKADDR_IN structure containing
                    the client's target address & port.

                xid - The transaction ID for this client.

    RETURNS:    BOOL - TRUE if command sent OK, FALSE otherwise.

********************************************************************/
BOOL Frame_SendNackCommand( SOCKET        sock,
                            LPSOCKADDR_IN paddr,
                            DWORD         xid )
{
    PACKET_NACK nackPacket;
    INT         cbWritten;

    //
    //  Send a NACK packet to the client.
    //

    nackPacket.type   = htonl( PACKET_TYPE_NACK );
    nackPacket.xid    = xid;
    nackPacket.reason = 0;      // reason codes not yet implemented

    cbWritten = sendto( sock,
                        (CHAR FAR *)&nackPacket,
                        sizeof(nackPacket),
                        0,
                        (SOCKADDR FAR *)paddr,
                        sizeof(SOCKADDR_IN) );

    return cbWritten == sizeof(nackPacket);

}   // Frame_SendNackCommand


/*******************************************************************

    NAME:       Frame_FindServerWindow

    SYNOPSIS:   Tries to locate an existing server-side window for
                the specified address/xid pair.

    ENTRY:      paddr - Points to a SOCKADDR_IN structure containing
                    the client's target address & port.

                xid - The transaction ID for this client.

    RETURNS:    HWND - The handle of the server-side window for this
                    session.  Will be NULL if no window exists yet.

********************************************************************/
HWND Frame_FindServerWindow( LPSOCKADDR_IN paddr,
                             DWORD         xid )
{
    HWND hwnd;

    hwnd = GetWindow( hwndMDIClient, GW_CHILD );

    while( hwnd != NULL )
    {
        LPSERVER_DATA pServerData = SERVERPTR(hwnd);

        if( ( pServerData != NULL ) &&
            ( pServerData->dwType == WINDOW_TYPE_SERVER ) &&
            ( pServerData->inetClient.s_addr == paddr->sin_addr.s_addr ) &&
            ( pServerData->xid == xid ) )
        {
            return hwnd;
        }

        hwnd = GetWindow( hwnd, GW_HWNDNEXT );
    }

    return NULL;

}   // Frame_FindServerWindow


/*******************************************************************

    NAME:       Frame_CreateServerData

    SYNOPSIS:   Creates a new SERVER_DATA structure describing the
                server-side of a connection.

    ENTRY:      ppacket - Points to the PACKET_FILE structure received
                    from the client.

                paddr - Points to a SOCKADDR_IN structure containing
                    the client's IP address & port number.

    RETURNS:    LPSERVER_DATA - Points to the new server data structure,
                    will be NULL if failed to create & initialize.

********************************************************************/
LPSERVER_DATA Frame_CreateServerData( LPPACKET_FILE ppacket,
                                      LPSOCKADDR_IN paddr )
{
    SOCKADDR_IN   addrServer;
    INT           cbAddrServer;
    SOCKERR       serr;
    LPSERVER_DATA pServerData = NULL;
    HWND          hwndServer  = NULL;
    HFILE         hFile       = HFILE_ERROR;
    SOCKET        sData       = INVALID_SOCKET;

    //
    //  Create the server-side data.
    //

    pServerData = (LPSERVER_DATA)GlobalAllocPtr( GPTR, sizeof(SERVER_DATA) );

    if( pServerData == NULL )
    {
        //
        //  Insufficient resources.
        //

        goto FatalError;
    }

    //
    //  Create the local file.
    //

    hFile = _lcreat( ppacket->szFile, 0 );

    if( hFile == HFILE_ERROR )
    {
        //
        //  Cannot create the local file.
        //

        goto FatalError;
    }

    //
    //  Create the TCP data socket.
    //

    serr = CreateSocket( &sData,
                         SOCK_STREAM,
                         htonl( INADDR_ANY ),
                         0 );

    if( serr != 0 )
    {
        //
        //  Cannot create socket.
        //

        goto FatalError;
    }

    cbAddrServer = sizeof(addrServer);

    if( getsockname( sData, (SOCKADDR FAR *)&addrServer, &cbAddrServer ) != 0 )
    {
        //
        //  Cannot get port data for the data socket.
        //

        goto FatalError;
    }

    if( listen( sData, 5 ) != 0 )
    {
        //
        //  Cannot listen on the socket.
        //

        goto FatalError;
    }

    //
    //  Initialize the server-side data.
    //

    pServerData->dwType     = WINDOW_TYPE_SERVER;
    pServerData->xid        = ppacket->xid;
    pServerData->hFile      = hFile;
    pServerData->sData      = sData;
    pServerData->inetClient = paddr->sin_addr;
    pServerData->portClient = paddr->sin_port;
    pServerData->portServer = addrServer.sin_port;
    pServerData->cbReceived = 0L;
    pServerData->timeout    = 0;

    _fstrcpy( pServerData->szFile, ppacket->szFile );

    //
    //  Success!
    //

    return pServerData;

FatalError:

    if( sData != INVALID_SOCKET )
    {
        ResetSocket( sData );
    }

    if( hFile != HFILE_ERROR )
    {
        _lclose( hFile );
    }

    if( pServerData != NULL )
    {
        GlobalFreePtr( pServerData );
    }

    return NULL;

}   // Frame_CreateServerData


/*******************************************************************

    NAME:       Frame_CreateClientData

    SYNOPSIS:   Creates a new CLIENT_DATA structure describing the
                client-side of a connection.

    ENTRY:      pszName - The target server's host name.

                paddr - Points to an IN_ADDR structure containing
                    the server's IP address.

    RETURNS:    LPCLIENT_DATA - Points to the new client data structure,
                    will be NULL if failed to create & initialize.

********************************************************************/
LPCLIENT_DATA Frame_CreateClientData( LPSTR     pszName,
                                      LPIN_ADDR paddr )
{
    LPCLIENT_DATA pClientData = NULL;

    //
    //  Allocate a new client data structure.
    //

    pClientData = (LPCLIENT_DATA)GlobalAllocPtr( GPTR, sizeof(CLIENT_DATA) );

    if( pClientData == NULL )
    {
        //
        //  Insufficient resources.
        //

        return NULL;
    }

    //
    //  Initialize the client-side data.
    //

    pClientData->dwType     = WINDOW_TYPE_CLIENT;
    pClientData->xid        = htonl( xidNext++ );
    pClientData->state      = Embryonic;
    pClientData->sReply     = INVALID_SOCKET;
    pClientData->timeout    = 0;
    pClientData->timeNext   = TIMEOUT_CLIENT_INITIAL;
    pClientData->inetServer = *paddr;
    pClientData->hFile      = HFILE_ERROR;
    pClientData->szFile[0]  = '\0';

    _fstrcpy( pClientData->szServer, pszName );

    //
    //  Success!
    //

    return pClientData;

}   // Frame_CreateClientData

