/*******************************************************************

    dialogs.c

    This file contains routines for displaying and managing the
    various dialogs used by the Windows Sockets Wormhole project.

********************************************************************/


#include "wormhole.h"


//
//  Private types.
//

typedef struct _NEW_HOST_PARAMS
{
    LPSTR     pszHostName;
    LPIN_ADDR paddr;
    HANDLE    hAsync;
    ULONG     laddr;
    BYTE      bBuffer[MAXGETHOSTSTRUCT];

} NEW_HOST_PARAMS, FAR * LPNEW_HOST_PARAMS;


//
//  Private globals.
//

NEW_HOST_PARAMS NewHostParams;          // Parameters for NewHost dialog.


//
//  Private prototypes.
//

BOOL CALLBACK About_DlgProc( HWND   hwnd,
                             UINT   nMessage,
                             WPARAM wParam,
                             LPARAM lParam );

VOID About_OnCommand( HWND hwnd,
                      INT  id,
                      HWND hwndCtl,
                      UINT codeNotify );

BOOL CALLBACK NewHost_DlgProc( HWND   hwnd,
                               UINT   nMessage,
                               WPARAM wParam,
                               LPARAM lParam );

BOOL NewHost_OnInitDialog( HWND   hwnd,
                           HWND   hwndFocus,
                           LPARAM lParam );

VOID NewHost_OnCommand( HWND hwnd,
                        INT  id,
                        HWND hwndCtl,
                        UINT codeNotify );

VOID NewHost_OnSocketAsync( HWND    hwnd,
                            HANDLE  hAsync,
                            SOCKERR serr,
                            WORD    cbBuffer );

VOID NewHost_EnableControls( HWND hwndParent,
                             BOOL fEnable );


//
//  Public functions.
//

/*******************************************************************

    NAME:       AboutDialog

    SYNOPSIS:   Displays the Wormhole "About" box.

    ENTRY:      hwndParent - The parent window for this dialog.

********************************************************************/
VOID AboutDialog( HWND hwndParent )
{
    FARPROC pfnAbout;

    pfnAbout = MakeProcInstance( (FARPROC)About_DlgProc, hInst );

    DialogBox( hInst,
               IDD_ABOUT,
               hwndParent,
               (DLGPROC)pfnAbout );

    FreeProcInstance( pfnAbout );

}   // AboutDialog


/*******************************************************************

    NAME:       NewHostDialog

    SYNOPSIS:   Prompts the user for a new host to connect to.  The
                user may enter either a host name or an IP address.
                It is up to this dialog to map either to a name\address
                pair.

    ENTRY:      hwndParent - The parent window for this dialog.

                pszHostName - Will receive the host name if successful.

                paddr - Will receive the IP address if successful.

    RETURNS:    BOOL - TRUE if we successfully received & mapped a
                    name\address pair, FALSE otherwise.

********************************************************************/
BOOL NewHostDialog( HWND      hwndParent,
                    LPSTR     pszHostName,
                    LPIN_ADDR paddr )

{
    FARPROC pfnDlg;
    BOOL    fResult;

    //
    //  Setup dialog parameters.
    //

    _fmemset( &NewHostParams, 0, sizeof(NewHostParams) );

    NewHostParams.pszHostName = pszHostName;
    NewHostParams.paddr       = paddr;
    NewHostParams.hAsync      = NULL;

    //
    //  Invoke the dialog.
    //

    pfnDlg = MakeProcInstance( (FARPROC)NewHost_DlgProc, hInst );

    fResult = DialogBox( hInst,
                         IDD_NEW,
                         hwndParent,
                         (DLGPROC)pfnDlg );

    //
    //  Cleanup & exit.
    //

    FreeProcInstance( pfnDlg );

    return fResult;

}   // NewHostDialog


//
//  Private functions.
//

/*******************************************************************

    NAME:       About_DlgProc

    SYNOPSIS:   Dialog procedure for the "About" dialog.

    ENTRY:      hwnd - Dialog box handle.

                nMessage - The message.

                wParam - The first message parameter.

                lParam - The second message parameter.

    RETURNS:    BOOL - TRUE if we handle the message, FALSE otherwise.

********************************************************************/
BOOL CALLBACK About_DlgProc( HWND   hwnd,
                             UINT   nMessage,
                             WPARAM wParam,
                             LPARAM lParam )
{
    switch( nMessage )
    {
        HANDLE_MSG( hwnd, WM_COMMAND, About_OnCommand );
    }

    return FALSE;

}   // AboutDlgProc


/*******************************************************************

    NAME:       About_OnCommand

    SYNOPSIS:   Handles WM_COMMAND messages set to the "About"
                dialog window.

    ENTRY:      hwnd - Dialog box handle.

                id - Identifies the menu/control/accelerator.

                hwndCtl - Identifies the control sending the command.

                codeNotify - A notification code.  Will be zero for
                    menus, one for accelerators.

********************************************************************/
VOID About_OnCommand( HWND hwnd,
                      INT  id,
                      HWND hwndCtl,
                      UINT codeNotify )
{
    if( ( id == IDOK ) || ( id == IDCANCEL ) )
    {
        EndDialog( hwnd, TRUE );
    }

}   // About_OnCommand


/*******************************************************************

    NAME:       NewHost_DlgProc

    SYNOPSIS:   Dialog procedure for the "GetNewHost" dialog.

    ENTRY:      hwnd - Dialog box handle.

                nMessage - The message.

                wParam - The first message parameter.

                lParam - The second message parameter.

    RETURNS:    BOOL - TRUE if we handle the message, FALSE otherwise.

********************************************************************/
BOOL CALLBACK NewHost_DlgProc( HWND   hwnd,
                               UINT   nMessage,
                               WPARAM wParam,
                               LPARAM lParam )
{
    switch( nMessage )
    {
        HANDLE_MSG( hwnd, WM_INITDIALOG,   NewHost_OnInitDialog  );
        HANDLE_MSG( hwnd, WM_COMMAND,      NewHost_OnCommand     );
        HANDLE_MSG( hwnd, WM_SOCKET_ASYNC, NewHost_OnSocketAsync );
    }

    return FALSE;

}   // AboutDlgProc


/*******************************************************************

    NAME:       NewHost_OnInitDialog

    SYNOPSIS:   Handles WM_INITDIALOG messages set to the "GetNewHost"
                dialog window.

    ENTRY:      hwnd - Dialog box handle.

                hwndFocus - Handle of control to receive input focus.

                lParam - Initialization parameter.  Is actually a
                    pointer to a NEW_HOST_PARAM structure.

    RETURNS:    BOOL - TRUE if Windows should set the initial input
                    focus to hwndFocus, FALSE if Windows should not
                    set the initial input focus.

********************************************************************/
BOOL NewHost_OnInitDialog( HWND   hwnd,
                           HWND   hwndFocus,
                           LPARAM lParam )
{
    //
    //  Limit the length of text allowed in the edit field.
    //

    Edit_LimitText( GetDlgItem( hwnd, IDD_NEW_HOST ),
                    MAX_HOST );

    return TRUE;

}   // NewHost_OnInitDialog


/*******************************************************************

    NAME:       NewHost_OnCommand

    SYNOPSIS:   Handles WM_COMMAND messages set to the "GetNewHost"
                dialog window.

    ENTRY:      hwnd - Dialog box handle.

                id - Identifies the menu/control/accelerator.

                hwndCtl - Identifies the control sending the command.

                codeNotify - A notification code.  Will be zero for
                    menus, one for accelerators.

********************************************************************/
VOID NewHost_OnCommand( HWND hwnd,
                        INT  id,
                        HWND hwndCtl,
                        UINT codeNotify )
{
    HANDLE  hAsyncTask;
    SOCKERR serr;

    //
    //  Interpret the command.
    //

    switch( id )
    {
    case IDOK :
        // handled below
        break;

    case IDCANCEL :
        //
        //  Cancel any pending async operation started by this dialog.
        //

        if( NewHostParams.hAsync != NULL )
        {
            WSACancelAsyncRequest( NewHostParams.hAsync );
            NewHostParams.hAsync = NULL;
        }

        EndDialog( hwnd, FALSE );
        return;

    default :
        return;
    }

    //
    //  We only make it to this point if the command was OK.
    //  Get the host name/address from the edit field.
    //

    Edit_GetText( GetDlgItem( hwnd, IDD_NEW_HOST ),
                  NewHostParams.pszHostName,
                  MAX_HOST );

    if( NewHostParams.pszHostName[0] == '\0' )
    {
        //
        //  User didn't really enter anything.
        //

        return;
    }

    //
    //  First check to see if it is a dotted IP address.
    //

    NewHostParams.laddr = inet_addr( NewHostParams.pszHostName );

    if( NewHostParams.laddr == INADDR_NONE )
    {
        //
        //  Assume the user gave us an actual host name.
        //

        hAsyncTask = WSAAsyncGetHostByName( hwnd,
                                            WM_SOCKET_ASYNC,
                                            NewHostParams.pszHostName,
                                            NewHostParams.bBuffer,
                                            sizeof(NewHostParams.bBuffer) );
    }
    else
    {
        //
        //  The user gave us a dotted IP address.
        //

        hAsyncTask = WSAAsyncGetHostByAddr( hwnd,
                                            WM_SOCKET_ASYNC,
                                            (CHAR FAR *)&NewHostParams.laddr,
                                            sizeof(NewHostParams.laddr),
                                            PF_INET,
                                            NewHostParams.bBuffer,
                                            sizeof(NewHostParams.bBuffer) );
    }

    if( hAsyncTask == NULL )
    {
        //
        //  Could not initiate the asynchronous API.
        //

        serr = WSAGetLastError();

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot initiate search for host %s, error %d: %s",
                NewHostParams.pszHostName,
                serr,
                SockerrToString( serr ) );

        return;
    }

    //
    //  The async command has been issued.  Disable all dialog
    //  controls except [Cancel].
    //

    NewHost_EnableControls( hwnd, FALSE );

}   // NewHost_OnCommand


/*******************************************************************

    NAME:       NewHost_OnSocketAsync

    SYNOPSIS:   Handles WM_SOCKET_ASYNC messages set to the
                "GetNewHost" dialog window by the asynchronous
                getXbyY socket API.

    ENTRY:      hwnd - Dialog box handle.

                hAsync - The asynchronous task handle returned by
                    the WSAAsyncGetXbyY API.

                serr - Any socket error that occurred during the
                    asynchronous API.

                cbBuffer - If serr == WSAENOBUFS, then this will
                    contain the required buffer size for the requested
                    operation.  Otherwise, cbBuffer is undefined.

********************************************************************/
VOID NewHost_OnSocketAsync( HWND    hwnd,
                            HANDLE  hAsync,
                            SOCKERR serr,
                            WORD    cbBuffer )
{
    LPHOSTENT phostent;

    phostent = (LPHOSTENT)NewHostParams.bBuffer;

    if( serr != 0 )
    {
        //
        //  Error retrieving host name/address.
        //

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot locate host %s, error %d: %s",
                NewHostParams.pszHostName,
                serr,
                SockerrToString( serr ) );

        //
        //  Reenable the dialog controls.
        //

        NewHost_EnableControls( hwnd, TRUE );

        return;
    }

    //
    //  Found the host.
    //

    _fstrcpy( NewHostParams.pszHostName, phostent->h_name );
    _fmemcpy( NewHostParams.paddr, phostent->h_addr, sizeof(IN_ADDR) );

    EndDialog( hwnd, TRUE );

}   // NewHost_OnSocketAsync


/*******************************************************************

    NAME:       NewHost_EnableControls

    SYNOPSIS:   Enables/disables "GetNewHost" dialog controls

    ENTRY:      hwnd - Dialog box handle.

                fEnable - If TRUE, enable controls, else disable controls.

********************************************************************/
VOID NewHost_EnableControls( HWND hwnd,
                             BOOL fEnable )
{
    EnableWindow( GetDlgItem( hwnd, IDD_NEW_HOST ),
                  fEnable );

    EnableWindow( GetDlgItem( hwnd, IDD_NEW_HOST_LABEL ),
                  fEnable );

    EnableWindow( GetDlgItem( hwnd, IDOK ),
                  fEnable );

    //
    //  If we enabling the dialog controls, it is due to a
    //  failure to retrieve the host data, so set the focus
    //  to the host edit field.  If we're disabling the controls,
    //  set the focus to the cancel button since that's the only
    //  thing we haven't disabled.
    //

    SetFocus( GetDlgItem( hwnd, fEnable ? IDD_NEW_HOST : IDCANCEL ) );

}   // NewHost_EnableControls

