/*******************************************************************

    client.c

    This file contains routines for displaying and managing the
    client-side MDI child windows.

********************************************************************/


#include "wormhole.h"


//
//  Private prototypes.
//

VOID Client_OnClose( HWND hwnd );

VOID Client_OnPaint( HWND hwnd );

VOID Client_OnTimer( HWND hwnd,
                     UINT id );

VOID Client_OnDropFiles( HWND  hwnd,
                         HDROP hdrop );

VOID Client_OnSocketSelect( HWND      hwnd,
                            SOCKET    sock,
                            SOCKERR   serr,
                            SOCKEVENT sevent );

LPSTR Client_StateToString( CLIENT_STATE state );

VOID Client_DestroyConnection( HWND          hwnd,
                               LPCLIENT_DATA pClientData );

SOCKERR Client_SendFileCommand( SOCKET        sock,
                                LPCLIENT_DATA pClientData );

VOID Client_HandleReply( HWND          hwnd,
                         LPCLIENT_DATA pClientData );

VOID Client_HandleData( HWND          hwnd,
                        LPCLIENT_DATA pClientData );

VOID Client_DragAcceptFiles( HWND hwnd,
                             BOOL fEnable );


//
//  Public functions.
//

/*******************************************************************

    NAME:       Client_WndProc

    SYNOPSIS:   Window procedure for the client-side MDI child windows.

    ENTRY:      hwnd - Window handle.

                nMessage - The message.

                wParam - The first message parameter.

                lParam - The second message parameter.

    RETURNS:    LRESULT - Depends on the actual message.

********************************************************************/
LRESULT CALLBACK Client_WndProc( HWND   hwnd,
                                 UINT   nMessage,
                                 WPARAM wParam,
                                 LPARAM lParam )
{
    switch( nMessage )
    {
        HANDLE_MSG( hwnd, WM_CLOSE,         Client_OnClose        );
        HANDLE_MSG( hwnd, WM_PAINT,         Client_OnPaint        );
        HANDLE_MSG( hwnd, WM_TIMER,         Client_OnTimer        );
        HANDLE_MSG( hwnd, WM_DROPFILES,     Client_OnDropFiles    );
        HANDLE_MSG( hwnd, WM_SOCKET_SELECT, Client_OnSocketSelect );
    }

    return DefMDIChildProc( hwnd, nMessage, wParam, lParam );

}   // Client_WndProc


/*******************************************************************

    NAME:       Client_CreateNew

    SYNOPSIS:   Creates a new client-side MDI child window.

    ENTRY:      pClientData - Points to a CLIENT_DATA structure
                    containing data for this new session.

    RETURNS:    HWND - Handle to the new child window.

********************************************************************/
HWND Client_CreateNew( LPCLIENT_DATA pClientData )
{
    MDICREATESTRUCT mcs;
    HWND            hwnd;

    mcs.szClass = pszClientClass;
    mcs.szTitle = pClientData->szServer,
    mcs.hOwner  = hInst;
    mcs.x       = CW_USEDEFAULT;
    mcs.y       = CW_USEDEFAULT;
    mcs.cx      = CW_USEDEFAULT;
    mcs.cy      = CW_USEDEFAULT;
    mcs.style   = 0;

    hwnd = FORWARD_WM_MDICREATE( hwndMDIClient,
                                 (LPMDICREATESTRUCT)&mcs,
                                 SendMessage );

    if( hwnd == NULL )
    {
        return NULL;
    }

    //
    //  Success!
    //

    pClientData->state = Idle;
    SetWindowLong( hwnd, GWL_CLIENT, (LONG)pClientData );

    Client_DragAcceptFiles( hwnd, TRUE );
    ShowWindow( hwnd, SW_SHOW );

    return hwnd;

}   // Client_CreateNew


//
//  Private functions.
//

/*******************************************************************

    NAME:       Client_OnClose

    SYNOPSIS:   Handles WM_CLOSE messages.

    ENTRY:      hwnd - Window handle.

********************************************************************/
VOID Client_OnClose( HWND hwnd )
{
    LPCLIENT_DATA pClientData;

    pClientData = CLIENTPTR(hwnd);

    if( pClientData != NULL )
    {
        //
        //  This window has client data.  Close any open
        //  socket/file handles before freeing the buffer.
        //

        Client_DestroyConnection( NULL, pClientData );
        GlobalFreePtr( pClientData );
        SetWindowLong( hwnd, GWL_CLIENT, 0L );
    }

    FORWARD_WM_CLOSE( hwnd, DefMDIChildProc );

}   // Client_OnClose


/*******************************************************************

    NAME:       Client_OnPaint

    SYNOPSIS:   Handles WM_PAINT messages.

    ENTRY:      hwnd - Window handle.

********************************************************************/
VOID Client_OnPaint( HWND hwnd )
{
    PAINTSTRUCT   psPaint;
    LPCLIENT_DATA pClientData;
    HDC           hdc;

    hdc = BeginPaint( hwnd, &psPaint );

    pClientData = CLIENTPTR(hwnd);

    if( pClientData != NULL )
    {
        WinPrintf( hdc, 1,  1, "Server:" );
        WinPrintf( hdc, 1, 20, pClientData->szServer );

        WinPrintf( hdc, 2,  1, "Address:" );
        WinPrintf( hdc, 2, 20, "%s", inet_ntoa( pClientData->inetServer ) );

        WinPrintf( hdc, 3,  1, "State:" );
        WinPrintf( hdc, 3, 20, "%s", Client_StateToString( pClientData->state ) );

        if( pClientData->state == Transferring )
        {
            WinPrintf( hdc, 4,  1, "Sending:" );
            WinPrintf( hdc, 4, 20, "%s", pClientData->szFile );

            WinPrintf( hdc, 5,  1, "Bytes Sent:" );
            WinPrintf( hdc, 5, 20, "%lu", pClientData->cbSent );
        }
    }

    EndPaint( hwnd, &psPaint );

}   // Client_OnPaint


/*******************************************************************

    NAME:       Client_OnTimer

    SYNOPSIS:   Handles WM_TIMER messages.

    ENTRY:      hwnd - Window handle.

                id - Timer ID.

********************************************************************/
VOID Client_OnTimer( HWND hwnd,
                     UINT id )
{
    LPCLIENT_DATA pClientData;
    SOCKERR       serr;

    pClientData = CLIENTPTR(hwnd);

    if( ( pClientData == NULL ) || ( pClientData->state != WaitingForPort ) )
    {
        //
        //  Either this window has no client data, or is not
        //  in a waiting state.  Ignore the timer.
        //

        return;
    }

    if( pClientData->timeout++ < pClientData->timeNext )
    {
        //
        //  No timeout yet.  Wait for another message.
        //

        return;
    }

    if( pClientData->timeNext >= TIMEOUT_CLIENT_MAX )
    {
        //
        //  We gave it our best shot.  Give up.
        //

        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Timeout waiting for server %s to respond",
                pClientData->szServer );

        return;
    }

    //
    //  Resend the FILE command.
    //

    serr = Client_SendFileCommand( pClientData->sReply,
                                   pClientData );

    if( serr != 0 )
    {
        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot resend FILE command to %s, error %d: %s",
                pClientData->szServer,
                serr,
                SockerrToString( serr ) );

        return;
    }

    //
    //  Restart the timeout counter.
    //

    pClientData->timeout   = 0;
    pClientData->timeNext += TIMEOUT_CLIENT_DELTA;

}   // Client_OnTimer


/*******************************************************************

    NAME:       Client_OnDropFiles

    SYNOPSIS:   Handles WM_DROPFILES messages.

    ENTRY:      hwnd - Window handle.

                hdrop - Handle to a structure containing the list of
                    files being dropped.  This handle must be passed
                    to DragFinish before leaving this routine.

********************************************************************/
VOID Client_OnDropFiles( HWND  hwnd,
                         HDROP hdrop )
{
    LPCLIENT_DATA pClientData;
    LPSTR         pszFile;
    HFILE         hFile  = HFILE_ERROR;
    SOCKET        sReply = INVALID_SOCKET;
    SOCKERR       serr   = 0;

    //
    //  Initialize some client-side data.
    //

    pClientData = CLIENTPTR(hwnd);
    pszFile     = pClientData->szFile;

    pClientData->sReply  = INVALID_SOCKET;
    pClientData->hFile   = HFILE_ERROR;
    pClientData->timeout = 0;
    pClientData->cbSent  = 0L;

    //
    //  Retrieve the dropped file.
    //

    DragQueryFile( hdrop,
                   0,
                   pszFile,
                   sizeof(pClientData->szFile) );

    DragFinish( hdrop );

    //
    //  Try to open it.
    //

    hFile = _lopen( pszFile, OF_READ | OF_SHARE_DENY_WRITE );

    if( hFile == HFILE_ERROR )
    {
        //
        //  Error opening file.
        //

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot open file %s for transfer to host %s",
                pszFile,
                pClientData->szServer );

        goto FatalError;
    }

    pClientData->cbFile = (DWORD)_llseek( hFile, 0L, 2 );
    _llseek( hFile, 0L, 0 );

    //
    //  Create the reply socket, then connect it to the server's
    //  address and the well-known Wormhole command port.  The
    //  server will send the "PORT" or "NACK" packet back to
    //  this socket.
    //

    serr = CreateSocket( &sReply,
                         SOCK_DGRAM,
                         htonl( INADDR_ANY ),
                         0 );

    if( serr == 0 )
    {
        SOCKADDR_IN addr;

        addr.sin_family = PF_INET;
        addr.sin_addr   = pClientData->inetServer;
        addr.sin_port   = htons( WORMHOLE_COMMAND_PORT );

        if( connect( sReply, (SOCKADDR *)&addr, sizeof(addr) ) != 0 )
        {
            serr = WSAGetLastError();
        }
    }

    if( serr != 0 )
    {
        //
        //  Fatal socket error.
        //

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot create reply socket for server %s, error %d: %s",
                pClientData->szServer,
                serr,
                SockerrToString( serr ) );

        goto FatalError;
    }

    if( WSAAsyncSelect( sReply,
                        hwnd,
                        WM_SOCKET_SELECT,
                        FD_READ ) != 0 )
    {
        //
        //  Cannot initiate an async select on the reply socket.
        //

        serr = WSAGetLastError();

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot initiate connection with server %s, error %d: %s",
                pClientData->szServer,
                serr,
                SockerrToString( serr ) );

        goto FatalError;
    }

    serr = Client_SendFileCommand( sReply, pClientData );

    if( serr != 0 )
    {
        //
        //  Cannot send FILE packet to server.
        //

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot send FILE command to server %s, error %d: %s",
                pClientData->szServer,
                serr,
                SockerrToString( serr ) );

        goto FatalError;
    }

    //
    //  Success!
    //

    pClientData->sReply   = sReply;
    pClientData->hFile    = hFile;
    pClientData->state    = WaitingForPort;
    pClientData->timeout  = 0;
    pClientData->timeNext = TIMEOUT_CLIENT_INITIAL;

    InvalidateRect( hwnd, NULL, TRUE );
    UpdateWindow( hwnd );
    Client_DragAcceptFiles( hwnd, FALSE );

    return;

FatalError:

    if( hFile != HFILE_ERROR )
    {
        _lclose( hFile );
    }

    if( sReply != INVALID_SOCKET )
    {
        ResetSocket( sReply );
    }

}   // Client_OnDropFiles


/*******************************************************************

    NAME:       Client_OnSocketSelect

    SYNOPSIS:   Handles WM_SOCKET_SELECT messages.

    ENTRY:      hwnd - Window handle.

                sock - The socket that generated the message.

                serr - A socket error code.

                sevent - One of the FD_* events that occurred.

********************************************************************/
VOID Client_OnSocketSelect( HWND      hwnd,
                            SOCKET    sock,
                            SOCKERR   serr,
                            SOCKEVENT sevent )
{
    LPCLIENT_DATA pClientData;

    pClientData = CLIENTPTR(hwnd);

    if( pClientData == NULL )
    {
        return;
    }

    if( serr != 0 )
    {
        //
        //  Fatal error.  Destroy the connection then give
        //  the user an appropriate error message.
        //

        Client_DestroyConnection( hwnd, pClientData );

        if( sock == pClientData->sReply )
        {
            MsgBox( hwnd,
                    MB_ICONSTOP | MB_OK,
                    "Cannot get response from server %s, error %d: %s",
                    pClientData->szServer,
                    serr,
                    SockerrToString( serr ) );
        }
        else
        {
            MsgBox( hwnd,
                    MB_ICONSTOP | MB_OK,
                    "Cannot send file %s to server %s, error %d: %s",
                    pClientData->szFile,
                    pClientData->szServer,
                    serr,
                    SockerrToString( serr ) );
        }

        return;
    }

    //
    //  Interpret the socket event.
    //

    switch( sevent )
    {
    case FD_CLOSE :
        //
        //  Server closed the connection.
        //

        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Server %s closed data socket",
                pClientData->szServer );
        break;

    case FD_CONNECT :
        //
        //  The connection to the server has completed.
        //

        if( WSAAsyncSelect( sock,
                            hwnd,
                            WM_SOCKET_SELECT,
                            FD_WRITE | FD_CLOSE ) != 0 )
        {
            serr = WSAGetLastError();
        }

        if( serr != 0 )
        {
            Client_DestroyConnection( hwnd, pClientData );

            MsgBox( hwnd,
                    MB_ICONSTOP | MB_OK,
                    "Cannot initiate transfer to server %s , error %d: %s",
                    pClientData->szServer,
                    serr,
                    SockerrToString( serr ) );
        }
        break;

    case FD_WRITE :
        //
        //  The data socket is writeable.
        //

        Client_HandleData( hwnd, pClientData );
        break;

    case FD_READ :
        //
        //  The reply socket is readable.
        //

        Client_HandleReply( hwnd, pClientData );
        break;

    default :
        //
        //  Who knows.  Ignore it.
        //

        break;
    }

}   // Client_OnSocketSelect


/*******************************************************************

    NAME:       Client_StateToString

    SYNOPSIS:   Maps a CLIENT_STATE value to a displayable form.

    ENTRY:      state - The client state to map.

    RETURNS:    LPSTR - The displayable form.  Will be "Unknown"
                    for unknown states.

********************************************************************/
LPSTR Client_StateToString( CLIENT_STATE state )
{
    LPSTR pszResult;

    switch( state )
    {
    case Embryonic :
        pszResult = "Starting";
        break;

    case Idle :
        pszResult = "Idle";
        break;

    case WaitingForPort :
        pszResult = "Waiting for Reply";
        break;

    case Transferring :
        pszResult = "Transfer in Progress";
        break;

    case Disconnected :
        pszResult = "Disconnected";
        break;

    default :
        pszResult = "Unknown";
        break;
    }

    return pszResult;

}   // Client_StateToString


/*******************************************************************

    NAME:       Client_DestroyConnection

    SYNOPSIS:   Destroys an existing connection.

    ENTRY:      hwnd - The client-side window handle.  If !NULL, then
                    force a repaint & enable drag/drop files.

                pClientData - Points to a CLIENT_DATA structure
                    describing the connection.

********************************************************************/
VOID Client_DestroyConnection( HWND          hwnd,
                               LPCLIENT_DATA pClientData )
{
    if( pClientData->sReply != INVALID_SOCKET )
    {
        ResetSocket( pClientData->sReply );
        pClientData->sReply = INVALID_SOCKET;
    }

    if( pClientData->sData != INVALID_SOCKET )
    {
        closesocket( pClientData->sData );
        pClientData->sData = INVALID_SOCKET;
    }

    if( pClientData->hFile != HFILE_ERROR )
    {
        _lclose( pClientData->hFile );
        pClientData->hFile = HFILE_ERROR;
    }

    pClientData->state = Idle;

    if( hwnd != NULL )
    {
        InvalidateRect( hwnd, NULL, TRUE );
        UpdateWindow( hwnd );
        Client_DragAcceptFiles( hwnd, TRUE );
    }

}   // Client_DestroyConnection


/*******************************************************************

    NAME:       Client_SendFileCommand

    SYNOPSIS:   Sends a FILE command to the target server.  This
                    command contains the name & size of the
                    file to transfer.

    ENTRY:      sock - The socket to use for the command.

                pClientData - Points to a CLIENT_DATA structure
                    describing the connection.

********************************************************************/
SOCKERR Client_SendFileCommand( SOCKET        sock,
                                LPCLIENT_DATA pClientData )
{
    PACKET_FILE filePacket;
    INT         cbWritten;
    LPSTR       pszTmp;

    //
    //  Send a FILE packet to the server.
    //

    filePacket.type   = htonl( PACKET_TYPE_FILE );
    filePacket.xid    = pClientData->xid;
    filePacket.cbFile = pClientData->cbFile;

    pszTmp = _fstrrchr( pClientData->szFile, '\\' );
    pszTmp = ( pszTmp ) ? pszTmp : _fstrrchr( pClientData->szFile, ':' );
    pszTmp = ( pszTmp ) ? pszTmp+1 : pClientData->szFile;

    _fstrcpy( filePacket.szFile, pszTmp );

    cbWritten = send( sock,
                      (CHAR FAR *)&filePacket,
                      sizeof(filePacket),
                      0 );

    return ( cbWritten == SOCKET_ERROR )
               ? WSAGetLastError()
               : 0;

}   // Client_SendFileCommand


/*******************************************************************

    NAME:       Client_HandleReply

    SYNOPSIS:   Handles async messages from the reply socket.

    ENTRY:      hwnd - Window handle.

                pClientData - Points to a CLIENT_DATA structure
                    describing this connection.

********************************************************************/
VOID Client_HandleReply( HWND          hwnd,
                         LPCLIENT_DATA pClientData )
{
    PACKET_PORT portPacket;
    INT         cbRead;
    SOCKET      sData;
    SOCKERR     serr;
    SOCKADDR_IN addr;
    INT         cbAddr;

    if( pClientData->state != WaitingForPort )
    {
        return;
    }

    //
    //  Server has acknowleged a FILE command.
    //

    cbRead = recv( pClientData->sReply,
                   (CHAR FAR *)&portPacket,
                   sizeof(portPacket),
                   0 );

    if( ( cbRead != sizeof(PACKET_PORT) ) &&
        ( cbRead != sizeof(PACKET_NACK) ) )
    {
        //
        //  Error.  Ignore it, wait for the timer
        //  and reset the request to the server.
        //

        return;
    }

    //
    //  Now that we've got a reply, close the reply socket.
    //

    ResetSocket( pClientData->sReply );
    pClientData->sReply = INVALID_SOCKET;

    //
    //  Interpret the reply.
    //

    if( portPacket.type != htonl( PACKET_TYPE_PORT ) )
    {
        //
        //  Server refused the request.
        //

        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Transfer of %s refused by server %s",
                pClientData->szFile,
                pClientData->szServer );

        return;
    }

    //
    //  Server has accepted the request.  Create a TCP
    //  socket to the port specified by the server.
    //

    serr = CreateSocket( &sData,
                         SOCK_STREAM,
                         htonl( INADDR_ANY ),
                         0 );

    if( serr != 0 )
    {
        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot connect to server %s, error %d: %s",
                pClientData->szServer,
                serr,
                SockerrToString( serr ) );

        return;
    }

    if( WSAAsyncSelect( sData,
                        hwnd,
                        WM_SOCKET_SELECT,
                        FD_CONNECT ) != 0 )
    {
        serr = WSAGetLastError();

        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Cannot initiate connection to server %s, error %d: %s",
                pClientData->szServer,
                serr,
                SockerrToString( serr ) );

        return;
    }

    addr.sin_family = AF_INET;
    addr.sin_addr   = pClientData->inetServer;
    addr.sin_port   = portPacket.port;

    cbAddr = sizeof(addr);

    if( connect( sData, (SOCKADDR FAR *)&addr, cbAddr ) != 0 )
    {
        serr = WSAGetLastError();

        if( serr != WSAEWOULDBLOCK )
        {
            Client_DestroyConnection( hwnd, pClientData );

            MsgBox( hwnd,
                    MB_ICONSTOP | MB_OK,
                    "Cannot initiate connection to server %s, error %d: %s",
                    pClientData->szServer,
                    serr,
                    SockerrToString( serr ) );

            return;
        }
    }

    //
    //  Success!
    //

    pClientData->sData = sData;
    pClientData->state = Transferring;

    InvalidateRect( hwnd, NULL, TRUE );
    UpdateWindow( hwnd );

}   // Client_HandleReply


/*******************************************************************

    NAME:       Client_HandleData

    SYNOPSIS:   Handles async messages from the data socket.

    ENTRY:      hwnd - Window handle.

                pClientData - Points to a CLIENT_DATA structure
                    describing this connection.

********************************************************************/
VOID Client_HandleData( HWND          hwnd,
                        LPCLIENT_DATA pClientData )
{
    INT  cbBuffer;
    INT  cbWritten;
    HDC  hdc;
    MSG  msg;

    if( pClientData->state != Transferring )
    {
        return;
    }

    //
    //  Read a chunk from the file.
    //

    cbBuffer = _lread( pClientData->hFile,
                       pClientData->bBuffer,
                       sizeof(pClientData->bBuffer) );

    if( cbBuffer == HFILE_ERROR )
    {
        Client_DestroyConnection( hwnd, pClientData );

        MsgBox( hwnd,
                MB_ICONSTOP | MB_OK,
                "Error reading file %s",
                pClientData->szFile );

        return;
    }

    if( cbBuffer == 0 )
    {
        //
        //  End of file.
        //

        Client_DestroyConnection( hwnd, pClientData );
        return;
    }

    //
    //  Send it to the server.
    //

    cbWritten = send( pClientData->sData,
                      (CHAR FAR *)pClientData->bBuffer,
                      cbBuffer,
                      0 );

    if( cbWritten == SOCKET_ERROR )
    {
        SOCKERR serr;

        serr = WSAGetLastError();

        if( serr == WSAEWOULDBLOCK )
        {
            //
            //  This would have blocked, so backup the
            //  file pointer and return.  We'll eventually
            //  get another FD_WRITE event.
            //

            _llseek( pClientData->hFile, (LONG)-cbBuffer, 1 );
        }
        else
        {
            Client_DestroyConnection( hwnd, pClientData );

            MsgBox( hwnd,
                    MB_ICONSTOP | MB_OK,
                    "Cannot send data to server %s, error %d: %s",
                    pClientData->szServer,
                    serr,
                    SockerrToString( serr ) );
        }

        return;
    }

    if( cbWritten < cbBuffer )
    {
        //
        //  Partial write, backup the file pointer.
        //

        _llseek( pClientData->hFile, (LONG)( cbWritten - cbBuffer ), 1 );
    }

    //
    //  Update the displayed transfer statistics.
    //

    pClientData->cbSent += cbWritten;
    hdc = GetDC( hwnd );
    WinPrintf( hdc, 5, 20, "%lu", pClientData->cbSent );
    ReleaseDC( hwnd, hdc );

    //
    //  If there are no WM_SOCKET_SELECT messages in the message
    //  queue, or if the first such message does not indicate an
    //  FD_WRITE event, then post ourselves such a message.  This
    //  will ensure this routine gets called again to send more
    //  data to the server.
    //

    if( !PeekMessage( &msg,
                      hwnd,
                      WM_SOCKET_SELECT,
                      WM_SOCKET_SELECT,
                      PM_NOREMOVE ) ||
        ( WSAGETSELECTEVENT(msg.lParam) != FD_WRITE ) )
    {
        FORWARD_WM_SOCKET_SELECT( hwnd,
                                  pClientData->sData,
                                  0,
                                  FD_WRITE,
                                  PostMessage );
    }

}   // Client_HandleData


/*******************************************************************

    NAME:       Client_DragAcceptFiles

    SYNOPSIS:   Enables/disables accepting of drag & drop files.

    ENTRY:      hwnd - Window handle.

                fEnable - TRUE to enable, FALSE to disable.

********************************************************************/
VOID Client_DragAcceptFiles( HWND hwnd,
                             BOOL fEnable )
{
    LONG style;

    style = GetWindowLong( hwnd, GWL_EXSTYLE );

    if( fEnable )
        style |= WS_EX_ACCEPTFILES;
    else
        style &= ~WS_EX_ACCEPTFILES;

    SetWindowLong( hwnd, GWL_EXSTYLE, style );

}   // Client_DragAcceptFiles

