/*++
/* NAME
/*	strcons,istrcmp,strvec,vecstr 3
/* SUMMARY
/*	string utility routines
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	general stuff
/* SYNOPSIS
/*	char *strcons(format,args)
/*	char *format;
/*
/*	int istrcmp(s1,s2)
/*	char *s1,s2;
/*
/*	char **strvec(string,separ)
/*	char *string;
/*	char *separ;
/*
/*	char *vecstr(vector,separ)
/*	char **vector;
/*	char *separ;
/* DESCRIPTION
/*	strcons() produces a formatted string, using printf()-like
/*	arguments. Basically it is an sprintf() that returns a 
/*	pointer to the result.
/*
/*	istrcmp() is a case-insensitive versions of the strcmp() functions.
/*
/*	strvec() breaks a null-terminated string using the separators given 
/*	in separ, and returns a null-terminated vector of pointers to the 
/*	resulting substrings. Memory for the vector and substrings are 
/*	allocated in dynamic memory. The original string is not modified.
/*
/*	vecstr() takes a null-terminated vector of string pointers
/*	and builds a string from the strings pointed to by the vector
/*	argument, separated by the string in the separ argument.
/*	Memory for the result is allocated in dynamic memory.
/* FUNCTIONS AND MACROS
/*	strtok(), malloc(), memcpy(), sprintf()
/* DIAGNOSTICS
/*	strvec(), vecstr() return a null pointer if there was not enough memory
/*	avaliable to hold the result.
/* BUGS
/*	strcons() does not do smart garbage collection; it just uses
/*	a circular buffer. The present implementation is not portable 
/*	to machines that pass arguments via registers.
/*
/*	strvec() cannot handle strings with more than BUFSIZ words.
/*	strvec() uses strtok(), which may have side effects.
/* AUTHOR(S)
/*	W.Z. Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Tue Apr  5 20:59:29 MET 1988
/* LAST MODIFICATION
/*	Wed Apr  6 00:23:08 MET 1988
/* VERSION/RELEASE
/*	1.1
/*--*/

#include <ctype.h>

#include "defs.h"

#define	NBUF	4

/* strcons - quick-and-dirty string constructor */

/* VARARGS1 */

char *strcons(fmt,a1,a2,a3,a4)
char *fmt;
long a1,a2,a3,a4;
{
    static char strbuf[NBUF][BUFSIZ];
    static int where = 0;
    register char *cp;

    sprintf(cp = strbuf[where = (where+1)%NBUF],fmt,a1,a2,a3,a4);
    return(cp);
}

/* istrcmp - case-insensitive string comparison */

#define	LOW(c)	(isascii(c)&&isupper(c)?tolower(c):(c))

int istrcmp(s1,s2)
register char *s1,*s2;
{
    while (*s1 && (LOW(*s1) == LOW(*s2)))
	s1++,s2++;
    return(LOW(*s1)-LOW(*s2));
}

/* strvec - make vector of substring pointers */

char **strvec(str,sep)
char *str;
char *sep;
{
    char *tmp[BUFSIZ];			/* scratch substring pointer storage */
    register char **cpp = tmp;
    char *sp;				/* ptr to private copy of original */
    register int bytec;

    /* make a copy of the original string */

    if ((sp = malloc(strlen(str)+1)) == 0)
	return(0);
    strcpy(sp,str);

    /* chop our copy at sequences of one or more separators */

    for (*cpp = strtok(sp,sep); *cpp; *++cpp = strtok((char *)0,sep))
	/* void */ ;

    /* now construct the vector of pointers to the substrings */

    if ((cpp = (char **) malloc(bytec = (cpp-tmp+1)*sizeof(*cpp))) == 0)
	return(0);
    return((char **)memcpy((char *)cpp,(char *)tmp,bytec));
}

/* vecstr - null-terminated vector of string pointers to one flat string */

public char *vecstr(vec,sep)
char **vec;
char *sep;
{
    register char **cpp;
    register int len = 0;		/* length of final string */
    register char *cp;
    register int flen = strlen(sep);	/* filler between substrings */

    /* find out how big the resulting string will be */

    for (cpp = vec; *cpp; cpp++)
	len += strlen(*cpp)+flen;

    /* allocate and initialize the result string */

    if ((cp = malloc(len+1)) == 0)
	return(0);
    *cp = '\0';

    /* fill the resulting string */

    for (cpp = vec; *cpp; cpp++) {
	strcat(cp,*cpp);
	strcat(cp,sep);
    }
    return(cp);
}
