/*++
/* NAME
/*	screen
/* SUMMARY
/*	structure of mail shell command windows
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	mailsh
/* SYNOPSIS
/*	#include "screen.h"
/* DESCRIPTION
/*	The data structures in this file are used by the interactive shell to 
/*	define what a screen looks like, and what commands the user can give.
/*
/*	For each screen, one has to define a table of (selector, command name, 
/*	help string, and function pointer) tuples.
/*	The command names are listed in the top window. If the user enters
/*	the type of input specified by the selector, the associated function
/*	is called. A null function pointer means go back to the calling screen. 
/*
/*	The table is terminated with a null selector entry; its help 
/*	string is displayed in the bottom window (as a prompt), and the 
/*	associated function is invoked upon entry of the screen.
/*
/*	The return value of an action function determines what happens next.
/*	An action function can signal an error condition, that the screen
/*	needs to be redrawn, and whether the calling screen should terminate.
/*
/*	User input can be of various forms: single-character, string or
/*	escape/enter.
/*
/*	In case of single-character input the selector fields should contain 
/*	for each key the (upper case) key code,
/*	a label that is displayed at the top of the screen, and a help
/*	text that explains the key's function. 
/*
/*	In case of string input the associated function is called with the 
/*	string input as argument. If that function returns an error status the
/*	text in the help field is printed in the error window and the 
/*	user is given another chance.
/*
/*	In case of escape/enter the interpreter invokes the action function when
/*	the user presses enter, and does nothing when escape is pressed. 
/* .nf

/* /* there is a Screen structure for each command for each screen */

typedef struct {
    short key;			/* type of input */
    char *name;			/* key label (for top window) */
    int  (*action)();		/* action when command is selected */
    char *help;			/* explanation (for H command) */
} Screen;

/* /* action function return masks */

#define	S_BREAK		1	/* return immediately after action */
#define	S_REDRAW	2	/* redraw screen */
#define	S_ERROR		4	/* action failed */

/* /* input types: ordinary character keys are encoded as themselves */

#define	BS	'\010'
#define	ENTER	'\015'
#define	ESC	'\033'
#define	DEL	'\177'

#define	ANY	256		/* press any key */
#define	UP	257		/* up-arrow key */
#define	DOWN	258		/* down-arrow key */
#define	LEFT	259		/* left-arrow key */
#define	RIGHT	260		/* right-arrow key */
#define	PGUP	261		/* page-up */
#define	PGDN	262		/* page-down */
#define	STRING	263		/* string input, ESC to quit */
#define	ESCCR	264		/* CR to confirm, ESC to quit */

#define	iskey(key)	(key > 0 && key < STRING)

/* /* system-dependent function-key labels */

#ifdef unix
#   define PgUp	"F1"
#   define PgDn "F2"
#endif

#ifdef MSDOS
#   define PgUp "PgUp"
#   define PgDn "PgDn"
#endif

/* /* often-used strings and messages */

extern char initscreen[];	/* Return to initial screen */
extern char int_error[];	/* The program is confused */
extern char pageup[];		/* Move screen one page upwards */
extern char pagedn[];		/* Move screen one page downwards */
extern char csrup[];		/* Move cursor upwards */
extern char csrdn[];		/* Move cursor downwards */
extern char getsummary[];	/* Press ESC to cancel.. */
extern char printcurr[];	/* Print current message */
extern char delcurr[];		/* Delete current message */
extern char *m_msgread[];	/* Cannot read that message */
/* SEE ALSO
/*	screen(3)	screen table implementation
/*	kbdinp(3)	screen table interpreter
/* AUTHOR(S)
/*	W.Z. Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Wed Apr  1 21:14:53 GMT+1:00 1987
/* LAST MODIFICATION
/*	Mon Apr  4 23:49:07 MET 1988
/* VERSION/RELEASE
/*	1.3
/*--*/
