/*++
/* NAME
/*	kio 3
/* SUMMARY
/*	interface between uucico and k-protocol driver
/* PACKAGE
/*	uucp on the TUEnet
/* SYNOPSIS
/*	kturnon()
/*
/*	kwrmsg(type,str,fn)
/*	char type, *str;
/*	int fn;
/*
/*	krdmsg(str,fn)
/*	char *str;
/*	int fn;
/*
/*	krddata(fn,fp)
/*	int fn;
/*	FILE *fp;
/*
/*	kwrdata(fp,fn)
/*	FILE *fp;
/*	int fn;
/*
/*	kturnoff()
/* IMPLICIT INPUTS
/*	Ifn, Ofn, file descriptors
/*	Debug, debugging level
/* DESCRIPTION
/*	The k protocol has been developed for the Sytek Localnet local area 
/*	network at the Eindhoven University of Technology (THE).
/*	Main features of this network are:
/*
/* .IP	o 
/*	Network partners may talk at different baudrates. This implies
/*	that the network does some buffering and that it needs flow control.
/* .IP	o 
/*	The network is normally not transparent for some control 
/*	characters (XON,XOFF and locally-defined others), independent 
/*	of the value of the eigth bit.
/* .IP	o 
/*	Some network stations are connected to telephone modems.
/*
/*	For these reasons, the k protocol must (i) rely on XON/XOFF flow 
/*	control, (ii) be suitable for 7-bit data paths, (iii) avoid
/*	sending of control characters and (iv) provide reliable operation
/*	over dial-in and dial-out telephone lines.
/*
/*	Data are sent as checksummed 512-byte packets, terminated by an 
/*	ASCII CR. Except for packet headers (^P), the k protocol only uses 
/*	ASCII codes 040 through 0137. Three data bytes are expanded to four 
/*	bytes upon transmission.
/*	
/*	The functions in kio.c form the interface between the uucico
/*	controlling functions, and the k-protocol packet driver.
/*
/*	kturnon() sets the terminal line characteristics (XON/XOFF). Always
/*	returns zero status.
/*
/*	krdmsg(), kwrmsg() exchange null-terminated strings.
/*	Exit status zero, or FAIL.
/*
/*	krddata(), kwrdata() perform file i/o, and accounting. Exit status 
/*	zero or FAIL.
/*
/*	kturnoff() sends a protocol abort sequence. Always returns zero 
/*	status.
/* FUNCTIONS AND MACROS
/*	kread, kread, kwrite, kclose, k-protocol presentation layer 
/* AUTHOR(S)
/*	Wietse Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Mon Feb  3 10:13:34 MET 1986
/* LAST MODIFICATION
/*	Mon Apr  4 23:42:46 MET 1988
/* VERSION/RELEASE
/*	2.4
/*--*/

#include "uucp.h"

#define	BUFLEN	BUFSIZ

kturnon()
{
    kopen(Ifn);
    return 0;
}


kturnoff()
{
    kclose(Ofn);
    return 0;
}


kwrmsg(type,str,fn)
char type,*str;
int fn;
{
    char bufr[BUFSIZ],*s;

    bufr[0] = type;
    s = &bufr[1];
    while (*str)
	*s++ = *str++;
    *s = '\0';
    if (*(--s) == '\n')
	*s = '\0';
    DEBUG(6," kwrmsg: \"%s\"\n",bufr);

    return (kwrite(fn,bufr,strlen(bufr)+1) > 0 ? 0 : FAIL);
}


krdmsg(str,fn)
char *str;
int fn;
{
    int len;

    for (;;) {
	if ((len = kread(fn,str,BUFSIZ)) == 0) {
	    continue;
	} else if (len > 0) {
	    str[len] = 0;
	    DEBUG(6," krdmsg: \"%s\"\n",str);
	    str += len;
	    if (str[-1] == '\0')
		return 0;
	} else {
	    return FAIL;
	}
    }
}


kwrdata(fp1,fn)
FILE *fp1;
{
    char bufr[BUFLEN];
    int len;
    int ret;
    time_t t1,t2;
    long bytes;
    char text[BUFSIZ];

    ret = FAIL;
    bytes = 0L;
    time(&t1);
    while ((len = fread(bufr,sizeof (char),BUFLEN,fp1)) > 0) {
	bytes += len;
	if (kwrblk(bufr,len,fn) != len)
	    goto acct;
	if (len != BUFLEN)
	    break;
    }
    ret = kwrblk(bufr,0,fn);
acct:
    time(&t2);
    sprintf(text,ret == 0 ?
    "sent data %ld bytes %ld secs" :
    "send failed after %ld bytes",
    bytes,t2 - t1);
    DEBUG(1,"%s\n",text);
    syslog(text);
    sysacct(bytes,t2 - t1);
    if (ret)
	sysaccf(NULL);		/* force accounting */
    return ret;
}


krddata(fn,fp2)
FILE *fp2;
{
    int len,ret;
    char bufr[BUFLEN];
    time_t t1,t2;
    long bytes;
    char text[BUFSIZ];

    ret = FAIL;
    bytes = 0L;
    time(&t1);
    for (;;) {
	len = krdblk(bufr,BUFLEN,fn);
	if (len < 0)
	    goto acct;
	bytes += len;
	if (fwrite(bufr,sizeof (char),len,fp2) != len)
	    goto acct;
	if (len < BUFLEN)
	    break;
    }
    ret = 0;
acct:
    time(&t2);
    sprintf(text,ret == 0 ? 
    "received data %ld bytes %ld secs" :
    "receive failed after %ld bytes",
    bytes,t2 - t1);
    DEBUG(1,"%s\n",text);
    syslog(text);
    sysacct(bytes,t2 - t1);
    if (ret)
	sysaccf(NULL);		/* force accounting */
    return ret;
}


krdblk(blk,len, fn)
char *blk;
int len,fn;
{
    int i,ret;

    for (i = 0; i < len; i += ret) {
	if ((ret = kread(fn,blk,len-i)) == 0) {
	    break;
	} else if (ret > 0) {
	    blk += ret;
	} else {
	    return FAIL;
	}
    }
    return i;
}


kwrblk(blk,len,fn)
char *blk;
int len,fn;
{
    return kwrite(fn,blk,len);
}
