/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSAnnotA.h
*********************************************************************/

#ifndef VSAnnotA_H
#define VSAnnotA_H
#ifndef __GNUC__ /* suppress gcc warning */
#pragma once
#endif /* __GNUC__ */

#include "VSCosA.h"
#include "VSDocA.h"
#include "VSTypes.h" 
#include "VSExcept.h"
#include "VSError.h"

#ifdef __cplusplus
extern "C"{
#endif

	typedef unsigned int annFlag; /* Annotations flags */

    /* Available types of the PDF sound encoding format */
	typedef enum {
		sefRaw = 0,          /* Unspecified or unsigned values in the range 0 to 2B ? 1 */
		sefSigned,           /* Twos-complement values */
		sefmuLaw,            /* -lawencoded samples */
		sefALaw              /* A-lawencoded samples */ 
	}  TSEFormat;

   /* Available types of the PDF line ending styles */
	typedef enum {
		lesNone = 0,            /* No line ending */ 
		lesSquare = 1,          /* A square filled with the annotations interior color */
		lesCircle = 2,          /* A circle filled with the annotations interior color, if any */
		lesDiamond = 3,         /* A diamond shape filled with the annotations interior color. */
		lesOpenArrow = 4,       /* Two short lines meeting in an acute angle, forming an open arrowhead */
		lesClosedArrow = 5      /* Two short lines meeting in an acute angle as in the OpenArrow
		                           style, connected by a third line to form a triangular closed
		                           arrowhead filled with the annotations interior color, if any */ 
	}  TLineEndingStyle;

    /* Available names of an icon to be used in displaying the file attach annotation. */
	typedef enum {
		atGraph = 1,        /* Graph icon */
		atPushPin = 2,      /* PushPin icon */
		atPaperclip = 3,    /* Paper clip icon */ 
		atTag = 4           /* Tag icon */
	}  TAttachType;

    /* Available names specifying a symbol to be associated with the caret. */
	typedef enum {
		csNone,            /* A new paragraph symbol () should be associated with the caret. */
		csNewPar           /* No symbol should be associated with the caret. */
	} TCaretSymbol;

    /* Available names of an icon to be used in displaying the rubber stamp annotation. */
	typedef enum {
		inApproved,				/* Approved icon type */
		inAsIs,					/* AsIs icon type */
		inConfidential,			/* Confidential icon type */
		inDepartmental,			/* Departmental icon type */
		inDraft,				/* Draft icon type */
		inExperimental,			/* Experimental icon type */
		inExpired,				/* Expired icon type */
		inFinal,				/* Final icon type */
		inForComment,			/* For comment icon type */
		inForPublicRelease,		/* For public release icon type */
		inNotApproved,			/* Not approved icon type */
		inNotForPublicRelease,	/* Not for public release icon type */
		inSold,					/* Sold icon type */
		inTopSecret				/* Top secret icon type */
	}TIconName;

	/* Available types of poly annotation */
	typedef enum {
		ptPolygon,    /* The polygon annotation */
		ptPolyline    /* The polyline annotation */
	} TPolyType;

	/* Available types of SC annotation */
	typedef enum {
		sctSquare,     /* Square annotation */
		sctCircle      /* Circle annotation */
	} TSCType;

    /* Available types of border styles */
	typedef enum {
		bsnSolid,       /* Solid style */ 
		bsnDashed,      /* Dashed style */
		bsnBeveled,     /* Beveled style */
		bsnInset,       /* Inset style */
		bsnUnderline    /* Underline style */
	} TBStyleName;

	/* Available names representing the border effects */
	typedef enum {
		enDefault,      /* Default border effects */
		enCloudy        /* Cloudy border effects */
	} * PEffectName, TEffectName;

    /* Available names representing the annotation flags */
	typedef enum {
		afDefault = 0,     /* Default annotation flag */
		afInvisible = 1,   /* Invisible annotation flag */
		afHidden = 2,      /* Hidden annotation flag */
		afPrint = 4,       /* Print annotation flag */
		afNoZoom = 8,      /* No Zoom annotation flag */
		afNoRotate = 16,   /* No Rotate annotation flag */
		afNoView = 32,     /* No View annotation flag */
		afReadOnly = 64    /* Read Only annotation flag */
	} TAnotFlags;

    /* Available annotations names */
	typedef enum {
		anDefault = 0,      /* Note annotation */
		anComment = 1,      /* Comment annotation */
		anHelp = 2,         /* Help annotation */
		anInsert = 3,       /* Insert annotation */
		anKey = 4,          /* Key annotation */
		anNewParagraph = 5, /* New paragraph annotation */
		anParagraph = 6     /* Paragraph annotation */
	} TAnotName;

	/* Available annotations text justify mode */
	typedef enum {
		jmLeft = 0,      /* Left text justify mode */
		jmCenter = 1,    /* Center text justify mode */
		jmRight = 2      /* Right text justify mode */
	} TJustifyMode;

	/* Point structure */
	typedef struct {
		int X;          	/* Point abscissa */
		int Y;           	/* Point ordinate */
	} * PPagePoint, TPagePoint;

	/* Boolean stream structure */
	typedef struct {
		ppBool isBool;           /* Stream is boolean */
		ppBool BoolVal;          /* Stream value */
		PDFCosHandle DictBool;   /* Stream body */
	} * PBoolStream, TBoolStream;

	/* Movie dictionary structure */
	typedef struct {
		char * fileSpecific;        /* Movie specification */
		int fsLength;               /* Movie specification length */
		TPagePoint Aspect;          /* The width and height of the movies bounding box, in pixels. */
		int Rotate;                 /* Rotation angle */ 
		TBoolStream FlagStream;     /* Boolean stream */
	} * PMovieDict, TMovieDict;

	/* Rectangle structure */
	typedef struct {
		 TPagePoint pt1; /* Top - left point */
		 TPagePoint pt2; /* Bottom - right point */
	} * PPageRect, TPageRect; 

    /* Border style dictionary */
	typedef struct {
		int width;                     /* (Optional) The border width in points. If this value is 0, no border is drawn. */
		TBStyleName name;              /*(Optional) The border style:
		                                  bsnSolid (Solid) A solid rectangle surrounding the annotation.
			                              bsnDashed (Dashed) A dashed rectangle surrounding the annotation. The dash pattern
			                                                 is specified by the D entry (see below).
			                              bsnBeveled (Beveled) A simulated embossed rectangle that appears to be 
										                       raised above the surface of the page.
			                              bsnInset (Inset) A simulated engraved rectangle that appears to be recessed below the
			                                               surface of the page.
			                              bsnUnderline (Underline) A single line along the bottom of the annotation rectangle. */
		int * array;                   /* A dash array defining a pattern of dashes and gaps to be used
		                                  in drawing dashed border                                      */
		int arrLength;                 /* Dash array length */
		TEffectName BordEffect;        /* (Optional) A name representing the border effect to apply. Possible values are:
									   enDefault: No effect: the border is as described by the annotation dictionarys BS entry.
									   enCloudy:  The border should appear cloudy. The width and dash array specified by BS
									                are honored. */ 
		int intensity;                 /*A number describing the intensity of the effect.
					                     Suggested values range from 0 to 2. */    
	} * PBSDict, TBSDict;

	/* Color structure */
	typedef struct {
		float red;/*(0.0 - 1.0)*/
		float green;/*(0.0 - 1.0)*/
		float blue;/*(0.0 - 1.0)*/
	} * PDeviceRGB, TDeviceRGB;
    
	/* Available highlight modes */
	typedef enum {
		hmInvert = 0,  /* Invert mode */
		hmNone,        /* None highlight */
		hmOutline,     /* Outline highlight */
		hmPush         /* Push highlight */
	} THighlighMode;

	/* Sound dictionary structure */
	typedef struct {
		int SamplingRate;     /* Sound dictionary sampling rate */
		int Channels;         /* Sound dictionary channels */
		int BitsPerSample;    /* Sound dictionary bits per sample */
		TSEFormat EncFormat;  /* Sound dictionary encoding format structure */
	} * PSoundDict, TSoundDict;

	/* Text annotation structure */
	typedef struct {
		PDFCosHandle    Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
												editing the text associated with this annotation. */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		PBSDict         BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												       the components of a color in the DeviceRGB color space. This color
												       will be used for the following purposes:
												          The background of the annotations icon when closed
												          The title bar of the annotations pop-up window
												          The border of a link annotation */
		char *          Contents;               /* (Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TAnotName       IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
								                    open. */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												              in default user space units */ 
		char *          TitleText;              /* (Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32         TTLength;               /* Length of title */
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */	
	} * PTextAnnotDict, TTextAnnotDict; 

	/* Link annotation structure */
	typedef struct {
		ppBool			Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
												   open. */
		TPageRect		Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */ 
		THighlighMode	AnnotHighLight;         /* (Optional) The annotations highlighting mode, the visual effect to be
												used when the mouse button is pressed or held down inside its active area: 
												                hmInvert   - (Invert) Invert the contents of the annotation rectangle. 
												                hmNone     - (None) No highlighting.  
												                hmOutline  - (Outline) Invert the annotations border.
																hmPush     - (Push) Display the annotations down appearance, if any. 
																             If no down appearance is defined, offset the contents of 
																			 the annotation rectangle to appear as if it were being
																             pushed below the surface of the page.			*/
		PDFActionHandle PageAction;             /* (Optional) A URI action formerly associated with this annotation. 
												When Web Capture changes an annotation from a URI to a go-to action, it uses this 
												entry to save the data from the original URI action so that it can be changed back 
												in case the target page for the go-to action is subsequently deleted. */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		char *			Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32			ContLength;             /* Count of characters in contents */
		PBSDict			BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TAnotName		IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		TDeviceRGB		Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												        the components of a color in the DeviceRGB color space. This color
												        will be used for the following purposes:
												            The background of the annotations icon when closed
												            The title bar of the annotations pop-up window
												            The border of a link annotation */
		char *			DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		annFlag			AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		char *			TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32			TTLength;               /* Length of title */
		float			Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */
		PDFCosHandle	Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
												 editing the text associated with this annotation. */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
	} * PLinkAnnotDict, TLinkAnnotDict; 

	/* Line annotation structure */
	typedef struct {
		ppBool			Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
									                 open. */
		TPageRect		Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */ 
		THighlighMode	AnnotHighLight;         /* (Optional) The annotations highlighting mode, the visual effect to be
												used when the mouse button is pressed or held down inside its active area: 
												                hmInvert   - (Invert) Invert the contents of the annotation rectangle. 
												                hmNone     - (None) No highlighting.  
												                hmOutline  - (Outline) Invert the annotations border.
																hmPush     - (Push) Display the annotations down appearance, if any. 
																             If no down appearance is defined, offset the contents of 
																			 the annotation rectangle to appear as if it were being
																             pushed below the surface of the page.			*/
		PDFActionHandle PageAction;             /* (Optional) A URI action formerly associated with this annotation. 
												When Web Capture changes an annotation from a URI to a go-to action, it uses this 
												entry to save the data from the original URI action so that it can be changed back 
												in case the target page for the go-to action is subsequently deleted. */
		ppInt32			LineCoordinates[4];     /* (Required) An array of four numbers, [x1 y1 x2 y2], specifying the starting and
												ending coordinates of the line in default user space. */
		float 			InteriorColor[3];          /* (Optional) An array of three numbers in the range 0.0 to 1.0 specifying
												the components, in the DeviceRGB color space, of the interior color with which to
												fill the annotations line endings (see Table 8.19). If this entry is absent, the interiors
												of the line endings are left transparent. */
		int 			LineEnding[2];          /* (Optional; PDF 1.4) An array of two names specifying the line ending styles to be
												used in drawing the line. The first and second elements of the array specify the
												line ending styles for the endpoints defined, respectively, by the first and second
												pairs of coordinates, (x1, y1) and (x2, y2), in the L array. */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		char *			Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32			ContLength;             /* Count of characters in contents */
		PBSDict			BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TAnotName		IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		TDeviceRGB		Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												      the components of a color in the DeviceRGB color space. This color
												      will be used for the following purposes:
												            The background of the annotations icon when closed
												            The title bar of the annotations pop-up window
												            The border of a link annotation */
		char *			DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		annFlag			AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		char *			TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32			TTLength;               /* Length of title */
		float			Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */
		PDFCosHandle	Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
												 editing the text associated with this annotation. */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
	} * PLineAnnotDict, TLineAnnotDict; 

    /* Square and circle annotation structure */
	typedef struct {
		ppBool			Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
									                open. */
		TPageRect		Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */ 
		THighlighMode	AnnotHighLight;         /* (Optional) The annotations highlighting mode, the visual effect to be
												used when the mouse button is pressed or held down inside its active area: 
												                hmInvert   - (Invert) Invert the contents of the annotation rectangle. 
												                hmNone     - (None) No highlighting.  
												                hmOutline  - (Outline) Invert the annotations border.
																hmPush     - (Push) Display the annotations down appearance, if any. 
																             If no down appearance is defined, offset the contents of 
																			 the annotation rectangle to appear as if it were being
																             pushed below the surface of the page.			*/
		PDFActionHandle PageAction;            /* (Optional) A URI action formerly associated with this annotation. 
												When Web Capture changes an annotation from a URI to a go-to action, it uses this 
												entry to save the data from the original URI action so that it can be changed back 
												in case the target page for the go-to action is subsequently deleted. */
		float *			InteriorColor;          /* (Optional) An array of three numbers in the range 0.0 to 1.0 specifying
												the components, in the DeviceRGB color space, of the interior color with which to
												fill the annotations line endings (see Table 8.19). If this entry is absent, the interiors
												of the line endings are left transparent. */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		char *			Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32			ContLength;             /* Count of characters in contents */
		PBSDict			BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TAnotName		IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		TDeviceRGB		Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												        the components of a color in the DeviceRGB color space. This color
												        will be used for the following purposes:
												              The background of the annotations icon when closed
												              The title bar of the annotations pop-up window
												              The border of a link annotation */
		char *			DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		annFlag			AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		char *			TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32			TTLength;               /* Length of title */
		float			Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */
		PDFCosHandle	Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
												 editing the text associated with this annotation. */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
	} * PSCAnnotDict, TSCAnnotDict; 

    /* Free annotation structure */
	typedef struct {
		ppBool			Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
									                 open. */
		TPageRect		Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */ 
		THighlighMode	AnnotHighLight;         /* (Optional) The annotations highlighting mode, the visual effect to be
												used when the mouse button is pressed or held down inside its active area: 
												                hmInvert   - (Invert) Invert the contents of the annotation rectangle. 
												                hmNone     - (None) No highlighting.  
												                hmOutline  - (Outline) Invert the annotations border.
																hmPush     - (Push) Display the annotations down appearance, if any. 
																             If no down appearance is defined, offset the contents of 
																			 the annotation rectangle to appear as if it were being
																             pushed below the surface of the page.			*/
		PDFActionHandle PageAction;             /* (Optional) A URI action formerly associated with this annotation. 
												When Web Capture changes an annotation from a URI to a go-to action, it uses this 
												entry to save the data from the original URI action so that it can be changed back 
												in case the target page for the go-to action is subsequently deleted. */
		char *			DA;                     /* (Required) The default appearance string to be used in formatting the text */
		ppInt32			DALength;               /* The default appearance string length */
		TJustifyMode	Quadding;               /* (Optional) A code specifying the form of quadding (justification) to be
												used in displaying the annotations text:
												          0 Left-justified
												          1 Centered
												          2 Right-justified */
		PBSDict         BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		char *			Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32			ContLength;             /* Count of characters in contents */
		TAnotName		IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		TDeviceRGB		Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		char *			DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		annFlag			AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		char *			TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32			TTLength;               /* Length of title */
		float			Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */
		PDFCosHandle	Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
												 editing the text associated with this annotation. */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
	} * PFreeAnnotDict, TFreeAnnotDict; 

    /* Poly annotation structure */
	typedef struct {
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		PBSDict         BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		PDFCosHandle    BorderEffect;           /* (Optional) A border effect dictionary describing an effect applied to the border 
												              described by the BS entry */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		int *           Vertices;               /* (Required) An array of numbers representing the alternating horizontal and vertical
												coordinates, respectively, of each vertex, in default user space. */
		int             VertLength;             /* Length of the vertices array */
		float *         InteriorColor;          /* (Optional) An array of three numbers in the range 0.0 to 1.0 specifying
												the components, in the DeviceRGB color space, of the interior color with which to
												fill the annotations line endings (see Table 8.19). If this entry is absent, the interiors
												of the line endings are left transparent. */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		int             LineEnding[2];             /* (Optional; PDF 1.4) An array of two names specifying the line ending styles to be
												used in drawing the line. The first and second elements of the array specify the
												line ending styles for the endpoints defined, respectively, by the first and second
												pairs of coordinates, (x1, y1) and (x2, y2), in the L array. */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TAnotName       IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		TPolyType       Type;                   /* (Required) The type of annotation that this dictionary describes; must be
												ptPolygon or ptPolyline for a polygon or polyline annotation, respectively. */
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
									               open. */
		PDFCosHandle    Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
												 editing the text associated with this annotation. */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		char *          TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32         TTLength;               /* Length of title */
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */	
	} * PPolyAnnotDict, TPolyAnnotDict; 

    /* Caret annotation structure */
	typedef struct {
		PDFCosHandle    Popup;                  /* (Optional) An indirect reference to a pop-up annotation for
		                                           entering or editing the text is associated with this
		                                           annotation.                                                 */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		PBSDict         BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to
		                                           1.0, representing the components of a color in the DeviceRGB
		                                           color space. This color will be used for the following
		                                           purposes:  The background of the annotations icon when its
		                                           closed  The title bar of the annotations pop-up window 
		                                           The border of a link annotation                              */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TAnotName       IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should
		                                           initially be opened.  */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		char *          TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32         TTLength;               /* Length of title */
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */	
		int *           RectBound;              /* (Optional) A set of 4 numbers describing the numerical differences between
												two rectangles: the Rect entry of the annotation and the actual boundaries
												of the underlying caret. Such a difference can occur, for example, when a paragraph
												symbol specified by Sy is displayed along with the caret.
												The 4 numbers correspond to the differences in default user space between the
												left, top, right and bottom coordinates of Rect and those of the caret, respectively.
												Each value must be greater than or equal to 0. The sum of the top and bottom
												differences must be less than the height of Rect, and the sum of the left and right
												differences must be less than the width of Rect. */ 
		TCaretSymbol    CaretSymbol;            /* (Optional) A name specifying a symbol to be associated with the caret:
												       csNewPar  - A new paragraph symbol () should be associated with the caret.
												       csNone    - No symbol should be associated with the caret. */
	} * PCaretAnnotDict, TCaretAnnotDict; 

	/* Rubber stamp annotation structure */
	typedef struct {
        PDFCosHandle    AppearanceStream;
		PDFCosHandle    Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
                                                editing the text associated with this annotation. */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		PBSDict         BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TAnotName       IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
									                 open. */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		char *          TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32         TTLength;               /* Length of title */
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */
		TIconName       IconStyleName;          /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												    applications should provide predefined icon appearances for at least the following
												    standard names:

													inApproved,          inAsIs,        inConfidential,
													inDepartmental,      inDraft,       inExperimental,
													inExpired,           inFinal,       inForComment,
													inForPublicRelease,  inNotApproved, inNotForPublicRelease,
													inSold, 			 inTopSecret

													Additional names may be supported as well. */
	} * PRubberStampAnnotDict, TRubberStampAnnotDict; 

	/* Popup annotation structure */
	typedef struct {
		PDFCosHandle    Popup;                  /* (Optional) An indirect reference to a pop-up annotation for entering or
											 	    editing the text associated with this annotation. */
		PDFActionHandle Action;                 /* (Optional) An action to be performed when the annotation is activated */
		PDFActionHandle AdditAction;            /* (Optional) An additional-actions dictionary defining the annotations
												behavior in response to various trigger events */
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		PBSDict         BSDict;                 /* (Optional) A border style dictionary specifying the 
											        characteristics of the annotations border */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TAnotName       IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
									              open. */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		char *          TitleText;              /*(Optional ) The text label to be displayed in the title bar of the annotations
												pop-up window when open and active. By convention, this entry identifies
												the user who added the annotation. */
		ppInt32         TTLength;               /* Length of title */
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */
		PDFCosHandle    Parent;                 /* (Optional) The parent annotation with which
												this pop-up annotation is associated. */
	} * PPopupAnnotDict, TPopupAnnotDict; 

	/* \File attached annotation structure */
	typedef struct {
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
										           open.  */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */	
		char *          FileName;               /* (Required) The filename associated with this annotation. */
		int             FNLength;               /* Filename Length  */
		TAttachType     FSIcon;                 /* (Optional) The name of an icon to be used in displaying the annotation.
												Viewer applications should provide predefined icon appearances for at least
												the following standard names:
												       atGraph,      atPushPin
												       atPaperclip,  atTag
												Additional names may be supported as well. */
	} * PFileAttachAnnotDict, TFileAttachAnnotDict; 
	
	/* Sound annotation structure */
	typedef struct {
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TAnotName       IconName;               /* (Optional) The name of an icon to be used in displaying the annotation. Viewer
												applications should provide predefined icon appearances for at least the following
												standard names: 
												            anDefault 	anComment   anHelp 
															anInsert 	anKey       anNewParagraph 
															anParagraph                                    */
		ppBool          Open;                   /* (Optional) A flag specifying whether the annotation should initially be displayed
										             open. */
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		char *          Filename;               /*(Required ) Sound filename. */
		ppInt32         FNLength;               /* Filename length */
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */	
	} * PSoundAnnotDict, TSoundAnnotDict; 

	/* Movie annotation structure */
	typedef struct {
		annFlag         AnFlags;                /*  (Optional ) A set of flags specifying various characteristics of the annotation. */
		TDeviceRGB      Color;                  /* (Optional ) An array of three numbers in the range 0.0 to 1.0, representing
												the components of a color in the DeviceRGB color space. This color
												will be used for the following purposes:
												 The background of the annotations icon when closed
												 The title bar of the annotations pop-up window
												 The border of a link annotation */
		char *          Contents;               /*(Required) Text to be displayed for the annotation or, if this type 
												of annotation does not display text, an alternate description of the 
												annotations contents in human-readable form. */
		ppInt32         ContLength;             /* Count of characters in contents */
		char *          DateTime;               /* (Optional ) The date and time when the annotation was created.*/
		TPageRect       Rectangle;              /* (Required) The annotation rectangle, defining the location of the annotation
												in default user space units */    
		char *          FileName;               /*(Optional ) Movie filename. */
		ppInt32         FNLength;               /* Length of filename */
		ppBool          Activation;             /* if it boolean value true, the movie should be played using default activation parameters;
												if it is false, the movie should not be played at all. */    
		float           Transparency;           /* (Optional) The constant opacity value to be used in painting the annotation 
												This value applies to all visible elements of the annotation
												in its closed state (including its background and border), but not to the popup
												window that appears when the annotation is opened. */	
	} * PMovieAnnotDict, TMovieAnnotDict; 

	/* Description
	   Adds text annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to text annotation.
	   Returns
	   Annotation handle.                                     */
    PDFAnnotationHandle PDFPageAddTextAnnotation( PDFDocHandle Doc, ppInt32 Page, TTextAnnotDict Annot );

	/* Description
	   Adds Link annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to text annotation.
	   Returns
	   Annotation handle.                                     */
    PDFAnnotationHandle PDFPageAddLinkAnnotation( PDFDocHandle Doc, ppInt32 Page, TLinkAnnotDict Annot );

	/* Description
	   Adds free annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to free annotation.
	   Returns
	   Annotation handle.                                     */
	PDFAnnotationHandle PDFPageAddFreeAnnotation( PDFDocHandle Doc, ppInt32 Page, TFreeAnnotDict Annot );

	/* Description
	   Adds line annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to line annotation.
	   Returns
	   Annotation handle.                                     */
	PDFAnnotationHandle PDFPageAddLineAnnotation( PDFDocHandle Doc, ppInt32 Page, TLineAnnotDict Annot );

	/* Description
	   Adds square and circle annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to square and circle
	                       annotation.
	   Returns
	   Annotation handle.                                      */
	PDFAnnotationHandle PDFPageAddSCAnnotation( PDFDocHandle Doc, ppInt32 Page, TSCAnnotDict Annot, TSCType Type );

	/* Description
	   Adds polygon annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to polygon annotation.
	   Returns
	   Annotation handle.                                        */
	PDFAnnotationHandle PDFPageAddPolyAnnotation( PDFDocHandle Doc, ppInt32 Page, TPolyAnnotDict Annot );

	/* Description
	   Adds caret annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to caret annotation.
	   Returns
	   Annotation handle.                                      */
	PDFAnnotationHandle PDFPageAddCaretAnnotation(PDFDocHandle Doc, ppInt32 Page, TCaretAnnotDict Annot );

	/* Description
	   Adds rubber stamp annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to rubber stamp annotation.
	   Returns
	   Annotation handle.                                             */
	PDFAnnotationHandle PDFPageAddRubberStampAnnotation(PDFDocHandle Doc, ppInt32 Page, TRubberStampAnnotDict Annot );

	/* Description
	   Adds popup annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to text annotation.
	   Returns
	   Annotation handle.                                     */
	PDFAnnotationHandle PDFPageAddPopupAnnotation(PDFDocHandle Doc, ppInt32 Page, TPopupAnnotDict Annot );

	/* Description
	   Adds attached file annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to attach file annotation.
	   Returns
	   Annotation handle.                                            */
	PDFAnnotationHandle PDFPageAddFileAttachAnnotation(PDFDocHandle Doc, ppInt32 Page, TFileAttachAnnotDict Annot );

	/* Description
	   Adds sound annotation to document from file.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to text annotation.
	   FileName :          [ in ] Sound filename.
	   SamplingRate :      [ in ] The sampling rate, in samples per
	                       second.
	   Channels :          [ in ] The number of sound channels.
	   BitsPerSample :     [ in ] The number of bits per sample value per
	                       channel. 
	   EncFormat :         [ in ] Sound encoding format.
	   Returns
	   Annotation handle.                                                 */
	PDFAnnotationHandle PDFPageAddSoundAnnotationFromFile(PDFDocHandle Doc, ppInt32 Page, TSoundAnnotDict Annot, 
														   int SamplingRate, int Channels, int BitsPerSample, TSEFormat EncFormat);

	/* Description
	   Adds movie annotation to document.
	   Parameters
	   Doc :               [ in ] Current document.
	   Page :              [ in ] Number of page.
	   AnnotDescription :  [ in ] Pointer to movie annotation.
	   Returns
	   Annotation handle.                                      */
	PDFAnnotationHandle PDFPageAddMovieAnnotation( PDFDocHandle Doc, ppInt32 Page, TMovieAnnotDict Annot );

#ifdef __cplusplus
}
#endif


#endif /* #ifndef VSAnnotA */

